# This is an example script for use with CMake projects for locating and configuring
# the nanopb library.
#
# The following variables can be set and are optional:
#
#
#   PROTOBUF_SRC_ROOT_FOLDER - When compiling with MSVC, if this cache variable is set
#                              the protobuf-default VS project build locations
#                              (vsprojects/Debug & vsprojects/Release) will be searched
#                              for libraries and binaries.
#
#   NANOPB_IMPORT_DIRS       - List of additional directories to be searched for
#                              imported .proto files.
#
#   NANOPB_OPTIONS           - List of options passed to nanopb.
#
#   Nanopb_FIND_COMPONENTS   - List of options to append to NANOPB_OPTIONS without the
#                              leading '--'.  This should not manually be set, but allows
#                              passing options to nanopb via find_package.  For example,
#                              'find_package(Nanopb REQUIRED COMPONENTS cpp-descriptors)'
#                              is equivalent to setting NANOPB_OPTIONS to --cpp-descriptors.
#
#   NANOPB_DEPENDS           - List of files to be used as dependencies
#                              for the generated source and header files. These
#                              files are not directly passed as options to
#                              nanopb but rather their directories.
#
#   NANOPB_GENERATE_CPP_APPEND_PATH - By default -I will be passed to protoc
#                                     for each directory where a proto file is referenced.
#                                     This causes all output files to go directly
#                                     under build directory, instead of mirroring
#                                     relative paths of source directories.
#                                     Set to FALSE if you want to disable this behaviour.
#   PROTOC_OPTIONS           - Pass options to protoc executable
#
# Defines the following variables:
#
#   NANOPB_FOUND - Found the nanopb library (source&header files, generator tool, protoc compiler tool)
#   NANOPB_INCLUDE_DIRS - Include directories for Google Protocol Buffers
#
# The following cache variables are also available to set or use:
#   PROTOBUF_PROTOC_EXECUTABLE - The protoc compiler
#   NANOPB_GENERATOR_SOURCE_DIR - The nanopb generator source
#
#  ====================================================================
#
# NANOPB_GENERATE_CPP (public function)
# NANOPB_GENERATE_CPP(SRCS HDRS [RELPATH <root-path-of-proto-files>]
#                     <proto-files>...)
#   SRCS = Variable to define with autogenerated source files
#   HDRS = Variable to define with autogenerated header files
# NANOPB_GENERATE_CPP(TARGET TGT [RELPATH <root-path-of-proto-files>]
#                     <proto-files>...)
#   TGT = Name of the static library to create with the autogenerated files
#
#   If you want to use relative paths in your import statements use the RELPATH
#   option. The argument to RELPATH should be the directory that all the
#   imports will be relative to.
#   When RELPATH is not specified then all proto files can be imported without
#   a path.
#
#
#  ====================================================================
#  Example using modern targets:
#
#   set(NANOPB_SRC_ROOT_FOLDER "/path/to/nanopb")
#   set(CMAKE_MODULE_PATH ${CMAKE_MODULE_PATH} ${NANOPB_SRC_ROOT_FOLDER}/extra)
#   find_package( Nanopb REQUIRED )
#
#   NANOPB_GENERATE_CPP(TARGET proto foo.proto)
#
#   add_executable(bar bar.cc)
#   target_link_libraries(bar proto)
#
#  Example with RELPATH:
#   Assume we have a layout like:
#    .../CMakeLists.txt
#    .../bar.cc
#    .../proto/
#    .../proto/foo.proto  (Which contains: import "sub/bar.proto"; )
#    .../proto/sub/bar.proto
#   Everything would be the same as the previous example, but the call to
#   NANOPB_GENERATE_CPP would change to:
#
#   NANOPB_GENERATE_CPP(TARGET proto RELPATH proto
#                       proto/foo.proto proto/sub/bar.proto)
#
#  Example using traditional variables:
#
#   set(NANOPB_SRC_ROOT_FOLDER "/path/to/nanopb")
#   set(CMAKE_MODULE_PATH ${CMAKE_MODULE_PATH} ${NANOPB_SRC_ROOT_FOLDER}/extra)
#   find_package( Nanopb REQUIRED )
#   include_directories(${NANOPB_INCLUDE_DIRS})
#
#   NANOPB_GENERATE_CPP(PROTO_SRCS PROTO_HDRS foo.proto)
#
#   include_directories(${CMAKE_CURRENT_BINARY_DIR})
#   add_executable(bar bar.cc ${PROTO_SRCS} ${PROTO_HDRS})
#
#  ====================================================================

#=============================================================================
# Copyright 2009 Kitware, Inc.
# Copyright 2009-2011 Philip Lowman <philip@yhbt.com>
# Copyright 2008 Esben Mose Hansen, Ange Optimization ApS
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
#
# * Redistributions of source code must retain the above copyright
#   notice, this list of conditions and the following disclaimer.
#
# * Redistributions in binary form must reproduce the above copyright
#   notice, this list of conditions and the following disclaimer in the
#   documentation and/or other materials provided with the distribution.
#
# * Neither the names of Kitware, Inc., the Insight Software Consortium,
#   nor the names of their contributors may be used to endorse or promote
#   products derived from this software without specific prior written
#   permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
# "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
# LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
# A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
# LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
# DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
# THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
# (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
# OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
#
#=============================================================================
#
# Changes
# 2013.01.31 - Pavlo Ilin - used Modules/FindProtobuf.cmake from cmake 2.8.10 to
#                           write FindNanopb.cmake
#
#=============================================================================


function(NANOPB_GENERATE_CPP)
  cmake_parse_arguments(NANOPB_GENERATE_CPP "" "RELPATH;TARGET" "" ${ARGN})
  if(NANOPB_GENERATE_CPP_TARGET)
    set(SRCS NANOPB_TARGET_SRCS)
    set(HDRS NANOPB_TARGET_HDRS)
  else()
    list(GET NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS 0 SRCS)
    list(GET NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS 1 HDRS)
    list(REMOVE_AT NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS 0 1)
  endif()
  if(NOT NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS)
    return()
  endif()
  set(NANOPB_OPTIONS_DIRS)

  if(MSVC)
    set(CUSTOM_COMMAND_PREFIX call)
  endif()

  if(NANOPB_GENERATE_CPP_RELPATH)
    get_filename_component(NANOPB_GENERATE_CPP_RELPATH ${NANOPB_GENERATE_CPP_RELPATH} ABSOLUTE)
    list(APPEND _nanopb_include_path "-I${NANOPB_GENERATE_CPP_RELPATH}")
    list(APPEND NANOPB_OPTIONS_DIRS ${NANOPB_GENERATE_CPP_RELPATH})
  endif()

  if(NANOPB_GENERATE_CPP_APPEND_PATH)
    # Create an include path for each file specified
    foreach(FIL ${NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS})
      get_filename_component(ABS_FIL ${FIL} ABSOLUTE)
      get_filename_component(ABS_PATH ${ABS_FIL} PATH)
      list(APPEND _nanopb_include_path "-I${ABS_PATH}")
    endforeach()
  else()
    list(APPEND _nanopb_include_path "-I${CMAKE_CURRENT_SOURCE_DIR}")
  endif()

  if(DEFINED NANOPB_IMPORT_DIRS)
    foreach(DIR ${NANOPB_IMPORT_DIRS})
      get_filename_component(ABS_PATH ${DIR} ABSOLUTE)
      list(APPEND _nanopb_include_path "-I${ABS_PATH}")
    endforeach()
  endif()

  list(REMOVE_DUPLICATES _nanopb_include_path)

  set(GENERATOR_PATH ${CMAKE_CURRENT_BINARY_DIR}/nanopb/generator)

  set(NANOPB_GENERATOR_EXECUTABLE ${GENERATOR_PATH}/nanopb_generator.py)
  if (CMAKE_HOST_WIN32)
    set(NANOPB_GENERATOR_PLUGIN ${GENERATOR_PATH}/protoc-gen-nanopb.bat)
  else()
    set(NANOPB_GENERATOR_PLUGIN ${GENERATOR_PATH}/protoc-gen-nanopb)
  endif()

  set(GENERATOR_CORE_DIR ${GENERATOR_PATH}/proto)
  set(GENERATOR_CORE_SRC
      ${GENERATOR_CORE_DIR}/nanopb.proto)

  # Set extensions according to NANOPB_OPTIONS
  string(REGEX MATCH "--extension=[^ ]+" _gen_ext "${NANOPB_OPTIONS}")
  string(REGEX MATCH "--header-extension=[^ ]+" _gen_hdr_ext
               "${NANOPB_OPTIONS}")
  string(REGEX MATCH "--source-extension=[^ ]+" _gen_src_ext
               "${NANOPB_OPTIONS}")
  if(_gen_ext)
    string(REPLACE "--extension=" "" GEN_EXTENSION "${_gen_ext}")
  else()
    set(GEN_EXTENSION ".pb")
  endif()
  if(_gen_hdr_ext)
    string(REPLACE "--header-extension=" "" GEN_HDR_EXTENSION "${_gen_hdr_ext}")
  else()
    set(GEN_HDR_EXTENSION ".h")
  endif()
  if(_gen_src_ext)
    string(REPLACE "--source-extension=" "" GEN_SRC_EXTENSION "${_gen_src_ext}")
  else()
    set(GEN_SRC_EXTENSION ".c")
  endif()

  # Treat the source directory as immutable.
  #
  # Copy the generator directory to the build directory before
  # compiling python and proto files.  Fixes issues when using the
  # same build directory with different python/protobuf versions
  # as the binary build directory is discarded across builds.
  #
  # Notice: copy_directory does not copy the content if the directory already exists.
  # We therefore append '/' to specify that we want to copy the content of the folder. See #847
  #
  add_custom_command(
      OUTPUT ${NANOPB_GENERATOR_EXECUTABLE} ${GENERATOR_CORE_SRC}
      COMMAND ${CMAKE_COMMAND} -E copy_directory
      ARGS ${NANOPB_GENERATOR_SOURCE_DIR}/ ${GENERATOR_PATH}
      VERBATIM)

  set(GENERATOR_CORE_PYTHON_SRC)
  foreach(FIL ${GENERATOR_CORE_SRC})
      get_filename_component(ABS_FIL ${FIL} ABSOLUTE)
      get_filename_component(FIL_WE ${FIL} NAME_WE)

      set(output "${GENERATOR_CORE_DIR}/${FIL_WE}_pb2.py")
      set(GENERATOR_CORE_PYTHON_SRC ${GENERATOR_CORE_PYTHON_SRC} ${output})
      add_custom_command(
        OUTPUT ${output}
        COMMAND ${CUSTOM_COMMAND_PREFIX} ${PROTOBUF_PROTOC_EXECUTABLE}
        ARGS -I${GENERATOR_PATH}/proto
          --python_out=${GENERATOR_CORE_DIR} ${ABS_FIL}
        DEPENDS ${ABS_FIL}
        VERBATIM)
  endforeach()

  foreach(FIL ${NANOPB_GENERATE_CPP_UNPARSED_ARGUMENTS})
    get_filename_component(ABS_FIL ${FIL} ABSOLUTE)
    get_filename_component(FIL_WE ${FIL} NAME_WLE)
    get_filename_component(FIL_DIR ${ABS_FIL} PATH)
    set(FIL_PATH_REL)
    if(NANOPB_GENERATE_CPP_RELPATH)
      # Check that the file is under the given "RELPATH"
      string(FIND ${ABS_FIL} ${NANOPB_GENERATE_CPP_RELPATH} LOC)
      if (${LOC} EQUAL 0)
        string(REPLACE "${NANOPB_GENERATE_CPP_RELPATH}/" "" FIL_REL ${ABS_FIL})
        get_filename_component(FIL_PATH_REL ${FIL_REL} PATH)
        file(MAKE_DIRECTORY ${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL})
      endif()
    endif()
    if(NOT FIL_PATH_REL)
      set(FIL_PATH_REL ".")
    endif()

    list(APPEND ${SRCS} "${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL}/${FIL_WE}${GEN_EXTENSION}${GEN_SRC_EXTENSION}")
    list(APPEND ${HDRS} "${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL}/${FIL_WE}${GEN_EXTENSION}${GEN_HDR_EXTENSION}")

    get_filename_component(ABS_OPT_IN_FIL ${FIL_DIR}/${FIL_WE}.options.in ABSOLUTE)
    if(EXISTS ${ABS_OPT_IN_FIL})
      set(ABS_OPT_FIL "${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL}/${FIL_WE}.options")
      configure_file(${ABS_OPT_IN_FIL} ${ABS_OPT_FIL})
    else()
      get_filename_component(ABS_OPT_FIL ${FIL_DIR}/${FIL_WE}.options ABSOLUTE)
    endif()

    # If there an options file in the same working directory, set it as a dependency
    if(EXISTS ${ABS_OPT_FIL})
        # Get directory as lookups for dependency options fail if an options
        # file is used. The options is still set as a dependency of the
        # generated source and header.
        get_filename_component(options_dir ${ABS_OPT_FIL} DIRECTORY)
        list(APPEND NANOPB_OPTIONS_DIRS ${options_dir})
    else()
        set(ABS_OPT_FIL)
    endif()

    # If the dependencies are options files, we need to pass the directories
    # as arguments to nanopb
    foreach(depends_file ${NANOPB_DEPENDS})
        get_filename_component(ext ${depends_file} EXT)
        if(ext STREQUAL ".options")
            get_filename_component(depends_dir ${depends_file} DIRECTORY)
            list(APPEND NANOPB_OPTIONS_DIRS ${depends_dir})
        endif()
    endforeach()

    if(NANOPB_OPTIONS_DIRS)
        list(REMOVE_DUPLICATES NANOPB_OPTIONS_DIRS)
    endif()

    set(NANOPB_PLUGIN_OPTIONS)
    foreach(options_path ${NANOPB_OPTIONS_DIRS})
        set(NANOPB_PLUGIN_OPTIONS "${NANOPB_PLUGIN_OPTIONS} -I${options_path}")
    endforeach()

    # Remove leading space before the first -I directive
    string(STRIP "${NANOPB_PLUGIN_OPTIONS}" NANOPB_PLUGIN_OPTIONS)

    if(NANOPB_OPTIONS)
        set(NANOPB_PLUGIN_OPTIONS "${NANOPB_PLUGIN_OPTIONS} ${NANOPB_OPTIONS}")
    endif()

    # based on the version of protoc it might be necessary to add "/${FIL_PATH_REL}" currently dealt with in #516
    set(NANOPB_OUT "${CMAKE_CURRENT_BINARY_DIR}")

    # We need to pass the path to the option files to the nanopb plugin. There are two ways to do it.
    # - An older hacky one using ':' as option separator in protoc args preventing the ':' to be used in path.
    # - Or a newer one, using --nanopb_opt which requires a version of protoc >= 3.6
    # Since nanopb 0.4.6, --nanopb_opt is the default.
    if(DEFINED NANOPB_PROTOC_OLDER_THAN_3_6_0)
      set(NANOPB_OPT_STRING "--nanopb_out=${NANOPB_PLUGIN_OPTIONS}:${NANOPB_OUT}")
    else()
      set(NANOPB_OPT_STRING "--nanopb_opt=${NANOPB_PLUGIN_OPTIONS}" "--nanopb_out=${NANOPB_OUT}")
    endif()

    add_custom_command(
      OUTPUT "${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL}/${FIL_WE}${GEN_EXTENSION}${GEN_SRC_EXTENSION}"
             "${CMAKE_CURRENT_BINARY_DIR}/${FIL_PATH_REL}/${FIL_WE}${GEN_EXTENSION}${GEN_HDR_EXTENSION}"
      COMMAND ${CUSTOM_COMMAND_PREFIX} ${PROTOBUF_PROTOC_EXECUTABLE}
      ARGS ${_nanopb_include_path} -I${GENERATOR_PATH}
           -I${GENERATOR_CORE_DIR} -I${CMAKE_CURRENT_BINARY_DIR}
           --plugin=protoc-gen-nanopb=${NANOPB_GENERATOR_PLUGIN}
           ${NANOPB_OPT_STRING}
           ${PROTOC_OPTIONS}
           ${ABS_FIL}
      DEPENDS ${ABS_FIL} ${GENERATOR_CORE_PYTHON_SRC}
           ${ABS_OPT_FIL} ${NANOPB_DEPENDS}
      COMMENT "Running C++ protocol buffer compiler using nanopb plugin on ${FIL}"
      VERBATIM )

  endforeach()

  set_source_files_properties(${${SRCS}} ${${HDRS}} PROPERTIES GENERATED TRUE)

  if(NANOPB_GENERATE_CPP_TARGET)
    add_library(${NANOPB_GENERATE_CPP_TARGET} STATIC EXCLUDE_FROM_ALL ${${SRCS}} ${${HDRS}})
    target_include_directories(${NANOPB_GENERATE_CPP_TARGET} PUBLIC ${CMAKE_CURRENT_BINARY_DIR})
    target_link_libraries(${NANOPB_GENERATE_CPP_TARGET} nanopb)
  endif()

  if(NOT DEFINED NANOPB_GENERATE_CPP_STANDALONE)
    set(NANOPB_GENERATE_CPP_STANDALONE TRUE)
  endif()

  if(MSVC)
      unset(CUSTOM_COMMAND_PREFIX)
  endif()

  if(NOT NANOPB_GENERATE_CPP_TARGET)
    if (NANOPB_GENERATE_CPP_STANDALONE)
      set(${SRCS} ${${SRCS}} ${NANOPB_SRCS} PARENT_SCOPE)
      set(${HDRS} ${${HDRS}} ${NANOPB_HDRS} PARENT_SCOPE)
    else()
      set(${SRCS} ${${SRCS}} PARENT_SCOPE)
      set(${HDRS} ${${HDRS}} PARENT_SCOPE)
    endif()
  endif()
endfunction()



#
# Main.
#

# By default have NANOPB_GENERATE_CPP macro pass -I to protoc
# for each directory where a proto file is referenced.
if(NOT DEFINED NANOPB_GENERATE_CPP_APPEND_PATH)
  set(NANOPB_GENERATE_CPP_APPEND_PATH TRUE)
endif()

# Make a really good guess regarding location of NANOPB_SRC_ROOT_FOLDER
if(NOT DEFINED NANOPB_SRC_ROOT_FOLDER)
  get_filename_component(NANOPB_SRC_ROOT_FOLDER
                         ${CMAKE_CURRENT_LIST_DIR}/.. ABSOLUTE)
endif()

# Parse any options given to find_package(... COMPONENTS ...)
foreach(component ${Nanopb_FIND_COMPONENTS})
  list(APPEND NANOPB_OPTIONS "--${component}")
endforeach()

# Find the include directory
find_path(NANOPB_INCLUDE_DIRS
    pb.h
    PATHS ${NANOPB_SRC_ROOT_FOLDER}
    NO_CMAKE_FIND_ROOT_PATH
)
mark_as_advanced(NANOPB_INCLUDE_DIRS)

# Find nanopb source files
set(NANOPB_SRCS)
set(NANOPB_HDRS)
list(APPEND _nanopb_srcs pb_decode.c pb_encode.c pb_common.c)
list(APPEND _nanopb_hdrs pb_decode.h pb_encode.h pb_common.h pb.h)

foreach(FIL ${_nanopb_srcs})
  find_file(${FIL}__nano_pb_file NAMES ${FIL} PATHS ${NANOPB_SRC_ROOT_FOLDER} ${NANOPB_INCLUDE_DIRS} NO_CMAKE_FIND_ROOT_PATH)
  list(APPEND NANOPB_SRCS "${${FIL}__nano_pb_file}")
  mark_as_advanced(${FIL}__nano_pb_file)
endforeach()

foreach(FIL ${_nanopb_hdrs})
  find_file(${FIL}__nano_pb_file NAMES ${FIL} PATHS ${NANOPB_INCLUDE_DIRS} NO_CMAKE_FIND_ROOT_PATH)
  mark_as_advanced(${FIL}__nano_pb_file)
  list(APPEND NANOPB_HDRS "${${FIL}__nano_pb_file}")
endforeach()

# Create the library target
add_library(nanopb STATIC EXCLUDE_FROM_ALL ${NANOPB_SRCS})
target_compile_features(nanopb PUBLIC c_std_11)
target_include_directories(nanopb PUBLIC ${NANOPB_INCLUDE_DIRS})

# Find the local protoc Executable
find_program(PROTOBUF_PROTOC_EXECUTABLE
    NAMES protoc
    DOC "The Google Protocol Buffers Compiler"
    PATHS
    ${PROTOBUF_SRC_ROOT_FOLDER}/vsprojects/Release
    ${PROTOBUF_SRC_ROOT_FOLDER}/vsprojects/Debug
    ${NANOPB_SRC_ROOT_FOLDER}/generator-bin
    ${NANOPB_SRC_ROOT_FOLDER}/generator
    NO_DEFAULT_PATH
)

# Test protoc, try to get version
execute_process(
    COMMAND ${PROTOBUF_PROTOC_EXECUTABLE} --version
    OUTPUT_QUIET
    ERROR_QUIET
    RESULT_VARIABLE ret
)
if(NOT ret EQUAL 0)
    # Fallback to system protoc
    unset(PROTOBUF_PROTOC_EXECUTABLE)
    find_program(PROTOBUF_PROTOC_EXECUTABLE
        NAMES protoc
        DOC "The Google Protocol Buffers Compiler"
    )
endif()

mark_as_advanced(PROTOBUF_PROTOC_EXECUTABLE)

# Find nanopb generator source dir
find_path(NANOPB_GENERATOR_SOURCE_DIR
    NAMES nanopb_generator.py
    DOC "nanopb generator source"
    PATHS
    ${NANOPB_SRC_ROOT_FOLDER}/generator
    NO_DEFAULT_PATH
    NO_CMAKE_FIND_ROOT_PATH
)
mark_as_advanced(NANOPB_GENERATOR_SOURCE_DIR)

include(FindPackageHandleStandardArgs)
find_package_handle_standard_args(Nanopb DEFAULT_MSG
  NANOPB_INCLUDE_DIRS
  NANOPB_SRCS NANOPB_HDRS
  NANOPB_GENERATOR_SOURCE_DIR
  PROTOBUF_PROTOC_EXECUTABLE
  )
