﻿using RimWorld;
using System.Collections.Generic;
using System.Linq;
using Verse;

namespace rjw.RMB
{
	/// <summary>
	/// Generator of RMB categories for "normal" sex
	/// </summary>
	static class RMB_Sex
	{
		/// <summary>
		/// Add have sex and reverse sex options to <paramref name="opts"/>
		/// </summary>
		/// <param name="pawn">Selected pawn</param>
		/// <param name="opts">List of RMB options</param>
		/// <param name="target">Target of right click</param>
		public static void AddFloatMenuOptions(Pawn pawn, List<FloatMenuOption> opts, LocalTargetInfo target)
		{
			AcceptanceReport canCreateEntries = DoBasicChecks(pawn, target.Pawn);

			if (!canCreateEntries)
			{
				if (RJWSettings.DevMode && !canCreateEntries.Reason.NullOrEmpty())
					opts.Add(new FloatMenuOption(canCreateEntries.Reason, null));

				return;
			}

			canCreateEntries = DoChecks(pawn, target.Pawn);

			if (!canCreateEntries)
			{
				opts.Add(new FloatMenuOption("RJW_RMB_Sex".Translate(target.Pawn) + ": " + canCreateEntries.Reason, null));
				return;
			}

			FloatMenuOption opt = GenerateCategoryOption(pawn, target);
			if (opt != null)
				opts.Add(opt);

			opt = GenerateCategoryOption(pawn, target, true);
			if (opt != null)
				opts.Add(opt);
		}

		/// <summary>
		/// Check for the things that should be obvious to the player or does not change for particular pawns.
		/// </summary>
		/// <returns>
		/// AcceptanceReport. Reason is an untranslated string and should only be shown in the DevMode
		/// </returns>
		private static AcceptanceReport DoBasicChecks(Pawn pawn, Pawn target)
		{
			if (target.Downed)
			{
				return "No sex: Target downed";
			}

			if (target.HostileTo(pawn))
			{
				return "No sex: Target hostile";
			}

			if (!xxx.can_be_fucked(target) && !xxx.can_fuck(target))
			{
				return "No sex: Target can't have sex";
			}

			return true;
		}

		/// <summary>
		/// Check for the things that can change for particular pawns.
		/// </summary>
		/// <returns>
		/// AcceptanceReport. Reason is a translated string and should not be null.
		/// </returns>
		private static AcceptanceReport DoChecks(Pawn pawn, Pawn target)
		{
			int opinionOf;

			if (!pawn.IsDesignatedHero() && !pawn.IsHeroOwner())
			{
				opinionOf = pawn.relations.OpinionOf(target);
				if (opinionOf < RJWHookupSettings.MinimumRelationshipToHookup)
				{
					if (!(opinionOf > 0 && xxx.is_nympho(pawn)))
					{
						return "RJW_RMB_ReasonLowOpinionOfTarget".Translate();
					}
				}
				if (SexAppraiser.would_fuck(pawn, target) < 0.1f)
				{
					return "RJW_RMB_ReasonUnappealingTarget".Translate();
				}
			}

			opinionOf = target.relations.OpinionOf(pawn);
			if (opinionOf < RJWHookupSettings.MinimumRelationshipToHookup)
			{
				if (!(opinionOf > 0 && xxx.is_nympho(target)))
				{
					return "RJW_RMB_ReasonLowOpinionOfPawn".Translate();
				}
			}

			if (SexAppraiser.would_fuck(target, pawn) < 0.1f)
			{
				return "RJW_RMB_ReasonUnappealingPawn".Translate();
			}

			return true;
		}

		/// <summary>
		/// Generate one FloatMenuOption
		/// </summary>
		/// <param name="pawn"></param>
		/// <param name="target"></param>
		/// <param name="reverse"></param>
		/// <returns>Category-level item that opens a sub-menu on click</returns>
		private static FloatMenuOption GenerateCategoryOption(Pawn pawn, LocalTargetInfo target, bool reverse = false)
		{
			string text = null;

			if (reverse)
			{
				text = "RJW_RMB_Sex_Reverse".Translate(target.Pawn);
			}
			else
			{
				text = "RJW_RMB_Sex".Translate(target.Pawn);
			}

			return FloatMenuUtility.DecoratePrioritizedTask(new FloatMenuOption(text, delegate ()
			{
				JobDef job = null;
				if (target.Pawn.InBed())
					job = xxx.casual_sex;
				else
					job = xxx.quick_sex;

				var validinteractions = RMB_Menu.GenerateNonSoloSexRoleOptions(pawn, target, job, false, reverse);
				if (validinteractions.Any())
					FloatMenuUtility.MakeMenu(validinteractions, (FloatMenuOption opt) => opt.Label, (FloatMenuOption opt) => opt.action);
				else
					Messages.Message("No valid interactions found for " + text, MessageTypeDefOf.RejectInput, false);
			}, MenuOptionPriority.High), pawn, target);
		}
	}
}