﻿using rjw.Modules.Shared.Extensions;
using System.Collections.Generic;
using System.Linq;
using Verse;

namespace rjw
{
	class RaceGroupDef_Helper
	{
		/// <summary>
		/// Cache for TryGetRaceGroupDef.
		/// </summary>
		static readonly IDictionary<PawnKindDef, RaceGroupDef> RaceGroupByPawnKind = new Dictionary<PawnKindDef, RaceGroupDef>();

		public static bool TryGetRaceGroupDef(PawnKindDef pawnKindDef, out RaceGroupDef raceGroupDef)
		{
			if (pawnKindDef == null)
			{
				ModLog.Error("Called TryGetRaceGroupDef with pawnKindDef = null");
				raceGroupDef = null;
				return false;
			}

			if (RaceGroupByPawnKind.TryGetValue(pawnKindDef, out raceGroupDef))
				{
					return raceGroupDef != null;
				}
				else
				{
					raceGroupDef = GetRaceGroupDefInternal(pawnKindDef);
					RaceGroupByPawnKind.Add(pawnKindDef, raceGroupDef);
					return raceGroupDef != null;
				}
		}
		public static bool TryGetRaceGroupDef(Pawn pawn, out RaceGroupDef raceGroupDef)
		{
			if (pawn == null)
			{
				ModLog.Error("Called TryGetRaceGroupDef with pawn = null");
				raceGroupDef = null;
				return false;
			}

			if (pawn.kindDef == null)
			{
				ModLog.Error($"{pawn} has kindDef = null");
				raceGroupDef = null;
				return false;
			}

			return TryGetRaceGroupDef(pawn.kindDef, out raceGroupDef);
		}

		/// <summary>
		/// Returns the best match RaceGroupDef for the given pawn, or null if none is found.
		/// </summary>
		static RaceGroupDef GetRaceGroupDefInternal(Pawn pawn)
		{
			return GetRaceGroupDefInternal(pawn.kindDef);
		}
		static RaceGroupDef GetRaceGroupDefInternal(PawnKindDef kindDef)
		{
			var raceName = kindDef.race.defName;
			var pawnKindName = kindDef.defName;
			var groups = DefDatabase<RaceGroupDef>.AllDefs;

			var kindMatches = groups.Where(group => group.pawnKindNames?.Contains(pawnKindName) ?? false).ToList();
			var raceMatches = groups.Where(group => group.raceNames?.Contains(raceName) ?? false).ToList();
			var count = kindMatches.Count() + raceMatches.Count();
			if (count == 0)
			{
				//ModLog.Message($"Pawn named '{pawn.Name}' matched no RaceGroupDef. If you want to create a matching RaceGroupDef you can use the raceName '{raceName}' or the pawnKindName '{pawnKindName}'.");
				return null;
			}
			else if (count == 1)
			{
				// ModLog.Message($"Pawn named '{pawn.Name}' matched 1 RaceGroupDef.");
				return kindMatches.Concat(raceMatches).Single();
			}
			else
			{
				// ModLog.Message($"Pawn named '{pawn.Name}' matched {count} RaceGroupDefs.");

				// If there are multiple RaceGroupDef matches, choose one of them.
				// First prefer defs NOT defined in rjw.
				// Then prefer a match by kind over a match by race.
				return kindMatches.FirstOrDefault(match => !IsThisMod(match))
					?? raceMatches.FirstOrDefault(match => !IsThisMod(match))
					?? kindMatches.FirstOrDefault()
					?? raceMatches.FirstOrDefault();
			}
		}

		static bool IsThisMod(Def def)
		{
			var rjwContent = LoadedModManager.RunningMods.Single(pack => pack.Name == "RimJobWorld");
			return rjwContent.AllDefs.Contains(def);
		}

		/// <summary>
		/// Returns true if a race part was chosen (even if that part is "no part").
		/// </summary>
		public static bool TryAddRacePart(Pawn pawn, BodyPartRecord partBPR, Gender gender = Gender.None)
		{
			if (!TryGetRaceGroupDef(pawn, out var raceGroupDef))
			{
				// No race, so nothing was chosen.
				return false;
			}

			if (!RacePartDef_Helper.TryChooseRacePartDef(pawn, raceGroupDef, partBPR, out var racePartDef, gender))
			{
				// Failed to find a part, so nothing was chosen.
				return false;
			}

			if (racePartDef.IsNone)
			{
				// "no part" was explicitly chosen.
				return true;
			}

			if (!racePartDef.TryGetHediffDef(out var hediffDef))
			{
				// Failed to find hediffDef.
				return false;
			}

			if (partBPR.IsMissingForPawn(pawn))
			{
				// Bodypart destroyed.
				return false;
			}

			var hediff = RacePartDef_Helper.MakePart(hediffDef, pawn, partBPR, racePartDef);
			pawn.health.AddHediff(hediff, partBPR);
			// A part was chosen and added.
			return true;
		}

		/// <summary>
		/// <para>Checks two pawns to see if they are compatible mates.</para>
		/// <para>This checks to see if they have similar kinds of genitalia for their
		/// race-groups; if there is any overlap at all, they're counted as compatible.
		/// This will fallback to comparing their leather if one lacks a race-group.</para>
		/// <para>IE: a dog and a warg both have canine anatomy, and so are compatible.</para>
		/// <para>Always returns true for human-like pawns, because fun!</para>
		/// </summary>
		/// <param name="pawn">The first pawn.</param>
		/// <param name="other">The second pawn.</param>
		/// <returns>Whether the two pawns are compatible.</returns>
		public static bool IsCompatibleForMating(Pawn pawn, Pawn other)
		{
			// Human-like pawns are always compatible.
			if (pawn.IsHumanLike() || other.IsHumanLike()) return true;
			// Same races are always compatible.
			if (pawn.kindDef.race == other.kindDef.race) return true;
			// Slimes can shapeshift to have any genitalia.
			if (xxx.is_slime(pawn) || xxx.is_slime(other)) return true;

			// Fallback to other properties if either lack a race-group.
			if (!TryGetRaceGroupDef(pawn.kindDef, out var pGroup))
				return MatingCompatibilityFallback(pawn, other);
			if (!TryGetRaceGroupDef(other.kindDef, out var oGroup))
				return MatingCompatibilityFallback(pawn, other);

			// We don't need to do anything special if they're in the same group.
			if (pGroup == oGroup) return true;

			// From here, we're just looking to see if the groups have overlap in genitals.
			foreach (var pPart in pGroup.maleGenitals)
				foreach (var oPart in oGroup.maleGenitals)
					if (pPart == oPart) return true;
			foreach (var pPart in pGroup.femaleGenitals)
				foreach (var oPart in oGroup.femaleGenitals)
					if (pPart == oPart) return true;

			return false;
		}

		private static bool MatingCompatibilityFallback(Pawn pawn, Pawn other)
		{
			var pProps = pawn.RaceProps;
			var oProps = other.RaceProps;
			// Do they differ in predatory behaviors?
			if (pProps.predator != oProps.predator) return false;
			// Do they eat different things?
			if (pProps.foodType != oProps.foodType) return false;
			// Do they have different leather types?
			if (pProps.leatherDef != oProps.leatherDef) return false;
			// Do they have different meat types?
			if (pProps.meatDef != oProps.meatDef) return false;
			return true;
		}
	}
}
