﻿using RimWorld;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Runtime.CompilerServices;
using System.Text;
using Verse;

namespace rjw
{
	public static class PawnExtensions
	{
		public static bool RaceHasFertility(this Pawn pawn)
		{
			// True by default.
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
				return raceGroupDef.hasFertility;
			return true;
		}

		public static bool RaceHasPregnancy(this Pawn pawn)
		{
			// True by default.
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
				return raceGroupDef.hasPregnancy;
			return true;
		}

		public static bool RaceHasOviPregnancy(this Pawn pawn)
		{
			// False by default.
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
				return raceGroupDef.oviPregnancy;
			return false;
		}

		public static bool RaceImplantEggs(this Pawn pawn)
		{
			// False by default.
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
				return raceGroupDef.ImplantEggs;
			return false;
		}

		public static bool RaceHasSexNeed(this Pawn pawn)
		{
			// True by default.
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
				return raceGroupDef.hasSexNeed;
			return true;
		}
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static bool TryAddRacePart(this Pawn pawn, BodyPartRecord partBPR, Gender gender = Gender.None)
		{
			return RaceGroupDef_Helper.TryAddRacePart(pawn, partBPR, gender);
		}

		public static bool Has(this Pawn pawn, RaceTagDef tag)
		{
			if (RaceGroupDef_Helper.TryGetRaceGroupDef(pawn, out var raceGroupDef))
			{
				return raceGroupDef.tags != null && raceGroupDef.tags.Contains(tag);
			}
			else
			{
				return tag.DefaultWhenNoRaceGroupDef(pawn);
			}
		}

		/// <summary>
		/// Checks if the pawn has a particular social memory of another pawn.
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <param name="other">The other pawn.</param>
		/// <param name="thought">The thought def to check for.</param>
		/// <returns>Whether the pawn has the memory.</returns>
		public static bool HasMemoryWith(this Pawn pawn, Pawn other, ThoughtDef thought)
		{
			if (pawn.needs.mood?.thoughts.memories.Memories is { } memories)
				foreach (var memory in memories)
					if (memory.otherPawn == other)
						if (memory.def == thought)
							return true;
			return false;
		}
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static bool IsSexyRobot(this Pawn pawn)
		{
			return AndroidsCompatibility.IsAndroid(pawn);
		}

		// In theory I think this should involve RaceGroupDef.
		public static bool IsUnsexyRobot(this Pawn pawn)
		{
			return !IsSexyRobot(pawn)
				&& (xxx.is_mechanoid(pawn) || pawn.kindDef.race.defName.ToLower().Contains("droid"));
		}
		
		/// <summary>
		/// <para>Checks to see if this pawn has animal intelligence.</para>
		/// <para>We generally treat all pawns of this intelligence as animals.</para>
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <returns>Whether the pawn has animal intelligence.</returns>
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static bool IsAnimal(this Pawn pawn) => xxx.is_animal(pawn);

		/// <summary>
		/// <para>Checks to see if this pawn has human-like intelligence.</para>
		/// <para>This is generally true of all the pawns we consider "human" enough.</para>
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <returns>Whether the pawn has human-like intelligence.</returns>
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static bool IsHumanLike(this Pawn pawn) => xxx.is_human(pawn);

		/// <summary>
		/// <para>Checks if this pawn is the same "species" as the other pawn.</para>
		/// <para>This takes the pawn's genes into consideration.  Yes, I know xenotypes
		/// don't suggest different species, but this is more about whether we can make
		/// assumptions about the physical differences between two pawns.</para>
		/// </summary>
		/// <param name="pawn">The first pawn.</param>
		/// <param name="other">The second pawn.</param>
		/// <returns>Whether the pawns are the same species.</returns>
		public static bool IsSameSpecies(this Pawn pawn, Pawn other)
		{
			// Different races are not the same species.
			if (pawn.kindDef.race != other.kindDef.race) return false;
			// Only compare genes if they're relevant.
			if (!ModsConfig.BiotechActive) return true;

			var pGeneTracker = pawn.genes;
			var oGeneTracker = other.genes;

			// Both not humanlike, so same species.
			if (pGeneTracker is null && oGeneTracker is null) return true;
			// One is humanlike and the other not, so not same species.
			if (pGeneTracker is null || oGeneTracker is null) return false;

			// Same xenotypes are considered the same species.
			if (pGeneTracker.Xenotype == oGeneTracker.Xenotype) return true;

			// So, that's all the fast checks we can do.  Now we get dirty and
			// compare genes directly.
			var pGenes = pGeneTracker.GenesListForReading.Select(ToGeneDef).ToHashSet();
			return pGenes.SetEquals(oGeneTracker.GenesListForReading.Select(ToGeneDef));

			static GeneDef ToGeneDef(Gene gene) => gene.def;
		}

		/// <summary>
		/// Checks if this pawn is considered disfigured to an observing pawn.
		/// </summary>
		/// <param name="observed">The pawn being observed.</param>
		/// <param name="observer">The pawn observing.</param>
		/// <returns>Whether the this pawn is considered disfigured.</returns>
		public static bool IsDisfiguredTo(this Pawn observed, Pawn observer)
		{
			var okWithBlindness = observer.Ideo?.IdeoApprovesOfBlindness() is true;
			return RelationsUtility.IsDisfigured(observed, observer, okWithBlindness);
		}

		/// <summary>
		/// Checks to see if this pawn is a manhunter.
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <returns>Whether the pawn is a manhunter.</returns>
		public static bool IsManhunter(this Pawn pawn)
		{
			if (pawn?.InMentalState is not true) return false;
			if (pawn.MentalStateDef == MentalStateDefOf.Manhunter) return true;
			if (pawn.MentalStateDef == MentalStateDefOf.ManhunterPermanent) return true;
			return false;
		}

		public static bool IsPoly(this Pawn pawn)
		{
			if (!pawn.IsHumanLike())
			{
				return true;
			}
			if (pawn.IsIdeologicallyPoly())
			{
				return true;
			}
			if (pawn.story?.traits == null)
			{
				return false;
			}
			if (xxx.AnyPolyamoryModIsActive && pawn.story.traits.HasTrait(xxx.polyamorous))
			{
				return true;
			}
			if (xxx.AnyPolygamyModIsActive && pawn.story.traits.HasTrait(xxx.polygamous))
			{
				return true;
			}
			return false;
		}

		/// <summary>
		/// <para>Checks to see if this pawn is a tamed animal.</para>
		/// <para>This is not specific to the player faction.</para>
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <returns>Whether the pawn is a tame animal.</returns>
		public static bool IsTameAnimal(this Pawn pawn) =>
			pawn.IsAnimal() && pawn.Faction?.def.humanlikeFaction is true;

		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static bool IsVisiblyPregnant(this Pawn pawn)
		{
			return pawn.IsPregnant(true);
		}

		public static bool IsPregnant(this Pawn pawn, bool mustBeVisible = false)
		{
			var set = pawn.health.hediffSet;
			return set.HasHediff(HediffDefOf.PregnantHuman, mustBeVisible) || 
				set.HasHediff(HediffDefOf.Pregnant, mustBeVisible) ||
				Hediff_BasePregnancy.KnownPregnancies().Any(x => set.HasHediff(HediffDef.Named(x), mustBeVisible));
		}

		/// <summary>
		/// Checks if this pawn has a psychic love connection with the given pawn.
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <param name="other">The other pawn.</param>
		/// <returns>Whether they make psychic love together.</returns>
		public static bool HasPsychicLoveWith(this Pawn pawn, Pawn other) {
			if (pawn.health.hediffSet.GetFirstHediffOfDef(HediffDefOf.PsychicLove) is HediffWithTarget hediff)
				return hediff.target == other;
			return false;
		}

		/// <summary>
		/// Checks if this pawn has their beer goggles on; IE they're drunk.
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <returns>Whether the pawn is drunk.</returns>
		public static bool HasBeerGoggles(this Pawn pawn) =>
			pawn.health.hediffSet.HasHediff(HediffDefOf.AlcoholHigh);

		/// <summary>
		/// <para>Checks if this pawn has their beer goggles on; IE they're drunk.</para>
		/// <para>This variant also gets the severity.</para>
		/// </summary>
		/// <param name="pawn">The pawn to inspect.</param>
		/// <param name="severity">The variable to store the severity into.</param>
		/// <returns>Whether the pawn is drunk.</returns>
		public static bool HasBeerGoggles(this Pawn pawn, out float severity)
		{
			severity = 0f;
			var hediff = pawn.health.hediffSet.GetFirstHediffOfDef(HediffDefOf.AlcoholHigh);
			if (hediff is null) return false;
			severity = hediff.Severity;
			return true;
		}

		public static List<Hediff> GetGenitalsList(this Pawn pawn)
		{
			List<Hediff> set;
			try//error at world gen
			{
				set = pawn.GetRJWPawnData().genitals;
				if (set.NullOrEmpty())
				{
					var partBPR = Genital_Helper.get_genitalsBPR(pawn);
					set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
					pawn.GetRJWPawnData().genitals = set;
				}

			}
			catch
			{
				var partBPR = Genital_Helper.get_genitalsBPR(pawn);
				set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
			}
			return set;
		}

		public static List<Hediff> GetBreastList(this Pawn pawn)
		{
			List<Hediff> set;
			try//error at world gen
			{
				set = pawn.GetRJWPawnData().breasts;
				if (set.NullOrEmpty())
				{
					var partBPR = Genital_Helper.get_breastsBPR(pawn);
					set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
					pawn.GetRJWPawnData().breasts = set;
				}

			}
			catch
			{
				var partBPR = Genital_Helper.get_breastsBPR(pawn);
				set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
			}
			return set;
		}
		public static List<Hediff> GetAnusList(this Pawn pawn)
		{
			List<Hediff> set;
			try//error at world gen
			{
				set = pawn.GetRJWPawnData().anus;
				if (set.NullOrEmpty())
				{
					var partBPR = Genital_Helper.get_anusBPR(pawn);
					set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
					pawn.GetRJWPawnData().anus = set;
				}

			}
			catch
			{
				var partBPR = Genital_Helper.get_anusBPR(pawn);
				set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
			}
			return set;
		}

		public static List<Hediff> GetTorsoList(this Pawn pawn)
		{
			List<Hediff> set;
			try//error at world gen
			{
				set = pawn.GetRJWPawnData().torso;
				if (set.NullOrEmpty())
				{
					var partBPR = Genital_Helper.get_torsoBPR(pawn);
					set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
					pawn.GetRJWPawnData().torso = set;
				}

			}
			catch
			{
				var partBPR = Genital_Helper.get_torsoBPR(pawn);
				set = Genital_Helper.get_PartsHediffList(pawn, partBPR);
			}
			return set;
		}

#nullable enable
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static SexProps? GetRMBSexPropsCache(this Pawn pawn)
		{
			return pawn.GetRJWPawnData()?.SexProps;
		}
		[MethodImpl(MethodImplOptions.AggressiveInlining)]
		public static PawnData? GetRJWPawnData(this Pawn pawn)
		{
			return pawn.GetCompRJW()?.PawnData;
		}
	}
}
