#nullable enable

using System;
using System.Collections.Generic;
using System.Linq;
using UnityEngine;
using Verse;
using rjw.Modules.Attraction;
using rjw.Modules.Shared.Extensions;

using ModifyPreferences = rjw.Modules.Attraction.AttractionRequest.ModifyPreferences;
using R_Orientation = rjw.Modules.Attraction.StandardPreferences.R_Orientation;
using R_Homewrecking = rjw.Modules.Attraction.StandardPreferences.R_Homewrecking;
using R_Relationship = rjw.Modules.Attraction.StandardPreferences.R_Relationship;
using Seeded = rjw.Modules.Rand.Seeded;

namespace rjw
{
	/// <summary>
	/// 
	/// </summary>
	/// <param name="Purpose">
	/// <para>How the observer is going to be evaluating the target.</para>
	/// <para>Factors can be calculated differently based on this value.</para>
	/// </param>
	/// <param name="PreQuirkModifier">
	/// Optional modifier that can be set to customize preferences before quirks.
	/// </param>
	/// <param name="PostQuirkModifier">
	/// Optional modifier that can be set to customize preferences after quirks.
	/// </param>
	/// <param name="IgnoreBleeding">
	/// <para>Option to ignore the bleeding pre-flight check.</para>
	/// <para>Only considered when the purpose is sex.</para>
	/// </param>
	public readonly record struct AppraisalSettings(
		AttractionPurpose Purpose = AttractionPurpose.ForFucking,
		ModifyPreferences? PreQuirkModifier = null,
		ModifyPreferences? PostQuirkModifier = null,
		bool IgnoreBleeding = false
	);

	/// <summary>
	/// <para>A parameters object that specifies everything needed for an appraisal.</para>
	/// <para>To supply unique settings for each pawn in an <see cref="AttractionRequest" />,
	/// use the object initializer syntax to set them.</para>
	/// <para>Use <see cref="SexAppraiser.CandidatesAsTargets" /> or
	/// <see cref="SexAppraiser.CandidatesAsObservers" /> to specify which settings
	/// apply to which pawn.</para>
	/// </summary>
	/// <param name="Pawn">
	/// The pawn at the center of the appraisal.
	/// </param>
	/// <param name="Candidates">
	/// The collection of candidates.
	/// </param>
	public readonly record struct AppraisalParams(
		Pawn Pawn,
		IEnumerable<Pawn> Candidates
		)
	{
		/// <summary>
		/// Settings to use when constructing an <see cref="AttractionRequest" />
		/// for the observer.
		/// </summary>
		public AppraisalSettings ObserverSettings { get; init; } = new();

		/// <summary>
		/// Settings to use when constructing an <see cref="AttractionRequest" />
		/// for the target.
		/// </summary>
		public AppraisalSettings TargetSettings { get; init; } = new();
	}

	/// <summary>
	/// A class containing information about an appraisal's result.
	/// </summary>
	public record AppraisalResult(
		Pawn Observer,
		AppraisalSettings ObserverSettings,
		Pawn Target,
		AppraisalSettings TargetSettings
		)
	{
		/// <summary>
		/// <para>The attraction of the observer to the target.</para>
		/// <para>The first access to this value will trigger an attraction evaluation,
		/// which may be expensive.  Avoid accessing this value unless it is needed.</para>
		/// </summary>
		public float ObserverAttraction
		{
			get
			{
				if (!observerAttraction.HasValue)
				{
					var request = new AttractionRequest(ObserverSettings.Purpose, Observer, Target)
					{
						preQuirkModifier = ObserverSettings.PreQuirkModifier,
						postQuirkModifier = ObserverSettings.PostQuirkModifier,
						ignoreBleeding = ObserverSettings.IgnoreBleeding
					};
					observerAttraction = AttractionUtility.Evaluate(ref request);
				}
				return observerAttraction.Value;
			}
		}

		private float? observerAttraction = null;

		/// <summary>
		/// <para>The attraction of the target to the observer.</para>
		/// <para>The first access to this value will trigger an attraction evaluation,
		/// which may be expensive.  Avoid accessing this value unless it is needed.</para>
		/// </summary>
		public float TargetAttraction
		{
			get
			{
				if (!targetAttraction.HasValue)
				{
					var request = new AttractionRequest(TargetSettings.Purpose, Target, Observer)
					{
						preQuirkModifier = TargetSettings.PreQuirkModifier,
						postQuirkModifier = TargetSettings.PostQuirkModifier,
						ignoreBleeding = TargetSettings.IgnoreBleeding
					};
					targetAttraction = AttractionUtility.Evaluate(ref request);
				}
				return targetAttraction.Value;
			}
		}
		private float? targetAttraction = null;

		/// <summary>
		/// <para>The combined attraction between both pawns.</para>
		/// <para>This will invoke both <see cref="ObserverAttraction" /> and
		/// <see cref="TargetAttraction" /> when accessed, which will evaluate
		/// attraction for both if it has not been done already, which may be
		/// expensive.  Avoid accessing this value unless it is needed.</para>
		/// </summary>
		public float CombinedAttraction => ObserverAttraction + TargetAttraction;
	}

	/// <summary>
	/// Responsible for judging the sexiness of potential partners.
	/// </summary>
	public static class SexAppraiser
	{
		public const float base_sat_per_fuck = 0.40f;
		public const float base_attraction = 0.60f;
		public const float no_partner_ability = 0.8f;

		public const float MinimumAttractionForSex = 0.1f;

		/// <summary>
		/// <para>Gets the chance that the pawn would rape, if an opportunity presents
		/// itself.</para>
		/// <para>Honestly, this should probably just influence the MTB-hours of
		/// the rape jobs instead of being run before a job.</para>
		/// </summary>
		/// <param name="rapist">The pawn considering rape.</param>
		/// <returns>A chance, between 0 and 1.</returns>
		public static float RapeChanceFor(Pawn rapist)
		{
			// Starts at 30% chance.
			var rapeChance = 0.3f;

			// More inclined to rape when horny.
			if (rapist.needs.TryGetNeed<Need_Sex>() is { } horniness)
				if (horniness.CurLevel <= horniness.thresh_horny())
					rapeChance += 0.25f;

			// Too drunk to care...
			if (rapist.HasBeerGoggles(out var severity))
				rapeChance *= GenMath.LerpDouble(0f, 1f, 1f, 1.3f, severity);

			// Increase factor from traits.
			if (xxx.is_rapist(rapist))
				rapeChance *= 1.5f;
			if (xxx.is_nympho(rapist))
				rapeChance *= 1.25f;
			if (xxx.is_bloodlust(rapist))
				rapeChance *= 1.2f;
			if (xxx.is_psychopath(rapist))
				rapeChance *= 1.2f;

			// Lower factor from traits.
			if (xxx.is_masochist(rapist))
				rapeChance *= 0.8f;

			// The rapist is really bored...
			if (rapist.needs.joy?.CurLevel is < 0.1f)
				rapeChance *= 1.2f;

			rapeChance = Mathf.Clamp01(rapeChance);

			// Log.Message($"rjw::xxx rape chance for {xxx.get_pawnname(rapist)} is {rapeChance}");

			return rapeChance;
		}

		/// <summary>
		/// Returns whether the pawn is in a raping mood.
		/// </summary>
		/// <param name="rapist">The pawn considering rape.</param>
		/// <returns>Whether the pawn is in a raping mood.</returns>
		public static bool InMoodForRape(Pawn rapist)
		{
			using (Seeded.ForHour(rapist))
				return Rand.Chance(RapeChanceFor(rapist));
		}

		/// <summary>
		/// <para>Finds partners for consensual sex with the observer.</para>
		/// <para>This uses a weighted-random algorithm that may not yield the best
		/// option first, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="observer">The pawn looking for sex.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <returns>A list of results.</returns>
		public static List<AppraisalResult> FindPartners(
			Pawn observer,
			IEnumerable<Pawn> candidates)
		{
			using (Seeded.ForHour(observer))
				return RNG.FindPartners(observer, candidates).ToList();
		}

		/// <summary>
		/// <para>Tries to find one partner for consensual sex with the observer
		/// from the list of given candidates.</para>
		/// <para>This uses a weighted-random algorithm that may not choose the
		/// very best option, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="observer">The pawn looking for sex.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <param name="result">A variable to store the result into.</param>
		/// <returns>Whether a result was found.</returns>
		public static bool FindPartner(
			Pawn observer,
			IEnumerable<Pawn> candidates,
			out AppraisalResult result)
		{
			using (Seeded.ForHour(observer))
				return RNG.FindPartners(observer, candidates).TryFirst(out result);
		}

		/// <summary>
		/// <para>Finds clients for whoring by the observer.</para>
		/// <para>This uses a weighted-random algorithm that may not yield the best
		/// option first, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="whore">The pawn looking for a job.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <returns>A list of results.</returns>
		public static List<AppraisalResult> FindTricks(
			Pawn whore,
			IEnumerable<Pawn> candidates)
		{
			using (Seeded.ForHour(whore))
				return RNG.FindTricks(whore, candidates).ToList();
		}

		/// <summary>
		/// <para>Tries to find one client for whoring by the observer from the
		/// list of given candidates.</para>
		/// <para>This uses a weighted-random algorithm that may not choose the
		/// very best option, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="whore">The pawn looking for a job.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <param name="result">A variable to store the result into.</param>
		/// <returns>Whether a result was found.</returns>
		public static bool FindTrick(
			Pawn whore,
			IEnumerable<Pawn> candidates,
			out AppraisalResult result)
		{
			using (Seeded.ForHour(whore))
				return RNG.FindTricks(whore, candidates).TryFirst(out result);
		}

		/// <summary>
		/// <para>Finds whores for use by the target.</para>
		/// <para>This is basically the inverse of <see cref="FindTricks" />.  In
		/// the results, the observer will be the whore and the target is the given
		/// `client`.</para>
		/// <para>This uses a weighted-random algorithm that may not yield the best
		/// option first, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="client">The pawn looking for a lay.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <returns>A list of results.</returns>
		public static List<AppraisalResult> FindWhores(
			Pawn client,
			IEnumerable<Pawn> candidates)
		{
			using (Seeded.ForHour(client))
				return RNG.FindWhores(client, candidates).ToList();
		}

		/// <summary>
		/// <para>Tries to find one whore for use by the target from the
		/// list of given candidates.</para>
		/// <para>This is basically the inverse of <see cref="FindTrick" />.  In
		/// the result, the observer will be the whore and the target is the given
		/// `client`.</para>
		/// <para>This uses a weighted-random algorithm that may not choose the
		/// very best option, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="client">The pawn looking for a lay.</param>
		/// <param name="candidates">The list of candidate pawns.</param>
		/// <param name="result">A variable to store the result into.</param>
		/// <returns>Whether a result was found.</returns>
		public static bool FindWhore(
			Pawn client,
			IEnumerable<Pawn> candidates,
			out AppraisalResult result)
		{
			using (Seeded.ForHour(client))
				return RNG.FindWhores(client, candidates).TryFirst(out result);
		}

		/// <summary>
		/// <para>Finds potential victims to be raped by a rapist.</para>
		/// <para>This uses a weighted-random algorithm that may not yield the best
		/// option first, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="rapist">The pawn looking for a good rape.</param>
		/// <param name="candidates">The list of candidate victims.</param>
		/// <param name="ignoreBleeding">Whether to ignore a bleeding victim.</param>
		/// <param name="ignoreOrienation">Whether to disable orientation checks.</param>
		/// <returns>A list of results.</returns>
		public static List<AppraisalResult> FindVictims(
			Pawn rapist,
			IEnumerable<Pawn> candidates,
			bool ignoreBleeding = false,
			bool ignoreOrienation = false)
		{
			using (Seeded.ForHour(rapist))
				return RNG.FindVictims(rapist, candidates, ignoreBleeding, ignoreOrienation).ToList();
		}

		/// <summary>
		/// <para>Tries to find one victim to be raped by a rapist from the
		/// list of given candidates.</para>
		/// <para>This uses a weighted-random algorithm that may not choose the
		/// very best option, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="rapist">The pawn looking for a good rape.</param>
		/// <param name="candidates">The list of candidate victims.</param>
		/// <param name="result">A variable to store the result into.</param>
		/// <param name="ignoreBleeding">Whether to ignore a bleeding victim.</param>
		/// <param name="ignoreOrienation">Whether to disable orientation checks.</param>
		/// <returns>Whether a result was found.</returns>
		public static bool FindVictim(
			Pawn rapist,
			IEnumerable<Pawn> candidates,
			out AppraisalResult result,
			bool ignoreBleeding = false,
			bool ignoreOrienation = false)
		{
			using (Seeded.ForHour(rapist))
				return RNG.FindVictims(rapist, candidates, ignoreBleeding, ignoreOrienation).TryFirst(out result);
		}

		/// <summary>
		/// <para>Finds potential rapists to rape by a specific victim.  Handy for
		/// getting a group of rapists together for a gangbang.</para>
		/// <para>This is basically the inverse of <see cref="FindVictims" />.  In
		/// the results, the observer will be the rapist and the target is the given
		/// `victim`.</para>
		/// <para>This uses a weighted-random algorithm that may not yield the best
		/// option first, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="victim">The pawn to be victimized.</param>
		/// <param name="candidates">The list of candidate rapists.</param>
		/// <param name="ignoreBleeding">Whether to ignore a bleeding victim.</param>
		/// <param name="ignoreOrienation">Whether to disable orientation checks.</param>
		/// <returns>A list of results.</returns>
		public static List<AppraisalResult> FindRapists(
			Pawn victim,
			IEnumerable<Pawn> candidates,
			bool ignoreBleeding = false,
			bool ignoreOrienation = false)
		{
			using (Seeded.ForHour(victim))
				return RNG.FindRapists(victim, candidates, ignoreBleeding, ignoreOrienation).ToList();
		}

		/// <summary>
		/// <para>Tries to find one rapist to rape by a specific victim from the
		/// list of given candidates.</para>
		/// <para>This is basically the inverse of <see cref="FindVictim" />.  In
		/// the result, the observer will be the rapist and the target is the given
		/// `victim`.</para>
		/// <para>This uses a weighted-random algorithm that may not choose the
		/// very best option, but does still prefer above average partners.</para>
		/// </summary>
		/// <param name="victim">The pawn to be victimized.</param>
		/// <param name="candidates">The list of candidate rapists.</param>
		/// <param name="result">A variable to store the result into.</param>
		/// <param name="ignoreBleeding">Whether to ignore bleeding from the victim.</param>
		/// <param name="ignoreOrienation">Whether to disable orientation checks.</param>
		/// <returns>Whether a result was found.</returns>
		public static bool FindRapist(
			Pawn victim,
			IEnumerable<Pawn> candidates,
			out AppraisalResult result,
			bool ignoreBleeding = false,
			bool ignoreOrienation = false)
		{
			using (Seeded.ForHour(victim))
				return RNG.FindRapists(victim, candidates, ignoreBleeding, ignoreOrienation).TryFirst(out result);
		}

		/// <summary>
		/// Creates a list of appraisal results from the given parameters struct,
		/// treating the <see cref="AppraisalParams.Pawn" /> as the observer and the
		/// list of <see cref="AppraisalParams.Candidates" /> as the targets.
		/// </summary>
		/// <param name="appraisalParams">The appraisal parameters.</param>
		/// <returns>The list of results.</returns>
		public static List<AppraisalResult> CandidatesAsTargets(in AppraisalParams appraisalParams)
		{
			if (appraisalParams.Candidates is not List<Pawn> candidateList)
				candidateList = appraisalParams.Candidates.ToList();

			var resultList = new List<AppraisalResult>(candidateList.Count);
			foreach (var candidate in candidateList)
				if (candidate != appraisalParams.Pawn)
					resultList.Add(new(
						appraisalParams.Pawn, appraisalParams.ObserverSettings,
						candidate, appraisalParams.TargetSettings
					));

			return resultList;
		}

		/// <summary>
		/// Creates a list of appraisal results from the given parameters struct,
		/// treating the <see cref="AppraisalParams.Pawn" /> as the target and the
		/// list of <see cref="AppraisalParams.Candidates" /> as the observers.
		/// </summary>
		/// <param name="appraisalParams">The appraisal parameters.</param>
		/// <returns>The list of results.</returns>
		public static List<AppraisalResult> CandidatesAsObservers(in AppraisalParams appraisalParams)
		{
			if (appraisalParams.Candidates is not List<Pawn> candidateList)
				candidateList = appraisalParams.Candidates.ToList();

			var resultList = new List<AppraisalResult>(candidateList.Count);
			foreach (var candidate in candidateList)
				if (candidate != appraisalParams.Pawn)
					resultList.Add(new(
						candidate, appraisalParams.ObserverSettings,
						appraisalParams.Pawn, appraisalParams.TargetSettings
					));

			return resultList;
		}

		/// <summary>
		/// <para>Given a collection of `AppraisalResult`, yields them back in the
		/// order of best-to-worst.</para>
		/// <para>Unlike <see cref="RNG.FindResults" /> and <see cref="RNG.FindBiasedResults" />,
		/// this provides no randomness.</para>
		/// </summary>
		/// <param name="results">The results to process.</param>
		/// <returns>An enumerable of results.</returns>
		public static IEnumerable<AppraisalResult> FindBestResults(IEnumerable<AppraisalResult> results)
		{
			return results
				.Where(ValidForObserver)
				.Where(ValidForTarget)
				.OrderByDescending(GetWeight);
		}

		/// <summary>
		/// Filter checking if the observer-to-target attraction meets the minimum
		/// value needed for consideration.
		/// </summary>
		static bool ValidForObserver(AppraisalResult r) =>
			r.ObserverAttraction >= MinimumAttractionForSex;

		/// <summary>
		/// <para>Filter checking if the target-to-observer attraction meets the minimum
		/// value needed for consideration.</para>
		/// <para>Note: target's attraction to observer is irrelevant for rape.</para>
		/// </summary>
		static bool ValidForTarget(AppraisalResult r) =>
			r.ObserverSettings.Purpose == AttractionPurpose.ForRape || r.TargetAttraction >= MinimumAttractionForSex;

		/// <summary>
		/// <para>Gets the weight for the given result.</para>
		/// <para>Note: target's attraction to observer is irrelevant for rape.</para>
		/// </summary>
		static float GetWeight(AppraisalResult r) =>
			r.ObserverSettings.Purpose == AttractionPurpose.ForRape ? r.ObserverAttraction : r.CombinedAttraction;

		/// <summary>
		/// These can be added to an attraction request to disable certain standard
		/// behaviors for special case scenarios.
		/// </summary>
		public static class PreferenceCustomizers
		{
			/// <summary>
			/// Customizes pawn preferences so orientation is ignored.
			/// </summary>
			public static void IgnoreOrientation(ref AttractionRequest request)
			{
				request.RemovePreference(nameof(R_Orientation));
			}

			/// <summary>
			/// Customizes pawn preferences so their current relationships are ignored.
			/// </summary>
			public static void IgnoreRelationships(ref AttractionRequest request)
			{
				request.RemovePreference(nameof(R_Homewrecking));
				request.RemovePreference(nameof(R_Relationship));
			}
		}

		/// <summary>
		/// These methods employ RNG.  You should be using a deterministic seed
		/// (using <see cref="Seeded" />) or using the Multiplayer API's
		/// `SyncMethod` attribute for methods responding to a player action.
		/// </summary>
		public static class RNG
		{
			/// <summary>
			/// <para>Given a collection of `AppraisalResult`, yields them back in a
			/// weighted-random order.  There is no bias in the options besides the
			/// weight from the attraction value, so any result could be first.</para>
			/// </summary>
			/// <param name="results">The results to process.</param>
			/// <returns>An enumerable of results.</returns>
			public static IEnumerable<AppraisalResult> FindResults(IEnumerable<AppraisalResult> results)
			{
				var options = results
					.Where(ValidForObserver)
					.Where(ValidForTarget)
					.ToHashSet();

				while (options.Count > 0)
				{
					var nextOption = options.RandomElementByWeight(GetWeight);
					options.Remove(nextOption);
					yield return nextOption;
				}
			}

			/// <summary>
			/// <para>Given a collection of `AppraisalResult`, yields them back in a
			/// weighted-random order, biased toward the above average options first.</para>
			/// </summary>
			/// <param name="results">The results to process.</param>
			/// <returns>An enumerable of results.</returns>
			public static IEnumerable<AppraisalResult> FindBiasedResults(IEnumerable<AppraisalResult> results)
			{
				var options = results
					.Where(ValidForObserver)
					.Where(ValidForTarget)
					.ToList();

				// The `Average` operation will fail if empty.
				if (options.Count == 0) yield break;

				// Split them into above and below average chunks.
				var averageWeight = options.Select(GetWeight).Average();
				var splitOptions = options.ToLookup((r) => GetWeight(r) >= averageWeight);

				// First, prefer the above-average options.
				var aboveAverage = splitOptions[true].ToHashSet();
				while (aboveAverage.Count > 0)
				{
					var nextOption = aboveAverage.RandomElementByWeight(GetWeight);
					aboveAverage.Remove(nextOption);
					yield return nextOption;
				}

				// If we get here, the consumer still wants more options, so fall back
				// to the below-average options.
				var belowAverage = splitOptions[false].ToHashSet();
				while (belowAverage.Count > 0)
				{
					var nextOption = belowAverage.RandomElementByWeight(GetWeight);
					belowAverage.Remove(nextOption);
					yield return nextOption;
				}
			}

			/// <summary>
			/// Unseeded implementation of <see cref="SexAppraiser.FindPartners" />.
			/// </summary>
			public static IEnumerable<AppraisalResult> FindPartners(
				Pawn observer,
				IEnumerable<Pawn> candidates)
			{
				// Use the defaults `ForFucking` for observer and candidates.
				var appraisalParams = new AppraisalParams(observer, candidates);

				return FindBiasedResults(CandidatesAsTargets(in appraisalParams));
			}

			/// <summary>
			/// Unseeded implementation of <see cref="SexAppraiser.FindTricks" />.
			/// </summary>
			public static IEnumerable<AppraisalResult> FindTricks(
				Pawn whore,
				IEnumerable<Pawn> candidates)
			{
				var appraisalParams = new AppraisalParams(whore, candidates)
				{
					// Ignore the service worker's relationships so it doesn't affect
					// their choices.
					ObserverSettings = new(
						Purpose: AttractionPurpose.ForFucking,
						PreQuirkModifier: PreferenceCustomizers.IgnoreRelationships
					),
					// Candidates will continue to use the usual `ForFucking` preferences.
				};

				return FindBiasedResults(CandidatesAsTargets(in appraisalParams));
			}

			/// <summary>
			/// Unseeded implementation of <see cref="SexAppraiser.FindWhores" />.
			/// </summary>
			public static IEnumerable<AppraisalResult> FindWhores(
				Pawn client,
				IEnumerable<Pawn> candidates)
			{
				var appraisalParams = new AppraisalParams(client, candidates)
				{
					// Ignore the service worker's relationships so it doesn't affect
					// their choices.
					ObserverSettings = new(
						Purpose: AttractionPurpose.ForFucking,
						PreQuirkModifier: PreferenceCustomizers.IgnoreRelationships
					),
					// Candidates will continue to use the usual `ForFucking` preferences.
				};

				return FindBiasedResults(CandidatesAsObservers(in appraisalParams));
			}

			/// <summary>
			/// Unseeded implementation of <see cref="SexAppraiser.FindVictims" />.
			/// </summary>
			public static IEnumerable<AppraisalResult> FindVictims(
				Pawn rapist,
				IEnumerable<Pawn> candidates,
				bool ignoreBleeding = false,
				bool ignoreOrienation = false)
			{
				var appraisalParams = new AppraisalParams(rapist, candidates)
				{
					// Rapist gotta rape.
					ObserverSettings = new(
						Purpose: AttractionPurpose.ForRape,
						PreQuirkModifier: ignoreOrienation ? PreferenceCustomizers.IgnoreOrientation : null,
						IgnoreBleeding: ignoreBleeding
					),
					// The attraction of the victim is not utilized for selection, however
					// the consumer can still request an attraction for the victim.  In this
					// case, we'll use the `General` purpose.
					TargetSettings = new(AttractionPurpose.General)
				};

				return FindBiasedResults(CandidatesAsTargets(in appraisalParams));
			}

			/// <summary>
			/// Unseeded implementation of <see cref="SexAppraiser.FindRapists" />.
			/// </summary>
			public static IEnumerable<AppraisalResult> FindRapists(
				Pawn victim,
				IEnumerable<Pawn> candidates,
				bool ignoreBleeding = false,
				bool ignoreOrienation = false)
			{
				var appraisalParams = new AppraisalParams(victim, candidates)
				{
					// Rapist gotta rape.
					ObserverSettings = new(
						Purpose: AttractionPurpose.ForRape,
						PreQuirkModifier: ignoreOrienation ? PreferenceCustomizers.IgnoreOrientation : null,
						IgnoreBleeding: ignoreBleeding
					),
					// The attraction of the victim is not utilized for selection, however
					// the consumer can still request an attraction for the victim.  In this
					// case, we'll use the `General` purpose.
					TargetSettings = new(AttractionPurpose.General)
				};

				return FindBiasedResults(CandidatesAsObservers(in appraisalParams));
			}
		}
	}
}
