#nullable enable

using Psychology;
using SyrTraits;
using Verse;
using RimWorld;
using Multiplayer.API;
using System;
using System.Collections.Generic;

namespace rjw
{
	public sealed class CompRJW : ThingComp
	{
		/// <summary>
		/// Core comp for genitalia and sexuality tracking.
		/// </summary>
		public CompRJW() { }

		public Orientation orientation = Orientation.None;
		public int NextHookupTick;
		private bool BootStrapTriggered = false;
		public Need_Sex? sexNeed;
		public bool drawNude = false;
		public bool keep_hat_on = false;
		private PawnData? pawnData = null;
		public PawnData PawnData => pawnData!;
		/// <summary>
		/// <para>Gets the pawn for this comp.</para>
		/// <para>May return null if the comp's parent is somehow not a pawn.</para>
		/// </summary>
		private Pawn? pawn;

		/// <summary>
		/// Initialize is called after parent initialized as a ThingWithComps but before it is initialized as a Pawn.
		/// Expect most Pawn to be null/empty.
		/// </summary>
		public override void Initialize(CompProperties props)
		{
			base.Initialize(props);
			pawn = parent as Pawn;

			if (pawn == null)
			{
				ModLog.Error($"Tried to initialize {nameof(CompRJW)} for the {parent}. {parent} is not a pawn");
				return;
			}
		}

		public override void CompTickInterval(int delta)
		{
			base.CompTickInterval(delta);

			if (pawn == null) return;
			
			if (pawn.IsHashIntervalTick(1500, delta) && !pawn.health.Dead)
			// The `NeedInterval` function is called every 150 ticks but was formerly
			// rate limited by a factor of 10.  This will make the upkeep execute at
			// the same interval.
			{
				if (sexNeed == null) 
				{
					sexNeed = pawn.needs?.TryGetNeed<Need_Sex>();
				}
				if (sexNeed != null)
				{
					DoUpkeep(pawn, sexNeed);
				}
			}
		}

		/// <summary>
		/// Performs some upkeep tasks originally handled in `Need_Sex.NeedInterval`.
		/// </summary>
		private void DoUpkeep(Pawn pawn, Need_Sex sexNeed)
		{
			if (pawn.Map == null) return;
			if (xxx.is_asexual(pawn)) return;

			var curLevel = sexNeed.CurLevel;

			// Update psyfocus if the pawn is awake.
			if (!RJWSettings.Disable_MeditationFocusDrain && pawn.Awake() && curLevel < sexNeed.thresh_frustrated())
				SexUtility.OffsetPsyfocus(pawn, -0.01f);
			//if (curLevel < sexNeed.thresh_horny())
			//	SexUtility.OffsetPsyfocus(pawn, -0.01f);
			//if (curLevel < sexNeed.thresh_frustrated() || curLevel > sexNeed.thresh_ahegao())
			//	SexUtility.OffsetPsyfocus(pawn, -0.05f);

			if (curLevel < sexNeed.thresh_horny() && (pawn.mindState.canLovinTick - Find.TickManager.TicksGame > 300))
				pawn.mindState.canLovinTick = Find.TickManager.TicksGame + 300;

			// This can probably all just go, since it's now a noop.
			// But I don't know if some mod is patching `xxx.bootstrap` or something.
			if (!BootStrapTriggered)
			{
				//--ModLog.Message("CompRJW::DoUpkeep::calling boostrap - pawn is " + xxx.get_pawnname(pawn));
				xxx.bootstrap(pawn.Map);
				BootStrapTriggered = true;
			}
			if (drawNude) {
				bool isFucking = pawn.jobs?.curDriver is JobDriver_Sex;
				if(!isFucking)
				{
					drawNude = false;
					keep_hat_on = false;
				}
			}
			
		}

		public override void PostExposeData()
		{
			base.PostExposeData();

			if (Scribe.mode == LoadSaveMode.Saving)
			{
				if (pawn == null) return;
			}
			else if (pawn == null && parent is Pawn parentPawn)
			{
				// Loading is done in several passes. Just grab the parent on the first pass when it becomes available.
				pawn = parentPawn;
			}

			// Saves/loads the data.
			Scribe_Values.Look(ref orientation, "RJW_Orientation");
			Scribe_Values.Look(ref NextHookupTick, "RJW_NextHookupTick");
			Scribe_Values.Look(ref drawNude, "RJW_drawNude");
			Scribe_Values.Look(ref keep_hat_on, "RJW_keep_hat_on");
			Scribe_Deep.Look(ref pawnData, "pawnData");
			//Log.Message("PostExposeData for " + pawn?.Name);

			if (Scribe.mode == LoadSaveMode.PostLoadInit && pawnData == null)
			{
				// RJW was added mid-save
				Sexualize();
			}
		}

		public override IEnumerable<Gizmo> CompGetGizmosExtra()
		{
			if (pawn == null)
			{
				yield break;
			}

			if (ModsConfig.RoyaltyActive)
			{
				if (pawn.jobs?.curDriver is JobDriver_Sex)
					yield return new SexGizmo(pawn) { order = 6 };
			}

			if (RJWSettings.show_RJW_designation_box && Find.Selector.NumSelected == 1)
			{
				// Compact button group containing rjw designations on pawn
				if (pawn.Faction == Faction.OfPlayer || pawn.IsPrisonerOfColony)
					yield return new RJWdesignations(pawn) { order = 6 };
			}

			if (RJWSettings.submit_button_enabled &&
				pawn.IsColonistPlayerControlled &&
				pawn.Drafted &&
				pawn.CanChangeDesignationColonist())
			{
				if (!(pawn.kindDef.race.defName.Contains("Droid") && !AndroidsCompatibility.IsAndroid(pawn)))
					yield return new SubmitGizmo(pawn) { order = 6 };
			}
		}

		/// <summary>
		/// Get the orientation of a <c>pawn</c> based on the kinsey rating added by Psychology
		/// </summary>
		/// <param name="pawn"></param>
		/// <returns></returns>
		public static Orientation? GetPsychologyOrientation(Pawn pawn)
		{
			CompRJW? compRJW = pawn.GetCompRJW();

			if (compRJW == null)
			{
				ModLog.Error($"CopyPsychologySexuality was called for {pawn}, pawn without the CompRJW");
				return null;
			}

			try
			{
				int kinsey = pawn.TryGetComp<CompPsychology>().Sexuality.kinseyRating;

				if (!Genital_Helper.has_genitals(pawn) && (pawn.kindDef.race.defName.ToLower().Contains("droid") || pawn.kindDef.race.defName.ToLower().Contains("drone")))
					return Orientation.Asexual;
				else if (kinsey == 0)
					return Orientation.Heterosexual;
				else if (kinsey == 1)
					return Orientation.MostlyHeterosexual;
				else if (kinsey == 2)
					return Orientation.LeaningHeterosexual;
				else if (kinsey == 3)
					return Orientation.Bisexual;
				else if (kinsey == 4)
					return Orientation.LeaningHomosexual;
				else if (kinsey == 5)
					return Orientation.MostlyHomosexual;
				else if (kinsey == 6)
					return Orientation.Homosexual;
				else
				{
					if (RJWSettings.DevMode) ModLog.Warning($"Unknown kinsey scale value {kinsey} on pawn {xxx.get_pawnname(pawn)}");
					return Orientation.Asexual;
				}
			}
			catch
			{
				if (!pawn.IsAnimal())
					ModLog.Warning("CopyPsychologySexuality " + pawn?.Name + ", def: " + pawn?.def?.defName + ", kindDef: " + pawn?.kindDef?.race.defName);
			}
			return null;
		}

		/// <summary>
		/// Get the orientation of a <c>pawn</c> based on the new sexuality traits added by Individuality
		/// </summary>
		/// <param name="pawn"></param>
		/// <returns></returns>
		public static Orientation? GetIndividualityOrientation(Pawn pawn)
		{
			CompRJW? compRJW = pawn.GetCompRJW();

			if (compRJW == null)
			{
				ModLog.Error($"CopyIndividualitySexuality was called for {pawn}, pawn without the CompRJW");
				return null;
			}

			try
			{
				CompIndividuality.Sexuality individualitySexuality = pawn.TryGetComp<CompIndividuality>().sexuality;

				if (individualitySexuality == CompIndividuality.Sexuality.Asexual)
					return Orientation.Asexual;
				else if (!Genital_Helper.has_genitals(pawn) && (pawn.kindDef.race.defName.ToLower().Contains("droid") || pawn.kindDef.race.defName.ToLower().Contains("drone")))
					return Orientation.Asexual;
				else if (individualitySexuality == CompIndividuality.Sexuality.Straight)
					return Orientation.Heterosexual;
				else if (individualitySexuality == CompIndividuality.Sexuality.Bisexual)
					return Orientation.Bisexual;
				else if (individualitySexuality == CompIndividuality.Sexuality.Gay)
					return Orientation.Homosexual;
				else
				{
					if (RJWSettings.DevMode) ModLog.Warning($"Pawn {xxx.get_pawnname(pawn)} had an odd sexuality: {individualitySexuality}");
					return Orientation.Asexual;
				}
			}
			catch
			{
				if (!pawn.IsAnimal())
					ModLog.Warning("CopyIndividualitySexuality " + pawn?.Name + ", def: " + pawn?.def?.defName + ", kindDef: " + pawn?.kindDef?.race.defName);
			}
			return null;
		}

		/// <summary>
		/// Get the orientation of a <c>pawn</c> based on vanilla traits
		/// </summary>
		/// <param name="pawn"></param>
		/// <param name="fallback">What orientation to return when no relevant traits are present</param>
		/// <returns></returns>
		public static Orientation? GetVanillaOrientation(Pawn pawn, Orientation? fallback = Orientation.Heterosexual)
		{
			CompRJW? compRJW = pawn.GetCompRJW();

			if (compRJW == null)
			{
				ModLog.Error($"VanillaTraitCheck was called for {pawn}, pawn without the CompRJW");
				return null;
			}

			if (pawn.story.traits.HasTrait(TraitDefOf.Asexual))
				return Orientation.Asexual;
			else if (!Genital_Helper.has_genitals(pawn) && (pawn.kindDef.race.defName.ToLower().Contains("droid") || pawn.kindDef.race.defName.ToLower().Contains("drone")))
				return Orientation.Asexual;
			else if (pawn.story.traits.HasTrait(TraitDefOf.Gay))
				return Orientation.Homosexual;
			else if (pawn.story.traits.HasTrait(TraitDefOf.Bisexual))
				return Orientation.Bisexual;
			return fallback;
		}

		/// <summary>
		/// Recalculates and applies the <c>pawn</c>'s sexual orientation based on the selected sexuality distribution source.
		/// </summary>
		/// <remarks>
		/// This is typically invoked when the <c>pawn</c>'s traits change (for example, when the pawn acquires the Gay trait).
		/// </remarks>
		/// <param name="pawn">The pawn whose orientation should be updated.</param>
		public static void UpdateOrientation(Pawn pawn)
		{
			CompRJW? compRJW = pawn.GetCompRJW();
			if (compRJW == null)
			{
				ModLog.Error($"Update_Orientation was called for {pawn}, pawn without the CompRJW");
				return;
			}

			Orientation? orientation = null;
			switch (RJWPreferenceSettings.sexuality_distribution)
			{
				case RJWPreferenceSettings.Rjw_sexuality.Vanilla:
					orientation = GetVanillaOrientation(pawn);
					break;
				case RJWPreferenceSettings.Rjw_sexuality.Psychology:
					orientation = GetPsychologyOrientation(pawn);
					break;
				case RJWPreferenceSettings.Rjw_sexuality.SYRIndividuality:
					orientation = GetIndividualityOrientation(pawn);
					break;
				case RJWPreferenceSettings.Rjw_sexuality.RimJobWorld:
					// Start by getting trait sexuality
					orientation = GetVanillaOrientation(pawn, fallback: null);
					// If orientation not yet set (e.g. upon creation of the pawn) and no relevant traits found, roll
					if (compRJW.orientation == Orientation.None && orientation == null)
					{
						if (RJWSettings.DevMode) ModLog.Message($"[RJWdist: {xxx.get_pawnname(pawn)}] Rolling new orientation");
						orientation = compRJW.RollOrientation(pawn);
					}
					// If orientation was already set before and agrees with traits, preserve it
					else if (compRJW.orientation != Orientation.None && orientation != null && orientation == compRJW.orientation)
					{
						if (RJWSettings.DevMode) ModLog.Message($"[RJWdist: {xxx.get_pawnname(pawn)}] Preserving orientation which agrees with traits");
						return;
					}
					// If orientation was already set before and there are no relevant traits, preserve it
					// Otherwise we reroll on every call for pawns with no relevant traits
					else if (compRJW.orientation != Orientation.None && orientation == null)
					{
						if (RJWSettings.DevMode) ModLog.Message($"[RJWdist: {xxx.get_pawnname(pawn)}] No relevant traits and orientation already set, skipping orientation update");
						return;
					}
					// If orientation was already set before and the traits do not agree, update to vanilla orientation
					else if (compRJW.orientation != Orientation.None && orientation != null && orientation != compRJW.orientation)
					{
						if (RJWSettings.DevMode) ModLog.Message($"[RJWdist: {xxx.get_pawnname(pawn)}] Traits disagree with orientation, updating for consistency");
					}
					break;
				default:
					if (RJWSettings.DevMode) ModLog.Error($"Unknown sexuality source. Orientation was not updated.");
					break;
			}
			if (orientation == null)
			{
				if (RJWSettings.DevMode) ModLog.Error($"Unable to update orientation for {xxx.get_pawnname(pawn)}");
				return;
			}
			compRJW.orientation = (Orientation)orientation;
		}

		/// <summary>
		/// The main method for adding genitalia and orientation.
		/// </summary>
		/// <param name="reroll"></param>
		public void Sexualize(bool reroll = false)
		{
			if (pawn == null)
			{
				ModLog.Error($"Tried to sexualize {parent}, it is not a pawn");
				return;
			}

			// Can't be in the Initialize or PostPostMake because pawn is not fully initialized at that stage
			pawnData ??= new PawnData(pawn);

			if (pawnData.sexualized && !reroll)
			{
				return;
			}

			if (reroll)
			{
				orientation = Orientation.None;
			}
			else if (orientation != Orientation.None)
			{
				return;
			}

			//Log.Message("Sexualizing pawn " + pawn?.Name + ", def: " + pawn?.def?.defName);

			if (!reroll)
				Sexualizer.sexualize_pawn(pawn);

			if (xxx.is_animal(pawn))
			{
				orientation = RollAnimalOrientation(pawn);
				//ModLog.Message($"Orientation for {pawn} is {orientation}, has_genitals = {Genital_Helper.has_genitals(pawn)}");
			}
			else if (xxx.has_traits(pawn) && Genital_Helper.has_genitals(pawn) && !(pawn.kindDef.race.defName.ToLower().Contains("droid") && !AndroidsCompatibility.IsAndroid(pawn)))
			{
				UpdateOrientation(pawn);
			}
			else if ((pawn.kindDef.race.defName.ToLower().Contains("droid") && !AndroidsCompatibility.IsAndroid(pawn)) || !Genital_Helper.has_genitals(pawn))
			{
				// Droids with no genitalia are set as asexual.
				// If player later adds genitalia to the droid, the droid 'sexuality' gets rerolled.
				orientation = Orientation.Asexual;
			}
			pawnData.sexualized = true;
		}

		[SyncMethod]
		public Orientation RollOrientation(Pawn pawn)
		{
			//Rand.PopState();
			//Rand.PushState(RJW_Multiplayer.PredictableSeed());
			float random = Rand.Range(0f, 1f);
			float checkpoint = RJWPreferenceSettings.asexual_ratio / RJWPreferenceSettings.GetTotal();

			float checkpoint_pan = checkpoint + (RJWPreferenceSettings.pansexual_ratio / RJWPreferenceSettings.GetTotal());
			float checkpoint_het = checkpoint_pan + (RJWPreferenceSettings.heterosexual_ratio / RJWPreferenceSettings.GetTotal());
			float checkpoint_bi = checkpoint_het + (RJWPreferenceSettings.bisexual_ratio / RJWPreferenceSettings.GetTotal());
			float checkpoint_gay = checkpoint_bi + (RJWPreferenceSettings.homosexual_ratio / RJWPreferenceSettings.GetTotal());

			if (random < checkpoint || !Genital_Helper.has_genitals(pawn))
				return Orientation.Asexual;
			else if (random < checkpoint_pan)
				return Orientation.Pansexual;
			else if (random < checkpoint_het)
				return Orientation.Heterosexual;
			else if (random < checkpoint_het + ((checkpoint_bi - checkpoint_het) * 0.33f))
				return Orientation.MostlyHeterosexual;
			else if (random < checkpoint_het + ((checkpoint_bi - checkpoint_het) * 0.66f))
				return Orientation.LeaningHeterosexual;
			else if (random < checkpoint_bi)
				return Orientation.Bisexual;
			else if (random < checkpoint_bi + ((checkpoint_gay - checkpoint_bi) * 0.33f))
				return Orientation.LeaningHomosexual;
			else if (random < checkpoint_bi + ((checkpoint_gay - checkpoint_bi) * 0.66f))
				return Orientation.MostlyHomosexual;
			else
				return Orientation.Homosexual;
		}

		// Simpler system for animals, with most of them being heterosexual.
		// Don't want to disturb player breeding projects by adding too many gay animals.
		[SyncMethod]
		public Orientation RollAnimalOrientation(Pawn pawn)
		{
			//Rand.PopState();
			//Rand.PushState(RJW_Multiplayer.PredictableSeed());
			float random = Rand.Range(0f, 1f);

			if (random < 0.03f || !Genital_Helper.has_genitals(pawn))
				return Orientation.Asexual;
			else if (random < 0.85f)
				return Orientation.Heterosexual;
			else if (random < 0.96f)
				return Orientation.Bisexual;
			else
				return Orientation.Homosexual;
		}
	}
}
