using System.Linq;
using Verse;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// Handles the genes of a target.  Requires biotech to be enabled.
	/// </summary>
	public sealed class P_HumanToGenes : AttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Requires biotech.
			if (!ModsConfig.BiotechActive) return;
			// Requires the observer to have genes.
			if (request.Pawn.genes is null) return;
			// Not applicable to animal targets.
			if (request.Category.AnimalTarget()) return;

			request.SetPreference(new P_HumanToGenes());
		}

		private P_HumanToGenes() : base(
			AttractionMode.Physical,
			nameof(P_HumanToGenes),
			FactorOperation.Multiply)
		{ }

		protected override float GetOperand(ref AttractionRequest request, float factor)
		{
			var pawn = request.Pawn;
			var target = request.Target;
			var operand = 1f;

			if (!ModsConfig.BiotechActive) return operand;
			if (target.genes is not { } genes) return operand;

			// Having some genes can make you prefer others with those genes.
			// The `SecondaryRomanceChanceFactor` exempts kind-hearted pawns from
			// taking a hit from lacking genes.  Mods could make these boosts
			// instead, so we'll take those, when possible.
			var onlyBoosts = xxx.is_kind(pawn) || request.HasLoveRelationWithTarget;

			foreach (var gene in genes.GenesListForReading.Where(gene => gene.Active))
				if (gene.def.missingGeneRomanceChanceFactor != 1f)
					if (!onlyBoosts || gene.def.missingGeneRomanceChanceFactor > 1f)
						if (pawn.genes?.HasActiveGene(gene.def) is not true)
							operand *= gene.def.missingGeneRomanceChanceFactor;

			return operand;
		}
	}
}