using UnityEngine;
using Verse;
using RimWorld;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// <para>Handles an animal (or wildman) observing another, prefering
	/// similar wildness to themselves.</para>
	/// <para>Similar to `S_HumanToWildness`, the target's training reduces
	/// the wildness perceived by the observing animal, though not as much
	/// as with humans.</para>
	/// </summary>
	public sealed class S_AnimalToWildness : CurveAttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Observer must be an animal or wildman.
			if (!request.Pawn.AnimalOrWildMan()) return;
			// Target must be an animal or wildman.
			if (!request.Target.AnimalOrWildMan()) return;
			// Observer must not be tame.
			if (request.Pawn.IsTameAnimal()) return;
			// Target must not be dead.
			if (request.Target.Dead) return;

			request.SetPreference(new S_AnimalToWildness());
		}

		/// <summary>
		/// Points of a curve that maps the difference in wildness to a scalar
		/// to apply. This uses the absolute value difference.
		/// </summary>
		private static readonly CurvePoint[] wildnessPoints = new CurvePoint[]
		{
			new(0f, 1f),
			new(0.1f, 1f),
			new(0.2f, 0.9f),
			new(0.4f, 0.7f)
		};

		private S_AnimalToWildness() : base(
			AttractionMode.Social,
			nameof(S_AnimalToWildness),
			FactorOperation.Multiply,
			GetWildnessDifference,
			wildnessPoints)
		{ }

		private static float GetWildnessDifference(ref AttractionRequest request) =>
			Mathf.Abs(GetWildness(request.Pawn) - GetWildness(request.Target));

		private static float GetWildness(Pawn target)
		{
			// Wildmen have a fixed wildness.
			if (target.IsWildMan()) return AttractionUtility.WildManWildness;

			var baseWildness = target.GetStatValue(StatDefOf.Wildness);
			if (baseWildness == 0f) return 0f;

			// Reduce wildness for each trainable.
			var trainingModifier = 0.1f * TrainingCount(target);
			return Mathf.Clamp01(baseWildness - trainingModifier);
		}

		private static int TrainingCount(Pawn target)
		{
			if (target.training is not { } training) return 0;
			if (!target.IsTameAnimal()) return 0;

			// Tamed animals will always have at least the "tamedness" trainable,
			// so are guaranteed at least 1 from this.
			var count = 0;
			foreach (var def in DefDatabase<TrainableDef>.AllDefsListForReading)
				if (training.HasLearned(def))
					count += 1;
			return count;
		}
	}
}