using UnityEngine;
using Verse;
using RimWorld;

namespace rjw.Modules.Attraction.StandardPreferences
{
	/// <summary>
	/// <para>Handles a human observing wildness in a pawn.</para>
	/// <para>This is a penalty applied based on the pawn's `SexWildnessAversion`
	/// stat, which reduces the penalty linearly as this stat is reduced toward
	/// zero.</para>
	/// <para>Training an animal will also help to reduce this penalty.</para>
	/// </summary>
	public sealed class S_HumanToWildness : AttractionPreference
	{
		[StandardPreference]
		public static void ApplyTo(ref AttractionRequest request)
		{
			// Observer must not be an animal or wildman.
			if (request.Pawn.AnimalOrWildMan()) return;
			// Target must be an animal or wildman.
			if (!request.Target.AnimalOrWildMan()) return;

			request.SetPreference(new S_HumanToWildness());
		}

		private S_HumanToWildness() : base(
			AttractionMode.Social,
			nameof(S_HumanToWildness),
			FactorOperation.Multiply)
		{ }

		protected override float GetOperand(ref AttractionRequest request, float factor)
		{
			var target = request.Target;
			var wildness = GetWildness(target);
			if (wildness == 0f) return 1f;
			// Reduce the wildness by the inverse of their aversion.
			var reduction = 1f - request.Pawn.GetStatValueForPawn(RJWStatDefOf.SexWildnessAversion, target);
			// Whatever wildness remains is inverted into our final operand.
			return 1f - Mathf.Clamp01(wildness - reduction);
		}

		private static float GetWildness(Pawn target)
		{
			// Dead targets obviously can't behave wild.
			if (target.Dead) return 0f;
			// Wildmen have a fixed wildness.
			if (target.IsWildMan()) return AttractionUtility.WildManWildness;

			var baseWildness = target.GetStatValue(StatDefOf.Wildness);
			if (baseWildness == 0f) return 0f;

			// Reduce wildness for each trainable.
			var trainingBonus = 0.2f * TrainingCount(target);
			return Mathf.Clamp01(baseWildness - trainingBonus);
		}

		private static int TrainingCount(Pawn target)
		{
			if (target.training is not { } training) return 0;
			if (!target.IsTameAnimal()) return 0;

			// Tamed animals will always have at least the "tamedness" trainable,
			// so are guaranteed at least 1 from this.
			var count = 0;
			foreach (var def in DefDatabase<TrainableDef>.AllDefsListForReading)
				if (training.HasLearned(def))
					count += 1;
			return count;
		}
	}
}