#nullable enable

using System;
using RimWorld;
using Verse;
using static rjw.Genital_Helper;
using static rjw.GenderHelper;

namespace rjw
{
	/// <summary>
	/// <para>Affects stats based on whether the pawn can only be penetrated by
	/// the other pawn in the stat request.</para>
	/// <para>This requires getting the stat with `pawn.GetStatValueForPawn`,
	/// so the other pawn can be provided.</para>
	/// <para>Although there is an explanation part, I don't believe it can
	/// ever be shown in the information panel under normal circumstances,
	/// since another pawn is required for this to ever activate and the
	/// information panel has no other pawn to provide.</para>
	/// </summary>
	public sealed class StatPart_WhenOnlyPenetrable : StatPart
	{
		public enum ValueMode { Factor, Offset }

		/// <summary>
		/// Whether to apply the values as factors or offsets.
		/// </summary>
		public ValueMode mode = ValueMode.Factor;

		/// <summary>
		/// The value to use when only penetrable.
		/// </summary>
		public float value = 0f;

		public override void TransformValue(StatRequest req, ref float val)
		{
			if (!req.HasThing) return;
			if (req.Thing is not Pawn pawn) return;
			if (req.Pawn is not Pawn otherPawn) return;
			if (!IsOnlyPenetrable(pawn, otherPawn)) return;

			val = mode switch
			{
				ValueMode.Factor => val * value,
				ValueMode.Offset => val + value,
				_ => throw new InvalidOperationException("bad value for `mode`")
			};
		}

		public override string? ExplanationPart(StatRequest req)
		{
			if (!req.HasThing) return null;
			if (req.Thing is not Pawn pawn) return null;
			if (req.Pawn is not Pawn otherPawn) return null;
			if (!IsOnlyPenetrable(pawn, otherPawn)) return null;

			// These do not affect the value, so no explanation needed.
			if (mode == ValueMode.Factor && value == 1f) return null;
			if (mode == ValueMode.Offset && value == 0f) return null;

			var numSense = mode switch
			{
				ValueMode.Factor => ToStringNumberSense.Factor,
				ValueMode.Offset => ToStringNumberSense.Offset,
				_ => throw new InvalidOperationException("bad value for `mode`")
			};

			var label = "StatPart_WhenOnlyPenetrable".Translate(otherPawn.LabelIndefinite());
			var text = value.ToStringByStyle(ToStringStyle.FloatMaxThree, numSense);
			return $"{label}: {text}";
		}

		private static bool IsOnlyPenetrable(Pawn pawn, Pawn otherPawn) =>
			(GetSex(otherPawn), GetSex(pawn)) switch
			{
				(Sex.Male or Sex.Trap or Sex.Futa, Sex.Female) => true,
				(Sex.Male or Sex.Trap or Sex.Futa, Sex.None) when has_anus(pawn) => true,
				_ => false
			};
	}
}