#!/usr/bin/env python3
# Script to generate icons.h header file for esp32-weather-epd.
# Copyright (C) 2025  Luke Marzen
#
# This program is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program.  If not, see <https://www.gnu.org/licenses/>.
import os

def generate_icons_h(directory):
    icons_h_content = []
    
    # Start the header file
    icons_h_content.append('// DO NOT MODIFY -- THIS FILE WAS GENERATED BY `python3 final_generate_icons_h.py`')
    icons_h_content.append('')
    icons_h_content.append('#ifndef __ICONS_H__')
    icons_h_content.append('#define __ICONS_H__')
    icons_h_content.append('')
    icons_h_content.append('#include <cstddef>')
    icons_h_content.append('')

    sizes = []
    icon_names = []

    # Loop through all directories in 'icons'
    for size_dir in sorted(os.listdir(directory)):
        size_path = os.path.join(directory, size_dir)
        if os.path.isdir(size_path):
            size = size_dir.split('x')[0]  # Assume format is SIZExSIZE (e.g., 16x16)
            sizes.append(int(size))

    for size_dir in sorted(os.listdir(directory)):
        size_path = os.path.join(directory, size_dir)
        if not os.path.isdir(size_path):
            continue

        size = size_dir.split('x')[0]  # Assume format is SIZExSIZE (e.g., 16x16)
        icons = sorted(os.listdir(size_path))
        for icon in icons:
            if icon.endswith(f"_{size}x{size}.h"):
                icon_name = icon.split(f"_{size}x{size}.h")[0]
                icon_names.append(icon_name)
        break

    sizes.sort()

    for size in sizes:
        icons_h_content.append(f'#include "icons_{size}x{size}.h"')
    icons_h_content.append('')

    # make enums
    icons_h_content.append('typedef enum icon_name {')
    for icon_name in icon_names:
        icons_h_content.append(f'  {icon_name},')
    icons_h_content.append('} icon_name_t;')
    icons_h_content.append('')

    # define the constexpr function
    icons_h_content.append('constexpr const unsigned char* getBitmap(icon_name_t icon, size_t size)')
    icons_h_content.append('{')
    icons_h_content.append('  switch (icon) {')
    for icon_name in icon_names:
        icons_h_content.append(f'  case {icon_name}:')
        icons_h_content.append('    switch (size) {')
        for size in sizes:
            icons_h_content.append(f'    case {size}: return {icon_name}_{size}x{size};')
        icons_h_content.append('    default:')
        icons_h_content.append('      return nullptr;')
        icons_h_content.append('    }')
    
    # Close the function and the header guard
    icons_h_content.append('  default:')
    icons_h_content.append('    return nullptr;')
    icons_h_content.append('  }')
    icons_h_content.append('}')
    icons_h_content.append('')
    icons_h_content.append('#endif')
    
    # Write the result to the icons.h file
    with open("icons/icons.h", "w") as f:
        f.write("\n".join(icons_h_content))
    print("icons.h file has been generated!")


generate_icons_h('icons')
