## 斜杠命令

> **🔥 专业提示**：将自定义命令与钩子结合以实现终极自动化。创建 `/deploy` 命令来触发安全钩子 + 后台构建。

[↑ 返回顶部](#快速导航)

### 内置斜杠命令
Claude Code 提供丰富的内置命令：

```bash
# 核心命令
/clear          # 清除对话历史
/help           # 获取使用帮助和可用命令
/review         # 请求代码审查
/model          # 选择或更改AI模型

# 后台进程管理
[注意：这些命令来自公告，尚未在官方文档中]
/bashes         # 列出所有后台进程（需验证）
/bash-output    # 获取后台进程输出（需验证）
/kill-bash      # 终止后台进程（需验证）

# 上下文管理（官方）
/compact        # 压缩对话，可选择焦点
/add-dir        # 添加工作目录到会话
[注意：/microcompact 来自公告，但不在文档中]

# 安全
[注意：为安全审查创建自定义命令]
# 示例：~/.claude/commands/security-review.md

# 自定义（官方）
/statusline     # 自定义终端状态行（已文档化）
/agents         # 管理自定义子代理（已文档化）

# 状态栏示例（新功能）
/statusline "git: $(git branch --show-current)"
/statusline "📍 $(pwd) | 🌡️ $(curl -s 'wttr.in?format=%t')"
/statusline "🤖 AI Buddy: Ready to help!"
```

### 自定义斜杠命令
为项目特定工作流创建自己的命令：

```bash
# 项目命令（存储在 .claude/commands/）
# 个人命令（存储在 ~/.claude/commands/）

# 命令结构（Markdown文件）：
# /my-command "argument"
# 使用 $ARGUMENTS 占位符
# 可以执行bash命令
# 可以使用 @ 前缀引用文件
# 支持frontmatter配置
```

### 高级命令功能
```bash
# 命名空间
/project:deploy     # 项目特定的部署命令
/team:review        # 团队工作流命令

# 扩展思考
# 命令可以触发扩展推理

# MCP 集成
# MCP 服务器可以动态暴露额外的斜杠命令
```

**关键理解**：斜杠命令为常见工作流提供快捷方式。内置命令处理核心功能，自定义命令适应您的项目需求。

## 钩子系统

> **🔥 协同力量**：钩子 + 后台任务 + MCP = 完全自动化。示例：Git提交钩子 → 触发后台测试 + 安全扫描 + 部署准备。

[↑ 返回顶部](#快速导航)

### 什么是钩子？
钩子是在 Claude Code 交互期间由特定事件触发的可配置脚本：

### 钩子事件类型
```bash
# 文件操作事件
on_file_read      # 读取文件后触发
on_file_write     # 写入文件后触发
on_file_edit      # 编辑文件后触发

# 命令执行事件
on_command_start  # 命令开始执行前触发
on_command_end    # 命令执行完成后触发

# 会话管理事件
on_session_start  # 会话开始时触发
on_session_end    # 会话结束时触发
on_compact        # 压缩对话时触发

# Git 操作事件
on_git_commit     # Git提交时触发
on_git_push       # Git推送时触发
on_git_pull       # Git拉取时触发

# 错误处理事件
on_error          # 发生错误时触发
on_tool_failure   # 工具失败时触发
```

### 钩子配置
钩子配置存储在项目或用户级别：

```bash
# 项目级钩子（.claude/hooks/）
.claude/hooks/
├── on_git_commit.md
├── on_file_write.md
└── on_error.md

# 用户级钩子（~/.claude/hooks/）
~/.claude/hooks/
├── on_session_start.md
├── on_command_end.md
└── security_checks.md
```

**钩子配置示例**：
```markdown
---
name: "Git Commit 安全检查"
description: "在每次Git提交前运行安全检查"
trigger: "on_git_commit"
---

# Git Commit 安全检查钩子

## 触发条件
- 用户执行 git commit 操作
- Claude Code 准备提交更改

## 执行的检查

### 1. 敏感信息扫描
```bash
# 检查是否包含API密钥、密码等
grep -r "api[_-]key\|password\|secret" --include="*.js" --include="*.py" --include="*.env" .
```

### 2. 代码质量检查
```bash
# 运行linting
npm run lint  # 或项目特定的lint命令

# 检查语法错误
python -m py_compile *.py  # Python项目
```

### 3. 安全漏洞扫描
```bash
# 依赖安全检查
npm audit  # Node.js项目
safety check  # Python项目
```

## 钩子响应
钩子可以返回JSON来控制行为：

```json
{
  "should_proceed": true,
  "messages": [
    "✅ 安全检查通过",
    "✅ 代码质量检查通过",
    "✅ 未发现敏感信息"
  ],
  "actions": [
    "记录审计日志",
    "更新项目状态"
  ]
}
```

## 钩子功能

### 钩子可以做什么：
- **验证检查**：在操作前验证条件
- **自动化任务**：触发构建、测试、部署
- **通知系统**：发送 Slack、电子邮件通知
- **日志记录**：记录操作和事件
- **安全扫描**：检查安全问题和漏洞
- **质量保证**：运行测试和代码检查
- **文档生成**：自动生成文档
- **备份操作**：创建文件和数据的备份

### 常见模式：
```bash
# 开发工作流自动化
git commit → 钩子触发 → 运行测试 → 更新文档 → 通知团队

# 安全监控
文件更改 → 钩子触发 → 安全扫描 → 记录结果 → 阻止或允许

# 部署自动化
代码推送 → 钩子触发 → 构建项目 → 运行测试 → 部署到生产环境

# 质量保证
代码提交 → 钩子触发 → 代码审查 → 性能测试 → 生成报告
```

### 钩子响应
钩子可以返回JSON来控制行为：

```json
{
  "should_proceed": true,           # 是否继续操作
  "messages": ["信息1", "信息2"],    # 显示给用户的消息
  "warnings": ["警告1"],            # 警告消息
  "errors": ["错误1"],              # 错误消息（可能阻止操作）
  "actions": [                      # 要执行的操作
    "run_tests",
    "update_documentation",
    "notify_team"
  ],
  "data": {                         # 传递给后续操作的数据
    "test_results": {...},
    "security_scan": {...}
  }
}
```

**实际钩子示例**：

**安全审查钩子**：
```markdown
---
name: "安全审查钩子"
trigger: "on_file_write"
files: ["*.js", "*.py", "*.php", "*.rb"]
---

# 安全审查钩子

## 检查内容
- SQL注入风险
- XSS漏洞
- 敏感信息暴露
- 不安全的函数使用

## 执行的命令
```bash
# 静态代码分析
semgrep --config=security .

# 依赖检查
npm audit --audit-level moderate

# 敏感信息扫描
git secrets --scan
```

## 响应格式
```json
{
  "should_proceed": true,
  "messages": ["安全审查完成"],
  "security_level": "medium",
  "recommendations": ["修复XSS漏洞", "更新依赖版本"]
}
```
```

**测试自动化钩子**：
```markdown
---
name: "测试自动化钩子"
trigger: "on_git_commit"
---

# 测试自动化钩子

## 触发测试
```bash
# 单元测试
npm test

# 集成测试
npm run test:integration

# 端到端测试
npm run test:e2e
```

## 结果处理
- 所有测试通过 → 允许提交
- 测试失败 → 阻止提交并显示错误
- 覆盖率不足 → 发出警告
```

## MCP 集成与子代理

> **🔥 集成优势**：MCP + 子代理 + 钩子 = 智能自动化生态系统。示例：GitHub MCP + 代码审查代理 + 安全钩子 = 完全自动化的代码质量保证。

[↑ 返回顶部](#快速导航)

### 模型上下文协议（MCP）

**MCP 是什么**：
MCP（Model Context Protocol）是一种标准化协议，允许 Claude Code 与外部工具和服务集成。它提供了一个统一的接口来访问各种数据源、API 和专业服务。

### MCP 启用的功能：
```bash
# 外部数据源集成
GitHub API → 仓库管理、问题跟踪、PR审查
Jira API → 项目管理、任务跟踪
Slack API → 团队沟通、通知
数据库连接 → 直接数据访问和分析

# 专业工具集成
Docker → 容器管理和部署
AWS/Azure/GCP → 云服务管理
Kubernetes → 容器编排
CI/CD 工具 → 持续集成和部署

# 开发工具集成
IDE 插件 → 开发环境集成
调试器 → 高级调试功能
性能分析器 → 代码性能优化
```

### 连接方法：
```bash
# MCP 服务器配置
.claude/mcp_servers/
├── github-mcp.json
├── database-mcp.json
└── slack-mcp.json

# 服务器配置示例
{
  "name": "github-integration",
  "server": "github-mcp",
  "config": {
    "api_token": "your_github_token",
    "repositories": ["owner/repo1", "owner/repo2"]
  }
}
```

### 身份验证：
```bash
# API 密钥管理
.claude/secrets/
├── github_token
├── jira_credentials
└── database_password

# 环境变量
export GITHUB_TOKEN="your_token"
export JIRA_API_KEY="your_key"
```

### 常见 MCP 集成

**流行的集成**：
```bash
# GitHub 集成
- 问题管理和跟踪
- Pull Request 审查
- 仓库分析
- 团队协作

# 数据库集成
- SQL 查询执行
- 数据分析
- 报表生成
- 数据迁移

# 项目管理工具
- Jira 任务创建和更新
- Trello 卡片管理
- Asana 项目同步
- Notion 文档管理

# 通信工具
- Slack 消息发送
- Teams 通知
- 邮件发送
- Discord 集成
```

### 使用示例：
```bash
# GitHub MCP 使用
"创建一个新的issue来跟踪这个bug"
→ Claude 使用 GitHub MCP 创建 issue

"审查所有开放的PR"
→ Claude 使用 GitHub MCP 获取并审查 PR

"部署到生产环境"
→ Claude 使用部署 MCP 触发部署流程

"分析数据库中的用户行为"
→ Claude 使用数据库 MCP 查询和分析数据
```

### 自定义子代理（增强版）

**创建自定义子代理**：
```javascript
// 定义专业代理
const codeReviewer = {
  name: "代码审查专家",
  expertise: ["代码质量", "性能优化", "安全审查"],
  tools: ["Read", "Grep", "Bash", "Task"],
  personality: "注重细节，建设性反馈",
  workflow: [
    "分析代码结构",
    "识别潜在问题",
    "提供改进建议",
    "验证修复方案"
  ]
}

const securityExpert = {
  name: "安全专家",
  expertise: ["漏洞扫描", "安全架构", "合规性"],
  tools: ["Read", "Grep", "Bash", "WebSearch"],
  personality: "谨慎，全面，注重安全最佳实践",
  workflow: [
    "威胁建模",
    "漏洞扫描",
    "风险评估",
    "安全建议"
  ]
}

const dataScientist = {
  name: "数据科学家",
  expertise: ["数据分析", "机器学习", "统计建模"],
  tools: ["Read", "Bash", "Task", "WebSearch"],
  personality: "分析性，数据驱动，注重准确性",
  workflow: [
    "数据探索",
    "统计分析",
    "模型开发",
    "结果验证"
  ]
}
```

### 使用子代理

**直接调用**：
```bash
@code-reviewer 分析这个函数的性能
@security-expert 审查这个身份验证系统
@data-scientist 分析这些销售数据
```

**团队协调**：
```bash
# 多个代理协作完成任务
@code-reviewer @security-expert "审查这个支付系统"
→ 代码审查专家分析代码结构
→ 安全专家评估安全风险
→ 两者协作提供综合反馈

# 顺序处理
@architect "设计系统架构" → 
@developer "实现核心功能" → 
@tester "编写测试用例" → 
@reviewer "最终审查"
```

### 自动代理选择
```javascript
// 基于任务类型自动选择代理
function selectOptimalAgent(task) {
  const taskType = analyzeTaskType(task)
  const complexity = assessComplexity(task)
  const domain = identifyDomain(task)
  
  // 选择规则
  if (domain === "security") return securityExpert
  if (taskType === "code_review") return codeReviewer
  if (domain === "data_science") return dataScientist
  
  // 复杂任务使用多个代理
  if (complexity > 0.8) {
    return [architect, developer, tester]
  }
  
  return generalDeveloper
}
```

### 每个代理的模型选择
```javascript
// 为不同代理选择合适的模型
const agentModelMapping = {
  "code-reviewer": "claude-3.5-sonnet",  // 代码分析
  "security-expert": "claude-3.5-sonnet", // 安全分析
  "data-scientist": "claude-3-opus",      // 复杂分析
  "architect": "claude-3.5-sonnet",       // 系统设计
  "general": "claude-3-haiku"             // 通用任务
}
```

### 顺序管道
```javascript
// 多代理顺序处理流水线
async function agentPipeline(task, agents) {
  let result = task
  const pipelineResults = []
  
  for (const agent of agents) {
    // 每个代理处理前一个代理的结果
    const agentResult = await agent.process(result)
    pipelineResults.push({
      agent: agent.name,
      result: agentResult,
      confidence: agentResult.confidence
    })
    
    result = agentResult.output
  }
  
  return {
    finalResult: result,
    pipeline: pipelineResults,
    overallConfidence: calculateOverallConfidence(pipelineResults)
  }
}
```

### 并行分析
```javascript
// 多代理并行分析
async function parallelAgentAnalysis(task, agents) {
  const promises = agents.map(agent => agent.analyze(task))
  const results = await Promise.all(promises)
  
  // 聚合结果
  const aggregated = aggregateResults(results)
  
  return {
    individualResults: results,
    consensus: aggregated.consensus,
    confidence: aggregated.confidence,
    recommendations: aggregated.recommendations
  }
}
```

### 专业调试
```javascript
// 专业调试代理
const debuggingSpecialist = {
  name: "调试专家",
  expertise: ["问题诊断", "性能分析", "错误修复"],
  approach: {
    systematic: true,
    data_driven: true,
    collaborative: true
  },
  tools: ["Read", "Bash", "Grep", "Task"],
  workflow: [
    "错误日志分析",
    "代码路径跟踪",
    "性能瓶颈识别",
    "修复方案验证"
  ]
}
```

### 安全审查系统（新功能）

### 临时安全审查
```bash
# 请求安全审查
/security-review "审查这个身份验证系统"

# 自动安全检查
"实现用户登录功能" → Claude 自动执行安全审查

# 持续安全监控
git commit → 自动安全扫描 → 报告生成
```

### 检测的常见漏洞
```bash
# OWASP Top 10
- SQL注入 (SQL Injection)
- 跨站脚本攻击 (XSS)
- 失效的访问控制 (Broken Access Control)
- 密码学缺陷 (Cryptographic Failures)
- 不安全的设计 (Insecure Design)
- 安全配置错误 (Security Misconfiguration)
- 脆弱和过时的组件 (Vulnerable Components)
- 身份验证失败 (Identification Failures)
- 软件和数据完整性失败 (Integrity Failures)
- 日志和监控失败 (Logging Failures)

# 其他安全问题
- API安全漏洞
- 敏感信息泄露
- 不安全的依赖项
- 配置错误
- 权限提升
- 业务逻辑漏洞
```

### GitHub Actions 集成
```yaml
# .github/workflows/security.yml
name: Security Scan
on: [push, pull_request]

jobs:
  security-scan:
    runs-on: ubuntu-latest
    steps:
      - uses: actions/checkout@v2
      
      - name: Run Security Scan
        run: |
          # 使用Claude Code进行安全审查
          claude security-review --target .
          
      - name: Generate Report
        run: |
          # 生成安全报告
          claude security-report --format markdown --output security-report.md
          
      - name: Upload Results
        uses: actions/upload-artifact@v2
        with:
          name: security-results
          path: security-report.md
```

### 安全开发工作流
```bash
# 开发阶段
1. 编写代码 → 实时安全检查
2. 本地测试 → 自动漏洞扫描
3. 代码审查 → 安全专家审查

# 部署阶段
4. CI/CD流水线 → 自动化安全测试
5. 部署前检查 → 合规性验证
6. 生产监控 → 持续安全监控

# 响应阶段
7. 漏洞检测 → 自动警报
8. 修复建议 → 自动生成修复方案
9. 验证修复 → 确认问题解决
```

### 持续安全监控
```bash
# 设置安全监控
git watch --pattern "*.js" --security-scan
git watch --pattern "*.py" --vulnerability-check

# Claude 监控文件变更并在检测到风险模式时发出警报
# 自动安全报告生成
# 集成到现有安全工具
```

### 增强文件支持（新功能）

### PDF 支持
```bash
# 直接PDF阅读
Read document.pdf                    # 完整PDF分析
Read research-paper.pdf offset=100 limit=50  # 特定页面

# PDF内容引用
@document.pdf "第3章讨论了机器学习方法"
@manual.pdf "安装步骤在第5页"

# PDF与其他格式结合
"总结这个PDF中的关键发现：@research.pdf"
"根据@manual.pdf中的说明实现这个功能"
```

### 用例
```bash
# 技术文档
API文档、用户手册、技术规范
→ 直接引用和实现

# 研究论文
学术论文、研究报告、白皮书
→ 理论实现和验证

# 商业文档
产品需求、项目计划、合同
→ 需求分析和实现

# 教育材料
教程、课程材料、书籍
→ 学习和技能应用
```

### 智能PDF处理
```javascript
// PDF内容智能分析
function analyzePDF(pdfContent) {
  return {
    structure: extractDocumentStructure(pdfContent),
    keyPoints: identifyKeyPoints(pdfContent),
    technicalContent: extractTechnicalContent(pdfContent),
    diagrams: extractDiagramDescriptions(pdfContent),
    codeExamples: extractCodeExamples(pdfContent),
    references: extractReferences(pdfContent)
  }
}

// 跨文档分析
function crossDocumentAnalysis(documents) {
  return {
    commonThemes: findCommonThemes(documents),
    contradictions: identifyContradictions(documents),
    complementaryInfo: findComplementaryInformation(documents),
    synthesis: createSynthesizedView(documents)
  }
}
```

**关键理解**：MCP 集成和子代理系统将 Claude Code 从代码助手转变为完整的开发平台，能够与外部工具协作并自动化复杂的开发工作流。