import Foundation
import UIKit
import Display
import Vision
import CoreImage
import CoreImage.CIFilterBuiltins
import SwiftSignalKit
import VideoToolbox

private let queue = Queue()

public func classifyImage(_ image: UIImage) -> Signal<[(String, Float)], NoError> {
    if #available(iOS 15.0, *) {
        let image = generateImage(image.size, contextGenerator: { size, context in
            context.setFillColor(UIColor.white.cgColor)
            context.fill(CGRect(origin: .zero, size: size))
            if let cgImage = image.cgImage {
                context.draw(cgImage, in: CGRect(origin: .zero, size: size), byTiling: false)
            }
        }, opaque: true, scale: 1.0)
        guard let cgImage = image?.cgImage else {
            return .single([])
        }
        return Signal { subscriber in
            let handler = VNImageRequestHandler(cgImage: cgImage, options: [:])
            let request = VNClassifyImageRequest { request, error in
                guard let results = request.results as? [VNClassificationObservation] else {
                    subscriber.putNext([])
                    subscriber.putCompletion()
                    return
                }
                
                let filteredResult = results.sorted(by: { $0.confidence > $1.confidence }).filter { $0.confidence > 0.1 }
                subscriber.putNext(filteredResult.map { ($0.identifier, $0.confidence) })
                subscriber.putCompletion()
            }
            try? handler.perform([request])
            return ActionDisposable {
                request.cancel()
            }
        }
        |> runOn(queue)
    } else {
        return .single([])
    }
}

public func emojiForClasses(_ classes: [String]) -> [String] {
    let map = [
        "abacus": "🧮",
        "accordion": "🪗",
        "acorn": "🌰",
        "acrobat": "🤸",
        "agriculture": "🌾",
        "aircraft": "✈️",
        "airplane": "✈️",
        "airport": "🛫",
        "airshow": "✈️",
        "alley": "🛣️",
        "alligator_crocodile": "🐊",
        "almond": "🌰",
        "ambulance": "🚑",
        "amusement_park": "🎡",
        "anchovy": "🐟",
        "angelfish": "🐠",
        "ant": "🐜",
        "antipasti": "🥗",
        "anvil": "⚒️",
        "apartment": "🏢",
        "apple": "🍎",
        "appliance": "🔌",
        "apricot": "🍑",
        "apron": "🧤",
        "aquarium": "🐠",
        "arachnid": "🕷️",
        "arch": "🏛️",
        "archery": "🏹",
        "arena": "🏟️",
        "armchair": "💺",
        "art": "🎨",
        "arthropods": "🐜",
        "artichoke": "🥦",
        "arugula": "🥬",
        "asparagus": "🌱",
        "athletics": "🏃",
        "atm": "🏧",
        "atv": "🚜",
        "auditorium": "🎭",
        "aurora": "🌌",
        "australian_shepherd": "🐶",
        "automobile": "🚗",
        "avocado": "🥑",
        "axe": "🪓",
        "baby": "👶",
        "backgammon": "🎲",
        "backhoe": "🚜",
        "backpack": "🎒",
        "bacon": "🥓",
        "badminton": "🏸",
        "bag": "👜",
        "bagel": "🥯",
        "baked_goods": "🥖",
        "baklava": "🍮",
        "balcony": "🏢",
        "ball": "⚽",
        "ballet": "🩰",
        "ballet_dancer": "💃",
        "ballgames": "🏀",
        "balloon": "🎈",
        "balloon_hotair": "🎈",
        "banana": "🍌",
        "banner": "🏳️",
        "bar": "🍺",
        "barbell": "🏋️",
        "barge": "🚢",
        "barn": "🏚️",
        "barnacle": "🦪",
        "barracuda": "🐟",
        "barrel": "🛢️",
        "baseball": "⚾",
        "baseball_bat": "🏏",
        "baseball_hat": "🧢",
        "basenji": "🐕",
        "basket_container": "🧺",
        "basketball": "🏀",
        "basset": "🐕",
        "bath": "🛁",
        "bathrobe": "🚿",
        "bathroom": "🚽",
        "bathroom_faucet": "🚿",
        "bathroom_room": "🚽",
        "beach": "🏖️",
        "beagle": "🐕",
        "bean": "🌱",
        "beanie": "🧢",
        "bear": "🐻",
        "bed": "🛏️",
        "bedding": "🛏️",
        "bedroom": "🛏️",
        "bee": "🐝",
        "beef": "🍖",
        "beehive": "🐝",
        "beekeeping": "🐝",
        "beer": "🍺",
        "beet": "🍆",
        "begonia": "🌺",
        "bell": "🔔",
        "bell_pepper": "🫑",
        "belltower": "🏰",
        "bellydance": "💃",
        "bench": "🪑",
        "bernese_mountain": "🐕",
        "berry": "🍓",
        "bib": "🍼",
        "bichon": "🐕",
        "bicycle": "🚲",
        "billboards": "🚧",
        "billiards": "🎱",
        "binoculars": "🔭",
        "bird": "🐦",
        "birdhouse": "🏠",
        "birthday_cake": "🎂",
        "biryani": "🍛",
        "biscotti": "🍪",
        "biscuit": "🍪",
        "bison": "🦬",
        "blackberry": "🍇",
        "bleachers": "🏟️",
        "blender": "🧉",
        "blizzard": "❄️",
        "blocks": "🧱",
        "blossom": "🌸",
        "blueberry": "🫐",
        "boar": "🐗",
        "board_game": "🎲",
        "boat": "🚤",
        "boathouse": "🏠",
        "bobcat": "🐆",
        "bodyboard": "🏄‍♂️",
        "bongo_drum": "🥁",
        "bonsai": "🌳",
        "book": "📖",
        "bookshelf": "📚",
        "boot": "👢",
        "bottle": "🍼",
        "bouquet": "💐",
        "bowl": "🍲",
        "bowling": "🎳",
        "bowtie": "🎀",
        "boxing": "🥊",
        "branch": "🌿",
        "brass_music": "🎺",
        "bread": "🍞",
        "breakdancing": "💃",
        "brick": "🧱",
        "brick_oven": "🍕",
        "bride": "👰",
        "bridesmaid": "👰‍♀️",
        "bridge": "🌉",
        "briefcase": "💼",
        "broccoli": "🥦",
        "broom": "🧹",
        "brownie": "🍫",
        "bruschetta": "🥖",
        "bubble_tea": "🧋",
        "bucket": "🪣",
        "building": "🏢",
        "bulldog": "🐶",
        "bulldozer": "🚜",
        "bullfighting": "🐂",
        "bungee": "🪢",
        "burrito": "🌯",
        "bus": "🚌",
        "butter": "🧈",
        "butterfly": "🦋",
        "cabinet": "🗄️",
        "cableway": "🚠",
        "cactus": "🌵",
        "cage": "🚪",
        "cake": "🍰",
        "cake_regular": "🍰",
        "cakestand": "🎂",
        "calendar": "📆",
        "caliper": "📏",
        "camel": "🐫",
        "camera": "📷",
        "camping": "🏕️",
        "candle": "🕯️",
        "candlestick": "🕯️",
        "candy": "🍬",
        "candy_cane": "🍬",
        "candy_other": "🍭",
        "canoe": "🛶",
        "cantaloupe": "🍈",
        "canyon": "🏞️",
        "caprese": "🥗",
        "car": "🚗",
        "caramel": "🍬",
        "cardboard_box": "📦",
        "carnation": "🌺",
        "carnival": "🎡",
        "carousel": "🎠",
        "carrot": "🥕",
        "cart": "🛒",
        "carton": "📦",
        "cashew": "🥜",
        "casino": "🎰",
        "casserole": "🍲",
        "cassette": "📼",
        "castle": "🏰",
        "cat": "🐱",
        "caterpillar": "🐛",
        "cauliflower": "🥦",
        "cave": "🕳️",
        "cd": "💿",
        "celebration": "🎉",
        "celery": "🌿",
        "celestial_body": "🌌",
        "celestial_body_other": "🪐",
        "cellar": "🏠",
        "cello": "🎻",
        "centipede": "🐛",
        "cephalopod": "🦑",
        "cereal": "🥣",
        "ceremony": "🎊",
        "cetacean": "🐳",
        "chainsaw": "🪚",
        "chair": "💺",
        "chair_other": "💺",
        "chairlift": "🚠",
        "chaise": "💺",
        "chalkboard": "🖼️",
        "chameleon": "🦎",
        "chandelier": "💡",
        "chart": "📊",
        "checkbook": "📔",
        "cheerleading": "📣",
        "cheese": "🧀",
        "cheesecake": "🍰",
        "cheetah": "🐆",
        "cherry": "🍒",
        "chess": "♟️",
        "chestnut": "🌰",
        "chewing_gum": "🍬",
        "chihuahua": "🐕",
        "child": "👶",
        "chimney": "🏠",
        "chinchilla": "🐭",
        "chives": "🌿",
        "chocolate": "🍫",
        "chocolate_chip": "🍪",
        "chopsticks": "🥢",
        "christmas_decoration": "🎄",
        "christmas_tree": "🎄",
        "chrysanthemum": "🌼",
        "cigar": "🚬",
        "cigarette": "🚬",
        "cilantro": "🌿",
        "circuit_board": "🔌",
        "circus": "🎪",
        "citrus_fruit": "🍊",
        "cityscape": "🏙️",
        "clam": "🐚",
        "clarinet": "🎼",
        "classroom": "🏫",
        "cliff": "🧗‍♂️",
        "cloak": "🧥",
        "clock": "⏰",
        "clock_tower": "🕰️",
        "closet": "🚪",
        "clothespin": "📌",
        "cloudy": "☁️",
        "clover": "🍀",
        "clown": "🤡",
        "clownfish": "🐠",
        "cockatoo": "🦜",
        "cocktail": "🍹",
        "coconut": "🥥",
        "coffee": "☕",
        "coffee_bean": "☕",
        "coin": "💰",
        "coleslaw": "🥗",
        "collie": "🐕",
        "compass": "🧭",
        "computer": "💻",
        "computer_keyboard": "⌨️",
        "computer_monitor": "🖥️",
        "computer_mouse": "🖱️",
        "computer_tower": "🖥️",
        "concert": "🎫",
        "conch": "🐚",
        "condiment": "🧂",
        "conference": "🏢",
        "container": "📦",
        "cookie": "🍪",
        "cookware": "🍳",
        "coral_reef": "🐠",
        "corgi": "🐕",
        "corkscrew": "🍾",
        "corn": "🌽",
        "cornflower": "🌼",
        "cosmetic_tool": "💄",
        "costume": "🎭",
        "cougar": "🐆",
        "coupon": "🎟️",
        "cow": "🐄",
        "cowboy_hat": "🤠",
        "coyote_wolf": "🐺",
        "crab": "🦀",
        "cranberry": "🍒",
        "crane_construction": "🏗️",
        "crate": "📦",
        "credit_card": "💳",
        "creek": "🏞️",
        "crepe": "🥞",
        "crib": "🛏️",
        "cricket_sport": "🏏",
        "croissant": "🥐",
        "crosswalk": "🚶‍♂️",
        "crowd": "👥",
        "cruise_ship": "🛳️",
        "crutch": "🩼",
        "cubicle": "🏢",
        "cucumber": "🥒",
        "cup": "🥤",
        "cupcake": "🧁",
        "currency": "💵",
        "curry": "🍛",
        "curtain": "🪟",
        "cutting_board": "🔪",
        "cycling": "🚴",
        "dachshund": "🐕",
        "daffodil": "🌼",
        "dahlia": "🌸",
        "daikon": "🥕",
        "daisy": "🌼",
        "dalmatian": "🐕",
        "dam": "🚧",
        "dancing": "💃",
        "dandelion": "🌼",
        "dartboard": "🎯",
        "dashboard": "🚗",
        "daytime": "☀️",
        "decanter": "🍾",
        "deck": "🛳️",
        "decoration": "🎀",
        "decorative_plant": "🌿",
        "deejay": "🎧",
        "deer": "🦌",
        "desert": "🏜️",
        "desk": "🖥️",
        "dessert": "🍰",
        "diagram": "📈",
        "dial": "⌚",
        "diaper": "👶",
        "dice": "🎲",
        "dill": "🌿",
        "dining_room": "🍴",
        "dinosaur": "🦕",
        "diorama": "🖼️",
        "dirt_road": "🛣️",
        "disco_ball": "🪩",
        "dishwasher": "🧼",
        "diskette": "💾",
        "diving": "🤿",
        "doberman": "🐕",
        "dock": "🚢",
        "document": "📄",
        "dog": "🐶",
        "doll": "🪆",
        "dolphin": "🐬",
        "dome": "🕌",
        "domicile": "🏠",
        "domino": "🁢",
        "donkey": "🐴",
        "donut": "🍩",
        "door": "🚪",
        "dove": "🕊️",
        "dragon_parade": "🐉",
        "dragonfly": "🐉",
        "dressage": "🏇",
        "drink": "🍹",
        "drinking_glass": "🥃",
        "driveway": "🚗",
        "drone_machine": "🚁",
        "drum": "🥁",
        "dumbbell": "🏋️",
        "dumpling": "🥟",
        "durian": "🤢",
        "eagle": "🦅",
        "earmuffs": "🧣",
        "easel": "🎨",
        "easter_egg": "🥚",
        "edamame": "🌱",
        "egg": "🥚",
        "eggplant": "🍆",
        "electric_fan": "💨",
        "elephant": "🐘",
        "elevator": "🛗",
        "elk": "🦌",
        "embers": "🔥",
        "engine_vehicle": "🚗",
        "entertainer": "🎤",
        "envelope": "✉️",
        "equestrian": "🐎",
        "escalator": "🚶‍♂️",
        "eucalyptus_tree": "🌿",
        "evergreen": "🌲",
        "extinguisher": "🧯",
        "eyeglasses": "👓",
        "fairground": "🎠",
        "falafel": "🧆",
        "farm": "🚜",
        "fedora": "🎩",
        "fencing_sport": "🤺",
        "ferns": "🌿",
        "ferret": "🐾",
        "ferris_wheel": "🎡",
        "fig": "🍇",
        "figurine": "🗿",
        "fire": "🔥",
        "firecracker": "🧨",
        "fireplace": "🔥",
        "firetruck": "🚒",
        "fireworks": "🎆",
        "fish": "🐟",
        "fishbowl": "🐠",
        "fishing": "🎣",
        "fishtank": "🐠",
        "flag": "🚩",
        "flagpole": "🚩",
        "flame": "🔥",
        "flamingo": "🦩",
        "flan": "🍮",
        "flashlight": "🔦",
        "flipchart": "📊",
        "flipper": "🦶",
        "flower": "🌸",
        "flower_arrangement": "💐",
        "flute": "🎶",
        "folding_chair": "🪑",
        "foliage": "🌿",
        "fondue": "🫕",
        "food": "🍲",
        "foosball": "⚽",
        "football": "🏈",
        "footwear": "👞",
        "forest": "🌲",
        "fork": "🍴",
        "forklift": "🚜",
        "formula_one_car": "🏎️",
        "fountain": "⛲",
        "fox": "🦊",
        "frame": "🖼️",
        "fried_chicken": "🍗",
        "fried_egg": "🍳",
        "fries": "🍟",
        "frisbee": "🥏",
        "frog": "🐸",
        "frozen": "❄️",
        "frozen_dessert": "🍨",
        "fruitcake": "🍰",
        "gamepad": "🎮",
        "games": "🕹️",
        "garage": "🚗",
        "garden": "🌷",
        "gargoyle": "🗿",
        "garlic": "🧄",
        "gas_mask": "😷",
        "gastropod": "🐌",
        "gazebo": "🏡",
        "gears": "⚙️",
        "gecko": "🦎",
        "gerbil": "🐭",
        "german_shepherd": "🐕",
        "geyser": "💦",
        "gift": "🎁",
        "gift_card": "💳",
        "gingerbread": "🍪",
        "giraffe": "🦒",
        "glacier": "🏔️",
        "glove": "🧤",
        "glove_other": "🥊",
        "go_kart": "🏎️",
        "goat": "🐐",
        "goggles": "🥽",
        "goldfish": "🐠",
        "golf": "🏌️‍♂️",
        "golf_ball": "⛳",
        "golf_club": "🏌️",
        "golf_course": "⛳️",
        "gown": "👗",
        "graduation": "🎓",
        "graffiti": "🎨",
        "grain": "🌾",
        "grand_prix": "🏁",
        "grape": "🍇",
        "grapefruit": "🍊",
        "grass": "🌱",
        "grater": "🧀",
        "grave": "⚰️",
        "green_beans": "🌱",
        "greenhouse": "🌿",
        "greyhound": "🐕",
        "grill": "🍖",
        "grilled_chicken": "🍗",
        "groom": "🤵",
        "guacamole": "🥑",
        "guava": "🍈",
        "guitar": "🎸",
        "gull": "🐦",
        "guppy": "🐟",
        "gymnastics": "🤸‍♀️",
        "gyoza": "🥟",
        "habanero": "🌶️",
        "ham": "🍖",
        "hamburger": "🍔",
        "hammer": "🔨",
        "hammock": "🏝️",
        "hamster": "🐹",
        "handwriting": "✍️",
        "hangar": "✈️",
        "hangglider": "🪂",
        "harbour": "⚓",
        "hardhat": "👷",
        "harp": "🎵",
        "hat": "🎩",
        "haze": "🌫️",
        "headgear": "👒",
        "headphones": "🎧",
        "health_club": "🏋️",
        "hedgehog": "🦔",
        "helicopter": "🚁",
        "helmet": "⛑️",
        "henna": "🌿",
        "herb": "🌿",
        "heron": "🐦",
        "high_chair": "🪑",
        "high_heel": "👠",
        "hiking": "🥾",
        "hill": "⛰️",
        "hippopotamus": "🦛",
        "hockey": "🏒",
        "holly": "🌿",
        "honey": "🍯",
        "honeydew": "🍈",
        "hoodie": "👕",
        "hookah": "💨",
        "horse": "🐴",
        "horseshoe": "🐴",
        "hospital": "🏥",
        "hotdog": "🌭",
        "hound": "🐕",
        "hourglass": "⏳",
        "house_single": "🏠",
        "houseboat": "🛳️",
        "housewares": "🏠",
        "hula": "💃",
        "hummingbird": "🐦",
        "hummus": "🍲",
        "hunting": "🏹",
        "hurdle": "🚧",
        "husky": "🐕‍🦺",
        "hydrant": "🚒",
        "hyena": "🦛",
        "ice": "❄️",
        "ice_cream": "🍨",
        "ice_skates": "⛸️",
        "ice_skating": "⛸️",
        "iceberg": "🧊",
        "igloo": "🏠",
        "iguana": "🦎",
        "illustrations": "🖼️",
        "insect": "🐞",
        "interior_room": "🛋️",
        "interior_shop": "🏬",
        "irish_wolfhound": "🐕",
        "iron_clothing": "🧴",
        "island": "🏝️",
        "ivy": "🌿",
        "jack_o_lantern": "🎃",
        "jack_russell_terrier": "🐕",
        "jacket": "🧥",
        "jacuzzi": "🛁",
        "jalapeno": "🌶️",
        "jar": "🏺",
        "jeans": "👖",
        "jeep": "🚙",
        "jello": "🍮",
        "jelly": "🍮",
        "jellyfish": "🎐",
        "jetski": "🚤",
        "jewelry": "💍",
        "jigsaw": "🧩",
        "jockey_horse": "🏇",
        "joystick": "🕹️",
        "jug": "🏺",
        "juggling": "🤹‍♂️",
        "juice": "🧃",
        "juicer": "🍹",
        "jungle": "🌴",
        "kangaroo": "🦘",
        "karaoke": "🎤",
        "kayak": "🚣",
        "kebab": "🍢",
        "keg": "🛢️",
        "kettle": "🍵",
        "keypad": "⌨️",
        "kickboxing": "🥊",
        "kilt": "👗",
        "kimono": "👘",
        "kitchen": "🍳",
        "kitchen_countertop": "🏠",
        "kitchen_faucet": "🚰",
        "kitchen_oven": "🍳",
        "kitchen_room": "🍳",
        "kitchen_sink": "🚰",
        "kite": "🪁",
        "kiteboarding": "🏄‍♂️",
        "kitten": "🐱",
        "kiwi": "🥝",
        "knife": "🔪",
        "koala": "🐨",
        "kohlrabi": "🥦",
        "koi": "🐟",
        "lab_coat": "🥼",
        "ladle": "🥄",
        "ladybug": "🐞",
        "lake": "🏞️",
        "lamp": "💡",
        "lamppost": "🚦",
        "land": "🏞️",
        "lantern": "🏮",
        "laptop": "💻",
        "laundry_machine": "🧺",
        "lava": "🌋",
        "leash": "🐕‍🦺",
        "leek": "🌱",
        "lemon": "🍋",
        "lemongrass": "🌱",
        "lemur": "🐒",
        "leopard": "🐆",
        "lettuce": "🥬",
        "library": "📚",
        "license_plate": "🚗",
        "lifejacket": "🦺",
        "lifesaver": "⚓",
        "light": "💡",
        "light_bulb": "💡",
        "lighter": "🔥",
        "lighthouse": "🏰",
        "lightning": "⚡",
        "lily": "🌸",
        "lime": "🍋",
        "limousine": "🚗",
        "lion": "🦁",
        "lionfish": "🐠",
        "liquid": "💧",
        "liquor": "🍾",
        "living_room": "🛋️",
        "lizard": "🦎",
        "llama": "🦙",
        "loafer": "👞",
        "lobster": "🦞",
        "lollipop": "🍭",
        "luggage": "🧳",
        "lychee": "🍈",
        "lynx": "🐈",
        "macadamia": "🌰",
        "mackerel": "🐟",
        "magazine": "📰",
        "mailbox": "📫",
        "malamute": "🐕‍🦺",
        "malinois": "🐕",
        "mallet": "🔨",
        "mammal": "🐘",
        "mandarine": "🍊",
        "mango": "🥭",
        "mangosteen": "🍈",
        "mangrove": "🌳",
        "manhole": "⛏️",
        "map": "🗺️",
        "maple_tree": "🍁",
        "margarita": "🍹",
        "marigold": "🌼",
        "marshmallow": "🍬",
        "marsupial": "🦘",
        "martial_arts": "🥋",
        "martini": "🍸",
        "mask": "😷",
        "mast": "⛵",
        "mastiff": "🐕",
        "matches": "🔥",
        "matzo": "🍪",
        "measuring_tape": "📏",
        "meat": "🍖",
        "meatball": "🍝",
        "medal": "🏅",
        "medicine": "💊",
        "megalith": "🪨",
        "megaphone": "📣",
        "melon": "🍈",
        "microphone": "🎤",
        "microscope": "🔬",
        "microwave": "🔥",
        "military_uniform": "🪖",
        "milkshake": "🥤",
        "millipede": "🐛",
        "mistletoe": "🌿",
        "mitten": "🧤",
        "moccasin": "👞",
        "mojito": "🍹",
        "mollusk": "🐚",
        "money": "💵",
        "monitor_lizard": "🦎",
        "monorail": "🚝",
        "moon": "🌕",
        "moose": "🦌",
        "mop": "🧹",
        "moss": "🌿",
        "moth": "🦋",
        "motocross": "🏍️",
        "motorcycle": "🏍️",
        "motorhome": "🚐",
        "motorsport": "🏎️",
        "mountain": "⛰️",
        "mousetrap": "🪤",
        "mower": "🚜",
        "muffin": "🧁",
        "mug": "☕",
        "museum": "🏛️",
        "mushroom": "🍄",
        "musical_instrument": "🎸",
        "mussel": "🐚",
        "mustard": "🌭",
        "naan": "🍞",
        "nachos": "🌮",
        "nascar": "🏎️",
        "necktie": "👔",
        "nectarine": "🍑",
        "nest": "🐦",
        "newspaper": "📰",
        "night_sky": "🌌",
        "nightclub": "🎤",
        "nut": "🥜",
        "oak_tree": "🌳",
        "oar": "🛶",
        "oatmeal": "🥣",
        "obelisk": "🗿",
        "ocean": "🌊",
        "omelet": "🍳",
        "onion": "🧅",
        "optical_equipment": "🔭",
        "oranges": "🍊",
        "orchard": "🌳",
        "orchestra": "🎻",
        "orchid": "🌺",
        "organ_instrument": "🎹",
        "origami": "🔖",
        "ostrich": "🦢",
        "otter": "🦦",
        "oven": "🍳",
        "owl": "🦉",
        "oyster": "🦪",
        "pacifier": "👶",
        "paella": "🥘",
        "paintball": "🔫",
        "paintbrush": "🖌️",
        "painting": "🖼️",
        "palm_tree": "🌴",
        "pan": "🍳",
        "pancake": "🥞",
        "panda": "🐼",
        "papaya": "🍈",
        "paper_bag": "🛍️",
        "parachute": "🪂",
        "parade": "🎉",
        "parakeet": "🦜",
        "parasailing": "🪂",
        "park": "🌲",
        "parking_lot": "🚗",
        "parrot": "🦜",
        "passionfruit": "🍈",
        "passport": "🛂",
        "pasta": "🍝",
        "pastry": "🥐",
        "path": "🛤️",
        "patio": "🪑",
        "payphone": "📞",
        "pea": "🟢",
        "peach": "🍑",
        "peacock": "🦚",
        "peanut": "🥜",
        "pear": "🍐",
        "pecan": "🌰",
        "pelican": "🐦",
        "pen": "🖊️",
        "penguin": "🐧",
        "people": "🧍‍♂️",
        "pepper_veggie": "🌶️",
        "pepperoni": "🍕",
        "peregrine": "🦅",
        "performance": "🎭",
        "pergola": "🌿",
        "persimmon": "🍅",
        "petunia": "🌺",
        "phone": "📱",
        "piano": "🎹",
        "pickle": "🥒",
        "pie": "🥧",
        "pier": "🛳️",
        "pierogi": "🥟",
        "pig": "🐖",
        "pigeon": "🐦",
        "piggybank": "🐷",
        "pillow": "🛌",
        "pineapple": "🍍",
        "ping_pong": "🏓",
        "pistachio": "🥜",
        "pita": "🥙",
        "pitbull": "🐕",
        "pizza": "🍕",
        "plant": "🌱",
        "plate": "🍽️",
        "play_card": "🃏",
        "playground": "🎠",
        "plum": "🍑",
        "podium": "🏆",
        "poinsettia": "🌺",
        "poker": "🃏",
        "pole": "🚩",
        "police_car": "🚓",
        "poncho": "🧥",
        "pool": "🏊",
        "popcorn": "🍿",
        "porch": "🏡",
        "porcupine": "🦔",
        "potato": "🥔",
        "poultry": "🍗",
        "pretzel": "🥨",
        "printer": "🖨️",
        "propeller": "🚁",
        "puffer_fish": "🐡",
        "puffin": "🐦",
        "pug": "🐶",
        "pumpkin": "🎃",
        "puppet": "🎭",
        "purse": "👛",
        "pyramid": "🔺",
        "rabbit": "🐰",
        "raccoon": "🦝",
        "rainbow": "🌈",
        "rake": "🍂",
        "raspberry": "🍇",
        "rat": "🐀",
        "raven": "🦅",
        "record": "💿",
        "rhinoceros": "🦏",
        "rice": "🍚",
        "river": "🌊",
        "road": "🛣️",
        "rocket": "🚀",
        "rodent": "🐭",
        "roof": "🏠",
        "rope": "🪢",
        "rose": "🌹",
        "rotisserie": "🍗",
        "roulette": "🎰",
        "rowboat": "🚣",
        "ruins": "🏚️",
        "sack": "🛍️",
        "saddle": "🐎",
        "sailboat": "⛵",
        "salad": "🥗",
        "salami": "🍕",
        "salmon": "🐟",
        "sand": "🏖️",
        "sandwich": "🥪",
        "sausage": "🌭",
        "saxophone": "🎷",
        "scallop": "🐚",
        "scarf": "🧣",
        "scissors": "✂️",
        "scooter": "🛴",
        "scoreboard": "📊",
        "scorpion": "🦂",
        "screwdriver": "🪛",
        "seahorse": "🌊",
        "seal": "🦭",
        "seashell": "🐚",
        "seat": "💺",
        "seaweed": "🌿",
        "seed": "🌱",
        "sequoia": "🌲",
        "sesame": "🌱",
        "shark": "🦈",
        "sheep": "🐑",
        "shower": "🚿",
        "shrub": "🌿",
        "sidewalk": "🚶‍♂️",
        "sign": "🪧",
        "silo": "🏭",
        "skateboard": "🛹",
        "skeleton": "💀",
        "skull": "💀",
        "skunk": "🦨",
        "sky": "☁️",
        "skyscraper": "🏙️",
        "sled": "🛷",
        "snail": "🐌",
        "snake": "🐍",
        "sneaker": "👟",
        "snow": "❄️",
        "snowboard": "🏂",
        "snowman": "☃️",
        "soccer": "⚽",
        "sock": "🧦",
        "soda": "🥤",
        "sofa": "🛋️",
        "softball": "🥎",
        "solar_panel": "⚡",
        "sombrero": "👒",
        "souffle": "🍮",
        "soup": "🍲",
        "spaghetti": "🍝",
        "spider": "🕷️",
        "spinach": "🥬",
        "spoon": "🥄",
        "sport": "🏀",
        "spotlight": "💡",
        "sprinkler": "💦",
        "squirrel": "🐿️",
        "stadium": "🏟️",
        "starfish": "🌟",
        "statue": "🗿",
        "steak": "🥩",
        "stethoscope": "🩺",
        "stove": "🔥",
        "straw_drinking": "🥤",
        "street": "🛣️",
        "streetcar": "🚋",
        "string_instrument": "🎻",
        "stroller": "👶",
        "suit": "👔",
        "suitcase": "🧳",
        "sun": "☀️",
        "sunflower": "🌻",
        "sunglasses": "🕶️",
        "surfboard": "🏄‍♂️",
        "sushi": "🍣",
        "swan": "🦢",
        "sword": "⚔️",
        "syringe": "💉",
        "table": "🪑",
        "taco": "🌮",
        "tambourine": "🥁",
        "tattoo": "💉",
        "teapot": "🫖",
        "telescope": "🔭",
        "television": "📺",
        "tennis": "🎾",
        "tent": "⛺",
        "theater": "🎭",
        "thermometer": "🌡️",
        "tiara": "👑",
        "ticket": "🎫",
        "tiger": "🐯",
        "tire": "🚗",
        "toad": "🐸",
        "toaster": "🍞",
        "tomato": "🍅",
        "toolbox": "🧰",
        "tornado": "🌪️",
        "tortilla": "🌮",
        "toucan": "🐦",
        "tower": "🗼",
        "toy": "🧸",
        "tractor": "🚜",
        "traffic_light": "🚦",
        "train": "🚂",
        "trampoline": "🤸‍♀️",
        "trash_can": "🗑️",
        "tree": "🌳",
        "tricycle": "🚲",
        "tripod": "📷",
        "trombone": "🎺",
        "trophy": "🏆",
        "truck": "🚚",
        "trumpet": "🎺",
        "tulip": "🌷",
        "tuna": "🐟",
        "turtle": "🐢",
        "typewriter": "📇",
        "ukulele": "🎸",
        "umbrella": "☔",
        "vacuum": "🧹",
        "vase": "🏺",
        "vegetable": "🥦",
        "vehicle": "🚗",
        "violin": "🎻",
        "volcano": "🌋",
        "volleyball": "🏐",
        "vulture": "🦅",
        "waffle": "🧇",
        "wagon": "🚚",
        "wallet": "👛",
        "walrus": "🦭",
        "watch": "⌚",
        "water": "💧",
        "waterfall": "🌊",
        "wedding": "💒",
        "whale": "🐋",
        "wheat": "🌾",
        "wheel": "🎡",
        "wheelbarrow": "🛒",
        "willow": "🌳",
        "windmill": "🌬️",
        "window": "🪟",
        "wine": "🍷",
        "woodpecker": "🐦",
        "worm": "🪱",
        "wreath": "🌲",
        "wrench": "🔧",
        "xylophone": "🎵",
        "yacht": "⛵",
        "yarn": "🧶",
        "yoga": "🧘‍♀️",
        "yogurt": "🥛",
        "zebra": "🦓",
        "zoo": "🦓",
        "zucchini": "🥒",
    ]
    
    var result: [String] = []
    for word in classes {
        if let emoji = map[word] {
            result.append(emoji)
        }
    }
    return result
}
