# Project Interface

> **Note:** This document is about the creation and usage of `ProjectInterface`.  
> In this document, **PI** refers to `ProjectInterface` and **Client** refers to the tool that can process PI.

## Introduction

The so-called `ProjectInterface` is a standardized project structure declaration for MaaFramework. It currently consists of a single file, `interface.json`. By defining a PI, you can leverage various derivative tools of MaaFramework. Therefore, even if you intend to integrate solely using a general-purpose programming language, it is recommended to define a PI containing the basic information.

* **MaaPiCli**  
  General CLI

* **[MFAWPF](https://github.com/SweetSmellFox/MFAWPF)**  
  General UI

* **[Maa Pipeline Support](https://marketplace.visualstudio.com/items?itemName=nekosu.maa-support)**  
  VSCode Plugin

* **[Project Template](https://github.com/MaaXYZ/MaaPracticeBoilerplate)**  
  Template organized based on PI

## `interface.json`

> This file can be validated and receive hints via the [schema](https://github.com/MaaXYZ/MaaFramework/blob/main/tools/interface.schema.json) file.  
> When opening the project template folder with VSCode, the schema and file will be automatically associated.

### Overall Structure

* **version** `string`  
  The project version, which the Client can display to the user.

* **message** `string`  
  A welcome message that the Client can show to the user.

* **controller** `object[]`  
  Controller configuration, an array of objects containing preset controller information.
  
  * **name** `string`  
    Unique name; the Client will display it for selection.
  
  * **type** `'Adb' | 'Win32'`  
    Controller type; valid values are `Adb` and `Win32`.
  
  * **adb** `object`  
    Specific configuration for the `Adb` controller.
    
    * **input** `number`  
      *Optional.* The control method for the Adb controller. If not provided, the default is used. See [Input and Screencap](#adb-input) for details.
    
    * **screencap** `number`  
      *Optional.* The screenshot method for the Adb controller. If not provided, the default is used. See [Input and Screencap](#adb-screencap) for details.
    
    * **config** `object`  
      *Optional.* Additional configuration for the Adb controller. If not provided, an empty object is used. See [Adb Config](#adb-config) for details.
  
  * **win32** `object`  
    Specific configuration for the `Win32` controller.
    
    * **class_regex** `string`  
      *Optional.* The regular expression used to search for window class names by the Win32 controller.
    
    * **window_regex** `string`  
      *Optional.* The regular expression used to search for window titles by the Win32 controller.
    
    * **input** `number`  
      *Optional.* The control method for the Win32 controller. If not provided, the default is used. See [Input and Screencap](#win32-input) for details.
    
    * **screencap** `number`  
      *Optional.* The screenshot method for the Win32 controller. If not provided, the default is used. See [Input and Screencap](#win32-screencap) for details.

* **resource** `object[]`  
  Resource configuration, an array of objects containing information about resource loading.
  
  * **name** `string`  
    Unique name; the Client will display it for selection.
  
  * **path** `string[]`  
    An array of paths to load. If multiple paths are provided, they will be loaded sequentially; resources loaded later will [override](#resource-override) those loaded earlier.  
    Use `{PROJECT_DIR}` to refer to the directory containing the `interface.json` file, for example:
  
    ```json
    "resource": [
        {
            "name": "Official",
            "path": [
                "{PROJECT_DIR}/resource"
            ]
        }
    ]
    ```
  
    **Note:** Resources include not only `pipeline` but also `image` and `model`. Therefore, do not specify the `pipeline` directory directly.

* **agent** `object`  
  Agent configuration, an object containing information about the subprocess (AgentServer).
  
  * **child_exec** `string`  
    The path to the subprocess, an executable file available in the system path. For example, if the Python path exists in the environment variables (system or user), you can simply write `"python"`.
  
  * **child_args** `string[]`  
    *Optional.* An array of arguments for the subprocess.  
    Use `{PROJECT_DIR}` to refer to the directory containing the `interface.json` file, for example:
  
    ```json
    "agent": {
        "child_exec": "python",
        "child_args": [
            "{PROJECT_DIR}/agent/main.py"
        ]
    }
    ```
  
    **Note:** Here, `{PROJECT_DIR}` refers to the directory where the installed `interface.json` file is located. Adjust accordingly for different projects.
  
  * **identifier** `string`  
    *Optional.* A connection identifier used to create a communication socket. If provided, it will be used; otherwise, one is created automatically.

* **task** `object[]`  
  Task configuration, an array of objects containing information about executable tasks.
  
  * **name** `string`  
    Unique name; the Client will display it for selection.
  
  * **entry** `string`  
    The task entry, which is the name of the Task in the `pipeline`.
  
  * **pipeline_override** `pipeline`  
    *Optional.* Task parameters that override the loaded resources when executing the task. This structure is exactly the same as the JSON file in the `pipeline` and must include the task name part, for example:
  
    ```json
    "pipeline_override": {
        "Quit": {
            "enabled": true
        }
    }
    ```
  
  * **option** `string[]`  
    *Optional.* An array of configuration option keys corresponding to subsequent `option` objects. The Client will prompt the user to make a selection based on these.
  
    The Client can display the options in the order provided in the `option` array.

* **option** `record<string, object>`  
  Definition of configuration options, an object mapping containing information about each option.
  
  * **_key_**  
    Unique name; tasks will reference this name.
  
  * **cases** `object[]`  
    *Optional.* An array of objects, each containing information about the available options.  
    The Client can display these options in the order provided in the `cases` array.
    
    * **name** `string`  
      Unique name; the Client will display it for selection.
    
    * **pipeline_override** `pipeline`  
      Same as the `pipeline_override` in the `task`. It takes effect when the option is activated.
  
  * **default_case** `string`  
    *Optional.* The default option, which the Client can use as the initial value.

### Input and Screencap

#### Input

Defines the methods MaaFramework will use for control.

##### Adb Input

> **Reference:** [MaaDef.h](https://github.com/MaaXYZ/MaaFramework/blob/main/include/MaaFramework/MaaDef.h#L227-L240)

Combine the selected methods below using bitwise OR to provide a single value. MaaFramework will try all provided methods in a fixed priority order, choosing the first available one.

By default, all methods except **EmulatorExtras** are attempted.

> **Priority:** EmulatorExtras > Maatouch > MinitouchAndAdbKey > AdbShell

* **AdbShell** `1`  
  Control using the adb process.

* **MinitouchAndAdbKey** `2`  
  Use the adb process for key control, and the minitouch tool for touch control.

* **Maatouch** `4`  
  Control using the maatouch tool.

* **EmulatorExtras** `8`  
  Control using emulator-specific tools. Currently supported emulators:
  
  * MuMu 12
  * LDPlayer 9

##### Win32 Input

> **Reference:** [MaaDef.h](https://github.com/MaaXYZ/MaaFramework/blob/main/include/MaaFramework/MaaDef.h#L252-L259)

Select one of the values below. There is no default value; the Client can choose one as the default.  
Different programs on Win32 handle input in various ways, so there is no universal method.

* **Seize** `1`  
  Exclusive control. In this mode, the user's cursor is directly moved by MaaFramework, and the target window remains active.

* **SendMessage** `2`  
  Control using SendMessage. In this mode, the target window may lose focus.

#### Screencap

Defines the method MaaFramework will use to take screenshots.

##### Adb Screencap

> **Reference:** [MaaDef.h](https://github.com/MaaXYZ/MaaFramework/blob/main/include/MaaFramework/MaaDef.h#L208-L225)

Combine the selected methods below using bitwise OR to provide a single value. MaaFramework will try all provided methods and choose the fastest available one.  
By default, all methods except **RawByNetcat**, **MinicapDirect**, and **MinicapStream** are attempted.

**Note:** Since **MinicapDirect** and **MinicapStream** encode to JPG (a lossy format), they can significantly reduce the effectiveness of template matching and are not recommended.

* **EncodeToFileAndPull** `1`  
  Takes a screenshot using an internal screencap process, encodes it as PNG and outputs it to a file, then pulls the file using the adb process and reads the file.

* **Encode** `2`  
  Takes a screenshot using an internal screencap process, encodes it as PNG, and transfers it via the adb process pipeline.

* **RawWithGzip** `4`  
  Takes a screenshot using an internal screencap process, compresses it with gzip, and transfers it via the adb process pipeline.

* **RawByNetcat** `8`  
  Takes a screenshot using an internal screencap process and transfers it via network using the nc process.

* **MinicapDirect** `16`  
  Uses the minicap tool to take a screenshot, encodes it as JPG, and transfers it via the adb process pipeline.

* **MinicapStream** `32`  
  Uses the minicap tool in streaming mode to take a screenshot, encodes it as JPG, and transfers it via the adb process pipeline.

* **EmulatorExtras** `64`  
  Takes a screenshot using emulator-specific tools.

##### Win32 Screencap

> **Reference:** [MaaDef.h](https://github.com/MaaXYZ/MaaFramework/blob/main/include/MaaFramework/MaaDef.h#L242-L250)

Select one of the values below. There is no default value; the Client can choose one as the default.  
Different programs on Win32 handle rendering in various ways, so there is no universal method.

* **GDI** `1`
* **FramePool** `2`
* **DXGI_DesktopDup** `4`

### Adb Config

The `config` object can be used to override some of the controller's default logic. Typically, specific configuration is only required when using **EmulatorExtras** in multi-instance scenarios.

### Resource Override

If a later-loaded resource contains a task with the same name as one already loaded, the tasks will be merged. Generally, the top-level keys of the new task will replace those of the old task. For example:

**Old Task**

```jsonc
{
    "task1": {
        "enabled": false,
        "recognition": "DirectHit",
        "next": [ "T1", "T2" ]
    }
}
```

New Task

```jsonc
{
    "task1": {
        "enabled": true,
        "action": "Click",
        "next": [ "T2", "T3" ]
    }
}
```

Merged Task

```jsonc
{
    "task1": {
        "enabled": true,
        "recognition": "DirectHit",
        "action": "Click",
        "next": [ "T2", "T3" ]       // Direct replacement; inner keys are not merged.
    }
}
```
