# Getting Started

> Note: This document is about integrating with NodeJS

## Install

Require NodeJS 20 or above. After creating the project directory, use the following command to install

```shell
npm install @maaxyz/maa-node
```

As the npm package contains MaaFramework libraries and AgentBinary, downloading would cost some time.

## Prepare Resource

Refer to [prepare resource files](../1.1-QuickStarted.md#Resource Preparation)

## Connect, Launch task

For a straightforward routine, we need the following steps:

* Scan devices
* Create controller and connect
* Create resource and load
* Create tasker and bind
* Launch tasks

```typescript
import * as maa from './maa'

console.log(maa.Global.version)

async function main() {
    // Query all adb devices
    const devices = await maa.AdbController.find()
    if (!devices) {
        return
    }

    // Use the first one to create the controller
    const [name, adb_path, address, screencap_methods, input_methods, config] = devices[0]
    const ctrl = new maa.AdbController(
        adb_path,
        address,
        screencap_methods,
        input_methods,
        config
    )
    ctrl.notify = (msg, detail) => {
        console.log(msg, detail)
    }
    // Connect to device
    await ctrl.post_connection()

    // Create resource
    const res = new maa.Resource()
    res.notify = (msg, detail) => {
        console.log(msg, detail)
    }
    // Load resource
    await res.post_bundle('./resource')

    // Create tasker
    const tskr = new maa.Tasker()
    tskr.notify = (msg, detail) => {
        console.log(msg, detail)
    }

    // Bind controller and resource
    tskr.bind(ctrl)
    tskr.bind(res)

    // Check if created successfully
    console.log(tskr.inited)

    // Launch task. Task1 is declared in pipeline/Task.json
    if (await tskr
        .post_task('Task1')
        .wait().success) {
        console.log('success!')
    }
}

main()
```

## Alter Resource Behavior on NodeJS Side

Take a look at this code `await tskr.post_task('task', 'Task1').wait()`

Function `post` can be called with three params. The third one is an object, which has exact the same structure to json in `pipeline`, and will override the original `pipeline`. Thus, you can pass an object here to control the task (even create new task).

```javascript
// With the third param, created a new task Task2, and the launch it
// The task created here will be available only inside current execution
// 通过第三个参数, 创建了一个新的任务Task2, 然后执行它
// 此处创建的任务仅在当前执行中有效
await tskr
    .post_task('Task2', {
        Task2: {
            next: [
                'Task1'
            ]
        }
    })
    .wait()
```

## Next Steps

Please refer to other documents to write resource.
