/**
 * 宝宝杀风格技能插件
 * 特点：合并区域布局、技能图标（限定技/觉醒技/转换技）
 */
import { lib, game, ui, get, ai, _status } from "noname";
import { createBaseSkillPlugin } from "./base.js";

const ASSETS_PATH = "extension/十周年UI/ui/assets/skill/baby";

export function createBabySkillPlugin(lib, game, ui, get, ai, _status, app) {
	const base = createBaseSkillPlugin(lib, game, ui, get, ai, _status, app);

	const plugin = {
		...base,

		precontent() {
			this.initCreateMethods();
			this.initUpdateMethods();
			base.initBaseRewrites.call(this);
			game.videoContent.updateSkillControl = (player, clear) => ui.updateSkillControl(player, clear);
			this.initTimer();
			ui.skillControlArea = ui.create.div();
		},

		recontent() {
			this.initDialogRewrites();
			this.initPlayerRewrites();
			this.initConfigRewrites();
			this.initEventListeners();
		},

		initCreateMethods() {
			Object.assign(ui.create, {
				skills: skills => {
					ui.skills = plugin.createSkills(skills, ui.skills);
					ui.skillControl?.update();
					return ui.skills;
				},
				skills2: skills => {
					ui.skills2 = plugin.createSkills(skills, ui.skills2);
					ui.skillControl?.update();
					return ui.skills2;
				},
				skills3: skills => {
					ui.skills3 = plugin.createSkills(skills, ui.skills3);
					ui.skillControl?.update();
					return ui.skills3;
				},
				skillControl: clear => {
					if (!ui.skillControl) {
						ui.skillControl = plugin.createSkillControl();
					}
					if (clear) {
						ui.skillControl.node.combined.innerHTML = "";
					}
					return ui.skillControl;
				},
			});
		},

		createSkillControl() {
			const isRight = lib.config["extension_十周年UI_rightLayout"] === "on";
			const cls = isRight ? ".skill-control" : ".skill-controlzuoshou";
			const node = ui.create.div(cls, ui.arena);
			node.node = {
				combined: ui.create.div(".combined", node),
			};
			Object.assign(node, plugin.controlElement);
			return node;
		},

		initUpdateMethods() {
			ui.updateSkillControl = (player, clear) => {
				const eSkills = player.getSkills("e", true, false).slice(0);
				let skills = player.getSkills("invisible", null, false);
				let gSkills = ui.skills2?.skills.length ? ui.skills2.skills : null;

				skills = skills.filter(s => {
					const info = get.info(s);
					return !info?.nopop || s.startsWith("olhedao_tianshu_");
				});

				const iSkills = player.invisibleSkills.slice(0);
				game.expandSkills(iSkills);
				skills.addArray(iSkills.filter(s => get.info(s)?.enable));

				// 过滤掉通过global关联的技能，避免重复
				if (gSkills) {
					const globalSkills = new Set();
					skills.forEach(s => {
						const info = get.info(s);
						if (info?.global) {
							const globals = Array.isArray(info.global) ? info.global : [info.global];
							globals.forEach(g => globalSkills.add(g));
						}
					});
					gSkills = gSkills.filter(s => !globalSkills.has(s));
				}

				if (player === game.me) {
					const skillControl = ui.create.skillControl(clear);
					skillControl.add(skills, eSkills);
					if (gSkills?.length) skillControl.add(gSkills);
					skillControl.update();
					game.addVideo("updateSkillControl", player, clear);
				}

				const { xiandingji, juexingji } = plugin.processSkillMarks(player);
				plugin.updateSkillMarks(player, xiandingji, juexingji);
			};
		},

		processSkillMarks(player) {
			const xiandingji = {};
			const juexingji = {};

			player.getSkills("invisible", null, false).forEach(skill => {
				const info = get.info(skill);
				if (!info) return;

				if (get.is.zhuanhuanji(skill, player) || info.limited || info.intro?.content === "limited") {
					xiandingji[skill] = player.awakenedSkills.includes(skill);
				}
				if ((info.juexingji || info.dutySkill) && player.awakenedSkills.includes(skill)) {
					juexingji[skill] = true;
				}
			});

			return { xiandingji, juexingji };
		},

		controlElement: {
			add(skill, eSkills) {
				if (Array.isArray(skill)) {
					skill.forEach(s => this.add(s, eSkills));
					return this;
				}

				if (lib.config["extension_十周年UI_aloneEquip"] && eSkills?.length) {
					const expandedE = game.expandSkills(eSkills.slice());
					const expandedS = game.expandSkills([skill]);
					if (expandedS.some(s => expandedE.includes(s))) return this;
				}

				// 展开技能(含group/global)
				const expandWithGlobal = skillId => {
					const result = [skillId];
					const info = get.info(skillId);
					if (info?.group) {
						const groups = Array.isArray(info.group) ? info.group : [info.group];
						groups.forEach(g => {
							if (lib.skill[g]) result.push(g);
						});
					}
					if (info?.global) {
						const globals = Array.isArray(info.global) ? info.global : [info.global];
						globals.forEach(g => {
							if (lib.skill[g]) result.push(g);
						});
					}
					return result;
				};

				const skills = expandWithGlobal(skill).map(s => app.get.skillInfo(s));
				const enableSkills = skills.filter(s => s.type === "enable");
				// 优先显示主动技能
				const showSkills = enableSkills.length ? enableSkills : skills;

				// 排序：主动技 > 被动技
				showSkills.sort((a, b) => {
					const aIsEnable = a.type === "enable";
					const bIsEnable = b.type === "enable";
					if (aIsEnable && !bIsEnable) return -1;
					if (!aIsEnable && bIsEnable) return 1;
					return 0;
				});

				showSkills.forEach(item => {
					if (this.hasExistingNode(item.id)) return;

					if (lib.config["extension_十周年UI_aloneEquip"] && eSkills?.length) {
						if (game.expandSkills(eSkills.slice()).includes(item.id)) return;
					}

					if (item.type === "enable") {
						this.createEnableSkillNode(item);
					} else {
						this.createTriggerSkillNode(item, eSkills);
					}
				});

				return this;
			},

			hasExistingNode(skillId) {
				return this.querySelector(`[data-id="${skillId}"]`);
			},

			createEnableSkillNode(item) {
				const skillName = get.translation(item.name).slice(0, 2);
				const cls = lib.skill[item.id].limited ? ".xiandingji.enable-skill" : ".skillitem.enable-skill";
				const node = ui.create.div(cls, this.node.combined, skillName);
				node.dataset.id = item.id;

				this.addSkillIcon(node, item.id);

				node.addEventListener("click", () => {
					if (lib.config["extension_十周年UI_bettersound"]) {
						game.playAudio("..", "extension", "十周年UI", "audio/SkillBtn");
					}
				});
				app.listen(node, plugin.clickSkill);
			},

			createTriggerSkillNode(item, eSkills) {
				if (!item.info || !item.translation) return;
				if (eSkills?.includes(item.id)) return;

				const skillName = get.translation(item.name).slice(0, 2);
				const node = ui.create.div(".skillitem.trigger-skill", this.node.combined, skillName);
				node.dataset.id = item.id;

				this.addSkillIcon(node, item.id);
			},

			addSkillIcon(node, skillId) {
				const info = lib.skill[skillId];
				if (!info) return;

				node.style.position = "relative";

				// 限定技图标
				if (info.limited || info.intro?.content === "limited") {
					const img = document.createElement("img");
					img.className = "skill-xianding-icon";
					img.src = `${ASSETS_PATH}/xiandingjihs.png`;
					Object.assign(img.style, {
						position: "absolute",
						top: "0",
						left: "0",
						width: "16px",
						height: "16px",
					});
					node.appendChild(img);
				}

				// 觉醒技图标
				if (info.juexingji || info.dutySkill) {
					const img = document.createElement("img");
					img.className = "skill-juexing-icon";
					img.src = `${ASSETS_PATH}/juexingjihs.png`;
					Object.assign(img.style, {
						position: "absolute",
						top: "0",
						left: "0",
						width: "16px",
						height: "16px",
					});
					node.appendChild(img);
				}

				// 转换技图标
				if (info.zhuanhuanji) {
					let imgType = "yang";
					const markNode = game.me?.node?.xSkillMarks?.querySelector(`.skillMarkItem.zhuanhuanji[data-id="${skillId}"]`);
					if (markNode?.classList.contains("yin")) imgType = "ying";

					const img = document.createElement("img");
					img.className = "skill-zhuanhuan-icon";
					img.src = imgType === "yang" ? `${ASSETS_PATH}/mark_yanghs.png` : `${ASSETS_PATH}/mark_yinghs.png`;
					Object.assign(img.style, {
						position: "absolute",
						top: "0",
						left: "0",
						width: "16px",
						height: "16px",
					});
					node.appendChild(img);
				}
			},

			update() {
				const skills = this.getAllSkills();

				// 重新排序
				const combinedNodes = Array.from(this.node.combined.childNodes);
				if (combinedNodes.length > 1) {
					combinedNodes.sort((a, b) => {
						const aIsEnable = a.classList.contains("enable-skill");
						const bIsEnable = b.classList.contains("enable-skill");
						if (aIsEnable && !bIsEnable) return -1;
						if (!aIsEnable && bIsEnable) return 1;
						return 0;
					});
					combinedNodes.forEach(node => this.node.combined.appendChild(node));
				}

				Array.from(this.node.combined.childNodes).forEach(item => {
					item.classList.toggle("usable", skills.includes(item.dataset.id));
					item.classList.toggle("select", _status.event.skill === item.dataset.id);
				});

				const count = this.node.combined.childNodes.length;
				const level = count > 2 ? 4 : count > 0 ? 2 : 0;
				ui.arena.dataset.sclevel = level;

				// 超过8个启用滚动
				this.node.combined.classList.toggle("scroll-enabled", count > 8);
			},

			getAllSkills() {
				const skills = [];
				[ui.skills, ui.skills2, ui.skills3].forEach(s => {
					if (s) skills.addArray(s.skills);
				});
				return skills;
			},
		},

		updateSkillMarks(player, xiandingji, juexingji) {
			let node = player.node.xSkillMarks;
			if (!node) {
				node = player.node.xSkillMarks = ui.create.div(".skillMarks", player);
			}
			node.style.display = "";

			this.cleanupSkillMarks(node, xiandingji, juexingji);
			this.createSkillMarks(node, xiandingji, juexingji, player);
		},

		cleanupSkillMarks(node, xiandingji, juexingji) {
			Array.from(node.childNodes).forEach(item => {
				if (!xiandingji.hasOwnProperty(item.dataset.id) && !juexingji[item.dataset.id]) {
					item.remove();
				}
			});

			Array.from(node.querySelectorAll(".juexingji")).forEach(item => {
				if (!juexingji[item.dataset.id]) item.remove();
			});
		},

		createSkillMarks(node, xiandingji, juexingji, player) {
			Object.entries(xiandingji).forEach(([skill, used]) => {
				if (player.hiddenSkills.includes(skill) && player !== game.me) return;

				const info = lib.skill[skill];
				let item = node.querySelector(`[data-id="${skill}"]`);

				if (!item) {
					const cls = info.zhuanhuanji ? ".skillMarkItem.zhuanhuanji" : ".skillMarkItem.xiandingji";
					item = ui.create.div(cls, node, "");
				}

				item.classList.toggle("used", used);
				item.dataset.id = skill;
			});

			Object.keys(juexingji).forEach(skill => {
				if (player.hiddenSkills.includes(skill) && player !== game.me) return;
				if (node.querySelector(`[data-id="${skill}"]`)) return;

				const info = lib.skill[skill];
				const cls = info.dutySkill ? ".skillMarkItem.duty" : ".skillMarkItem.juexingji";
				const item = ui.create.div(cls, node, "");
				item.dataset.id = skill;
			});
		},

		initDialogRewrites() {
			app.reWriteFunction(ui.create, {
				dialog: [
					null,
					function (dialog) {
						dialog.classList.add("xdialog");
						app.reWriteFunction(dialog, {
							hide: [
								null,
								function () {
									app.emit("dialog:change", dialog);
								},
							],
						});
					},
				],
			});

			app.reWriteFunction(lib.element.dialog, {
				open: [
					null,
					function () {
						app.emit("dialog:change", this);
					},
				],
				close: [
					null,
					function () {
						app.emit("dialog:change", this);
					},
				],
			});
		},

		initPlayerRewrites() {
			app.reWriteFunction(lib.element.player, {
				markSkill: [
					function (args, name) {
						const info = lib.skill[name];
						if (!info) return;
						if (info.limited || info.intro?.content === "limited") return this;
					},
				],
			});
		},

		initConfigRewrites() {
			app.reWriteFunction(lib.configMenu.appearence.config, {
				update: [
					null,
					function (res, config, map) {
						map.button_press.hide();
					},
				],
			});
		},

		initEventListeners() {
			app.on("playerUpdateE", player => plugin.updateMark(player));
		},
	};

	return plugin;
}
