/*
 * Copyright (c) 2016 Intel Corporation
 * Copyright (c) 2023 Nordic Semiconductor ASA
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef ZEPHYR_DRIVERS_USB_COMMON_USB_DWC2_HW
#define ZEPHYR_DRIVERS_USB_COMMON_USB_DWC2_HW

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/* This file describes register set for the DesignWare USB 2.0 controller IP */

/* IN endpoint register block */
struct usb_dwc2_in_ep {
	volatile uint32_t diepctl;
	uint32_t reserved;
	volatile uint32_t diepint;
	uint32_t reserved1;
	volatile uint32_t dieptsiz;
	volatile uint32_t diepdma;
	volatile uint32_t dtxfsts;
	volatile uint32_t diepdmab;
};

/* OUT endpoint register block */
struct usb_dwc2_out_ep {
	volatile uint32_t doepctl;
	uint32_t reserved;
	volatile uint32_t doepint;
	uint32_t reserved1;
	volatile uint32_t doeptsiz;
	volatile uint32_t doepdma;
	uint32_t reserved2;
	volatile uint32_t doepdmab;
};

/* DWC2 register map
 * TODO: This should probably be split into global, host, and device register
 * blocks
 */
struct usb_dwc2_reg {
	volatile uint32_t gotgctl;
	volatile uint32_t gotgint;
	volatile uint32_t gahbcfg;
	volatile uint32_t gusbcfg;
	volatile uint32_t grstctl;
	volatile uint32_t gintsts;
	volatile uint32_t gintmsk;
	volatile uint32_t grxstsr;
	volatile uint32_t grxstsp;
	volatile uint32_t grxfsiz;
	volatile uint32_t gnptxfsiz;
	volatile uint32_t gnptxsts;
	volatile uint32_t gi2cctl;
	volatile uint32_t gpvndctl;
	volatile uint32_t ggpio;
	volatile uint32_t guid;
	volatile uint32_t gsnpsid;
	volatile uint32_t ghwcfg1;
	volatile uint32_t ghwcfg2;
	volatile uint32_t ghwcfg3;
	volatile uint32_t ghwcfg4;
	volatile uint32_t glpmcfg;
	volatile uint32_t gpwrdn;
	volatile uint32_t gdfifocfg;
	volatile uint32_t gadpctl;
	volatile uint32_t grefclk;
	volatile uint32_t gintmsk2;
	volatile uint32_t gintsts2;
	volatile uint32_t reserved1[36];
	volatile uint32_t hptxfsiz;
	union {
		volatile uint32_t dptxfsiz[15];
		volatile uint32_t dieptxf[15];
	};
	volatile uint32_t reserved2[176];
	/* Host mode register 0x0400 .. 0x0670 */
	uint32_t reserved3[256];
	/* Device mode register 0x0800 .. 0x0D00 */
	volatile uint32_t dcfg;
	volatile uint32_t dctl;
	volatile uint32_t dsts;
	uint32_t reserved4;
	volatile uint32_t diepmsk;
	volatile uint32_t doepmsk;
	volatile uint32_t daint;
	volatile uint32_t daintmsk;
	volatile uint32_t dtknqr1;
	volatile uint32_t dtknqr2;
	volatile uint32_t dvbusdis;
	volatile uint32_t dvbuspulse;
	union {
		volatile uint32_t dtknqr3;
		volatile uint32_t dthrctl;
	};
	union {
		volatile uint32_t dtknqr4;
		volatile uint32_t diepempmsk;
	};
	volatile uint32_t deachint;
	volatile uint32_t deachintmsk;
	volatile uint32_t diepeachmsk[16];
	volatile uint32_t doepeachmsk[16];
	volatile uint32_t reserved5[16];
	struct usb_dwc2_in_ep in_ep[16];
	struct usb_dwc2_out_ep out_ep[16];
	volatile uint32_t reserved6[64];
	volatile uint32_t pcgcctl;
};

/* The last register (PCGCCTL) must be at offset 0xE00. */
BUILD_ASSERT(offsetof(struct usb_dwc2_reg, pcgcctl) == 0x0E00);

/*
 * GET_FIELD/SET_FIELD macros below are intended to be used to define functions
 * to get/set a bitfield of a register from/into a value. They should not be
 * used to get/set a bitfield consisting of only one bit.
 */
#define USB_DWC2_GET_FIELD_DEFINE(name, reg_name_and_field)			\
	static inline uint32_t usb_dwc2_get_##name(const uint32_t value)	\
	{									\
		return (value & USB_DWC2_##reg_name_and_field##_MASK) >>	\
			USB_DWC2_##reg_name_and_field##_POS;			\
	}

#define USB_DWC2_SET_FIELD_DEFINE(name, reg_name_and_field)			\
	static inline uint32_t usb_dwc2_set_##name(const uint32_t value)	\
	{									\
		return (value << USB_DWC2_##reg_name_and_field##_POS) &		\
			USB_DWC2_##reg_name_and_field##_MASK;			\
	}

#define USB_DWC2_GET_FIELD_AND_IDX_DEFINE(name, reg_name_and_field)		\
	static inline uint32_t usb_dwc2_get_##name(const uint32_t value,	\
						   const uint32_t idx)		\
	{									\
		return (value & USB_DWC2_##reg_name_and_field##_MASK(idx)) >>	\
			USB_DWC2_##reg_name_and_field##_POS(idx);		\
	}

/* AHB configuration register */
#define USB_DWC2_GAHBCFG			0x0008UL
#define USB_DWC2_GAHBCFG_LOA_EOP_WORD_POS	27UL
#define USB_DWC2_GAHBCFG_LOA_EOP_WORD_MASK	(0x3UL << USB_DWC2_GAHBCFG_LOA_EOP_WORD_POS)
#define USB_DWC2_GAHBCFG_LOA_EOP_WORD_ONE	1
#define USB_DWC2_GAHBCFG_LOA_EOP_WORD_TWO	2
#define USB_DWC2_GAHBCFG_LOA_EOP_BYTE_POS	25UL
#define USB_DWC2_GAHBCFG_LOA_EOP_BYTE_MASK	(0x3UL << USB_DWC2_GAHBCFG_LOA_EOP_BYTE_POS)
#define USB_DWC2_GAHBCFG_LOA_EOP_BYTE_ONE	1
#define USB_DWC2_GAHBCFG_LOA_EOP_BYTE_TWO	2
#define USB_DWC2_GAHBCFG_INVDESCENDIANESS_POS	24UL
#define USB_DWC2_GAHBCFG_INVDESCENDIANESS	BIT(USB_DWC2_GAHBCFG_INVDESCENDIANESS_POS)
#define USB_DWC2_GAHBCFG_AHBSINGLE_POS		23UL
#define USB_DWC2_GAHBCFG_AHBSINGLE		BIT(USB_DWC2_GAHBCFG_AHBSINGLE_POS)
#define USB_DWC2_GAHBCFG_NOTIALLDMAWRIT_POS	22UL
#define USB_DWC2_GAHBCFG_NOTIALLDMAWRIT		BIT(USB_DWC2_GAHBCFG_NOTIALLDMAWRIT_POS)
#define USB_DWC2_GAHBCFG_REMMEMSUPP_POS		21UL
#define USB_DWC2_GAHBCFG_REMMEMSUPP		BIT(USB_DWC2_GAHBCFG_REMMEMSUPP_POS)
#define USB_DWC2_GAHBCFG_PTXFEMPLVL_POS		8UL
#define USB_DWC2_GAHBCFG_PTXFEMPLVL		BIT(USB_DWC2_GAHBCFG_PTXFEMPLVL_POS)
#define USB_DWC2_GAHBCFG_NPTXFEMPLVL_POS	7UL
#define USB_DWC2_GAHBCFG_NPTXFEMPLVL		BIT(USB_DWC2_GAHBCFG_NPTXFEMPLVL_POS)
#define USB_DWC2_GAHBCFG_DMAEN_POS		5UL
#define USB_DWC2_GAHBCFG_DMAEN			BIT(USB_DWC2_GAHBCFG_DMAEN_POS)
#define USB_DWC2_GAHBCFG_HBSTLEN_POS		1UL
#define USB_DWC2_GAHBCFG_HBSTLEN_MASK		(0xFUL << USB_DWC2_GAHBCFG_HBSTLEN_POS)
#define USB_DWC2_GAHBCFG_HBSTLEN_SINGLE		0
#define USB_DWC2_GAHBCFG_HBSTLEN_INCR		1
#define USB_DWC2_GAHBCFG_HBSTLEN_INCR4		3
#define USB_DWC2_GAHBCFG_HBSTLEN_INCR8		5
#define USB_DWC2_GAHBCFG_HBSTLEN_INCR16		7
#define USB_DWC2_GAHBCFG_GLBINTRMASK_POS	0UL
#define USB_DWC2_GAHBCFG_GLBINTRMASK		BIT(USB_DWC2_GAHBCFG_GLBINTRMASK_POS)

USB_DWC2_SET_FIELD_DEFINE(gahbcfg_loa_eop_word, GAHBCFG_LOA_EOP_WORD)
USB_DWC2_SET_FIELD_DEFINE(gahbcfg_loa_eop_byte, GAHBCFG_LOA_EOP_BYTE)
USB_DWC2_SET_FIELD_DEFINE(gahbcfg_hbstlen, GAHBCFG_HBSTLEN)
USB_DWC2_GET_FIELD_DEFINE(gahbcfg_loa_eop_word, GAHBCFG_LOA_EOP_WORD)
USB_DWC2_GET_FIELD_DEFINE(gahbcfg_loa_eop_byte, GAHBCFG_LOA_EOP_BYTE)
USB_DWC2_GET_FIELD_DEFINE(gahbcfg_hbstlen, GAHBCFG_HBSTLEN)

/* USB configuration register */
#define USB_DWC2_GUSBCFG			0x000CUL
#define USB_DWC2_GUSBCFG_FORCEDEVMODE_POS	30UL
#define USB_DWC2_GUSBCFG_FORCEDEVMODE		BIT(USB_DWC2_GUSBCFG_FORCEDEVMODE_POS)
#define USB_DWC2_GUSBCFG_FORCEHSTMODE_POS	29UL
#define USB_DWC2_GUSBCFG_FORCEHSTMODE		BIT(USB_DWC2_GUSBCFG_FORCEHSTMODE_POS)
#define USB_DWC2_GUSBCFG_PHYSEL_POS		6UL
#define USB_DWC2_GUSBCFG_PHYSEL_USB11		BIT(USB_DWC2_GUSBCFG_PHYSEL_POS)
#define USB_DWC2_GUSBCFG_PHYSEL_USB20		0UL
#define USB_DWC2_GUSBCFG_ULPI_UTMI_SEL_POS	4UL
#define USB_DWC2_GUSBCFG_ULPI_UTMI_SEL_ULPI	BIT(USB_DWC2_GUSBCFG_ULPI_UTMI_SEL_POS)
#define USB_DWC2_GUSBCFG_ULPI_UTMI_SEL_UTMI	0UL
#define USB_DWC2_GUSBCFG_PHYIF_POS		3UL
#define USB_DWC2_GUSBCFG_PHYIF_16_BIT		BIT(USB_DWC2_GUSBCFG_PHYIF_POS)
#define USB_DWC2_GUSBCFG_PHYIF_8_BIT		0UL

/* Reset register */
#define USB_DWC2_GRSTCTL			0x0010UL
#define USB_DWC2_GRSTCTL_AHBIDLE_POS		31UL
#define USB_DWC2_GRSTCTL_AHBIDLE		BIT(USB_DWC2_GRSTCTL_AHBIDLE_POS)
#define USB_DWC2_GRSTCTL_CSFTRSTDONE_POS	29UL
#define USB_DWC2_GRSTCTL_CSFTRSTDONE		BIT(USB_DWC2_GRSTCTL_CSFTRSTDONE_POS)
#define USB_DWC2_GRSTCTL_TXFNUM_POS		6UL
#define USB_DWC2_GRSTCTL_TXFNUM_MASK		(0x1FUL << USB_DWC2_GRSTCTL_TXFNUM_POS)
#define USB_DWC2_GRSTCTL_TXFFLSH_POS		5UL
#define USB_DWC2_GRSTCTL_TXFFLSH		BIT(USB_DWC2_GRSTCTL_TXFFLSH_POS)
#define USB_DWC2_GRSTCTL_RXFFLSH_POS		4UL
#define USB_DWC2_GRSTCTL_RXFFLSH		BIT(USB_DWC2_GRSTCTL_RXFFLSH_POS)
#define USB_DWC2_GRSTCTL_CSFTRST_POS		0UL
#define USB_DWC2_GRSTCTL_CSFTRST		BIT(USB_DWC2_GRSTCTL_CSFTRST_POS)

USB_DWC2_SET_FIELD_DEFINE(grstctl_txfnum, GRSTCTL_TXFNUM)

/* Core interrupt registers */
#define USB_DWC2_GINTSTS			0x0014UL
#define USB_DWC2_GINTMSK			0x0018UL
#define USB_DWC2_GINTSTS_WKUPINT_POS		31UL
#define USB_DWC2_GINTSTS_WKUPINT		BIT(USB_DWC2_GINTSTS_WKUPINT_POS)
#define USB_DWC2_GINTSTS_SESSREQINT_POS		30UL
#define USB_DWC2_GINTSTS_SESSREQINT		BIT(USB_DWC2_GINTSTS_SESSREQINT_POS)
#define USB_DWC2_GINTSTS_DISCONNINT_POS		29UL
#define USB_DWC2_GINTSTS_DISCONNINT		BIT(USB_DWC2_GINTSTS_DISCONNINT_POS)
#define USB_DWC2_GINTSTS_CONIDSTSCHNG_POS	28UL
#define USB_DWC2_GINTSTS_CONIDSTSCHNG		BIT(USB_DWC2_GINTSTS_CONIDSTSCHNG_POS)
#define USB_DWC2_GINTSTS_LPM_INT_POS		27UL
#define USB_DWC2_GINTSTS_LPM_INT		BIT(USB_DWC2_GINTSTS_LPM_INT_POS)
#define USB_DWC2_GINTSTS_HCHINT_POS		25UL
#define USB_DWC2_GINTSTS_HCHINT			BIT(USB_DWC2_GINTSTS_HCHINT_POS)
#define USB_DWC2_GINTSTS_PRTINT_POS		24UL
#define USB_DWC2_GINTSTS_PRTINT			BIT(USB_DWC2_GINTSTS_PRTINT_POS)
#define USB_DWC2_GINTSTS_RESETDET_POS		23UL
#define USB_DWC2_GINTSTS_RESETDET		BIT(USB_DWC2_GINTSTS_RESETDET_POS)
#define USB_DWC2_GINTSTS_FETSUSP_POS		22UL
#define USB_DWC2_GINTSTS_FETSUSP		BIT(USB_DWC2_GINTSTS_FETSUSP_POS)
#define USB_DWC2_GINTSTS_INCOMPIP_POS		21UL
#define USB_DWC2_GINTSTS_INCOMPIP		BIT(USB_DWC2_GINTSTS_INCOMPIP_POS)
#define USB_DWC2_GINTSTS_INCOMPISOOUT		USB_DWC2_GINTSTS_INCOMPIP
#define USB_DWC2_GINTSTS_INCOMPISOIN_POS	20UL
#define USB_DWC2_GINTSTS_INCOMPISOIN		BIT(USB_DWC2_GINTSTS_INCOMPISOIN_POS)
#define USB_DWC2_GINTSTS_OEPINT_POS		19UL
#define USB_DWC2_GINTSTS_OEPINT			BIT(USB_DWC2_GINTSTS_OEPINT_POS)
#define USB_DWC2_GINTSTS_IEPINT_POS		18UL
#define USB_DWC2_GINTSTS_IEPINT			BIT(USB_DWC2_GINTSTS_IEPINT_POS)
#define USB_DWC2_GINTSTS_EPMIS_POS		17UL
#define USB_DWC2_GINTSTS_EPMIS			BIT(USB_DWC2_GINTSTS_EPMIS_POS)
#define USB_DWC2_GINTSTS_RSTRDONEINT_POS	16UL
#define USB_DWC2_GINTSTS_RSTRDONEINT		BIT(USB_DWC2_GINTSTS_RSTRDONEINT_POS)
#define USB_DWC2_GINTSTS_EOPF_POS		15UL
#define USB_DWC2_GINTSTS_EOPF			BIT(USB_DWC2_GINTSTS_EOPF_POS)
#define USB_DWC2_GINTSTS_ISOOUTDROP_POS		14UL
#define USB_DWC2_GINTSTS_ISOOUTDROP		BIT(USB_DWC2_GINTSTS_ISOOUTDROP_POS)
#define USB_DWC2_GINTSTS_ENUMDONE_POS		13UL
#define USB_DWC2_GINTSTS_ENUMDONE		BIT(USB_DWC2_GINTSTS_ENUMDONE_POS)
#define USB_DWC2_GINTSTS_USBRST_POS		12UL
#define USB_DWC2_GINTSTS_USBRST			BIT(USB_DWC2_GINTSTS_USBRST_POS)
#define USB_DWC2_GINTSTS_USBSUSP_POS		11UL
#define USB_DWC2_GINTSTS_USBSUSP		BIT(USB_DWC2_GINTSTS_USBSUSP_POS)
#define USB_DWC2_GINTSTS_ERLYSUSP_POS		10UL
#define USB_DWC2_GINTSTS_ERLYSUSP		BIT(USB_DWC2_GINTSTS_ERLYSUSP_POS)
#define USB_DWC2_GINTSTS_GOUTNAKEFF_POS		7UL
#define USB_DWC2_GINTSTS_GOUTNAKEFF		BIT(USB_DWC2_GINTSTS_GOUTNAKEFF_POS)
#define USB_DWC2_GINTSTS_GINNAKEFF_POS		6UL
#define USB_DWC2_GINTSTS_GINNAKEFF		BIT(USB_DWC2_GINTSTS_GINNAKEFF_POS)
#define USB_DWC2_GINTSTS_NPTXFEMP_POS		5UL
#define USB_DWC2_GINTSTS_NPTXFEMP		BIT(USB_DWC2_GINTSTS_NPTXFEMP_POS)
#define USB_DWC2_GINTSTS_RXFLVL_POS		4UL
#define USB_DWC2_GINTSTS_RXFLVL			BIT(USB_DWC2_GINTSTS_RXFLVL_POS)
#define USB_DWC2_GINTSTS_SOF_POS		3UL
#define USB_DWC2_GINTSTS_SOF			BIT(USB_DWC2_GINTSTS_SOF_POS)
#define USB_DWC2_GINTSTS_OTGINT_POS		2UL
#define USB_DWC2_GINTSTS_OTGINT			BIT(USB_DWC2_GINTSTS_OTGINT_POS)
#define USB_DWC2_GINTSTS_MODEMIS_POS		1UL
#define USB_DWC2_GINTSTS_MODEMIS		BIT(USB_DWC2_GINTSTS_MODEMIS_POS)
#define USB_DWC2_GINTSTS_CURMOD_POS		0UL
#define USB_DWC2_GINTSTS_CURMOD			BIT(USB_DWC2_GINTSTS_CURMOD_POS)

/* Status read and pop registers */
#define USB_DWC2_GRXSTSR			0x001CUL
#define USB_DWC2_GRXSTSP			0x0020UL
#define USB_DWC2_GRXSTSR_FN_POS			21UL
#define USB_DWC2_GRXSTSR_FN_MASK		(0xFUL << USB_DWC2_GRXSTSR_FN_POS)
#define USB_DWC2_GRXSTSR_PKTSTS_POS		17UL
#define USB_DWC2_GRXSTSR_PKTSTS_MASK		(0xFUL << USB_DWC2_GRXSTSR_PKTSTS_POS)
#define USB_DWC2_GRXSTSR_PKTSTS_GLOBAL_OUT_NAK	1
#define USB_DWC2_GRXSTSR_PKTSTS_OUT_DATA	2
#define USB_DWC2_GRXSTSR_PKTSTS_OUT_DATA_DONE	3
#define USB_DWC2_GRXSTSR_PKTSTS_SETUP_DONE	4
#define USB_DWC2_GRXSTSR_PKTSTS_SETUP		6
#define USB_DWC2_GRXSTSR_DPID_POS		15UL
#define USB_DWC2_GRXSTSR_DPID_MASK		(0x3UL << USB_DWC2_GRXSTSR_DPID_POS)
#define USB_DWC2_GRXSTSR_DPID_DATA0		0
#define USB_DWC2_GRXSTSR_DPID_DATA2		1
#define USB_DWC2_GRXSTSR_DPID_DATA1		2
#define USB_DWC2_GRXSTSR_DPID_MDATA		3
#define USB_DWC2_GRXSTSR_BCNT_POS		4UL
#define USB_DWC2_GRXSTSR_BCNT_MASK		(0x000007FFUL << USB_DWC2_GRXSTSR_BCNT_POS)
#define USB_DWC2_GRXSTSR_EPNUM_POS		0UL
#define USB_DWC2_GRXSTSR_EPNUM_MASK		0x0000000FUL
#define USB_DWC2_GRXSTSR_CHNUM_POS		0UL
#define USB_DWC2_GRXSTSR_CHNUM_MASK		0x0000000FUL

USB_DWC2_GET_FIELD_DEFINE(grxstsp_fn, GRXSTSR_FN)
USB_DWC2_GET_FIELD_DEFINE(grxstsp_pktsts, GRXSTSR_PKTSTS)
USB_DWC2_GET_FIELD_DEFINE(grxstsp_bcnt, GRXSTSR_BCNT)
USB_DWC2_GET_FIELD_DEFINE(grxstsp_epnum, GRXSTSR_EPNUM)

/* Receive FIFO size register (device mode) */
#define USB_DWC2_GRXFSIZ			0x0024UL
#define USB_DWC2_GRXFSIZ_RXFDEP_POS		0UL
#define USB_DWC2_GRXFSIZ_RXFDEP_MASK		(0xFFFFUL << USB_DWC2_GRXFSIZ_RXFDEP_POS)

USB_DWC2_GET_FIELD_DEFINE(grxfsiz, GRXFSIZ_RXFDEP)
USB_DWC2_SET_FIELD_DEFINE(grxfsiz, GRXFSIZ_RXFDEP)

/* Non-periodic transmit FIFO size register (device mode) */
#define USB_DWC2_GNPTXFSIZ			0x0028UL
#define USB_DWC2_GNPTXFSIZ_NPTXFDEP_POS		16UL
#define USB_DWC2_GNPTXFSIZ_NPTXFDEP_MASK	(0xFFFFUL << USB_DWC2_GNPTXFSIZ_NPTXFDEP_POS)
#define USB_DWC2_GNPTXFSIZ_NPTXFSTADDR_POS	0UL
#define USB_DWC2_GNPTXFSIZ_NPTXFSTADDR_MASK	(0xFFFFUL << USB_DWC2_GNPTXFSIZ_NPTXFSTADDR_POS)

USB_DWC2_GET_FIELD_DEFINE(gnptxfsiz_nptxfdep, GNPTXFSIZ_NPTXFDEP)
USB_DWC2_GET_FIELD_DEFINE(gnptxfsiz_nptxfstaddr, GNPTXFSIZ_NPTXFSTADDR)
USB_DWC2_SET_FIELD_DEFINE(gnptxfsiz_nptxfdep, GNPTXFSIZ_NPTXFDEP)
USB_DWC2_SET_FIELD_DEFINE(gnptxfsiz_nptxfstaddr, GNPTXFSIZ_NPTXFSTADDR)

/* Application (vendor) general purpose registers */
#define USB_DWC2_GGPIO				0x0038UL
#define USB_DWC2_GGPIO_STM32_VBDEN_POS		21UL
#define USB_DWC2_GGPIO_STM32_VBDEN		BIT(USB_DWC2_GGPIO_STM32_VBDEN_POS)
#define USB_DWC2_GGPIO_STM32_PWRDWN_POS		16UL
#define USB_DWC2_GGPIO_STM32_PWRDWN		BIT(USB_DWC2_GGPIO_STM32_PWRDWN_POS)

/* GSNPSID register */
#define USB_DWC2_GSNPSID_REV_5_00A		0x4F54500AUL

/* GHWCFG1 register */
#define USB_DWC2_GHWCFG1			0x0044UL
#define USB_DWC2_GHWCFG1_EPDIR_POS(i)		(i * 2)
#define USB_DWC2_GHWCFG1_EPDIR_MASK(i)		(0x3UL << USB_DWC2_GHWCFG1_EPDIR_POS(i))
#define USB_DWC2_GHWCFG1_EPDIR_OUT		2
#define USB_DWC2_GHWCFG1_EPDIR_IN		1
#define USB_DWC2_GHWCFG1_EPDIR_BDIR		0

USB_DWC2_GET_FIELD_AND_IDX_DEFINE(ghwcfg1_epdir, GHWCFG1_EPDIR)

/* GHWCFG2 register */
#define USB_DWC2_GHWCFG2			0x0048UL
#define USB_DWC2_GHWCFG2_TKNQDEPTH_POS		26UL
#define USB_DWC2_GHWCFG2_TKNQDEPTH_MASK		(0x1FUL << USB_DWC2_GHWCFG2_TKNQDEPTH_POS)
#define USB_DWC2_GHWCFG2_PTXQDEPTH_POS		24UL
#define USB_DWC2_GHWCFG2_PTXQDEPTH_MASK		(0x3UL << USB_DWC2_GHWCFG2_PTXQDEPTH_POS)
#define USB_DWC2_GHWCFG2_PTXQDEPTH_QUE16	3
#define USB_DWC2_GHWCFG2_PTXQDEPTH_QUE8		2
#define USB_DWC2_GHWCFG2_PTXQDEPTH_QUE4		1
#define USB_DWC2_GHWCFG2_PTXQDEPTH_QUE2		0
#define USB_DWC2_GHWCFG2_NPTXQDEPTH_POS		22UL
#define USB_DWC2_GHWCFG2_NPTXQDEPTH_MASK	(0x3UL << USB_DWC2_GHWCFG2_NPTXQDEPTH_POS)
#define USB_DWC2_GHWCFG2_NPTXQDEPTH_EIGHT	2
#define USB_DWC2_GHWCFG2_NPTXQDEPTH_FOUR	1
#define USB_DWC2_GHWCFG2_NPTXQDEPTH_TWO		0
#define USB_DWC2_GHWCFG2_MULTIPROCINTRPT_POS	20UL
#define USB_DWC2_GHWCFG2_MULTIPROCINTRPT	BIT(USB_DWC2_GHWCFG2_MULTIPROCINTRPT_POS)
#define USB_DWC2_GHWCFG2_DYNFIFOSIZING_POS	19UL
#define USB_DWC2_GHWCFG2_DYNFIFOSIZING		BIT(USB_DWC2_GHWCFG2_DYNFIFOSIZING_POS)
#define USB_DWC2_GHWCFG2_PERIOSUPPORT_POS	18UL
#define USB_DWC2_GHWCFG2_PERIOSUPPORT		BIT(USB_DWC2_GHWCFG2_PERIOSUPPORT_POS)
#define USB_DWC2_GHWCFG2_NUMHSTCHNL_POS		14UL
#define USB_DWC2_GHWCFG2_NUMHSTCHNL_MASK	(0xFUL << USB_DWC2_GHWCFG2_NUMHSTCHNL_POS)
#define USB_DWC2_GHWCFG2_NUMDEVEPS_POS		10UL
#define USB_DWC2_GHWCFG2_NUMDEVEPS_MASK		(0xFUL << USB_DWC2_GHWCFG2_NUMDEVEPS_POS)
#define USB_DWC2_GHWCFG2_FSPHYTYPE_POS		8UL
#define USB_DWC2_GHWCFG2_FSPHYTYPE_MASK		(0x3UL << USB_DWC2_GHWCFG2_FSPHYTYPE_POS)
#define USB_DWC2_GHWCFG2_FSPHYTYPE_FSPLUSULPI	3
#define USB_DWC2_GHWCFG2_FSPHYTYPE_FSPLUSUTMI	2
#define USB_DWC2_GHWCFG2_FSPHYTYPE_FS		1
#define USB_DWC2_GHWCFG2_FSPHYTYPE_NO_FS	0
#define USB_DWC2_GHWCFG2_HSPHYTYPE_POS		6UL
#define USB_DWC2_GHWCFG2_HSPHYTYPE_MASK		(0x3UL << USB_DWC2_GHWCFG2_HSPHYTYPE_POS)
#define USB_DWC2_GHWCFG2_HSPHYTYPE_UTMIPLUSULPI	3
#define USB_DWC2_GHWCFG2_HSPHYTYPE_ULPI		2
#define USB_DWC2_GHWCFG2_HSPHYTYPE_UTMIPLUS	1
#define USB_DWC2_GHWCFG2_HSPHYTYPE_NO_HS	0
#define USB_DWC2_GHWCFG2_SINGPNT_POS		5UL
#define USB_DWC2_GHWCFG2_SINGPNT		BIT(USB_DWC2_GHWCFG2_SINGPNT_POS)
#define USB_DWC2_GHWCFG2_OTGARCH_POS		3UL
#define USB_DWC2_GHWCFG2_OTGARCH_MASK		(0x3UL << USB_DWC2_GHWCFG2_OTGARCH_POS)
#define USB_DWC2_GHWCFG2_OTGARCH_INTERNALDMA	2
#define USB_DWC2_GHWCFG2_OTGARCH_EXTERNALDMA	1
#define USB_DWC2_GHWCFG2_OTGARCH_SLAVEMODE	0
#define USB_DWC2_GHWCFG2_OTGMODE_POS		0UL
#define USB_DWC2_GHWCFG2_OTGMODE_MASK		(0x7UL << USB_DWC2_GHWCFG2_OTGMODE_POS)
#define USB_DWC2_GHWCFG2_OTGMODE_NONOTGH	6
#define USB_DWC2_GHWCFG2_OTGMODE_SRPCAPH	5
#define USB_DWC2_GHWCFG2_OTGMODE_NONOTGD	4
#define USB_DWC2_GHWCFG2_OTGMODE_SRPCAPD	3
#define USB_DWC2_GHWCFG2_OTGMODE_NHNPNSRP	2
#define USB_DWC2_GHWCFG2_OTGMODE_SRPOTG		1
#define USB_DWC2_GHWCFG2_OTGMODE_HNPSRP		0

USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_tknqdepth, GHWCFG2_TKNQDEPTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_ptxqdepth, GHWCFG2_PTXQDEPTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_nptxqdepth, GHWCFG2_NPTXQDEPTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_numhstchnl, GHWCFG2_NUMHSTCHNL)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_numdeveps, GHWCFG2_NUMDEVEPS)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_fsphytype, GHWCFG2_FSPHYTYPE)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_hsphytype, GHWCFG2_HSPHYTYPE)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_otgarch, GHWCFG2_OTGARCH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg2_otgmode, GHWCFG2_OTGMODE)

/* GHWCFG3 register */
#define USB_DWC2_GHWCFG3			0x004CUL
#define USB_DWC2_GHWCFG3_DFIFODEPTH_POS		16UL
#define USB_DWC2_GHWCFG3_DFIFODEPTH_MASK	(0xFFFFUL << USB_DWC2_GHWCFG3_DFIFODEPTH_POS)
#define USB_DWC2_GHWCFG3_LPMMODE_POS		15UL
#define USB_DWC2_GHWCFG3_LPMMODE		BIT(USB_DWC2_GHWCFG3_LPMMODE_POS)
#define USB_DWC2_GHWCFG3_BCSUPPORT_POS		14UL
#define USB_DWC2_GHWCFG3_BCSUPPORT		BIT(USB_DWC2_GHWCFG3_BCSUPPORT_POS)
#define USB_DWC2_GHWCFG3_HSICMODE_POS		13UL
#define USB_DWC2_GHWCFG3_HSICMODE		BIT(USB_DWC2_GHWCFG3_HSICMODE_POS)
#define USB_DWC2_GHWCFG3_ADPSUPPORT_POS		12UL
#define USB_DWC2_GHWCFG3_ADPSUPPORT		BIT(USB_DWC2_GHWCFG3_ADPSUPPORT_POS)
#define USB_DWC2_GHWCFG3_RSTTYPE_POS		11UL
#define USB_DWC2_GHWCFG3_RSTTYPE		BIT(USB_DWC2_GHWCFG3_RSTTYPE_POS)
#define USB_DWC2_GHWCFG3_OPTFEATURE_POS		10UL
#define USB_DWC2_GHWCFG3_OPTFEATURE		BIT(USB_DWC2_GHWCFG3_OPTFEATURE_POS)
#define USB_DWC2_GHWCFG3_VNDCTLSUPT_POS		9UL
#define USB_DWC2_GHWCFG3_VNDCTLSUPT		BIT(USB_DWC2_GHWCFG3_VNDCTLSUPT_POS)
#define USB_DWC2_GHWCFG3_I2CINTSEL_POS		8UL
#define USB_DWC2_GHWCFG3_I2CINTSEL		BIT(USB_DWC2_GHWCFG3_I2CINTSEL)
#define USB_DWC2_GHWCFG3_OTGEN_POS		7UL
#define USB_DWC2_GHWCFG3_OTGEN			BIT(USB_DWC2_GHWCFG3_OTGEN_POS)
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_POS	4UL
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_MASK	(0x7UL << USB_DWC2_GHWCFG3_PKTSIZEWIDTH_POS)
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS10	6U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS9	5U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS8	4U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS7	3U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS6	2U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS5	1U
#define USB_DWC2_GHWCFG3_PKTSIZEWIDTH_BITS4	0U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_POS	0UL
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_MASK	(0xFUL << USB_DWC2_GHWCFG3_XFERSIZEWIDTH_POS)
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH19	8U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH18	7U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH17	6U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH16	5U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH15	4U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH14	3U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH13	2U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH12	1U
#define USB_DWC2_GHWCFG3_XFERSIZEWIDTH_WIDTH11	0U

#define GHWCFG3_PKTCOUNT(pktsizewidth) BIT_MASK(pktsizewidth + 4)
#define GHWCFG3_XFERSIZE(xfersizewidth) BIT_MASK(xfersizewidth + 11)

USB_DWC2_GET_FIELD_DEFINE(ghwcfg3_dfifodepth, GHWCFG3_DFIFODEPTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg3_pktsizewidth, GHWCFG3_PKTSIZEWIDTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg3_xfersizewidth, GHWCFG3_XFERSIZEWIDTH)

/* GHWCFG4 register */
#define USB_DWC2_GHWCFG4			0x0050UL
#define USB_DWC2_GHWCFG4_DESCDMA_POS		31UL
#define USB_DWC2_GHWCFG4_DESCDMA		BIT(USB_DWC2_GHWCFG4_DESCDMA_POS)
#define USB_DWC2_GHWCFG4_DESCDMAENABLED_POS	30UL
#define USB_DWC2_GHWCFG4_DESCDMAENABLED		BIT(USB_DWC2_GHWCFG4_DESCDMAENABLED_POS)
#define USB_DWC2_GHWCFG4_INEPS_POS		26UL
#define USB_DWC2_GHWCFG4_INEPS_MASK		(0xFUL << USB_DWC2_GHWCFG4_INEPS_POS)
#define USB_DWC2_GHWCFG4_DEDFIFOMODE_POS	25UL
#define USB_DWC2_GHWCFG4_DEDFIFOMODE		BIT(USB_DWC2_GHWCFG4_DEDFIFOMODE_POS)
#define USB_DWC2_GHWCFG4_SESSENDFLTR_POS	24UL
#define USB_DWC2_GHWCFG4_SESSENDFLTR		BIT(USB_DWC2_GHWCFG4_SESSENDFLTR_POS)
#define USB_DWC2_GHWCFG4_BVALIDFLTR_POS		23UL
#define USB_DWC2_GHWCFG4_BVALIDFLTR		BIT(USB_DWC2_GHWCFG4_BVALIDFLTR_POS)
#define USB_DWC2_GHWCFG4_AVALIDFLTR_POS		22UL
#define USB_DWC2_GHWCFG4_AVALIDFLTR		BIT(USB_DWC2_GHWCFG4_AVALIDFLTR_POS)
#define USB_DWC2_GHWCFG4_VBUSVALIDFLTR_POS	21UL
#define USB_DWC2_GHWCFG4_VBUSVALIDFLTR		BIT(USB_DWC2_GHWCFG4_VBUSVALIDFLTR_POS)
#define USB_DWC2_GHWCFG4_IDDGFLTR_POS		20UL
#define USB_DWC2_GHWCFG4_IDDGFLTR		BIT(USB_DWC2_GHWCFG4_IDDGFLTR_POS)
#define USB_DWC2_GHWCFG4_NUMCTLEPS_POS		16UL
#define USB_DWC2_GHWCFG4_NUMCTLEPS_MASK		(0xFUL << USB_DWC2_GHWCFG4_NUMCTLEPS_POS)
#define USB_DWC2_GHWCFG4_PHYDATAWIDTH_POS	14UL
#define USB_DWC2_GHWCFG4_PHYDATAWIDTH_MASK	(0x3UL << USB_DWC2_GHWCFG4_PHYDATAWIDTH_POS)
#define USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT_POS	13UL
#define USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT	BIT(USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT_POS)
#define USB_DWC2_GHWCFG4_ACGSUPT_POS		12UL
#define USB_DWC2_GHWCFG4_ACGSUPT		BIT(USB_DWC2_GHWCFG4_ACGSUPT_POS)
#define USB_DWC2_GHWCFG4_IPGISOCSUPT_POS	11UL
#define USB_DWC2_GHWCFG4_IPGISOCSUPT		BIT(USB_DWC2_GHWCFG4_IPGISOCSUPT_POS)
#define USB_DWC2_GHWCFG4_SERVINTFLOW_POS	10UL
#define USB_DWC2_GHWCFG4_SERVINTFLOW		BIT(USB_DWC2_GHWCFG4_SERVINTFLOW_POS)
#define USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT1_POS	9UL
#define USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT1	BIT(USB_DWC2_GHWCFG4_ENHANCEDLPMSUPT1_POS)
#define USB_DWC2_GHWCFG4_EXT_HIBERNATION_POS	7UL
#define USB_DWC2_GHWCFG4_EXT_HIBERNATION	BIT(USB_DWC2_GHWCFG4_EXT_HIBERNATION_POS)
#define USB_DWC2_GHWCFG4_HIBERNATION_POS	6UL
#define USB_DWC2_GHWCFG4_HIBERNATION		BIT(USB_DWC2_GHWCFG4_HIBERNATION_POS)
#define USB_DWC2_GHWCFG4_AHBFREQ_POS		5UL
#define USB_DWC2_GHWCFG4_AHBFREQ		BIT(USB_DWC2_GHWCFG4_AHBFREQ_POS)
#define USB_DWC2_GHWCFG4_PARTIALPWRDN_POS	4UL
#define USB_DWC2_GHWCFG4_PARTIALPWRDN		BIT(USB_DWC2_GHWCFG4_PARTIALPWRDN_POS)
#define USB_DWC2_GHWCFG4_NUMDEVPERIOEPS_POS	0UL
#define USB_DWC2_GHWCFG4_NUMDEVPERIOEPS_MASK	(0xFUL << USB_DWC2_GHWCFG4_NUMDEVPERIOEPS_POS)

USB_DWC2_GET_FIELD_DEFINE(ghwcfg4_ineps, GHWCFG4_INEPS)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg4_numctleps, GHWCFG4_NUMCTLEPS)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg4_phydatawidth, GHWCFG4_PHYDATAWIDTH)
USB_DWC2_GET_FIELD_DEFINE(ghwcfg4_numdevperioeps, GHWCFG4_NUMDEVPERIOEPS)

/* LPM Config Register */
#define USB_DWC2_GLPMCFG			0x0054UL
#define USB_DWC2_GLPMCFG_LPM_RESTORESLPSTS_POS	29UL
#define USB_DWC2_GLPMCFG_LPM_RESTORESLPSTS	BIT(USB_DWC2_GLPMCFG_LPM_RESTORESLPSTS_POS)
#define USB_DWC2_GLPMCFG_LPM_ENBESL_POS		28UL
#define USB_DWC2_GLPMCFG_LPM_ENBESL		BIT(USB_DWC2_GLPMCFG_LPM_ENBESL_POS)
#define USB_DWC2_GLPMCFG_LPM_RETRYCNT_STS_POS	25UL
#define USB_DWC2_GLPMCFG_LPM_RETRYCNT_STS_MASK	(0x7UL << USB_DWC2_GLPMCFG_LPM_RETRYCNT_STS_POS)
#define USB_DWC2_GLPMCFG_SNDLPM_POS		24UL
#define USB_DWC2_GLPMCFG_SNDLPM			BIT(USB_DWC2_GLPMCFG_SNDLPM_POS)
/* Host mode LPM Retry Count and LPM Channel Index */
#define USB_DWC2_GLPMCFG_LPM_RETRY_CNT_POS	21UL
#define USB_DWC2_GLPMCFG_LPM_RETRY_CNT_MASK	(0x7UL << USB_DWC2_GLPMCFG_LPM_RETRY_CNT_POS)
#define USB_DWC2_GLPMCFG_LPM_CHNL_INDX_POS	17UL
#define USB_DWC2_GLPMCFG_LPM_CHNL_INDX_MASK	(0xFUL << USB_DWC2_GLPMCFG_LPM_CHNL_INDX_POS)
/* Device mode LPM Accept Control */
#define USB_DWC2_GLPMCFG_LPM_ACK_BULK_POS	23UL
#define USB_DWC2_GLPMCFG_LPM_ACK_BULK		BIT(USB_DWC2_GLPMCFG_LPM_ACK_BULK_POS)
#define USB_DWC2_GLPMCFG_LPM_ACK_ISO_POS	22UL
#define USB_DWC2_GLPMCFG_LPM_ACK_ISO		BIT(USB_DWC2_GLPMCFG_LPM_ACK_ISO_POS)
#define USB_DWC2_GLPMCFG_LPM_NYET_CTRL_POS	21UL
#define USB_DWC2_GLPMCFG_LPM_NYET_CTRL		BIT(USB_DWC2_GLPMCFG_LPM_NYET_CTRL_POS)
#define USB_DWC2_GLPMCFG_LPM_ACK_INTR_POS	20UL
#define USB_DWC2_GLPMCFG_LPM_ACK_INTR		BIT(USB_DWC2_GLPMCFG_LPM_ACK_INTR_POS)
#define USB_DWC2_GLPMCFG_L1RESUMEOK_POS		16UL
#define USB_DWC2_GLPMCFG_L1RESUMEOK		BIT(USB_DWC2_GLPMCFG_L1RESUMEOK_POS)
#define USB_DWC2_GLPMCFG_SLPSTS_POS		15UL
#define USB_DWC2_GLPMCFG_SLPSTS			BIT(USB_DWC2_GLPMCFG_SLPSTS_POS)
#define USB_DWC2_GLPMCFG_COREL1RES_POS		13UL
#define USB_DWC2_GLPMCFG_COREL1RES_MASK		(0x3UL << USB_DWC2_GLPMCFG_COREL1RES_POS)
#define USB_DWC2_GLPMCFG_COREL1RES_ERROR	0
#define USB_DWC2_GLPMCFG_COREL1RES_STALL	1
#define USB_DWC2_GLPMCFG_COREL1RES_NYET		2
#define USB_DWC2_GLPMCFG_COREL1RES_ACK		3
#define USB_DWC2_GLPMCFG_HIRD_THRES_POS		8UL
#define USB_DWC2_GLPMCFG_HIRD_THRES_MASK	(0x1FUL << USB_DWC2_GLPMCFG_HIRD_THRES_POS)
#define USB_DWC2_GLPMCFG_ENBLSLPM_POS		7UL
#define USB_DWC2_GLPMCFG_ENBLSLPM		BIT(USB_DWC2_GLPMCFG_ENBLSLPM_POS)
#define USB_DWC2_GLPMCFG_BREMOTEWAKE_POS	6UL
#define USB_DWC2_GLPMCFG_BREMOTEWAKE		BIT(USB_DWC2_GLPMCFG_BREMOTEWAKE_POS)
#define USB_DWC2_GLPMCFG_HIRD_POS		2UL
#define USB_DWC2_GLPMCFG_HIRD_MASK		(0xFUL << USB_DWC2_GLPMCFG_HIRD_POS)
#define USB_DWC2_GLPMCFG_APPL1RES_POS		1UL
#define USB_DWC2_GLPMCFG_APPL1RES		BIT(USB_DWC2_GLPMCFG_APPL1RES_POS)
#define USB_DWC2_GLPMCFG_LPMCAP_POS		0UL
#define USB_DWC2_GLPMCFG_LPMCAP			BIT(USB_DWC2_GLPMCFG_LPMCAP_POS)

USB_DWC2_GET_FIELD_DEFINE(glpmcfg_lpm_retrycnt_sts, GLPMCFG_LPM_RETRYCNT_STS)
USB_DWC2_GET_FIELD_DEFINE(glpmcfg_lpm_retry_cnt, GLPMCFG_LPM_RETRY_CNT)
USB_DWC2_GET_FIELD_DEFINE(glpmcfg_lpm_chnl_indx, GLPMCFG_LPM_CHNL_INDX)
USB_DWC2_GET_FIELD_DEFINE(glpmcfg_corel1res, GLPMCFG_COREL1RES)
USB_DWC2_GET_FIELD_DEFINE(glpmcfg_hird_thres, GLPMCFG_HIRD_THRES)
USB_DWC2_GET_FIELD_DEFINE(glpmcfg_hird, GLPMCFG_HIRD)
USB_DWC2_SET_FIELD_DEFINE(glpmcfg_lpm_retry_cnt, GLPMCFG_LPM_RETRY_CNT)
USB_DWC2_SET_FIELD_DEFINE(glpmcfg_lpm_chnl_indx, GLPMCFG_LPM_CHNL_INDX)
USB_DWC2_SET_FIELD_DEFINE(glpmcfg_hird_thres, GLPMCFG_HIRD_THRES)
USB_DWC2_SET_FIELD_DEFINE(glpmcfg_hird, GLPMCFG_HIRD)

/* Global Power Down Register */
#define USB_DWC2_GPWRDN				0x0058UL
#define USB_DWC2_GPWRDN_MULTVALIDBC_POS		24UL
#define USB_DWC2_GPWRDN_MULTVALIDBC_MASK	(0x1FUL << USB_DWC2_GPWRDN_MULTVALIDBC_POS)
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_0	0
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_C	1
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_B	2
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_A	4
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_GND	8
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_A_GND	12
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_FLOAT	16
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_C_FLOAT	17
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_B_FLOAT	18
#define USB_DWC2_GPWRDN_MULTVALIDBC_RID_1	31
#define USB_DWC2_GPWRDN_BSESSVLD_POS		22UL
#define USB_DWC2_GPWRDN_BSESSVLD		BIT(USB_DWC2_GPWRDN_BSESSVLD_POS)
#define USB_DWC2_GPWRDN_IDDIG_POS		21UL
#define USB_DWC2_GPWRDN_IDDIG			BIT(USB_DWC2_GPWRDN_IDDIG_POS)
#define USB_DWC2_GPWRDN_LINESTATE_POS		19UL
#define USB_DWC2_GPWRDN_LINESTATE_MASK		(0x3UL << USB_DWC2_GPWRDN_LINESTATE_POS)
#define USB_DWC2_GPWRDN_LINESTATE_DM0DP0	0
#define USB_DWC2_GPWRDN_LINESTATE_DM0DP1	1
#define USB_DWC2_GPWRDN_LINESTATE_DM1DP0	2
#define USB_DWC2_GPWRDN_LINESTATE_NOT_DEFINED	3
#define USB_DWC2_GPWRDN_STSCHNGINTMSK_POS	18UL
#define USB_DWC2_GPWRDN_STSCHNGINTMSK		BIT(USB_DWC2_GPWRDN_STSCHNGINTMSK_POS)
#define USB_DWC2_GPWRDN_STSCHNGINT_POS		17UL
#define USB_DWC2_GPWRDN_STSCHNGINT		BIT(USB_DWC2_GPWRDN_STSCHNGINT_POS)
#define USB_DWC2_GPWRDN_SRPDETECTMSK_POS	16UL
#define USB_DWC2_GPWRDN_SRPDETECTMSK		BIT(USB_DWC2_GPWRDN_SRPDETECTMSK_POS)
#define USB_DWC2_GPWRDN_SRPDETECT_POS		15UL
#define USB_DWC2_GPWRDN_SRPDETECT		BIT(USB_DWC2_GPWRDN_SRPDETECT_POS)
#define USB_DWC2_GPWRDN_CONNDETMSK_POS		14UL
#define USB_DWC2_GPWRDN_CONNDETMSK		BIT(USB_DWC2_GPWRDN_CONNDETMSK_POS)
#define USB_DWC2_GPWRDN_CONNECTDET_POS		13UL
#define USB_DWC2_GPWRDN_CONNECTDET		BIT(USB_DWC2_GPWRDN_CONNECTDET_POS)
#define USB_DWC2_GPWRDN_DISCONNECTDETECTMSK_POS	12UL
#define USB_DWC2_GPWRDN_DISCONNECTDETECTMSK	BIT(USB_DWC2_GPWRDN_DISCONNECTDETECTMSK_POS)
#define USB_DWC2_GPWRDN_DISCONNECTDETECT_POS	11UL
#define USB_DWC2_GPWRDN_DISCONNECTDETECT	BIT(USB_DWC2_GPWRDN_DISCONNECTDETECT_POS)
#define USB_DWC2_GPWRDN_RESETDETMSK_POS		10UL
#define USB_DWC2_GPWRDN_RESETDETMSK		BIT(USB_DWC2_GPWRDN_RESETDETMSK_POS)
#define USB_DWC2_GPWRDN_RESETDETECTED_POS	9UL
#define USB_DWC2_GPWRDN_RESETDETECTED		BIT(USB_DWC2_GPWRDN_RESETDETECTED_POS)
#define USB_DWC2_GPWRDN_LINESTAGECHANGEMSK_POS	8UL
#define USB_DWC2_GPWRDN_LINESTAGECHANGEMSK	BIT(USB_DWC2_GPWRDN_LINESTAGECHANGEMSK_POS)
#define USB_DWC2_GPWRDN_LNSTSCHNG_POS		7UL
#define USB_DWC2_GPWRDN_LNSTSCHNG		BIT(USB_DWC2_GPWRDN_LNSTSCHNG_POS)
#define USB_DWC2_GPWRDN_DISABLEVBUS_POS		6UL
#define USB_DWC2_GPWRDN_DISABLEVBUS		BIT(USB_DWC2_GPWRDN_DISABLEVBUS_POS)
#define USB_DWC2_GPWRDN_PWRDNSWTCH_POS		5UL
#define USB_DWC2_GPWRDN_PWRDNSWTCH		BIT(USB_DWC2_GPWRDN_PWRDNSWTCH_POS)
#define USB_DWC2_GPWRDN_PWRDNRST_N_POS		4UL
#define USB_DWC2_GPWRDN_PWRDNRST_N		BIT(USB_DWC2_GPWRDN_PWRDNRST_N_POS)
#define USB_DWC2_GPWRDN_PWRDNCLMP_POS		3UL
#define USB_DWC2_GPWRDN_PWRDNCLMP		BIT(USB_DWC2_GPWRDN_PWRDNCLMP_POS)
#define USB_DWC2_GPWRDN_RESTORE_POS		2UL
#define USB_DWC2_GPWRDN_RESTORE			BIT(USB_DWC2_GPWRDN_RESTORE_POS)
#define USB_DWC2_GPWRDN_PMUACTV_POS		1UL
#define USB_DWC2_GPWRDN_PMUACTV			BIT(USB_DWC2_GPWRDN_PMUACTV_POS)
#define USB_DWC2_GPWRDN_PMUINTSEL_POS		0UL
#define USB_DWC2_GPWRDN_PMUINTSEL		BIT(USB_DWC2_GPWRDN_PMUINTSEL_POS)

USB_DWC2_GET_FIELD_DEFINE(gpwrdn_multvalidbc, GPWRDN_MULTVALIDBC)
USB_DWC2_GET_FIELD_DEFINE(gpwrdn_linestate, GPWRDN_LINESTATE)

/* GDFIFOCFG register */
#define USB_DWC2_GDFIFOCFG			0x005CUL
#define USB_DWC2_GDFIFOCFG_EPINFOBASEADDR_POS	16UL
#define USB_DWC2_GDFIFOCFG_EPINFOBASEADDR_MASK	(0xFFFFUL << USB_DWC2_GDFIFOCFG_EPINFOBASEADDR_POS)
#define USB_DWC2_GDFIFOCFG_GDFIFOCFG_POS	0UL
#define USB_DWC2_GDFIFOCFG_GDFIFOCFG_MASK	(0xFFFFUL << USB_DWC2_GDFIFOCFG_GDFIFOCFG_POS)

USB_DWC2_GET_FIELD_DEFINE(gdfifocfg_epinfobaseaddr, GDFIFOCFG_EPINFOBASEADDR)
USB_DWC2_GET_FIELD_DEFINE(gdfifocfg_gdfifocfg, GDFIFOCFG_GDFIFOCFG)
USB_DWC2_SET_FIELD_DEFINE(gdfifocfg_epinfobaseaddr, GDFIFOCFG_EPINFOBASEADDR)
USB_DWC2_SET_FIELD_DEFINE(gdfifocfg_gdfifocfg, GDFIFOCFG_GDFIFOCFG)

/* Device IN endpoint transmit FIFO size register */
#define USB_DWC2_DIEPTXF1			0x0104UL
#define USB_DWC2_DIEPTXF_INEPNTXFDEP_POS	16UL
#define USB_DWC2_DIEPTXF_INEPNTXFDEP_MASK	(0xFFFFUL << USB_DWC2_DIEPTXF_INEPNTXFDEP_POS)
#define USB_DWC2_DIEPTXF_INEPNTXFSTADDR_POS	0UL
#define USB_DWC2_DIEPTXF_INEPNTXFSTADDR_MASK	(0xFFFFUL << USB_DWC2_DIEPTXF_INEPNTXFSTADDR_POS)

USB_DWC2_GET_FIELD_DEFINE(dieptxf_inepntxfdep, DIEPTXF_INEPNTXFDEP)
USB_DWC2_GET_FIELD_DEFINE(dieptxf_inepntxfstaddr, DIEPTXF_INEPNTXFSTADDR)
USB_DWC2_SET_FIELD_DEFINE(dieptxf_inepntxfdep, DIEPTXF_INEPNTXFDEP)
USB_DWC2_SET_FIELD_DEFINE(dieptxf_inepntxfstaddr, DIEPTXF_INEPNTXFSTADDR)

/* Device configuration registers */
#define USB_DWC2_DCFG				0x0800UL
#define USB_DWC2_DCFG_RESVALID_POS		26UL
#define USB_DWC2_DCFG_RESVALID_MASK		(0x3FUL << USB_DWC2_DCFG_RESVALID_POS)
#define USB_DWC2_DCFG_PERSCHINTVL_POS		24UL
#define USB_DWC2_DCFG_PERSCHINTVL_MASK		(0x3UL << USB_DWC2_DCFG_PERSCHINTVL_POS)
#define USB_DWC2_DCFG_PERSCHINTVL_MF25		0
#define USB_DWC2_DCFG_PERSCHINTVL_MF50		1
#define USB_DWC2_DCFG_PERSCHINTVL_MF75		2
#define USB_DWC2_DCFG_PERSCHINTVL_RESERVED	3
#define USB_DWC2_DCFG_DESCDMA_POS		23UL
#define USB_DWC2_DCFG_DESCDMA			BIT(USB_DWC2_DCFG_DESCDMA_POS)
#define USB_DWC2_DCFG_EPMISCNT_POS		18UL
#define USB_DWC2_DCFG_EPMISCNT_MASK		(0x1FUL << USB_DWC2_DCFG_EPMISCNT_POS)
#define USB_DWC2_DCFG_IPGISOCSUPT_POS		17UL
#define USB_DWC2_DCFG_IPGISOCSUPT		BIT(USB_DWC2_DCFG_IPGISOCSUPT_POS)
#define USB_DWC2_DCFG_ERRATICINTMSK_POS		15UL
#define USB_DWC2_DCFG_ERRATICINTMSK		BIT(USB_DWC2_DCFG_ERRATICINTMSK_POS)
#define USB_DWC2_DCFG_XCVRDLY_POS		14UL
#define USB_DWC2_DCFG_XCVRDLY			BIT(USB_DWC2_DCFG_XCVRDLY_POS)
#define USB_DWC2_DCFG_ENDEVOUTNAK_POS		13UL
#define USB_DWC2_DCFG_ENDEVOUTNAK		BIT(USB_DWC2_DCFG_ENDEVOUTNAK_POS)
#define USB_DWC2_DCFG_PERFRINT_POS		11UL
#define USB_DWC2_DCFG_PERFRINT_MASK		(0x3UL << USB_DWC2_DCFG_PERFRINT_POS)
#define USB_DWC2_DCFG_PERFRINT_EOPF80		0
#define USB_DWC2_DCFG_PERFRINT_EOPF85		1
#define USB_DWC2_DCFG_PERFRINT_EOPF90		2
#define USB_DWC2_DCFG_PERFRINT_EOPF95		3
#define USB_DWC2_DCFG_DEVADDR_POS		4UL
#define USB_DWC2_DCFG_DEVADDR_MASK		(0x7FUL << USB_DWC2_DCFG_DEVADDR_POS)
#define USB_DWC2_DCFG_ENA32KHZSUSP_POS		3UL
#define USB_DWC2_DCFG_ENA32KHZSUSP		BIT(USB_DWC2_DCFG_ENA32KHZSUSP_POS)
#define USB_DWC2_DCFG_NZSTSOUTHSHK_POS		2UL
#define USB_DWC2_DCFG_NZSTSOUTHSHK		BIT(USB_DWC2_DCFG_NZSTSOUTHSHK_POS)
#define USB_DWC2_DCFG_DEVSPD_POS		0UL
#define USB_DWC2_DCFG_DEVSPD_MASK		(0x03UL << USB_DWC2_DCFG_DEVSPD_POS)
#define USB_DWC2_DCFG_DEVSPD_USBHS20		0
#define USB_DWC2_DCFG_DEVSPD_USBFS20		1
#define USB_DWC2_DCFG_DEVSPD_USBLS116		2
#define USB_DWC2_DCFG_DEVSPD_USBFS1148		3

USB_DWC2_SET_FIELD_DEFINE(dcfg_resvalid, DCFG_RESVALID)
USB_DWC2_SET_FIELD_DEFINE(dcfg_perschintvl, DCFG_PERSCHINTVL)
USB_DWC2_SET_FIELD_DEFINE(dcfg_epmiscnt, DCFG_EPMISCNT)
USB_DWC2_SET_FIELD_DEFINE(dcfg_perfrint, DCFG_PERFRINT)
USB_DWC2_SET_FIELD_DEFINE(dcfg_devaddr, DCFG_DEVADDR)
USB_DWC2_SET_FIELD_DEFINE(dcfg_devspd, DCFG_DEVSPD)
USB_DWC2_GET_FIELD_DEFINE(dcfg_resvalid, DCFG_RESVALID)
USB_DWC2_GET_FIELD_DEFINE(dcfg_perschintvl, DCFG_PERSCHINTVL)
USB_DWC2_GET_FIELD_DEFINE(dcfg_epmiscnt, DCFG_EPMISCNT)
USB_DWC2_GET_FIELD_DEFINE(dcfg_perfrint, DCFG_PERFRINT)
USB_DWC2_GET_FIELD_DEFINE(dcfg_devaddr, DCFG_DEVADDR)
USB_DWC2_GET_FIELD_DEFINE(dcfg_devspd, DCFG_DEVSPD)

/* Device control register */
#define USB_DWC2_DCTL				0x0804UL
#define USB_DWC2_DCTL_SERVINT_POS		19UL
#define USB_DWC2_DCTL_SERVINT			BIT(USB_DWC2_DCTL_SERVINT_POS)
#define USB_DWC2_DCTL_DEEPSLEEPBESLREJECT_POS	18UL
#define USB_DWC2_DCTL_DEEPSLEEPBESLREJECT	BIT(USB_DWC2_DCTL_DEEPSLEEPBESLREJECT_POS)
#define USB_DWC2_DCTL_NAKONBBLE_POS		16UL
#define USB_DWC2_DCTL_NAKONBBLE			BIT(USB_DWC2_DCTL_NAKONBBLE_POS)
#define USB_DWC2_DCTL_IGNRFRMNUM_POS		15UL
#define USB_DWC2_DCTL_IGNRFRMNUM		BIT(USB_DWC2_DCTL_IGNRFRMNUM_POS)
#define USB_DWC2_DCTL_PWRONPRGDONE_POS		11UL
#define USB_DWC2_DCTL_PWRONPRGDONE		BIT(USB_DWC2_DCTL_PWRONPRGDONE_POS)
#define USB_DWC2_DCTL_CGOUTNAK_POS		10UL
#define USB_DWC2_DCTL_CGOUTNAK			BIT(USB_DWC2_DCTL_CGOUTNAK_POS)
#define USB_DWC2_DCTL_SGOUTNAK_POS		9UL
#define USB_DWC2_DCTL_SGOUTNAK			BIT(USB_DWC2_DCTL_SGOUTNAK_POS)
#define USB_DWC2_DCTL_CGNPINNAK_POS		8UL
#define USB_DWC2_DCTL_CGNPINNAK			BIT(USB_DWC2_DCTL_CGNPINNAK_POS)
#define USB_DWC2_DCTL_SGNPINNAK_POS		7UL
#define USB_DWC2_DCTL_SGNPINNAK			BIT(USB_DWC2_DCTL_SGNPINNAK_POS)
#define USB_DWC2_DCTL_TSTCTL_POS		4UL
#define USB_DWC2_DCTL_TSTCTL_MASK		(0x7UL << USB_DWC2_DCTL_TSTCTL_POS)
#define USB_DWC2_DCTL_TSTCTL_TESTFE		5UL
#define USB_DWC2_DCTL_TSTCTL_TESTPM		4UL
#define USB_DWC2_DCTL_TSTCTL_TESTSN		3UL
#define USB_DWC2_DCTL_TSTCTL_TESTK		2UL
#define USB_DWC2_DCTL_TSTCTL_TESTJ		1UL
#define USB_DWC2_DCTL_TSTCTL_DISABLED		0UL
#define USB_DWC2_DCTL_GOUTNAKSTS_POS		3UL
#define USB_DWC2_DCTL_GOUTNAKSTS		BIT(USB_DWC2_DCTL_GOUTNAKSTS_POS)
#define USB_DWC2_DCTL_GNPINNAKSTS_POS		2UL
#define USB_DWC2_DCTL_GNPINNAKSTS		BIT(USB_DWC2_DCTL_GNPINNAKSTS_POS)
#define USB_DWC2_DCTL_SFTDISCON_POS		1UL
#define USB_DWC2_DCTL_SFTDISCON			BIT(USB_DWC2_DCTL_SFTDISCON_POS)
#define USB_DWC2_DCTL_RMTWKUPSIG_POS		0UL
#define USB_DWC2_DCTL_RMTWKUPSIG		BIT(USB_DWC2_DCTL_RMTWKUPSIG_POS)

USB_DWC2_GET_FIELD_DEFINE(dctl_tstctl, DCTL_TSTCTL)
USB_DWC2_SET_FIELD_DEFINE(dctl_tstctl, DCTL_TSTCTL)

/* Device status register */
#define USB_DWC2_DSTS				0x0808UL
#define USB_DWC2_DSTS_DEVLNSTS_POS		22UL
#define USB_DWC2_DSTS_DEVLNSTS_MASK		(0x3UL << USB_DWC2_DSTS_DEVLNSTS_POS)
#define USB_DWC2_DSTS_SOFFN_POS			8UL
#define USB_DWC2_DSTS_SOFFN_MASK		(0x3FFFUL << USB_DWC2_DSTS_SOFFN_POS)
#define USB_DWC2_DSTS_ERRTICERR_POS		3UL
#define USB_DWC2_DSTS_ERRTICERR			BIT(USB_DWC2_DSTS_ERRTICERR_POS)
#define USB_DWC2_DSTS_ENUMSPD_POS		1UL
#define USB_DWC2_DSTS_ENUMSPD_MASK		(0x3UL << USB_DWC2_DSTS_ENUMSPD_POS)
#define USB_DWC2_DSTS_ENUMSPD_HS3060		0
#define USB_DWC2_DSTS_ENUMSPD_FS3060		1
#define USB_DWC2_DSTS_ENUMSPD_LS6		2
#define USB_DWC2_DSTS_ENUMSPD_FS48		3
#define USB_DWC2_DSTS_SUSPSTS_POS		0UL
#define USB_DWC2_DSTS_SUSPSTS			BIT(USB_DWC2_DSTS_SUSPSTS_POS)

USB_DWC2_GET_FIELD_DEFINE(dsts_devlnsts, DSTS_DEVLNSTS)
USB_DWC2_GET_FIELD_DEFINE(dsts_soffn, DSTS_SOFFN)
USB_DWC2_GET_FIELD_DEFINE(dsts_enumspd, DSTS_ENUMSPD)

/* Device all endpoints interrupt registers */
#define USB_DWC2_DAINT				0x0818UL
#define USB_DWC2_DAINT_OUTEPINT_POS		16UL
#define USB_DWC2_DAINT_OUTEPINT_MASK		(0xFFFFUL << USB_DWC2_DAINT_OUTEPINT_POS)
#define USB_DWC2_DAINT_INEPINT_POS		0UL
#define USB_DWC2_DAINT_INEPINT_MASK		(0xFFFFUL << USB_DWC2_DAINT_INEPINT_POS)

USB_DWC2_GET_FIELD_DEFINE(daint_outepint, DAINT_OUTEPINT)
USB_DWC2_GET_FIELD_DEFINE(daint_inepint, DAINT_INEPINT)

#define USB_DWC2_DAINTMSK			0x081CUL
#define USB_DWC2_DAINT_OUTEPINT(ep_num)		BIT(16UL + ep_num)
#define USB_DWC2_DAINT_INEPINT(ep_num)		BIT(ep_num)

/* Device threshold control register */
#define USB_DWC2_DTHRCTL			0x0830UL
#define USB_DWC2_DTHRCTL_ARBPRKEN_POS		27UL
#define USB_DWC2_DTHRCTL_ARBPRKEN		BIT(USB_DWC2_DTHRCTL_ARBPRKEN_POS)
#define USB_DWC2_DTHRCTL_RXTHRLEN_POS		17UL
#define USB_DWC2_DTHRCTL_RXTHRLEN_MASK		(0x1FFUL << USB_DWC2_DTHRCTL_RXTHRLEN_POS)
#define USB_DWC2_DTHRCTL_RXTHREN_POS		16UL
#define USB_DWC2_DTHRCTL_RXTHREN		BIT(USB_DWC2_DTHRCTL_RXTHREN_POS)
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_POS	11UL
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_MASK	(0x3UL << USB_DWC2_DTHRCTL_AHBTHRRATIO_POS)
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_THRESZERO	0
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_THRESONE	1
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_THRESTWO	2
#define USB_DWC2_DTHRCTL_AHBTHRRATIO_THRESTHREE	3
#define USB_DWC2_DTHRCTL_TXTHRLEN_POS		2UL
#define USB_DWC2_DTHRCTL_TXTHRLEN_MASK		(0x1FFUL << USB_DWC2_DTHRCTL_TXTHRLEN_POS)
#define USB_DWC2_DTHRCTL_ISOTHREN_POS		1UL
#define USB_DWC2_DTHRCTL_ISOTHREN		BIT(USB_DWC2_DTHRCTL_ISOTHREN_POS)
#define USB_DWC2_DTHRCTL_NONISOTHREN_POS	0UL
#define USB_DWC2_DTHRCTL_NONISOTHREN		BIT(USB_DWC2_DTHRCTL_NONISOTHREN_POS)

USB_DWC2_GET_FIELD_DEFINE(dthrctl_rxthrlen, DTHRCTL_RXTHRLEN)
USB_DWC2_GET_FIELD_DEFINE(dthrctl_ahbthrratio, DTHRCTL_AHBTHRRATIO)
USB_DWC2_GET_FIELD_DEFINE(dthrctl_txthrlen, DTHRCTL_TXTHRLEN)
USB_DWC2_SET_FIELD_DEFINE(dthrctl_rxthrlen, DTHRCTL_RXTHRLEN)
USB_DWC2_SET_FIELD_DEFINE(dthrctl_ahbthrratio, DTHRCTL_AHBTHRRATIO)
USB_DWC2_SET_FIELD_DEFINE(dthrctl_txthrlen, DTHRCTL_TXTHRLEN)

/*
 * Device IN/OUT endpoint control register
 * IN endpoint offsets 0x0900 + (0x20 * n), n = 0 .. x,
 * OUT endpoint offsets 0x0B00 + (0x20 * n), n = 0 .. x,
 *
 */
#define USB_DWC2_DIEPCTL0			0x0900UL
#define USB_DWC2_DOEPCTL0			0x0B00UL
#define USB_DWC2_DEPCTL_EPENA_POS		31UL
#define USB_DWC2_DEPCTL_EPENA			BIT(USB_DWC2_DEPCTL_EPENA_POS)
#define USB_DWC2_DEPCTL_EPDIS_POS		30UL
#define USB_DWC2_DEPCTL_EPDIS			BIT(USB_DWC2_DEPCTL_EPDIS_POS)
#define USB_DWC2_DEPCTL_SETD1PID_POS		29UL
#define USB_DWC2_DEPCTL_SETD1PID		BIT(USB_DWC2_DEPCTL_SETD1PID_POS)
#define USB_DWC2_DEPCTL_SETODDFR		USB_DWC2_DEPCTL_SETD1PID
#define USB_DWC2_DEPCTL_SETD0PID_POS		28UL
#define USB_DWC2_DEPCTL_SETD0PID		BIT(USB_DWC2_DEPCTL_SETD0PID_POS)
#define USB_DWC2_DEPCTL_SETEVENFR		USB_DWC2_DEPCTL_SETD0PID
#define USB_DWC2_DEPCTL_SNAK_POS		27UL
#define USB_DWC2_DEPCTL_SNAK			BIT(USB_DWC2_DEPCTL_SNAK_POS)
#define USB_DWC2_DEPCTL_CNAK_POS		26UL
#define USB_DWC2_DEPCTL_CNAK			BIT(USB_DWC2_DEPCTL_CNAK_POS)
#define USB_DWC2_DEPCTL_TXFNUM_POS		22UL
#define USB_DWC2_DEPCTL_TXFNUM_MASK		(0xFUL << USB_DWC2_DEPCTL_TXFNUM_POS)
#define USB_DWC2_DEPCTL_STALL_POS		21UL
#define USB_DWC2_DEPCTL_STALL			BIT(USB_DWC2_DEPCTL_STALL_POS)
#define USB_DWC2_DEPCTL_EPTYPE_POS		18UL
#define USB_DWC2_DEPCTL_EPTYPE_MASK		(0x3UL << USB_DWC2_DEPCTL_EPTYPE_POS)
#define USB_DWC2_DEPCTL_EPTYPE_INTERRUPT	3
#define USB_DWC2_DEPCTL_EPTYPE_BULK		2
#define USB_DWC2_DEPCTL_EPTYPE_ISO		1
#define USB_DWC2_DEPCTL_EPTYPE_CONTROL		0
#define USB_DWC2_DEPCTL_NAKSTS_POS		17UL
#define USB_DWC2_DEPCTL_NAKSTS			BIT(USB_DWC2_DEPCTL_NAKSTS_POS)
#define USB_DWC2_DEPCTL_DPID_POS		16UL
#define USB_DWC2_DEPCTL_DPID			BIT(USB_DWC2_DEPCTL_DPID_POS)
#define USB_DWC2_DEPCTL_USBACTEP_POS		15UL
#define USB_DWC2_DEPCTL_USBACTEP		BIT(USB_DWC2_DEPCTL_USBACTEP_POS)
#define USB_DWC2_DEPCTL0_MPS_POS		0UL
#define USB_DWC2_DEPCTL0_MPS_MASK		(0x3UL << USB_DWC2_DEPCTL0_MPS_POS)
#define USB_DWC2_DEPCTL0_MPS_8			3
#define USB_DWC2_DEPCTL0_MPS_16			2
#define USB_DWC2_DEPCTL0_MPS_32			1
#define USB_DWC2_DEPCTL0_MPS_64			0
#define USB_DWC2_DEPCTL_MPS_POS			0UL
#define USB_DWC2_DEPCTL_MPS_MASK		(0x7FF << USB_DWC2_DEPCTL_MPS_POS)

USB_DWC2_GET_FIELD_DEFINE(depctl_txfnum, DEPCTL_TXFNUM)
USB_DWC2_SET_FIELD_DEFINE(depctl_txfnum, DEPCTL_TXFNUM)
USB_DWC2_GET_FIELD_DEFINE(depctl_eptype, DEPCTL_EPTYPE)
USB_DWC2_SET_FIELD_DEFINE(depctl_eptype, DEPCTL_EPTYPE)
USB_DWC2_GET_FIELD_DEFINE(depctl0_mps, DEPCTL0_MPS)
USB_DWC2_SET_FIELD_DEFINE(depctl0_mps, DEPCTL0_MPS)
USB_DWC2_GET_FIELD_DEFINE(depctl_mps, DEPCTL_MPS)
USB_DWC2_SET_FIELD_DEFINE(depctl_mps, DEPCTL_MPS)

/*
 * Device IN endpoint interrupt register
 * offsets 0x0908 + (0x20 * n), n = 0 .. x
 */
#define USB_DWC2_DIEPINT0			0x0908UL
#define USB_DWC2_DIEPINT_NYETINTRPT_POS		14UL
#define USB_DWC2_DIEPINT_NYETINTRPT		BIT(USB_DWC2_DIEPINT_NYETINTRPT_POS)
#define USB_DWC2_DIEPINT_NAKINTRPT_POS		13UL
#define USB_DWC2_DIEPINT_NAKINTRPT		BIT(USB_DWC2_DIEPINT_NAKINTRPT_POS)
#define USB_DWC2_DIEPINT_BBLEERR_POS		12UL
#define USB_DWC2_DIEPINT_BBLEERR		BIT(USB_DWC2_DIEPINT_BBLEERR_POS)
#define USB_DWC2_DIEPINT_PKTDRPSTS_POS		11UL
#define USB_DWC2_DIEPINT_PKTDRPSTS		BIT(USB_DWC2_DIEPINT_PKTDRPSTS_POS)
#define USB_DWC2_DIEPINT_BNAINTR_POS		9UL
#define USB_DWC2_DIEPINT_BNAINTR		BIT(USB_DWC2_DIEPINT_BNAINTR_POS)
#define USB_DWC2_DIEPINT_TXFIFOUNDRN_POS	8UL
#define USB_DWC2_DIEPINT_TXFIFOUNDRN		BIT(USB_DWC2_DIEPINT_TXFIFOUNDRN_POS)
#define USB_DWC2_DIEPINT_TXFEMP_POS		7UL
#define USB_DWC2_DIEPINT_TXFEMP			BIT(USB_DWC2_DIEPINT_TXFEMP_POS)
#define USB_DWC2_DIEPINT_INEPNAKEFF_POS		6UL
#define USB_DWC2_DIEPINT_INEPNAKEFF		BIT(USB_DWC2_DIEPINT_INEPNAKEFF_POS)
#define USB_DWC2_DIEPINT_INTKNEPMIS_POS		5UL
#define USB_DWC2_DIEPINT_INTKNEPMIS		BIT(USB_DWC2_DIEPINT_INTKNEPMIS_POS)
#define USB_DWC2_DIEPINT_INTKNTXFEMP_POS	4UL
#define USB_DWC2_DIEPINT_INTKNTXFEMP		BIT(USB_DWC2_DIEPINT_INTKNTXFEMP_POS)
#define USB_DWC2_DIEPINT_TIMEOUT_POS		3UL
#define USB_DWC2_DIEPINT_TIMEOUT		BIT(USB_DWC2_DIEPINT_TIMEOUT_POS)
#define USB_DWC2_DIEPINT_AHBERR_POS		2UL
#define USB_DWC2_DIEPINT_AHBERR			BIT(USB_DWC2_DIEPINT_AHBERR_POS)
#define USB_DWC2_DIEPINT_EPDISBLD_POS		1UL
#define USB_DWC2_DIEPINT_EPDISBLD		BIT(USB_DWC2_DIEPINT_EPDISBLD_POS)
#define USB_DWC2_DIEPINT_XFERCOMPL_POS		0UL
#define USB_DWC2_DIEPINT_XFERCOMPL		BIT(USB_DWC2_DIEPINT_XFERCOMPL_POS)

/*
 * Device OUT endpoint interrupt register
 * offsets 0x0B08 + (0x20 * n), n = 0 .. x
 */
#define USB_DWC2_DOEPINT0			0x0B08UL
#define USB_DWC2_DOEPINT_STUPPKTRCVD_POS	15UL
#define USB_DWC2_DOEPINT_STUPPKTRCVD		BIT(USB_DWC2_DOEPINT_STUPPKTRCVD_POS)
#define USB_DWC2_DOEPINT_NYETINTRPT_POS		14UL
#define USB_DWC2_DOEPINT_NYETINTRPT		BIT(USB_DWC2_DOEPINT_NYETINTRPT_POS)
#define USB_DWC2_DOEPINT_NAKINTRPT_POS		13UL
#define USB_DWC2_DOEPINT_NAKINTRPT		BIT(USB_DWC2_DOEPINT_NAKINTRPT_POS)
#define USB_DWC2_DOEPINT_BBLEERR_POS		12UL
#define USB_DWC2_DOEPINT_BBLEERR		BIT(USB_DWC2_DOEPINT_BBLEERR_POS)
#define USB_DWC2_DOEPINT_PKTDRPSTS_POS		11UL
#define USB_DWC2_DOEPINT_PKTDRPSTS		BIT(USB_DWC2_DOEPINT_PKTDRPSTS_POS)
#define USB_DWC2_DOEPINT_BNAINTR_POS		9UL
#define USB_DWC2_DOEPINT_BNAINTR		BIT(USB_DWC2_DOEPINT_BNAINTR_POS)
#define USB_DWC2_DOEPINT_OUTPKTERR_POS		8UL
#define USB_DWC2_DOEPINT_OUTPKTERR		BIT(USB_DWC2_DOEPINT_OUTPKTERR_POS)
#define USB_DWC2_DOEPINT_BACK2BACKSETUP_POS	6UL
#define USB_DWC2_DOEPINT_BACK2BACKSETUP		BIT(USB_DWC2_DOEPINT_BACK2BACKSETUP_POS)
#define USB_DWC2_DOEPINT_STSPHSERCVD_POS	5UL
#define USB_DWC2_DOEPINT_STSPHSERCVD		BIT(USB_DWC2_DOEPINT_STSPHSERCVD_POS)
#define USB_DWC2_DOEPINT_OUTTKNEPDIS_POS	4UL
#define USB_DWC2_DOEPINT_OUTTKNEPDIS		BIT(USB_DWC2_DOEPINT_OUTTKNEPDIS_POS)
#define USB_DWC2_DOEPINT_SETUP_POS		3UL
#define USB_DWC2_DOEPINT_SETUP			BIT(USB_DWC2_DOEPINT_SETUP_POS)
#define USB_DWC2_DOEPINT_AHBERR_POS		2UL
#define USB_DWC2_DOEPINT_AHBERR			BIT(USB_DWC2_DOEPINT_AHBERR_POS)
#define USB_DWC2_DOEPINT_EPDISBLD_POS		1UL
#define USB_DWC2_DOEPINT_EPDISBLD		BIT(USB_DWC2_DOEPINT_EPDISBLD_POS)
#define USB_DWC2_DOEPINT_XFERCOMPL_POS		0UL
#define USB_DWC2_DOEPINT_XFERCOMPL		BIT(USB_DWC2_DOEPINT_XFERCOMPL_POS)

/* Device IN control endpoint transfer size register */
#define USB_DWC2_DIEPTSIZ0			0x0910UL
#define USB_DWC2_DIEPTSIZ0_PKTCNT_POS		19UL
#define USB_DWC2_DIEPTSIZ0_PKTCNT_MASK		(0x3UL << USB_DWC2_DIEPTSIZ0_PKTCNT_POS)
#define USB_DWC2_DIEPTSIZ0_XFERSIZE_POS		0UL
#define USB_DWC2_DIEPTSIZ0_XFERSIZE_MASK	0x7FUL

USB_DWC2_GET_FIELD_DEFINE(dieptsiz0_pktcnt, DIEPTSIZ0_PKTCNT)
USB_DWC2_GET_FIELD_DEFINE(dieptsiz0_xfersize, DIEPTSIZ0_XFERSIZE)

/* Device OUT control endpoint transfer size register */
#define USB_DWC2_DOEPTSIZ0			0x0B10UL
#define USB_DWC2_DOEPTSIZ0_SUPCNT_POS		29UL
#define USB_DWC2_DOEPTSIZ0_SUPCNT_MASK		(0x3UL << USB_DWC2_DOEPTSIZ0_SUPCNT_POS)
#define USB_DWC2_DOEPTSIZ0_PKTCNT_POS		19UL
#define USB_DWC2_DOEPTSIZ0_PKTCNT_MASK		(0x1UL << USB_DWC2_DOEPTSIZ0_PKTCNT_POS)
#define USB_DWC2_DOEPTSIZ0_XFERSIZE_POS		0UL
#define USB_DWC2_DOEPTSIZ0_XFERSIZE_MASK	0x7FUL

USB_DWC2_GET_FIELD_DEFINE(doeptsiz0_supcnt, DOEPTSIZ0_SUPCNT)
USB_DWC2_GET_FIELD_DEFINE(doeptsiz0_pktcnt, DOEPTSIZ0_PKTCNT)
USB_DWC2_GET_FIELD_DEFINE(doeptsiz0_xfersize, DOEPTSIZ0_XFERSIZE)

/*
 * Device IN endpoint transfer size register
 * at offsets 0x0910 + (0x20 * n), n = 1 .. x
 */
#define USB_DWC2_DIEPTSIZN_MC_POS		29UL
#define USB_DWC2_DIEPTSIZN_MC_MASK		(0x3UL << USB_DWC2_DIEPTSIZN_MC_POS)
#define USB_DWC2_DIEPTSIZN_PKTCNT_POS		19UL
#define USB_DWC2_DIEPTSIZN_PKTCNT_MASK		(0x3FFUL << USB_DWC2_DIEPTSIZN_PKTCNT_POS)
#define USB_DWC2_DIEPTSIZN_XFERSIZE_POS		0UL
#define USB_DWC2_DIEPTSIZN_XFERSIZE_MASK	0x7FFFFUL

USB_DWC2_GET_FIELD_DEFINE(dieptsizn_mc, DIEPTSIZN_MC)
USB_DWC2_GET_FIELD_DEFINE(dieptsizn_pktcnt, DIEPTSIZN_PKTCNT)
USB_DWC2_GET_FIELD_DEFINE(dieptsizn_xfersize, DIEPTSIZN_XFERSIZE)
USB_DWC2_SET_FIELD_DEFINE(dieptsizn_mc, DIEPTSIZN_MC)
USB_DWC2_SET_FIELD_DEFINE(dieptsizn_pktcnt, DIEPTSIZN_PKTCNT)
USB_DWC2_SET_FIELD_DEFINE(dieptsizn_xfersize, DIEPTSIZN_XFERSIZE)

/*
 * Device OUT endpoint transfer size register
 * at offsets 0x0B10 + (0x20 * n), n = 1 .. x
 */
#define USB_DWC2_DOEPTSIZN_RXDPID_POS		29UL
#define USB_DWC2_DOEPTSIZN_RXDPID_MASK		(0x3UL << USB_DWC2_DOEPTSIZN_RXDPID_POS)
#define USB_DWC2_DOEPTSIZN_RXDPID_MDATA		3
#define USB_DWC2_DOEPTSIZN_RXDPID_DATA1		2
#define USB_DWC2_DOEPTSIZN_RXDPID_DATA2		1
#define USB_DWC2_DOEPTSIZN_RXDPID_DATA0		0
#define USB_DWC2_DOEPTSIZN_PKTCNT_POS		19UL
#define USB_DWC2_DOEPTSIZN_PKTCNT_MASK		(0x3FFUL << USB_DWC2_DOEPTSIZN_PKTCNT_POS)
#define USB_DWC2_DOEPTSIZN_XFERSIZE_POS		0UL
#define USB_DWC2_DOEPTSIZN_XFERSIZE_MASK	0x7FFFFUL

USB_DWC2_GET_FIELD_DEFINE(doeptsizn_rxdpid, DOEPTSIZN_RXDPID)
USB_DWC2_GET_FIELD_DEFINE(doeptsizn_pktcnt, DOEPTSIZN_PKTCNT)
USB_DWC2_GET_FIELD_DEFINE(doeptsizn_xfersize, DOEPTSIZN_XFERSIZE)
USB_DWC2_SET_FIELD_DEFINE(doeptsizn_pktcnt, DOEPTSIZN_PKTCNT)
USB_DWC2_SET_FIELD_DEFINE(doeptsizn_xfersize, DOEPTSIZN_XFERSIZE)

/* Power and Clock Gating Control Register */
#define USB_DWC2_PCGCCTL			0x0E00UL
#define USB_DWC2_PCGCCTL_IF_DEV_MODE_POS	31UL
#define USB_DWC2_PCGCCTL_IF_DEV_MODE		BIT(USB_DWC2_PCGCCTL_IF_DEV_MODE_POS)
#define USB_DWC2_PCGCCTL_P2HD_PRT_SPD_POS	29UL
#define USB_DWC2_PCGCCTL_P2HD_PRT_SPD_MASK	(0x3UL << USB_DWC2_PCGCCTL_P2HD_PRT_SPD_POS)
#define USB_DWC2_PCGCCTL_P2HD_PRT_SPD_LS	2
#define USB_DWC2_PCGCCTL_P2HD_PRT_SPD_FS	1
#define USB_DWC2_PCGCCTL_P2HD_PRT_SPD_HS	0
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_POS	27UL
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_MASK	(0x3UL << USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_POS)
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_FS48	3
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_LS	2
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_FS	1
#define USB_DWC2_PCGCCTL_P2HD_DEV_ENUM_SPD_HS	0
#define USB_DWC2_PCGCCTL_MAC_DEV_ADDR_POS	20UL
#define USB_DWC2_PCGCCTL_MAC_DEV_ADDR_MASK	(0x7FUL << USB_DWC2_PCGCCTL_MAC_DEV_ADDR_POS)
#define USB_DWC2_PCGCCTL_MAX_TERMSELECT_POS	19UL
#define USB_DWC2_PCGCCTL_MAX_TERMSELECT		BIT(USB_DWC2_PCGCCTL_MAX_TERMSELECT_POS)
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_POS	17UL
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_MASK	(0x3UL << USB_DWC2_PCGCCTL_MAC_XCVRSELECT_POS)
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_LFS	3
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_LS	2
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_FS	1
#define USB_DWC2_PCGCCTL_MAC_XCVRSELECT_HS	0
#define USB_DWC2_PCGCCTL_SH2PL_PRT_CTL0_POS	16UL
#define USB_DWC2_PCGCCTL_SH2PL_PRT_CTL0		BIT(USB_DWC2_PCGCCTL_SH2PL_PRT_CTL0_POS)
#define USB_DWC2_PCGCCTL_PRT_CLK_SEL_POS	14UL
#define USB_DWC2_PCGCCTL_PRT_CLK_SEL_MASK	(0x3UL << USB_DWC2_PCGCCTL_PRT_CLK_SEL_POS)
#define USB_DWC2_PCGCCTL_RESTOREVALUE_POS	14UL
#define USB_DWC2_PCGCCTL_RESTOREVALUE_MASK	(0x3FFFFUL << USB_DWC2_PCGCCTL_RESTOREVALUE_POS)
#define USB_DWC2_PCGCCTL_ESSREGRESTORED_POS	13UL
#define USB_DWC2_PCGCCTL_ESSREGRESTORED		BIT(USB_DWC2_PCGCCTL_ESSREGRESTORED_POS)
#define USB_DWC2_PCGCCTL_RESTOREMODE_POS	9UL
#define USB_DWC2_PCGCCTL_RESTOREMODE		BIT(USB_DWC2_PCGCCTL_RESTOREMODE_POS)
#define USB_DWC2_PCGCCTL_L1SUSPENDED_POS	7UL
#define USB_DWC2_PCGCCTL_L1SUSPENDED		BIT(USB_DWC2_PCGCCTL_L1SUSPENDED_POS)
#define USB_DWC2_PCGCCTL_PHYSLEEP_POS		6UL
#define USB_DWC2_PCGCCTL_PHYSLEEP		BIT(USB_DWC2_PCGCCTL_PHYSLEEP_POS)
#define USB_DWC2_PCGCCTL_ENBL_L1GATING_POS	5UL
#define USB_DWC2_PCGCCTL_ENBL_L1GATING		BIT(USB_DWC2_PCGCCTL_ENBL_L1GATING_POS)
#define USB_DWC2_PCGCCTL_RSTPDWNMODULE_POS	3UL
#define USB_DWC2_PCGCCTL_RSTPDWNMODULE		BIT(USB_DWC2_PCGCCTL_RSTPDWNMODULE_POS)
#define USB_DWC2_PCGCCTL_GATEHCLK_POS		1UL
#define USB_DWC2_PCGCCTL_GATEHCLK		BIT(USB_DWC2_PCGCCTL_GATEHCLK_POS)
#define USB_DWC2_PCGCCTL_STOPPCLK_POS		0UL
#define USB_DWC2_PCGCCTL_STOPPCLK		BIT(USB_DWC2_PCGCCTL_STOPPCLK_POS)

USB_DWC2_GET_FIELD_DEFINE(pcgcctl_p2hd_prt_spd, PCGCCTL_P2HD_PRT_SPD)
USB_DWC2_GET_FIELD_DEFINE(pcgcctl_p2hd_dev_enum_spd, PCGCCTL_P2HD_DEV_ENUM_SPD)
USB_DWC2_GET_FIELD_DEFINE(pcgcctl_mac_dev_addr, PCGCCTL_MAC_DEV_ADDR)
USB_DWC2_GET_FIELD_DEFINE(pcgcctl_mac_xcvrselect, PCGCCTL_MAC_XCVRSELECT)
USB_DWC2_GET_FIELD_DEFINE(pcgcctl_prt_clk_sel, PCGCCTL_PRT_CLK_SEL)
USB_DWC2_GET_FIELD_DEFINE(pcgcctl_restorevalue, PCGCCTL_RESTOREVALUE)
USB_DWC2_SET_FIELD_DEFINE(pcgcctl_restorevalue, PCGCCTL_RESTOREVALUE)

/*
 * Device IN/OUT endpoint transfer size register
 * IN at offsets 0x0910 + (0x20 * n), n = 0 .. x,
 * OUT at offsets 0x0B10 + (0x20 * n), n = 0 .. x
 *
 * Note: Legacy definitions for the usb_dc_dw.c driver only.
 */
#define USB_DWC2_DEPTSIZ_PKT_CNT_POS		19UL
#define USB_DWC2_DIEPTSIZ0_PKT_CNT_MASK		(0x3 << 19)
#define USB_DWC2_DIEPTSIZn_PKT_CNT_MASK		(0x3FF << 19)
#define USB_DWC2_DOEPTSIZn_PKT_CNT_MASK		(0x3FF << 19)
#define USB_DWC2_DOEPTSIZ0_PKT_CNT_MASK		(0x1 << 19)
#define USB_DWC2_DOEPTSIZ_SUP_CNT_POS		29UL
#define USB_DWC2_DOEPTSIZ_SUP_CNT_MASK		(0x3 << 29)
#define USB_DWC2_DEPTSIZ_XFER_SIZE_POS		0UL
#define USB_DWC2_DEPTSIZ0_XFER_SIZE_MASK	0x7F
#define USB_DWC2_DEPTSIZn_XFER_SIZE_MASK	0x7FFFF

/*
 * Device IN endpoint transmit FIFO status register,
 * offsets 0x0918 + (0x20 * n), n = 0 .. x
 */
#define USB_DWC2_DTXFSTS0			0x0918UL
#define USB_DWC2_DTXFSTS_INEPTXFSPCAVAIL_POS	0UL
#define USB_DWC2_DTXFSTS_INEPTXFSPCAVAIL_MASK	0xFFFFUL

USB_DWC2_GET_FIELD_DEFINE(dtxfsts_ineptxfspcavail, DTXFSTS_INEPTXFSPCAVAIL)

#ifdef __cplusplus
}
#endif

#endif /* ZEPHYR_DRIVERS_USB_COMMON_USB_DWC2_HW */
