/*
 * Copyright (c) 2022 Nordic Semiconductor ASA
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef ZEPHYR_INCLUDE_UHC_MAX3421E_H
#define ZEPHYR_INCLUDE_UHC_MAX3421E_H

#include <zephyr/sys/util_macro.h>

#define MAX3421E_MAX_EP_SIZE		64U


/* SPI command byte format macros */
#define MAX3421E_CMD_REG_SHIFT		3U
#define MAX3421E_CMD_DIR_WR		BIT(1)
#define MAX3421E_CMD_DIR_RD		0U

#define MAX3421E_CMD_SPI_READ(reg) \
	(((reg) << MAX3421E_CMD_REG_SHIFT) | MAX3421E_CMD_DIR_RD)

#define MAX3421E_CMD_SPI_WRITE(reg) \
	(((reg) << MAX3421E_CMD_REG_SHIFT) | MAX3421E_CMD_DIR_WR)


/* Below are all the register definitions for the host mode. */

/* Register RCVFIFO */
#define MAX3421E_REG_RCVFIFO		1U

/* Register SNDFIFO */
#define MAX3421E_REG_SNDFIFO		2U

/* Register SUDFIFO */
#define MAX3421E_REG_SUDFIFO		4U

/* Register RCVBC */
#define MAX3421E_REG_RCVBC		6U
#define MAX3421E_RCVBC_MAX		0x7FU

/* Register SNDBC */
#define MAX3421E_REG_SNDBC		7U
#define MAX3421E_SNDBC_MAX		0x7FU

/* Register USBIRQ */
#define MAX3421E_REG_USBIRQ		13U
#define MAX3421E_VBUSIRQ		BIT(6)
#define MAX3421E_NOVBUSIRQ		BIT(5)
#define MAX3421E_OSCOKIRQ		BIT(0)

/* Register USBIEN */
#define MAX3421E_REG_USBIEN		14U
#define MAX3421E_VBUSIE			BIT(6)
#define MAX3421E_NOVBUSIE		BIT(5)
#define MAX3421E_OSCOKIE		BIT(0)

/* Register USBCTL */
#define MAX3421E_REG_USBCTL		15U
#define MAX3421E_CHIPRES		BIT(5)
#define MAX3421E_PWRDOWN		BIT(4)

/* Register CPUCTL */
#define MAX3421E_REG_CPUCTL		16U
#define MAX3421E_PULSEWID1		BIT(7)
#define MAX3421E_PULSEWID0		BIT(6)
#define MAX3421E_IE			BIT(0)

/* Register PINCTL */
#define MAX3421E_REG_PINCTL		17U
#define MAX3421E_FDUPSPI		BIT(4)
#define MAX3421E_INTLEVEL		BIT(3)
#define MAX3421E_POSINT			BIT(2)
#define MAX3421E_GPXB			BIT(1)
#define MAX3421E_GPXA			BIT(0)

/* Register REVISION */
#define MAX3421E_REG_REVISION		18U

/* Register IOPINS1, IOPINS2, GPINIRQ, GPINIEN, GPINPOL */
#define MAX3421E_REG_IOPINS1		20U
#define MAX3421E_REG_IOPINS2		21U
#define MAX3421E_REG_GPINIRQ		22U
#define MAX3421E_REG_GPINIE		23U
#define MAX3421E_REG_GPINPOL		24U

/* Register HIRQ and HIEN */
#define MAX3421E_REG_HIRQ		25U
#define MAX3421E_REG_HIEN		26U
#define MAX3421E_HXFRDN			BIT(7)
#define MAX3421E_FRAME			BIT(6)
#define MAX3421E_CONDET			BIT(5)
#define MAX3421E_SUSDN			BIT(4)
#define MAX3421E_SNDBAV			BIT(3)
#define MAX3421E_RCVDAV			BIT(2)
#define MAX3421E_RWU			BIT(1)
#define MAX3421E_BUSEVENT		BIT(0)

/* Register MODE */
#define MAX3421E_REG_MODE		27U
#define MAX3421E_DPPULLDN		BIT(7)
#define MAX3421E_DMPULLDN		BIT(6)
#define MAX3421E_DELAYISO		BIT(5)
#define MAX3421E_SEPIRQ			BIT(4)
#define MAX3421E_SOFKAENAB		BIT(3)
#define MAX3421E_HUBPRE			BIT(2)
#define MAX3421E_LOWSPEED		BIT(1)
#define MAX3421E_HOST			BIT(0)

/* Register PERADDR */
#define MAX3421E_REG_PERADDR		28U
#define MAX3421E_PERADDR_MASK		0x7FU

/* Register HCTL */
#define MAX3421E_REG_HCTL		29U
#define MAX3421E_SNDTOG1		BIT(7)
#define MAX3421E_SNDTOG0		BIT(6)
#define MAX3421E_RCVTOG1		BIT(5)
#define MAX3421E_RCVTOG0		BIT(4)
#define MAX3421E_SIGRSM			BIT(3)
#define MAX3421E_SAMPLEBUS		BIT(2)
#define MAX3421E_FRMRST			BIT(1)
#define MAX3421E_BUSRST			BIT(0)

/* Register HXFR */
#define MAX3421E_REG_HXFR		30U
#define MAX3421E_HS			BIT(7)
#define MAX3421E_ISO			BIT(6)
#define MAX3421E_OUTNIN			BIT(5)
#define MAX3421E_SETUP			BIT(4)
#define MAX3421E_EP_MASK		0x0FU
#define MAX3421E_EP(ep)			((ep) & MAX3421E_EP_MASK)
#define MAX3421E_HXFR_TYPE(hxfr)	((hxfr) & 0xF0U)

#define MAX3421E_HXFR_SETUP(ep)		(MAX3421E_SETUP | MAX3421E_EP(ep))

#define MAX3421E_HXFR_BULKIN(ep)	MAX3421E_EP(ep)
#define MAX3421E_HXFR_ISOIN(ep)		(MAX3421E_ISO | MAX3421E_EP(ep))
#define MAX3421E_HXFR_HSIN(ep)		(MAX3421E_HS | MAX3421E_EP(ep))

#define MAX3421E_HXFR_BULKOUT(ep)	(MAX3421E_OUTNIN | MAX3421E_HXFR_BULKIN(ep))
#define MAX3421E_HXFR_ISOOUT(ep)	(MAX3421E_OUTNIN | MAX3421E_HXFR_ISOIN(ep))
#define MAX3421E_HXFR_HSOUT(ep)		(MAX3421E_OUTNIN | MAX3421E_HXFR_HSIN(ep))

#define MAX3421E_HXFR_TYPE_SETUP	MAX3421E_SETUP
#define MAX3421E_HXFR_TYPE_HSIN		MAX3421E_HS
#define MAX3421E_HXFR_TYPE_HSOUT	(MAX3421E_OUTNIN | MAX3421E_HXFR_TYPE_HSIN)
#define MAX3421E_HXFR_TYPE_ISOIN	MAX3421E_ISO
#define MAX3421E_HXFR_TYPE_ISOOUT	(MAX3421E_OUTNIN | MAX3421E_HXFR_TYPE_ISOIN)
#define MAX3421E_HXFR_TYPE_BULKIN	0
#define MAX3421E_HXFR_TYPE_BULKOUT	MAX3421E_OUTNIN

/* Register HRSL */
#define MAX3421E_REG_HRSL		31U
#define MAX3421E_JKSTATUS_MASK		(BIT(7) | BIT(6))
#define MAX3421E_JSTATUS		BIT(7)
#define MAX3421E_KSTATUS		BIT(6)
#define MAX3421E_SNDTOGRD		BIT(5)
#define MAX3421E_RCVTOGRD		BIT(4)
#define MAX3421E_HRSLT_MASK		0x0FU
#define MAX3421E_HRSLT(hr)		((hr) & MAX3421E_HRSLT_MASK)

#define MAX3421E_HR_SUCCESS		0x00U
#define MAX3421E_HR_BUSY		0x01U
#define MAX3421E_HR_BADREQ		0x02U
#define MAX3421E_HR_UNDEF		0x03U
#define MAX3421E_HR_NAK			0x04U
#define MAX3421E_HR_STALL		0x05U
#define MAX3421E_HR_TOGERR		0x06U
#define MAX3421E_HR_WRONGPID		0x07U
#define MAX3421E_HR_BADBC		0x08U
#define MAX3421E_HR_PIDERR		0x09U
#define MAX3421E_HR_PKTERR		0x0AU
#define MAX3421E_HR_CRCERR		0x0BU
#define MAX3421E_HR_KERR		0x0CU
#define MAX3421E_HR_JERR		0x0DU
#define MAX3421E_HR_TIMEOUT		0x0EU
#define MAX3421E_HR_BABBLE		0x0FU

/* Successful Transfer */
#define HRSLT_IS_SUCCESS(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_SUCCESS)
/* SIE is busy, transfer pending */
#define HRSLT_IS_BUSY(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_BUSY)
/* Bad value in HXFR reg */
#define HRSLT_IS_BADREQ(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_BADREQ)
/* reserved */
#define HRSLT_IS_UNDEF(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_UNDEF)
/* Peripheral returned NAK */
#define HRSLT_IS_NAK(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_NAK)
/* Peripheral returned STALL */
#define HRSLT_IS_STALL(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_STALL)
/* Toggle error/ISO over-underrun */
#define HRSLT_IS_TOGERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_TOGERR)
/* Received the wrong PID */
#define HRSLT_IS_WRONGPID(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_WRONGPID)
/* Bad byte count */
#define HRSLT_IS_BADBC(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_BADBC)
/* Receive PID is corrupted */
#define HRSLT_IS_PIDERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_PIDERR)
/* Packet error (stuff, EOP) */
#define HRSLT_IS_PKTERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_PKTERR)
/* CRC error */
#define HRSLT_IS_CRCERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_CRCERR)
/* K-state instead of response */
#define HRSLT_IS_KERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_KERR)
/* J-state instead of response */
#define HRSLT_IS_JERR(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_JERR)
/* Device did not respond in time */
#define HRSLT_IS_TIMEOUT(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_TIMEOUT)
/* Device talked too long */
#define HRSLT_IS_BABBLE(hr)		(MAX3421E_HRSLT(hr) == MAX3421E_HR_BABBLE)

#endif /* ZEPHYR_INCLUDE_UHC_MAX3421E_H */
