/*
 * Copyright (c) 2023 Microchip Technolofy, Inc.
 *
 * SPDX-License-Identifier: Apache-2.0
 */


#include <zephyr/kernel.h>
#include <zephyr/device.h>
#include <zephyr/devicetree.h>
#include <zephyr/crypto/crypto.h>
#include <zephyr/crypto/hash.h>
#include <zephyr/sys/util.h>

#include <stdio.h>
#include <string.h>
#include <soc.h>

struct hash_tp {
	enum hash_algo hash_id;
	size_t msgsz;
	const uint8_t *msg;
	const uint8_t *digest;
};

const uint8_t msg37[] = {
	0x45, 0x73, 0x66, 0x72, 0x44, 0x36, 0x70, 0x73,
	0x47, 0x7a, 0x66, 0x70, 0x78, 0x56, 0x78, 0x6b,
	0x70, 0x65, 0x4a, 0x56, 0x55, 0x4a, 0x58, 0x62,
	0x41, 0x61, 0x53, 0x33, 0x6e, 0x55, 0x78, 0x41,
	0x46, 0x6d, 0x34, 0x77, 0x42,
};
const uint8_t msg37_sha224_digest[] = {
	0xe4, 0x70, 0xa9, 0x89, 0xc5, 0x37, 0xda, 0x0d,
	0x9f, 0x55, 0x9a, 0x4e, 0x9d, 0xed, 0xaa, 0x75,
	0xf8, 0xe0, 0x58, 0x0f, 0xc4, 0x2e, 0x0d, 0x23,
	0x03, 0x7c, 0x0f, 0x18,
};
const uint8_t msg37_sha256_digest[] = {
	0xe2, 0x82, 0x23, 0xfb, 0x3f, 0x6a, 0x49, 0x17,
	0xb0, 0x97, 0xba, 0x60, 0x51, 0xd0, 0xb6, 0x38,
	0x1d, 0xd0, 0x85, 0xfe, 0xd0, 0x6b, 0x99, 0x17,
	0x81, 0x32, 0x89, 0x1b, 0xbb, 0x18, 0x33, 0xfe,
};
const uint8_t msg37_sha384_digest[] = {
	0xce, 0x92, 0x9e, 0x57, 0xd5, 0xc1, 0x6a, 0x73,
	0xf2, 0xfd, 0xcc, 0xed, 0x8b, 0xec, 0xf4, 0xd7,
	0xf5, 0x61, 0x1b, 0x57, 0xaa, 0x74, 0x80, 0x3f,
	0x76, 0x64, 0x83, 0x06, 0x8a, 0x47, 0x04, 0x41,
	0x83, 0x21, 0x83, 0xd7, 0xb5, 0x44, 0xb7, 0xe4,
	0xeb, 0xe4, 0xb0, 0xba, 0xbb, 0x7b, 0x92, 0x75,
};
const uint8_t msg37_sha512_digest[] = {
	0xe1, 0x7c, 0x02, 0xe7, 0x74, 0x44, 0x2d, 0x37,
	0xd9, 0x6c, 0xf9, 0x12, 0xfd, 0xb3, 0x64, 0xcc,
	0x00, 0x68, 0xbd, 0x0b, 0x79, 0xb6, 0x52, 0x00,
	0x37, 0x9f, 0xa2, 0xbe, 0x4d, 0x1f, 0x4a, 0x67,
	0x46, 0xc6, 0x74, 0x01, 0x60, 0x46, 0x32, 0x5e,
	0x35, 0x4f, 0x36, 0x26, 0x84, 0xd7, 0x2e, 0x47,
	0x58, 0x75, 0xb1, 0x94, 0x77, 0x3d, 0x11, 0xd1,
	0x3f, 0x69, 0x5b, 0x80, 0x36, 0x9c, 0x69, 0x48,
};

const uint8_t msg185[] = {
	0x35, 0x61, 0x42, 0x65, 0x38, 0x65, 0x70, 0x50,
	0x62, 0x4b, 0x74, 0x47, 0x4d, 0x4a, 0x79, 0x65,
	0x64, 0x53, 0x6a, 0x4e, 0x7a, 0x39, 0x70, 0x43,
	0x62, 0x53, 0x67, 0x64, 0x6b, 0x32, 0x65, 0x6f,
	0x42, 0x6a, 0x38, 0x4d, 0x54, 0x79, 0x62, 0x6b,
	0x4c, 0x4e, 0x73, 0x34, 0x34, 0x70, 0x59, 0x78,
	0x4a, 0x45, 0x37, 0x33, 0x69, 0x77, 0x71, 0x39,
	0x72, 0x41, 0x70, 0x6f, 0x65, 0x70, 0x66, 0x42,
	0x39, 0x33, 0x6d, 0x54, 0x6d, 0x48, 0x6f, 0x51,
	0x57, 0x57, 0x53, 0x4d, 0x37, 0x6c, 0x30, 0x6a,
	0x78, 0x46, 0x35, 0x6f, 0x65, 0x61, 0x46, 0x44,
	0x39, 0x55, 0x66, 0x49, 0x56, 0x4a, 0x42, 0x38,
	0x4e, 0x64, 0x6f, 0x64, 0x6a, 0x63, 0x61, 0x32,
	0x74, 0x34, 0x38, 0x69, 0x36, 0x7a, 0x56, 0x31,
	0x31, 0x37, 0x4b, 0x55, 0x57, 0x64, 0x72, 0x61,
	0x4b, 0x55, 0x33, 0x72, 0x58, 0x61, 0x31, 0x34,
	0x4e, 0x44, 0x59, 0x6a, 0x7a, 0x69, 0x73, 0x34,
	0x70, 0x50, 0x51, 0x62, 0x57, 0x41, 0x39, 0x4d,
	0x68, 0x45, 0x4e, 0x35, 0x49, 0x4f, 0x41, 0x41,
	0x67, 0x52, 0x31, 0x57, 0x78, 0x76, 0x36, 0x4b,
	0x34, 0x6b, 0x57, 0x31, 0x75, 0x75, 0x38, 0x4c,
	0x75, 0x41, 0x30, 0x77, 0x64, 0x4c, 0x4d, 0x41,
	0x5a, 0x46, 0x46, 0x51, 0x59, 0x79, 0x51, 0x34,
	0x63,
};
const uint8_t msg185_sha224_digest[] = {
	0x24, 0xf4, 0xa2, 0x16, 0x52, 0x80, 0x9f, 0xc0,
	0x84, 0x5e, 0x17, 0xd4, 0x2c, 0xe0, 0x50, 0x74,
	0x69, 0xdc, 0xec, 0x21, 0x29, 0x58, 0xc2, 0x2f,
	0xf4, 0x34, 0x89, 0x48,
};
const uint8_t msg185_sha256_digest[] = {
	0x79, 0x2a, 0x14, 0xc2, 0x97, 0x90, 0x1d, 0xb4,
	0x97, 0x73, 0xaa, 0xc8, 0x18, 0x50, 0x84, 0xf6,
	0xab, 0xf4, 0x10, 0x88, 0x9c, 0x30, 0x6a, 0x44,
	0x40, 0x29, 0x01, 0x16, 0xc7, 0x14, 0x79, 0xd1,
};
const uint8_t msg185_sha384_digest[] = {
	0xff, 0x99, 0x7a, 0x44, 0x3b, 0xe1, 0x96, 0x84,
	0x74, 0x7d, 0x32, 0x96, 0x76, 0xd1, 0x10, 0x85,
	0x91, 0x30, 0xc3, 0x4b, 0xde, 0x4d, 0xb7, 0x6f,
	0xf2, 0xbe, 0x69, 0xaa, 0xe1, 0x2c, 0x80, 0xd4,
	0x1d, 0x8f, 0xf3, 0x3a, 0x51, 0x86, 0xb9, 0x8c,
	0xc4, 0xb6, 0x1b, 0x45, 0xfc, 0x86, 0xac, 0x31,
};
const uint8_t msg185_sha512_digest[] = {
	0xaa, 0xba, 0x04, 0xc9, 0x58, 0x37, 0xaa, 0xa8,
	0xb9, 0x1c, 0x42, 0x66, 0x4f, 0x64, 0xa3, 0x52,
	0x6f, 0xf9, 0x94, 0x75, 0x40, 0x91, 0x5c, 0x1d,
	0x20, 0x6d, 0xa2, 0xd5, 0x59, 0x0c, 0x8b, 0xdc,
	0xb7, 0x83, 0x80, 0x76, 0xa6, 0xe6, 0x4c, 0xcf,
	0xe3, 0x8c, 0x4c, 0x4e, 0xfc, 0xdf, 0x16, 0xb8,
	0xdf, 0x95, 0x42, 0xb0, 0x8d, 0x25, 0x9a, 0x19,
	0x91, 0x5e, 0x79, 0x32, 0xf1, 0x74, 0xc8, 0x62,
};

const uint8_t msg238[] = {
	0x75, 0x79, 0x30, 0x65, 0x45, 0x4a, 0x59, 0x4d,
	0x36, 0x76, 0x4d, 0x41, 0x4f, 0x6b, 0x47, 0x38,
	0x74, 0x63, 0x61, 0x39, 0x46, 0x41, 0x73, 0x4d,
	0x4e, 0x76, 0x6c, 0x35, 0x61, 0x39, 0x65, 0x39,
	0x4c, 0x7a, 0x53, 0x70, 0x72, 0x72, 0x51, 0x6b,
	0x73, 0x68, 0x4e, 0x6a, 0x6a, 0x79, 0x65, 0x63,
	0x69, 0x70, 0x65, 0x6a, 0x76, 0x53, 0x67, 0x47,
	0x33, 0x59, 0x57, 0x42, 0x38, 0x43, 0x72, 0x39,
	0x71, 0x36, 0x7a, 0x47, 0x76, 0x74, 0x67, 0x4f,
	0x30, 0x54, 0x71, 0x78, 0x38, 0x41, 0x70, 0x49,
	0x51, 0x75, 0x30, 0x7a, 0x4e, 0x52, 0x53, 0x75,
	0x50, 0x68, 0x31, 0x67, 0x77, 0x53, 0x59, 0x46,
	0x43, 0x69, 0x52, 0x4d, 0x4e, 0x58, 0x53, 0x54,
	0x49, 0x30, 0x78, 0x6c, 0x63, 0x4d, 0x42, 0x50,
	0x58, 0x64, 0x30, 0x71, 0x57, 0x44, 0x56, 0x34,
	0x6e, 0x79, 0x4d, 0x4d, 0x30, 0x69, 0x55, 0x56,
	0x62, 0x53, 0x50, 0x49, 0x79, 0x70, 0x55, 0x55,
	0x54, 0x42, 0x58, 0x4a, 0x63, 0x77, 0x6d, 0x5a,
	0x31, 0x30, 0x4a, 0x66, 0x4e, 0x79, 0x55, 0x66,
	0x44, 0x7a, 0x56, 0x71, 0x56, 0x56, 0x72, 0x4c,
	0x4b, 0x38, 0x4e, 0x35, 0x45, 0x79, 0x4c, 0x65,
	0x4c, 0x67, 0x59, 0x44, 0x46, 0x42, 0x78, 0x47,
	0x76, 0x4d, 0x76, 0x74, 0x4a, 0x57, 0x57, 0x4f,
	0x4f, 0x79, 0x52, 0x4b, 0x55, 0x52, 0x6a, 0x69,
	0x4a, 0x4b, 0x74, 0x6b, 0x79, 0x50, 0x50, 0x78,
	0x6c, 0x77, 0x63, 0x4d, 0x50, 0x70, 0x55, 0x6e,
	0x4e, 0x41, 0x52, 0x56, 0x66, 0x79, 0x35, 0x6e,
	0x31, 0x66, 0x32, 0x69, 0x65, 0x6d, 0x77, 0x46,
	0x4f, 0x31, 0x31, 0x6f, 0x56, 0x49, 0x67, 0x76,
	0x72, 0x6f, 0x55, 0x74, 0x51, 0x52,
};
const uint8_t msg238_sha224_digest[] = {
	0xfb, 0xe5, 0xec, 0x23, 0xab, 0x7f, 0x71, 0x99,
	0xf1, 0xb7, 0x26, 0x7a, 0x6c, 0xcf, 0x68, 0xae,
	0x2f, 0xe5, 0x00, 0x0c, 0x58, 0x70, 0x5e, 0xe0,
	0x88, 0x3b, 0x51, 0x12,
};
const uint8_t msg238_sha256_digest[] = {
	0x1f, 0xd2, 0x20, 0xe1, 0xd4, 0x6b, 0xeb, 0x6c,
	0xaf, 0x3b, 0x18, 0xc5, 0x6a, 0xfa, 0x83, 0x6e,
	0xb9, 0xfc, 0x05, 0xff, 0x7c, 0x77, 0xc0, 0xb6,
	0x5f, 0x30, 0xd5, 0xd8, 0x98, 0x24, 0xdf, 0xcf,
};
const uint8_t msg238_sha384_digest[] = {
	0x87, 0x2c, 0xd9, 0xee, 0xd1, 0x18, 0xdf, 0x0c,
	0xdb, 0xbb, 0x76, 0xcf, 0xdd, 0xc5, 0x8d, 0xcb,
	0x14, 0xf1, 0x68, 0x7d, 0x18, 0x51, 0x37, 0xbb,
	0x55, 0x79, 0xd9, 0x59, 0xca, 0x49, 0xdb, 0x61,
	0x16, 0xf2, 0x4b, 0xeb, 0xa8, 0xce, 0xc8, 0xcd,
	0x26, 0x48, 0xf6, 0xe4, 0x20, 0xd1, 0x75, 0x4d,
};
const uint8_t msg238_sha512_digest[] = {
	0x24, 0x08, 0x8a, 0x1c, 0xc3, 0x44, 0x04, 0x79,
	0x56, 0xab, 0x65, 0xad, 0xf1, 0xdb, 0x99, 0x9c,
	0x54, 0x36, 0x84, 0xf0, 0xa8, 0x9c, 0x9b, 0x6d,
	0xc5, 0x13, 0x14, 0xb2, 0xff, 0x15, 0x17, 0x9d,
	0x4f, 0xf8, 0x61, 0x0f, 0x39, 0x28, 0x91, 0xe2,
	0x94, 0x3b, 0x32, 0x0f, 0xc5, 0xba, 0x31, 0xeb,
	0x41, 0xf4, 0xbe, 0x28, 0x0b, 0x39, 0x41, 0x26,
	0x17, 0x8c, 0x76, 0x1d, 0x4c, 0x9d, 0x4f, 0x47,
};

const struct hash_tp hash_test_tbl2[] = {
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA224,
		.msgsz = sizeof(msg37),
		.msg = msg37,
		.digest = msg37_sha224_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA256,
		.msgsz = sizeof(msg37),
		.msg = msg37,
		.digest = msg37_sha256_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA384,
		.msgsz = sizeof(msg37),
		.msg = msg37,
		.digest = msg37_sha384_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA512,
		.msgsz = sizeof(msg37),
		.msg = msg37,
		.digest = msg37_sha512_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA224,
		.msgsz = sizeof(msg185),
		.msg = msg185,
		.digest = msg185_sha224_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA256,
		.msgsz = sizeof(msg185),
		.msg = msg185,
		.digest = msg185_sha256_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA384,
		.msgsz = sizeof(msg185),
		.msg = msg185,
		.digest = msg185_sha384_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA512,
		.msgsz = sizeof(msg185),
		.msg = msg185,
		.digest = msg185_sha512_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA224,
		.msgsz = sizeof(msg238),
		.msg = msg238,
		.digest = msg238_sha224_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA256,
		.msgsz = sizeof(msg238),
		.msg = msg238,
		.digest = msg238_sha256_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA384,
		.msgsz = sizeof(msg238),
		.msg = msg238,
		.digest = msg238_sha384_digest,
	},
	{
		.hash_id = CRYPTO_HASH_ALGO_SHA512,
		.msgsz = sizeof(msg238),
		.msg = msg238,
		.digest = msg238_sha512_digest,
	},
};

/* zephyr crypto driver capability flags, hash context, and hash packet */
static uint32_t cap_flags;
static struct hash_ctx zhash_ctx;
static struct hash_pkt zhash_pkt;

/* Hash digest buffer sized for largest digest (SHA-512) */
static uint8_t digest[64];

#define MCHP_XEC_ROM_API_NODE DT_NODELABEL(rom_api)
#define MCHP_XEC_SYMCR_NODE DT_NODELABEL(symcr)

static const struct device *const symcr_dev = DEVICE_DT_GET(MCHP_XEC_SYMCR_NODE);

static int test_zephyr_hash_chunk_block_size(const struct hash_tp *htbl, size_t nentries);
static int test_zephyr_hash_chunk(const struct hash_tp *htbl, size_t nentries, size_t chunksz);
static int validate_hw_compatibility(const struct device *dev);

static int pr_hash_algo_name(enum hash_algo algo);
static size_t hash_digest_size(enum hash_algo algo);
static size_t hash_block_size(enum hash_algo algo);

#define MCHP_XEC_STRUCT_HASH_STATE_STRUCT_SIZE	8
#define MCHP_XEC_STRUCT_HASH_STATE_MEM_SIZE	256
#define MCHP_XEC_STRUCT_HASH_STRUCT_SIZE	240
#define MCHP_XEC_STRUCT_HMAC2_STRUCT_SIZE	256

int main(void)
{
	int ret;
	size_t chunk_size = 0;

	printf("It lives! %s\n", CONFIG_BOARD);
	printf("ROM API say GIVE MEMORY, MORE MEMORY!\n");
	printf("Size of MEC172x ROM API hash state save memory is %u bytes!!!!\n",
		MCHP_XEC_STRUCT_HASH_STATE_MEM_SIZE);
	printf("Size of MEC172x ROM API hash context structure is %u bytes!!!!\n",
		MCHP_XEC_STRUCT_HASH_STRUCT_SIZE);
	printf("Size of MEC172x ROM API HMAC context structure is %u bytes!!!!\n",
		MCHP_XEC_STRUCT_HMAC2_STRUCT_SIZE);

	if (!device_is_ready(symcr_dev)) {
		printf("ERROR: symcr device is not ready!\n");
		return 0;
	}

	/* Check zephyr crypto drivers capabilities */
	ret = validate_hw_compatibility(symcr_dev);
	if (ret) {
		printf("ERROR: symcr driver capabilties failure\n");
		return 0;
	}

	printf("\nTest Zephyr crypto hash API for multiblock plus remainder\n");
	ret = test_zephyr_hash_chunk_block_size(hash_test_tbl2, ARRAY_SIZE(hash_test_tbl2));
	printf("Test Zephyr crypto hash API for multiblock plus remainder returned %d\n", ret);

	printf("\nTest Zephyr crypto arbitrary chunk size = %u\n", chunk_size);
	ret = test_zephyr_hash_chunk(hash_test_tbl2, ARRAY_SIZE(hash_test_tbl2), chunk_size);
	printf("Test Zephyr crypto arbitrary chunk size returned %d\n", ret);

	chunk_size = 8u;
	printf("\nTest Zephyr crypto arbitrary chunk size = %u\n", chunk_size);
	ret = test_zephyr_hash_chunk(hash_test_tbl2, ARRAY_SIZE(hash_test_tbl2), chunk_size);
	printf("Test Zephyr crypto arbitrary chunk size returned %d\n", ret);

	chunk_size = 23u;
	printf("\nTest Zephyr crypto arbitrary chunk size = %u\n", chunk_size);
	ret = test_zephyr_hash_chunk(hash_test_tbl2, ARRAY_SIZE(hash_test_tbl2), chunk_size);
	printf("Test Zephyr crypto arbitrary chunk size returned %d\n", ret);

	printf("Application done\n");
	return 0;
}

static size_t hash_digest_size(enum hash_algo algo)
{
	size_t digestsz;

	switch (algo) {
	case CRYPTO_HASH_ALGO_SHA224:
		digestsz = 28u;
		break;
	case CRYPTO_HASH_ALGO_SHA256:
		digestsz = 32u;
		break;
	case CRYPTO_HASH_ALGO_SHA384:
		digestsz = 48u;
		break;
	case CRYPTO_HASH_ALGO_SHA512:
		digestsz = 64u;
		break;
	default:
		digestsz = 0u;
	}

	return digestsz;
}

static size_t hash_block_size(enum hash_algo algo)
{
	size_t blocksz;

	switch (algo) {
	case CRYPTO_HASH_ALGO_SHA224:
	case CRYPTO_HASH_ALGO_SHA256:
		blocksz = 64u;
		break;
	case CRYPTO_HASH_ALGO_SHA384:
	case CRYPTO_HASH_ALGO_SHA512:
		blocksz = 128u;
		break;
	default:
		blocksz = 0u;
	}

	return blocksz;
}

static int pr_hash_algo_name(enum hash_algo algo)
{
	int ret = 0;

	switch (algo) {
	case CRYPTO_HASH_ALGO_SHA224:
		printf("SHA-224\n");
		break;
	case CRYPTO_HASH_ALGO_SHA256:
		printf("SHA-256\n");
		break;
	case CRYPTO_HASH_ALGO_SHA384:
		printf("SHA-384\n");
		break;
	case CRYPTO_HASH_ALGO_SHA512:
		printf("SHA-512\n");
		break;
	default:
		printf("Uknown Hash algorithm\n");
		ret = -ENOTSUP;
	}

	return ret;
}

static int test_zephyr_hash_chunk_block_size(const struct hash_tp *htbl, size_t nentries)
{
	int fail_cnt = 0, ret = 0;
	size_t blocksz = 0, digestsz = 0, msgsz = 0, updatesz = 0;

	for (size_t n = 0; n < nentries; n++) {
		const struct hash_tp *tp = &htbl[n];
		const uint8_t *msgptr = tp->msg;
		const uint8_t *exp_digest = tp->digest;

		memset(digest, 0, sizeof(digest));

		ret = pr_hash_algo_name(tp->hash_id);
		if (ret) {
			continue;
		}

		msgsz = tp->msgsz;
		blocksz = hash_block_size(tp->hash_id);
		digestsz = hash_digest_size(tp->hash_id);
		printf("Message size is %u, block size is %u, digest size is %u byte\n",
			msgsz, blocksz, digestsz);

		/* Open a hash session from our crypto driver */
		zhash_ctx.device = NULL;
		zhash_ctx.drv_sessn_state = NULL;
		zhash_ctx.hash_hndlr = NULL;
		zhash_ctx.started = false;
		/* application must supply correct flags */
		zhash_ctx.flags = CAP_SYNC_OPS | CAP_SEPARATE_IO_BUFS;

		ret = hash_begin_session(symcr_dev, &zhash_ctx, tp->hash_id);
		if (ret) {
			printf("ERROR: zephyr crypto begin hash session: %d\n", ret);
			fail_cnt++;
			continue;
		}

		if (msgsz > blocksz) {
			/* SHA algorithms block sizes are powers of 2 */
			updatesz = msgsz & ~(blocksz - 1u);
			printf("  Update size is %d\n", updatesz);
			zhash_pkt.in_buf = msgptr;
			zhash_pkt.in_len = updatesz;
			zhash_pkt.out_buf = digest;
			zhash_pkt.ctx = &zhash_ctx;

			ret = hash_update(&zhash_ctx, &zhash_pkt);
			if (ret) {
				printf("ERROR: zephyr crypto hash update: %d\n", ret);
				fail_cnt++;
				continue;
			}

			msgptr += updatesz;
			msgsz -= updatesz;
		}

		printf("  Final size is %u\n", msgsz);

		zhash_pkt.in_buf = msgptr;
		zhash_pkt.in_len = msgsz;
		zhash_pkt.out_buf = digest;
		zhash_pkt.ctx = &zhash_ctx;

		/* final */
		ret = hash_compute(&zhash_ctx, &zhash_pkt);
		if (ret) {
			printf("ERROR: zephyr crypto hash compute: %d\n", ret);
			fail_cnt++;
			continue;
		}

		ret = hash_free_session(symcr_dev, &zhash_ctx);
		if (ret) {
			printf("ERROR: zephyr crypto free hash session: %d\n", ret);
			fail_cnt++;
			continue;
		}

		ret = strncmp(exp_digest, digest, digestsz);
		if (ret == 0) {
			printf("Hash computation PASS\n");
		} else {
			printf("Hash computation FAIL\n");
			fail_cnt++;
		}
	}

	return fail_cnt;
}

static int test_zephyr_hash_chunk(const struct hash_tp *htbl, size_t nentries, size_t chunksz)
{
	int fail_cnt = 0, ret = 0, chunk_cnt = 0;
	size_t blocksz = 0, digestsz = 0, msgsz = 0;
	size_t remsz = 0, updatesz = 0, total_updatesz = 0;

	for (size_t n = 0; n < nentries; n++) {
		const struct hash_tp *tp = &htbl[n];
		const uint8_t *msgptr = tp->msg;
		const uint8_t *exp_digest = tp->digest;

		memset(digest, 0, sizeof(digest));

		ret = pr_hash_algo_name(tp->hash_id);
		if (ret) {
			continue;
		}

		msgsz = tp->msgsz;
		blocksz = hash_block_size(tp->hash_id);
		digestsz = hash_digest_size(tp->hash_id);
		printf("Chunk size is %u Message size is %u, block size is %u, digest size is %u\n",
			chunksz, msgsz, blocksz, digestsz);

		/* Open a hash session from our crypto driver */
		zhash_ctx.device = NULL;
		zhash_ctx.drv_sessn_state = NULL;
		zhash_ctx.hash_hndlr = NULL;
		zhash_ctx.started = false;
		/* application must supply correct flags */
		zhash_ctx.flags = CAP_SYNC_OPS | CAP_SEPARATE_IO_BUFS;

		ret = hash_begin_session(symcr_dev, &zhash_ctx, tp->hash_id);
		if (ret) {
			printf("ERROR: zephyr crypto begin hash session: %d\n", ret);
			fail_cnt++;
			continue;
		}

		chunk_cnt = 0;
		total_updatesz = 0;
		updatesz = chunksz;
		while (chunksz && ((msgsz - total_updatesz) > chunksz)) {
			chunk_cnt++;

			zhash_pkt.in_buf = msgptr;
			zhash_pkt.in_len = updatesz;
			zhash_pkt.out_buf = digest;
			zhash_pkt.ctx = &zhash_ctx;

			printf("  Chunk %d update size is %u\n", chunk_cnt, updatesz);
			ret = hash_update(&zhash_ctx, &zhash_pkt);
			if (ret) {
				printf("ERROR: zephyr crypto hash update: %d\n", ret);
				return ret;
			}

			msgptr += updatesz;
			total_updatesz += updatesz;
		}

		remsz = msgsz - total_updatesz;
		zhash_pkt.in_buf = msgptr;
		zhash_pkt.in_len = remsz;
		zhash_pkt.out_buf = digest;
		zhash_pkt.ctx = &zhash_ctx;

		printf("  Compute hash final digest. Remaining size is %u\n", remsz);
		ret = hash_compute(&zhash_ctx, &zhash_pkt);
		if (ret) {
			printf("ERROR: zephyr crypto hash compute final: %d\n", ret);
			return ret;
		}

		ret = hash_free_session(symcr_dev, &zhash_ctx);
		if (ret) {
			printf("ERROR: zephyr crypto free hash session: %d\n", ret);
			return ret;
		}

		ret = strncmp(exp_digest, digest, digestsz);
		if (ret == 0) {
			printf("Hash computation PASS\n");
		} else {
			printf("Hash computation FAIL\n");
			fail_cnt++;
		}
	}

	return fail_cnt;
}

static int validate_hw_compatibility(const struct device *dev)
{
	uint32_t flags = 0U;

	flags = crypto_query_hwcaps(dev);
	if ((flags & CAP_RAW_KEY) == 0U) {
		printf("Please provision the key separately "
			"as the module doesn't support a raw key\n");
		return -1;
	}

	if ((flags & CAP_SYNC_OPS) == 0U) {
		printf("The app assumes sync semantics. "
			"Please rewrite the app accordingly before proceeding\n");
		return -1;
	}

	if ((flags & CAP_SEPARATE_IO_BUFS) == 0U) {
		printf("The app assumes distinct IO buffers. "
			"Please rewrite the app accordingly before proceeding\n");
		return -1;
	}

	cap_flags = CAP_RAW_KEY | CAP_SYNC_OPS | CAP_SEPARATE_IO_BUFS;

	return 0;

}
