/*
 * Copyright (c) 2025 Renesas Electronics Corporation
 *
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @brief Option setting header file for Renesas RX130
 */

/*
 * Option-Setting Memory for the RX. This region of memory (located in flash)
 * determines the state of the MCU after reset and can not be changed on runtime
 *
 * All registers are set to 0xffffffff by default, which are "safe" settings.
 * Please refer to the Renesas RX Group User's Manual before changing any of
 * the values as some changes can be permanent or lock access to the device.
 *
 * Address range: 0xFFFFFF80 to 0xFFFFFF8F
 */

/* Endian Select Register (MDE) at 0xFFFFFF80
 *
 * b2 to b0: endian select between (0 0 0) for big endian and (1 1 1) for little
 * endian. Set this according to __BYTE_ORDER__ (cf. include\toolchain\gcc.h)
 *
 * all other bits are reserved and have to be set to 1
 */
#define SOC_RX_MDE (0xFFFFFFFFUL) /* little endian */

/* Option Function Select Register 0 (OFS0)
 * This section sets the IWDT (Independent Watchdog Timer) behavior immediately after reset
 * by programming the OFS0 register. When enabled, IWDT starts counting automatically
 * starts after a reset.
 */

#ifdef CONFIG_WDT_RENESAS_RX_IWDT_AUTO_START_MODE
#define SOC_RX_OFS0                                                                                \
	((CONFIG_WDT_RENESAS_RX_OFS0_IWDTSTRT << 1) |                      /* bit 1 */             \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_TIMEOUT_IWDTTOPS & 0x3) << 2) |      /* bits 2-3 */          \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_CLKDIV_IWDTCKS & 0xF) << 4) |        /* bits 4-7 */          \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_WINDOW_END_IWDTRPES & 0x3) << 8) |   /* bits 8-9 */          \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_WINDOW_STRT_IWDTRPSS & 0x3) << 10) | /* bits 10-11 */        \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_IRQ_SEL_IWDTRSTIRQS & 0x1) << 12) |  /* bit 12 */            \
	 ((CONFIG_WDT_RENESAS_RX_OFS0_IWDTSLCSTP & 0x1) << 14) |           /* bit 14 */            \
	 0xFFFFA001UL /* reserved bits 0,13,15..31 */                                              \
	)
#else
#define SOC_RX_OFS0 (0xFFFFFFFFUL)
#endif

/* Option Function Select Register 1 (OFS1) (Voltage detection and HOCO)
 */
#define SOC_RX_OFS1 (0xFFFFFFFFUL)
