/*
 * Copyright (c) 2025 Microchip Technology Inc.
 *
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * @file mchp_pic32cm_jh_clock.h
 * @brief List clock subsystem IDs for pic32cm_jh family.
 *
 * Clock subsystem IDs. To be used in devicetree nodes, and as argument for clock API.
 */

#ifndef INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_PIC32CM_JH_CLOCK_H_
#define INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_PIC32CM_JH_CLOCK_H_

/**
 * Encode clock type, mclk bus, mclk mask bit, gclk pch and instance number,
 * to clock subsystem.
 *
 * - 00..07 (8 bits): inst
 *
 * - 08..13 (6 bits): gclkperiph
 * (values from 0 to 47)
 *
 * - 14..19 (6 bits): mclkmaskbit
 * (values from 0 to 31)
 *
 * - 20..25 (6 bits): mclkbus
 * following values
 * MBUS_AHB  (0)
 * MBUS_APBA (1)
 * MBUS_APBB (2)
 * MBUS_APBC (3)
 * MBUS_APBD (4)
 *
 * - 26..31 (6 bits): type
 * following values
 * SUBSYS_TYPE_XOSC       (0)
 * SUBSYS_TYPE_OSC48M     (1)
 * SUBSYS_TYPE_FDPLL      (2)
 * SUBSYS_TYPE_RTC        (3)
 * SUBSYS_TYPE_XOSC32K    (4)
 * SUBSYS_TYPE_OSC32K     (5)
 * SUBSYS_TYPE_GCLKGEN    (6)
 * SUBSYS_TYPE_GCLKPERIPH (7)
 * SUBSYS_TYPE_MCLKCPU    (8)
 * SUBSYS_TYPE_MCLKPERIPH (9)
 *
 * @param type clock subsystem type
 * @param mclkbus select from the AHBx and the APBx buses
 * @param mclkmaskbit select the module connected to AHBx or APBx bus (0 to 31)
 * @param gclkperiph gclk peripheral channel number m in PCHTRLm (0 to 42)
 * @param inst instance number of the specified clock type
 */
#define MCHP_CLOCK_DERIVE_ID(type, mclkbus, mclkmaskbit, gclkperiph, inst)                         \
	(((type) << 26) | ((mclkbus) << 20) | ((mclkmaskbit) << 14) | ((gclkperiph) << 8) | inst)

/* XOSC_TYPE ids */
#define CLOCK_MCHP_XOSC_ID     MCHP_CLOCK_DERIVE_ID(0, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_XOSC_ID_MAX (0)

/* OSC48M_TYPE ids */
#define CLOCK_MCHP_OSC48M_ID     MCHP_CLOCK_DERIVE_ID(1, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_OSC48M_ID_MAX (0)

/* FDPLL_TYPE id */
#define CLOCK_MCHP_FDPLL_ID     MCHP_CLOCK_DERIVE_ID(2, 0x3f, 0x3f, 0, 0)
#define CLOCK_MCHP_FDPLL_ID_MAX (0)

/* RTC_TYPE ids */
#define CLOCK_MCHP_RTC_ID     MCHP_CLOCK_DERIVE_ID(3, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_RTC_ID_MAX (0)

/* XOSC32K_TYPE ids */
#define CLOCK_MCHP_XOSC32K_ID_XOSC1K  MCHP_CLOCK_DERIVE_ID(4, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_XOSC32K_ID_XOSC32K MCHP_CLOCK_DERIVE_ID(4, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_XOSC32K_ID_MAX     (1)

/* OSC32K_TYPE ids */
#define CLOCK_MCHP_OSC32K_ID_OSC1K  MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_OSC32K_ID_OSC32K MCHP_CLOCK_DERIVE_ID(5, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_OSC32K_ID_MAX    (1)

/* GCLKGEN_TYPE ids */
#define CLOCK_MCHP_GCLKGEN_ID_GEN0 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_GCLKGEN_ID_GEN1 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 1)
#define CLOCK_MCHP_GCLKGEN_ID_GEN2 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 2)
#define CLOCK_MCHP_GCLKGEN_ID_GEN3 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 3)
#define CLOCK_MCHP_GCLKGEN_ID_GEN4 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 4)
#define CLOCK_MCHP_GCLKGEN_ID_GEN5 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 5)
#define CLOCK_MCHP_GCLKGEN_ID_GEN6 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 6)
#define CLOCK_MCHP_GCLKGEN_ID_GEN7 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 7)
#define CLOCK_MCHP_GCLKGEN_ID_GEN8 MCHP_CLOCK_DERIVE_ID(6, 0x3f, 0x3f, 0x3f, 8)
#define CLOCK_MCHP_GCLKGEN_ID_MAX  (8)

/* GCLKPERIPH_TYPE ids */
#define CLOCK_MCHP_GCLKPERIPH_ID_EIC              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 2, 0)
#define CLOCK_MCHP_GCLKPERIPH_ID_FREQM_MSR        MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 3, 1)
#define CLOCK_MCHP_GCLKPERIPH_ID_FREQM_REF        MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 4, 2)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_0  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 5, 3)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_1  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 6, 4)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_2  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 7, 5)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_3  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 8, 6)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_4  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 9, 7)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_5  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 10, 8)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_6  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 11, 9)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_7  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 12, 10)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_8  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 13, 11)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_9  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 14, 12)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_10 MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 15, 13)
#define CLOCK_MCHP_GCLKPERIPH_ID_EVSYS_CHANNEL_11 MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 16, 14)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM_0_7_SLOW  MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 17, 15)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM0_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 18, 16)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM1_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 19, 17)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM2_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 20, 18)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM3_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 21, 19)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM4_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 22, 20)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM5_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 23, 21)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM6_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 24, 22)
#define CLOCK_MCHP_GCLKPERIPH_ID_SERCOM7_CORE     MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 25, 23)
#define CLOCK_MCHP_GCLKPERIPH_ID_CAN0             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 26, 24)
#define CLOCK_MCHP_GCLKPERIPH_ID_CAN1             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 27, 25)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC0             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 28, 26)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC1             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 28, 27)
#define CLOCK_MCHP_GCLKPERIPH_ID_TCC2             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 29, 28)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC0              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 30, 29)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC1              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 30, 30)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC2              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 31, 31)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC3              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 31, 32)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC4              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 32, 33)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC5              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 33, 34)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC6              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 34, 35)
#define CLOCK_MCHP_GCLKPERIPH_ID_TC7              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 35, 36)
#define CLOCK_MCHP_GCLKPERIPH_ID_ADC0             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 36, 37)
#define CLOCK_MCHP_GCLKPERIPH_ID_ADC1             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 37, 38)
#define CLOCK_MCHP_GCLKPERIPH_ID_DAC              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 38, 39)
#define CLOCK_MCHP_GCLKPERIPH_ID_PTC              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 39, 40)
#define CLOCK_MCHP_GCLKPERIPH_ID_CCL              MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 40, 41)
#define CLOCK_MCHP_GCLKPERIPH_ID_PDEC             MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 41, 42)
#define CLOCK_MCHP_GCLKPERIPH_ID_AC               MCHP_CLOCK_DERIVE_ID(7, 0x3f, 0x3f, 42, 43)
#define CLOCK_MCHP_GCLKPERIPH_ID_MAX              (43)

/* MCLKCPU_TYPE ids */
#define CLOCK_MCHP_MCLKCPU_ID  MCHP_CLOCK_DERIVE_ID(8, 0x3f, 0x3f, 0x3f, 0)
#define CLOCK_MCHP_MCLKCPU_MAX (0)

/* MCLKPERIPH_TYPE ids */
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_APBA      MCHP_CLOCK_DERIVE_ID(9, 0, 0, 0x3f, 0)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_APBB      MCHP_CLOCK_DERIVE_ID(9, 0, 1, 0x3f, 1)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_APBC      MCHP_CLOCK_DERIVE_ID(9, 0, 2, 0x3f, 2)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_DSU       MCHP_CLOCK_DERIVE_ID(9, 0, 3, 0x3f, 3)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_HMATRIXHS MCHP_CLOCK_DERIVE_ID(9, 0, 4, 0x3f, 4)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_NVMCTRL   MCHP_CLOCK_DERIVE_ID(9, 0, 5, 0x3f, 5)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_MCRAMC    MCHP_CLOCK_DERIVE_ID(9, 0, 6, 0x3f, 6)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_DMAC      MCHP_CLOCK_DERIVE_ID(9, 0, 7, 0x3f, 7)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_CAN0      MCHP_CLOCK_DERIVE_ID(9, 0, 8, 0x3f, 8)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_CAN1      MCHP_CLOCK_DERIVE_ID(9, 0, 9, 0x3f, 9)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_PAC       MCHP_CLOCK_DERIVE_ID(9, 0, 10, 0x3f, 10)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_DIVAS     MCHP_CLOCK_DERIVE_ID(9, 0, 12, 0x3f, 11)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_APBD      MCHP_CLOCK_DERIVE_ID(9, 0, 13, 0x3f, 12)
#define CLOCK_MCHP_MCLKPERIPH_ID_AHB_ICM       MCHP_CLOCK_DERIVE_ID(9, 0, 14, 0x3f, 13)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_PAC        MCHP_CLOCK_DERIVE_ID(9, 1, 0, 0x3f, 14)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_PM         MCHP_CLOCK_DERIVE_ID(9, 1, 1, 0x3f, 15)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_MCLK       MCHP_CLOCK_DERIVE_ID(9, 1, 2, 0x3f, 16)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_RSTC       MCHP_CLOCK_DERIVE_ID(9, 1, 3, 0x3f, 17)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_OSCCTRL    MCHP_CLOCK_DERIVE_ID(9, 1, 4, 0x3f, 18)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_OSC32KCTRL MCHP_CLOCK_DERIVE_ID(9, 1, 5, 0x3f, 19)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_SUPC       MCHP_CLOCK_DERIVE_ID(9, 1, 6, 0x3f, 20)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_GCLK       MCHP_CLOCK_DERIVE_ID(9, 1, 7, 0x3f, 21)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_WDT        MCHP_CLOCK_DERIVE_ID(9, 1, 8, 0x3f, 22)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_RTC        MCHP_CLOCK_DERIVE_ID(9, 1, 9, 0x3f, 23)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_EIC        MCHP_CLOCK_DERIVE_ID(9, 1, 10, 0x3f, 24)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_FREQM      MCHP_CLOCK_DERIVE_ID(9, 1, 11, 0x3f, 25)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBA_MCRAMC     MCHP_CLOCK_DERIVE_ID(9, 1, 12, 0x3f, 26)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_PORT      MCHP_CLOCK_DERIVE_ID(9, 2, 0, 0x3f, 27)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_DSU       MCHP_CLOCK_DERIVE_ID(9, 2, 1, 0x3f, 28)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_NVMCTRL   MCHP_CLOCK_DERIVE_ID(9, 2, 2, 0x3f, 29)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBB_HMATRIXHS MCHP_CLOCK_DERIVE_ID(9, 2, 5, 0x3f, 30)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_EVSYS   MCHP_CLOCK_DERIVE_ID(9, 3, 0, 0x3f, 31)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM0 MCHP_CLOCK_DERIVE_ID(9, 3, 1, 0x3f, 32)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM1 MCHP_CLOCK_DERIVE_ID(9, 3, 2, 0x3f, 33)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM2 MCHP_CLOCK_DERIVE_ID(9, 3, 3, 0x3f, 34)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM3 MCHP_CLOCK_DERIVE_ID(9, 3, 4, 0x3f, 35)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM4 MCHP_CLOCK_DERIVE_ID(9, 3, 5, 0x3f, 36)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SERCOM5 MCHP_CLOCK_DERIVE_ID(9, 3, 6, 0x3f, 37)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TCC0    MCHP_CLOCK_DERIVE_ID(9, 3, 9, 0x3f, 38)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TCC1    MCHP_CLOCK_DERIVE_ID(9, 3, 10, 0x3f, 39)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TCC2    MCHP_CLOCK_DERIVE_ID(9, 3, 11, 0x3f, 40)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC0     MCHP_CLOCK_DERIVE_ID(9, 3, 12, 0x3f, 41)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC1     MCHP_CLOCK_DERIVE_ID(9, 3, 13, 0x3f, 42)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC2     MCHP_CLOCK_DERIVE_ID(9, 3, 14, 0x3f, 43)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC3     MCHP_CLOCK_DERIVE_ID(9, 3, 15, 0x3f, 44)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_TC4     MCHP_CLOCK_DERIVE_ID(9, 3, 16, 0x3f, 45)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_ADC0    MCHP_CLOCK_DERIVE_ID(9, 3, 17, 0x3f, 46)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_ADC1    MCHP_CLOCK_DERIVE_ID(9, 3, 18, 0x3f, 47)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_AC      MCHP_CLOCK_DERIVE_ID(9, 3, 19, 0x3f, 48)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_DAC     MCHP_CLOCK_DERIVE_ID(9, 3, 20, 0x3f, 49)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_PTC     MCHP_CLOCK_DERIVE_ID(9, 3, 21, 0x3f, 50)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_CCL     MCHP_CLOCK_DERIVE_ID(9, 3, 22, 0x3f, 51)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_ICM     MCHP_CLOCK_DERIVE_ID(9, 3, 25, 0x3f, 52)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_PDEC    MCHP_CLOCK_DERIVE_ID(9, 3, 26, 0x3f, 53)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBC_SMBIST  MCHP_CLOCK_DERIVE_ID(9, 3, 27, 0x3f, 54)

#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM6 MCHP_CLOCK_DERIVE_ID(9, 4, 0, 0x3f, 55)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_SERCOM7 MCHP_CLOCK_DERIVE_ID(9, 4, 1, 0x3f, 56)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TC5     MCHP_CLOCK_DERIVE_ID(9, 4, 2, 0x3f, 57)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TC6     MCHP_CLOCK_DERIVE_ID(9, 4, 3, 0x3f, 58)
#define CLOCK_MCHP_MCLKPERIPH_ID_APBD_TC7     MCHP_CLOCK_DERIVE_ID(9, 4, 4, 0x3f, 59)
#define CLOCK_MCHP_MCLKPERIPH_ID_MAX          (59)

#endif /* INCLUDE_ZEPHYR_DT_BINDINGS_CLOCK_MCHP_PIC32CM_JH_CLOCK_H_ */
