/*
 * Copyright (c) 2025, FocalTech Systems CO.,Ltd
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef ZEPHYR_INCLUDE_DT_BINDINGS_RESET_FOCALTECH_FT9001_RESET_H_
#define ZEPHYR_INCLUDE_DT_BINDINGS_RESET_FOCALTECH_FT9001_RESET_H_

/**
 * @defgroup focaltech_reset_macros FocalTech Reset Configuration Macros
 * @brief Macros for encoding reset register and bit information
 * @{
 */

#define FOCALTECH_RESET_SHIFT 6U
#define FOCALTECH_RESET_MASK  ((1U << FOCALTECH_RESET_SHIFT) - 1U)

#define FOCALTECH_RESET_CONFIG(reg, bit)                                                           \
	(((FOCALTECH_##reg##_OFFSET) << FOCALTECH_RESET_SHIFT) | ((bit) & FOCALTECH_RESET_MASK))

/** @} */

/**
 * @defgroup focaltech_reset_regs FocalTech Reset Control Module Registers
 * @brief Reset Control Module Register offsets
 * @{
 */

/** EPORT Reset Control Register */
#define FOCALTECH_EPORTRSTCR_OFFSET 0x88U
/** Multi Reset Control Register */
#define FOCALTECH_MULTIRSTCR_OFFSET 0xACU
/** System Reset Control Register */
#define FOCALTECH_SYSRSTCR_OFFSET   0xB0U
/** AHB3 Reset Control Register */
#define FOCALTECH_AHB3RSTCR_OFFSET  0xB4U
/** Algorithm Reset Control Register */
#define FOCALTECH_ARITHRSTCR_OFFSET 0xB8U
/** IPS Reset Control Register */
#define FOCALTECH_IPSRSTCR_OFFSET   0xBCU

/** @} */

/**
 * @defgroup focaltech_reset_enables FocalTech Reset Enable/Disable Definitions
 * @brief Reset enable/disable definitions for peripherals
 * @{
 */

/**
 * @defgroup focaltech_eportrstr_resets EPORTRSTR Reset Control
 * @brief EPORT Reset Control Register peripherals
 * @{
 */

/** EPORT0 reset */
#define FOCALTECH_RESET_EPORT0 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 0U)
/** EPORT1 reset */
#define FOCALTECH_RESET_EPORT1 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 1U)
/** EPORT2 reset */
#define FOCALTECH_RESET_EPORT2 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 2U)
/** EPORT3 reset */
#define FOCALTECH_RESET_EPORT3 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 3U)
/** EPORT4 reset */
#define FOCALTECH_RESET_EPORT4 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 4U)
/** EPORT5 reset */
#define FOCALTECH_RESET_EPORT5 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 5U)
/** EPORT6 reset */
#define FOCALTECH_RESET_EPORT6 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 6U)
/** EPORT7 reset */
#define FOCALTECH_RESET_EPORT7 FOCALTECH_RESET_CONFIG(EPORTRSTCR, 7U)
/** ACMP1 reset */
#define FOCALTECH_RESET_ACMP1  FOCALTECH_RESET_CONFIG(EPORTRSTCR, 8U)
/** ACMP2 reset */
#define FOCALTECH_RESET_ACMP2  FOCALTECH_RESET_CONFIG(EPORTRSTCR, 9U)

/** @} */

/**
 * @defgroup focaltech_multirstcr_resets MULTIRSTCR Reset Control
 * @brief Multi Reset Control Register peripherals
 * @{
 */

/** OTP AHB reset */
#define FOCALTECH_RESET_OTP_AHB   FOCALTECH_RESET_CONFIG(MULTIRSTCR, 0U)
/** MCC reset */
#define FOCALTECH_RESET_MCC       FOCALTECH_RESET_CONFIG(MULTIRSTCR, 1U)
/** MCC address reset */
#define FOCALTECH_RESET_MCCADR    FOCALTECH_RESET_CONFIG(MULTIRSTCR, 2U)
/** ADC reset */
#define FOCALTECH_RESET_ADC       FOCALTECH_RESET_CONFIG(MULTIRSTCR, 3U)
/** MESH reset */
#define FOCALTECH_RESET_MESH      FOCALTECH_RESET_CONFIG(MULTIRSTCR, 4U)
/** TC reset */
#define FOCALTECH_RESET_TC        FOCALTECH_RESET_CONFIG(MULTIRSTCR, 5U)
/** Clock output reset */
#define FOCALTECH_RESET_CLKOUT    FOCALTECH_RESET_CONFIG(MULTIRSTCR, 6U)
/** Key controller reset */
#define FOCALTECH_RESET_KEY_CTRL  FOCALTECH_RESET_CONFIG(MULTIRSTCR, 7U)
/** CCM reset */
#define FOCALTECH_RESET_CCM_RESET FOCALTECH_RESET_CONFIG(MULTIRSTCR, 8U)
/** OTP IPS reset */
#define FOCALTECH_RESET_OTP_IPS   FOCALTECH_RESET_CONFIG(MULTIRSTCR, 9U)
/** CPM IPS reset */
#define FOCALTECH_RESET_CPM_IPS   FOCALTECH_RESET_CONFIG(MULTIRSTCR, 10U)
/** Trace reset */
#define FOCALTECH_RESET_TRACE     FOCALTECH_RESET_CONFIG(MULTIRSTCR, 11U)
/** I2S master reset */
#define FOCALTECH_RESET_I2S_M     FOCALTECH_RESET_CONFIG(MULTIRSTCR, 12U)
/** I2S slave reset */
#define FOCALTECH_RESET_I2S_S     FOCALTECH_RESET_CONFIG(MULTIRSTCR, 13U)
/** CRC0 reset */
#define FOCALTECH_RESET_CRC0      FOCALTECH_RESET_CONFIG(MULTIRSTCR, 14U)
/** CRC1 reset */
#define FOCALTECH_RESET_CRC1      FOCALTECH_RESET_CONFIG(MULTIRSTCR, 15U)

/** @} */

/**
 * @defgroup focaltech_sysrstcr_resets SYSRSTCR Reset Control
 * @brief System Reset Control Register peripherals
 * @{
 */

/** SSI slave 1 reset */
#define FOCALTECH_RESET_SSISLV1   FOCALTECH_RESET_CONFIG(SYSRSTCR, 0U)
/** DMAC1 reset */
#define FOCALTECH_RESET_DMAC1     FOCALTECH_RESET_CONFIG(SYSRSTCR, 1U)
/** DMAC2 reset */
#define FOCALTECH_RESET_DMAC2     FOCALTECH_RESET_CONFIG(SYSRSTCR, 2U)
/** AHB2 MUX reset */
#define FOCALTECH_RESET_AHB2_MUX  FOCALTECH_RESET_CONFIG(SYSRSTCR, 3U)
/** PXLP1 AHB reset */
#define FOCALTECH_RESET_PXLP1_AHB FOCALTECH_RESET_CONFIG(SYSRSTCR, 10U)
/** PXLP2 AHB reset */
#define FOCALTECH_RESET_PXLP2_AHB FOCALTECH_RESET_CONFIG(SYSRSTCR, 11U)
/** SRAM data reset */
#define FOCALTECH_RESET_SRAMD     FOCALTECH_RESET_CONFIG(SYSRSTCR, 12U)
/** SRAM0 reset */
#define FOCALTECH_RESET_SRAM0     FOCALTECH_RESET_CONFIG(SYSRSTCR, 13U)
/** SRAM1 reset */
#define FOCALTECH_RESET_SRAM1     FOCALTECH_RESET_CONFIG(SYSRSTCR, 14U)
/** SRAM2 reset */
#define FOCALTECH_RESET_SRAM2     FOCALTECH_RESET_CONFIG(SYSRSTCR, 15U)
/** SRAM3/4 reset */
#define FOCALTECH_RESET_SRAM3_4   FOCALTECH_RESET_CONFIG(SYSRSTCR, 16U)
/** SSI4 reset */
#define FOCALTECH_RESET_SSI4      FOCALTECH_RESET_CONFIG(SYSRSTCR, 17U)
/** SSI5 reset */
#define FOCALTECH_RESET_SSI5      FOCALTECH_RESET_CONFIG(SYSRSTCR, 18U)
/** SSI6 reset */
#define FOCALTECH_RESET_SSI6      FOCALTECH_RESET_CONFIG(SYSRSTCR, 19U)

/** @} */

/**
 * @defgroup focaltech_ahb3rstcr_resets AHB3RSTCR Reset Control
 * @brief AHB3 Reset Control Register peripherals
 * @{
 */

/** ROM reset */
#define FOCALTECH_RESET_ROM         FOCALTECH_RESET_CONFIG(AHB3RSTCR, 0U)
/** ROM OTP MUX reset */
#define FOCALTECH_RESET_ROM_OTP_MUX FOCALTECH_RESET_CONFIG(AHB3RSTCR, 1U)
/** AHB3 MUX reset */
#define FOCALTECH_RESET_AHB3_MUX    FOCALTECH_RESET_CONFIG(AHB3RSTCR, 2U)
/** USB controller reset */
#define FOCALTECH_RESET_USBC        FOCALTECH_RESET_CONFIG(AHB3RSTCR, 3U)
/** PXLP1 APB reset */
#define FOCALTECH_RESET_PXLP1_APB   FOCALTECH_RESET_CONFIG(AHB3RSTCR, 4U)
/** PXLP2 APB reset */
#define FOCALTECH_RESET_PXLP2_APB   FOCALTECH_RESET_CONFIG(AHB3RSTCR, 5U)
/** I2S1 reset */
#define FOCALTECH_RESET_I2S1        FOCALTECH_RESET_CONFIG(AHB3RSTCR, 6U)
/** I2S2 reset */
#define FOCALTECH_RESET_I2S2        FOCALTECH_RESET_CONFIG(AHB3RSTCR, 7U)

/** @} */

/**
 * @defgroup focaltech_arithrstcr_resets ARITHRSTCR Reset Control
 * @brief Algorithm Reset Control Register peripherals
 * @{
 */

/** AES reset */
#define FOCALTECH_RESET_AES        FOCALTECH_RESET_CONFIG(ARITHRSTCR, 1U)
/** SM4 reset */
#define FOCALTECH_RESET_SM4        FOCALTECH_RESET_CONFIG(ARITHRSTCR, 2U)
/** RF AES SM4 reset */
#define FOCALTECH_RESET_RF_AES_SM4 FOCALTECH_RESET_CONFIG(ARITHRSTCR, 4U)
/** Crypto reset */
#define FOCALTECH_RESET_CRYPTO     FOCALTECH_RESET_CONFIG(ARITHRSTCR, 5U)
/** SHA reset */
#define FOCALTECH_RESET_SHA        FOCALTECH_RESET_CONFIG(ARITHRSTCR, 6U)
/** EDMAC0 reset */
#define FOCALTECH_RESET_EDMAC0     FOCALTECH_RESET_CONFIG(ARITHRSTCR, 7U)
/** DES reset */
#define FOCALTECH_RESET_DES        FOCALTECH_RESET_CONFIG(ARITHRSTCR, 8U)
/** ZUC reset */
#define FOCALTECH_RESET_ZUC        FOCALTECH_RESET_CONFIG(ARITHRSTCR, 9U)
/** AHB2MLB reset */
#define FOCALTECH_RESET_AHB2MLB    FOCALTECH_RESET_CONFIG(ARITHRSTCR, 10U)
/** AHB2IPS2 reset */
#define FOCALTECH_RESET_AHB2IPS2   FOCALTECH_RESET_CONFIG(ARITHRSTCR, 11U)
/** TRNG reset */
#define FOCALTECH_RESET_TRNG       FOCALTECH_RESET_CONFIG(ARITHRSTCR, 12U)

/** @} */

/**
 * @defgroup focaltech_ipsrstcr_resets IPSRSTCR Reset Control
 * @brief IPS Reset Control Register peripherals
 * @{
 */

/** IO controller reset */
#define FOCALTECH_RESET_IO_CTRL FOCALTECH_RESET_CONFIG(IPSRSTCR, 0U)
/** Watchdog timer reset */
#define FOCALTECH_RESET_WDT     FOCALTECH_RESET_CONFIG(IPSRSTCR, 1U)
/** RTC reset */
#define FOCALTECH_RESET_RTC     FOCALTECH_RESET_CONFIG(IPSRSTCR, 2U)
/** PIT1 reset */
#define FOCALTECH_RESET_PIT1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 3U)
/** PIT2 reset */
#define FOCALTECH_RESET_PIT2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 4U)
/** USI1 reset */
#define FOCALTECH_RESET_USI1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 5U)
/** EDMAC1 reset */
#define FOCALTECH_RESET_EDMAC1  FOCALTECH_RESET_CONFIG(IPSRSTCR, 6U)
/** SPI1 reset */
#define FOCALTECH_RESET_SPI1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 7U)
/** SPI2 reset */
#define FOCALTECH_RESET_SPI2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 8U)
/** SPI3 reset */
#define FOCALTECH_RESET_SPI3    FOCALTECH_RESET_CONFIG(IPSRSTCR, 9U)
/** SCI1 reset */
#define FOCALTECH_RESET_SCI1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 10U)
/** SCI2 reset */
#define FOCALTECH_RESET_SCI2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 11U)
/** USI2 reset */
#define FOCALTECH_RESET_USI2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 12U)
/** CAN1 reset */
#define FOCALTECH_RESET_CAN1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 13U)
/** I2C1 reset */
#define FOCALTECH_RESET_I2C1    FOCALTECH_RESET_CONFIG(IPSRSTCR, 14U)
/** PWM0 reset */
#define FOCALTECH_RESET_PWM0    FOCALTECH_RESET_CONFIG(IPSRSTCR, 15U)
/** I2C2 reset */
#define FOCALTECH_RESET_I2C2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 16U)
/** CAN2 reset */
#define FOCALTECH_RESET_CAN2    FOCALTECH_RESET_CONFIG(IPSRSTCR, 17U)
/** SCI3 reset */
#define FOCALTECH_RESET_SCI3    FOCALTECH_RESET_CONFIG(IPSRSTCR, 18U)
/** QADC reset */
#define FOCALTECH_RESET_QADC    FOCALTECH_RESET_CONFIG(IPSRSTCR, 20U)
/** DAC reset */
#define FOCALTECH_RESET_DAC     FOCALTECH_RESET_CONFIG(IPSRSTCR, 21U)
/** MCC IPS reset */
#define FOCALTECH_RESET_MCC_IPS FOCALTECH_RESET_CONFIG(IPSRSTCR, 22U)
/** PWMT1 reset */
#define FOCALTECH_RESET_PWMT1   FOCALTECH_RESET_CONFIG(IPSRSTCR, 23U)
/** LD reset */
#define FOCALTECH_RESET_LD      FOCALTECH_RESET_CONFIG(IPSRSTCR, 24U)
/** PWMT2 reset */
#define FOCALTECH_RESET_PWMT2   FOCALTECH_RESET_CONFIG(IPSRSTCR, 25U)
/** PGD reset */
#define FOCALTECH_RESET_PGD     FOCALTECH_RESET_CONFIG(IPSRSTCR, 26U)
/** Security detector reset */
#define FOCALTECH_RESET_SEC_DET FOCALTECH_RESET_CONFIG(IPSRSTCR, 27U)
/** PCI reset */
#define FOCALTECH_RESET_PCI     FOCALTECH_RESET_CONFIG(IPSRSTCR, 28U)
/** PMU RTC reset */
#define FOCALTECH_RESET_PMURTC  FOCALTECH_RESET_CONFIG(IPSRSTCR, 29U)
/** AHB2IPS reset */
#define FOCALTECH_RESET_AHB2IPS FOCALTECH_RESET_CONFIG(IPSRSTCR, 30U)
/** PWMT3 reset */
#define FOCALTECH_RESET_PWMT3   FOCALTECH_RESET_CONFIG(IPSRSTCR, 31U)

/** @} */

/** @} */

#endif /* ZEPHYR_INCLUDE_DT_BINDINGS_RESET_FOCALTECH_FT9001_RESET_H_ */
