// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <XCAFDoc_Datum.hxx>

#include <Standard_Type.hxx>
#include <TDF_Attribute.hxx>
#include <TDF_Label.hxx>
#include <TDF_ChildIterator.hxx>
#include <TDF_RelocationTable.hxx>
#include <TDataStd_AsciiString.hxx>
#include <TDataStd_IntegerArray.hxx>
#include <TDataStd_RealArray.hxx>
#include <TDataStd_Integer.hxx>
#include <TDataStd_Real.hxx>
#include <TNaming_Builder.hxx>
#include <TNaming_Tool.hxx>
#include <TDataStd_Name.hxx>

#include <XCAFDimTolObjects_DatumObject.hxx>

IMPLEMENT_STANDARD_RTTIEXT(XCAFDoc_Datum, TDF_Attribute)

enum ChildLab
{
  ChildLab_Begin = 1,
  ChildLab_Name  = ChildLab_Begin,
  ChildLab_Position,
  ChildLab_Modifiers,
  ChildLab_ModifierWithValue,
  ChildLab_IsDTarget,
  ChildLab_DTargetType,
  ChildLab_AxisLoc,
  ChildLab_AxisN,
  ChildLab_AxisRef,
  ChildLab_DTargetLength,
  ChildLab_DTargetWidth,
  ChildLab_DTargetNumber,
  ChildLab_DatumTarget,
  ChildLab_PlaneLoc,
  ChildLab_PlaneN,
  ChildLab_PlaneRef,
  ChildLab_Pnt,
  ChildLab_PntText,
  ChildLab_Presentation,
  ChildLab_End
};

//=================================================================================================

XCAFDoc_Datum::XCAFDoc_Datum() = default;

//=================================================================================================

const Standard_GUID& XCAFDoc_Datum::GetID()
{
  static Standard_GUID DID("58ed092e-44de-11d8-8776-001083004c77");
  // static Standard_GUID ID("efd212e2-6dfd-11d4-b9c8-0060b0ee281b");
  return DID;
  // return ID;
}

//=================================================================================================

occ::handle<XCAFDoc_Datum> XCAFDoc_Datum::Set(
  const TDF_Label&                             theLabel,
  const occ::handle<TCollection_HAsciiString>& theName,
  const occ::handle<TCollection_HAsciiString>& theDescription,
  const occ::handle<TCollection_HAsciiString>& theIdentification)
{
  occ::handle<XCAFDoc_Datum> aDatum;
  if (!theLabel.FindAttribute(XCAFDoc_Datum::GetID(), aDatum))
  {
    aDatum = new XCAFDoc_Datum();
    theLabel.AddAttribute(aDatum);
  }
  aDatum->Set(theName, theDescription, theIdentification);
  return aDatum;
}

//=================================================================================================

occ::handle<XCAFDoc_Datum> XCAFDoc_Datum::Set(const TDF_Label& theLabel)
{
  occ::handle<XCAFDoc_Datum> aDatum;
  if (!theLabel.FindAttribute(XCAFDoc_Datum::GetID(), aDatum))
  {
    aDatum = new XCAFDoc_Datum();
    theLabel.AddAttribute(aDatum);
  }
  return aDatum;
}

//=================================================================================================

void XCAFDoc_Datum::Set(const occ::handle<TCollection_HAsciiString>& theName,
                        const occ::handle<TCollection_HAsciiString>& theDescription,
                        const occ::handle<TCollection_HAsciiString>& theIdentification)
{
  Backup();
  myName           = theName;
  myDescription    = theDescription;
  myIdentification = theIdentification;
}

//=================================================================================================

occ::handle<TCollection_HAsciiString> XCAFDoc_Datum::GetName() const
{
  if (myName.IsNull())
    return new TCollection_HAsciiString();
  return myName;
}

//=================================================================================================

occ::handle<TCollection_HAsciiString> XCAFDoc_Datum::GetDescription() const
{
  return myDescription;
}

//=================================================================================================

occ::handle<TCollection_HAsciiString> XCAFDoc_Datum::GetIdentification() const
{
  return myIdentification;
}

//=================================================================================================

void XCAFDoc_Datum::SetObject(const occ::handle<XCAFDimTolObjects_DatumObject>& theObject)
{
  Backup();

  if (theObject->GetSemanticName())
  {
    TCollection_ExtendedString str(theObject->GetSemanticName()->String());
    TDataStd_Name::Set(Label(), str);
  }

  for (int aChild = ChildLab_Begin; aChild < ChildLab_End; aChild++)
  {
    Label().FindChild(aChild).ForgetAllAttributes();
  }
  if (!theObject->GetName().IsNull() && !theObject->GetName()->IsEmpty())
    occ::handle<TDataStd_AsciiString> anAttName =
      TDataStd_AsciiString::Set(Label().FindChild(ChildLab_Name), theObject->GetName()->String());

  occ::handle<TDataStd_Integer> aPosition =
    TDataStd_Integer::Set(Label().FindChild(ChildLab_Position), theObject->GetPosition());

  if (theObject->GetModifiers().Length() > 0)
  {
    occ::handle<NCollection_HArray1<int>> anArr =
      new NCollection_HArray1<int>(1, theObject->GetModifiers().Length());
    for (int i = 1; i <= theObject->GetModifiers().Length(); i++)
      anArr->SetValue(i, theObject->GetModifiers().Value(i));
    occ::handle<TDataStd_IntegerArray> aModifiers =
      TDataStd_IntegerArray::Set(Label().FindChild(ChildLab_Modifiers),
                                 1,
                                 theObject->GetModifiers().Length());
    if (!aModifiers.IsNull())
      aModifiers->ChangeArray(anArr);
  }

  XCAFDimTolObjects_DatumModifWithValue aM;
  double                                aV;
  theObject->GetModifierWithValue(aM, aV);
  if (aM != XCAFDimTolObjects_DatumModifWithValue_None)
  {
    occ::handle<TDataStd_Integer> aModifierWithValueM =
      TDataStd_Integer::Set(Label().FindChild(ChildLab_ModifierWithValue), aM);

    occ::handle<TDataStd_Real> aModifierWithValueV =
      TDataStd_Real::Set(Label().FindChild(ChildLab_ModifierWithValue), aV);
  }

  occ::handle<TDataStd_Integer> aIsTarget =
    TDataStd_Integer::Set(Label().FindChild(ChildLab_IsDTarget), theObject->IsDatumTarget());

  if (theObject->IsDatumTarget())
  {
    occ::handle<TDataStd_Integer> aType =
      TDataStd_Integer::Set(Label().FindChild(ChildLab_DTargetType),
                            theObject->GetDatumTargetType());

    if (theObject->GetDatumTargetType() == XCAFDimTolObjects_DatumTargetType_Area)
    {
      if (!theObject->GetDatumTarget().IsNull())
      {
        TNaming_Builder tnBuild(Label().FindChild(ChildLab_DatumTarget));
        tnBuild.Generated(theObject->GetDatumTarget());
      }
    }
    else if (theObject->HasDatumTargetParams())
    {
      gp_Ax2                                   anAx    = theObject->GetDatumTargetAxis();
      occ::handle<NCollection_HArray1<double>> aLocArr = new NCollection_HArray1<double>(1, 3);
      for (int i = 1; i <= 3; i++)
        aLocArr->SetValue(i, anAx.Location().Coord(i));
      occ::handle<TDataStd_RealArray> aLoc =
        TDataStd_RealArray::Set(Label().FindChild(ChildLab_AxisLoc), 1, 3);
      if (!aLoc.IsNull())
        aLoc->ChangeArray(aLocArr);

      occ::handle<NCollection_HArray1<double>> aNArr = new NCollection_HArray1<double>(1, 3);
      for (int i = 1; i <= 3; i++)
        aNArr->SetValue(i, anAx.Direction().Coord(i));
      occ::handle<TDataStd_RealArray> aN =
        TDataStd_RealArray::Set(Label().FindChild(ChildLab_AxisN), 1, 3);
      if (!aN.IsNull())
        aN->ChangeArray(aNArr);

      occ::handle<NCollection_HArray1<double>> aRArr = new NCollection_HArray1<double>(1, 3);
      for (int i = 1; i <= 3; i++)
        aRArr->SetValue(i, anAx.XDirection().Coord(i));
      occ::handle<TDataStd_RealArray> aR =
        TDataStd_RealArray::Set(Label().FindChild(ChildLab_AxisRef), 1, 3);
      if (!aR.IsNull())
        aR->ChangeArray(aRArr);

      if (theObject->GetDatumTargetType() != XCAFDimTolObjects_DatumTargetType_Point)
      {
        occ::handle<TDataStd_Real> aLen =
          TDataStd_Real::Set(Label().FindChild(ChildLab_DTargetLength),
                             theObject->GetDatumTargetLength());
        if (theObject->GetDatumTargetType() == XCAFDimTolObjects_DatumTargetType_Rectangle)
          occ::handle<TDataStd_Real> aWidth =
            TDataStd_Real::Set(Label().FindChild(ChildLab_DTargetWidth),
                               theObject->GetDatumTargetWidth());
      }
    }
    occ::handle<TDataStd_Integer> aNum =
      TDataStd_Integer::Set(Label().FindChild(ChildLab_DTargetNumber),
                            theObject->GetDatumTargetNumber());
  }

  if (theObject->HasPlane())
  {
    gp_Ax2 anAx = theObject->GetPlane();

    occ::handle<NCollection_HArray1<double>> aLocArr = new NCollection_HArray1<double>(1, 3);
    for (int i = 1; i <= 3; i++)
      aLocArr->SetValue(i, anAx.Location().Coord(i));
    occ::handle<TDataStd_RealArray> aLoc =
      TDataStd_RealArray::Set(Label().FindChild(ChildLab_PlaneLoc), 1, 3);
    if (!aLoc.IsNull())
      aLoc->ChangeArray(aLocArr);

    occ::handle<NCollection_HArray1<double>> aNArr = new NCollection_HArray1<double>(1, 3);
    for (int i = 1; i <= 3; i++)
      aNArr->SetValue(i, anAx.Direction().Coord(i));
    occ::handle<TDataStd_RealArray> aN =
      TDataStd_RealArray::Set(Label().FindChild(ChildLab_PlaneN), 1, 3);
    if (!aN.IsNull())
      aN->ChangeArray(aNArr);

    occ::handle<NCollection_HArray1<double>> aRArr = new NCollection_HArray1<double>(1, 3);
    for (int i = 1; i <= 3; i++)
      aRArr->SetValue(i, anAx.XDirection().Coord(i));
    occ::handle<TDataStd_RealArray> aR =
      TDataStd_RealArray::Set(Label().FindChild(ChildLab_PlaneRef), 1, 3);
    if (!aR.IsNull())
      aR->ChangeArray(aRArr);
  }

  if (theObject->HasPoint())
  {
    gp_Pnt aPnt = theObject->GetPoint();

    occ::handle<NCollection_HArray1<double>> aLocArr = new NCollection_HArray1<double>(1, 3);
    for (int i = 1; i <= 3; i++)
      aLocArr->SetValue(i, aPnt.Coord(i));
    occ::handle<TDataStd_RealArray> aLoc =
      TDataStd_RealArray::Set(Label().FindChild(ChildLab_Pnt), 1, 3);
    if (!aLoc.IsNull())
      aLoc->ChangeArray(aLocArr);
  }

  if (theObject->HasPointText())
  {
    gp_Pnt aPntText = theObject->GetPointTextAttach();

    occ::handle<NCollection_HArray1<double>> aLocArr = new NCollection_HArray1<double>(1, 3);
    for (int i = 1; i <= 3; i++)
      aLocArr->SetValue(i, aPntText.Coord(i));
    occ::handle<TDataStd_RealArray> aLoc =
      TDataStd_RealArray::Set(Label().FindChild(ChildLab_PntText), 1, 3);
    if (!aLoc.IsNull())
      aLoc->ChangeArray(aLocArr);
  }

  TopoDS_Shape aPresentation = theObject->GetPresentation();
  if (!aPresentation.IsNull())
  {
    TDF_Label       aLPres = Label().FindChild(ChildLab_Presentation);
    TNaming_Builder tnBuild(aLPres);
    tnBuild.Generated(aPresentation);
    occ::handle<TCollection_HAsciiString> aName = theObject->GetPresentationName();
    if (!aName.IsNull())
    {
      TCollection_ExtendedString str(aName->String());
      TDataStd_Name::Set(aLPres, str);
    }
  }
}

//=================================================================================================

occ::handle<XCAFDimTolObjects_DatumObject> XCAFDoc_Datum::GetObject() const
{
  occ::handle<XCAFDimTolObjects_DatumObject> anObj = new XCAFDimTolObjects_DatumObject();

  occ::handle<TDataStd_Name>            aSemanticNameAttr;
  occ::handle<TCollection_HAsciiString> aSemanticName;
  if (Label().FindAttribute(TDataStd_Name::GetID(), aSemanticNameAttr))
  {
    const TCollection_ExtendedString& aName = aSemanticNameAttr->Get();
    if (!aName.IsEmpty())
      aSemanticName = new TCollection_HAsciiString(aName);
  }
  anObj->SetSemanticName(aSemanticName);

  occ::handle<TDataStd_AsciiString> anAttName;
  if (Label().FindChild(ChildLab_Name).FindAttribute(TDataStd_AsciiString::GetID(), anAttName))
  {
    occ::handle<TCollection_HAsciiString> aStr = new TCollection_HAsciiString(anAttName->Get());
    anObj->SetName(aStr);
  }

  occ::handle<TDataStd_IntegerArray> anArr;
  if (Label().FindChild(ChildLab_Modifiers).FindAttribute(TDataStd_IntegerArray::GetID(), anArr)
      && !anArr->Array().IsNull())
  {
    NCollection_Sequence<XCAFDimTolObjects_DatumSingleModif> aModifiers;
    for (int i = 1; i <= anArr->Length(); i++)
      aModifiers.Append((XCAFDimTolObjects_DatumSingleModif)anArr->Value(i));
    anObj->SetModifiers(aModifiers);
  }

  occ::handle<TDataStd_Integer> aModifierWithValueM;
  if (Label()
        .FindChild(ChildLab_ModifierWithValue)
        .FindAttribute(TDataStd_Integer::GetID(), aModifierWithValueM))
  {
    occ::handle<TDataStd_Real> aModifierWithValueV;
    if (Label()
          .FindChild(ChildLab_ModifierWithValue)
          .FindAttribute(TDataStd_Real::GetID(), aModifierWithValueV))
    {
      anObj->SetModifierWithValue((XCAFDimTolObjects_DatumModifWithValue)aModifierWithValueM->Get(),
                                  aModifierWithValueV->Get());
    }
  }

  occ::handle<TDataStd_Integer> aPosition;
  if (Label().FindChild(ChildLab_Position).FindAttribute(TDataStd_Integer::GetID(), aPosition))
  {
    anObj->SetPosition(aPosition->Get());
  }

  occ::handle<TDataStd_RealArray> aLoc, aN, aR;
  if (Label().FindChild(ChildLab_PlaneLoc).FindAttribute(TDataStd_RealArray::GetID(), aLoc)
      && aLoc->Length() == 3
      && Label().FindChild(ChildLab_PlaneN).FindAttribute(TDataStd_RealArray::GetID(), aN)
      && aN->Length() == 3
      && Label().FindChild(ChildLab_PlaneRef).FindAttribute(TDataStd_RealArray::GetID(), aR)
      && aR->Length() == 3)
  {
    gp_Pnt aL(aLoc->Value(aLoc->Lower()),
              aLoc->Value(aLoc->Lower() + 1),
              aLoc->Value(aLoc->Lower() + 2));
    gp_Dir aD(aN->Value(aN->Lower()), aN->Value(aN->Lower() + 1), aN->Value(aN->Lower() + 2));
    gp_Dir aDR(aR->Value(aR->Lower()), aR->Value(aR->Lower() + 1), aR->Value(aR->Lower() + 2));
    gp_Ax2 anAx(aL, aD, aDR);
    anObj->SetPlane(anAx);
  }

  occ::handle<TDataStd_RealArray> aPnt;
  if (Label().FindChild(ChildLab_Pnt).FindAttribute(TDataStd_RealArray::GetID(), aPnt)
      && aPnt->Length() == 3)
  {
    gp_Pnt aP(aLoc->Value(aPnt->Lower()),
              aPnt->Value(aPnt->Lower() + 1),
              aPnt->Value(aPnt->Lower() + 2));
    anObj->SetPoint(aP);
  }

  occ::handle<TDataStd_RealArray> aPntText;
  if (Label().FindChild(ChildLab_PntText).FindAttribute(TDataStd_RealArray::GetID(), aPntText)
      && aPntText->Length() == 3)
  {
    gp_Pnt aP(aPntText->Value(aPntText->Lower()),
              aPntText->Value(aPntText->Lower() + 1),
              aPntText->Value(aPntText->Lower() + 2));
    anObj->SetPointTextAttach(aP);
  }

  occ::handle<TDataStd_Integer> aIsDTarget;
  if (Label().FindChild(ChildLab_IsDTarget).FindAttribute(TDataStd_Integer::GetID(), aIsDTarget))
  {
    anObj->IsDatumTarget((aIsDTarget->Get() != 0));
  }
  else
  {
    return anObj;
  }

  if (aIsDTarget->Get() != 0)
  {
    occ::handle<TDataStd_Integer> aDTargetType;
    if (Label()
          .FindChild(ChildLab_DTargetType)
          .FindAttribute(TDataStd_Integer::GetID(), aDTargetType))
    {
      anObj->SetDatumTargetType((XCAFDimTolObjects_DatumTargetType)aDTargetType->Get());
      if (anObj->GetDatumTargetType() == XCAFDimTolObjects_DatumTargetType_Area)
      {
        occ::handle<TNaming_NamedShape> aDatumTarget;
        if (Label()
              .FindChild(ChildLab_DatumTarget)
              .FindAttribute(TNaming_NamedShape::GetID(), aDatumTarget))
        {
          anObj->SetDatumTarget(aDatumTarget->Get());
        }
      }
      else
      {
        if (Label().FindChild(ChildLab_AxisLoc).FindAttribute(TDataStd_RealArray::GetID(), aLoc)
            && aLoc->Length() == 3
            && Label().FindChild(ChildLab_AxisN).FindAttribute(TDataStd_RealArray::GetID(), aN)
            && aN->Length() == 3
            && Label().FindChild(ChildLab_AxisRef).FindAttribute(TDataStd_RealArray::GetID(), aR)
            && aR->Length() == 3)
        {
          gp_Pnt aL(aLoc->Value(aLoc->Lower()),
                    aLoc->Value(aLoc->Lower() + 1),
                    aLoc->Value(aLoc->Lower() + 2));
          gp_Dir aD(aN->Value(aN->Lower()), aN->Value(aN->Lower() + 1), aN->Value(aN->Lower() + 2));
          gp_Dir aDR(aR->Value(aR->Lower()),
                     aR->Value(aR->Lower() + 1),
                     aR->Value(aR->Lower() + 2));
          gp_Ax2 anAx(aL, aD, aDR);
          anObj->SetDatumTargetAxis(anAx);
        }

        if (anObj->GetDatumTargetType() != XCAFDimTolObjects_DatumTargetType_Point)
        {
          occ::handle<TDataStd_Real> aLen;
          if (Label().FindChild(ChildLab_DTargetLength).FindAttribute(TDataStd_Real::GetID(), aLen))
          {
            anObj->SetDatumTargetLength(aLen->Get());
          }
          if (anObj->GetDatumTargetType() == XCAFDimTolObjects_DatumTargetType_Rectangle)
          {
            occ::handle<TDataStd_Real> aWidth;
            if (Label()
                  .FindChild(ChildLab_DTargetWidth)
                  .FindAttribute(TDataStd_Real::GetID(), aWidth))
            {
              anObj->SetDatumTargetWidth(aWidth->Get());
            }
          }
        }
      }
    }
    occ::handle<TDataStd_Integer> aNum;
    if (Label().FindChild(ChildLab_DTargetNumber).FindAttribute(TDataStd_Integer::GetID(), aNum))
    {
      anObj->SetDatumTargetNumber(aNum->Get());
    }
    else
    {
      anObj->SetDatumTargetNumber(0);
    }
  }

  occ::handle<TNaming_NamedShape> aNS;
  TDF_Label                       aLPres = Label().FindChild(ChildLab_Presentation);
  if (aLPres.FindAttribute(TNaming_NamedShape::GetID(), aNS))
  {

    TopoDS_Shape aPresentation = TNaming_Tool::GetShape(aNS);
    if (!aPresentation.IsNull())
    {
      occ::handle<TDataStd_Name>            aNameAtrr;
      occ::handle<TCollection_HAsciiString> aPresentName;
      if (aLPres.FindAttribute(TDataStd_Name::GetID(), aNameAtrr))
      {
        const TCollection_ExtendedString& aName = aNameAtrr->Get();

        if (!aName.IsEmpty())
          aPresentName = new TCollection_HAsciiString(aName);
      }

      anObj->SetPresentation(aPresentation, aPresentName);
    }
  }
  return anObj;
}

//=================================================================================================

const Standard_GUID& XCAFDoc_Datum::ID() const
{
  return GetID();
}

//=================================================================================================

void XCAFDoc_Datum::Restore(const occ::handle<TDF_Attribute>& theWith)
{
  myName           = occ::down_cast<XCAFDoc_Datum>(theWith)->GetName();
  myDescription    = occ::down_cast<XCAFDoc_Datum>(theWith)->GetDescription();
  myIdentification = occ::down_cast<XCAFDoc_Datum>(theWith)->GetIdentification();
}

//=================================================================================================

occ::handle<TDF_Attribute> XCAFDoc_Datum::NewEmpty() const
{
  return new XCAFDoc_Datum();
}

//=================================================================================================

void XCAFDoc_Datum::Paste(const occ::handle<TDF_Attribute>& theInto,
                          const occ::handle<TDF_RelocationTable>& /*RT*/) const
{
  occ::down_cast<XCAFDoc_Datum>(theInto)->Set(myName, myDescription, myIdentification);
}

//=================================================================================================

void XCAFDoc_Datum::DumpJson(Standard_OStream& theOStream, int theDepth) const
{
  OCCT_DUMP_TRANSIENT_CLASS_BEGIN(theOStream)

  OCCT_DUMP_BASE_CLASS(theOStream, theDepth, TDF_Attribute)

  OCCT_DUMP_FIELD_VALUE_STRING(theOStream, myName.get())
  OCCT_DUMP_FIELD_VALUE_STRING(theOStream, myDescription.get())
  OCCT_DUMP_FIELD_VALUE_STRING(theOStream, myIdentification.get())

  occ::handle<XCAFDimTolObjects_DatumObject> anObject = GetObject();
  OCCT_DUMP_FIELD_VALUES_DUMPED(theOStream, theDepth, anObject.get())
}
