// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <Interface_Check.hxx>
#include <Interface_CheckIterator.hxx>
#include <MoniTool_Macros.hxx>
#include <Interface_Static.hxx>
#include <Message_Messenger.hxx>
#include <Standard_Failure.hxx>
#include <Standard_Transient.hxx>
#include <Standard_ErrorHandler.hxx>
#include <Standard_Type.hxx>
#include <Transfer_ActorOfFinderProcess.hxx>
#include <Transfer_SimpleBinderOfTransient.hxx>
#include <Transfer_TransientMapper.hxx>
#include <XSControl_Controller.hxx>
#include <XSControl_TransferWriter.hxx>
#include <XSControl_Utils.hxx>

IMPLEMENT_STANDARD_RTTIEXT(XSControl_TransferWriter, Standard_Transient)

//=================================================================================================

void XSControl_TransferWriter::Clear(const int mode)
{
  if (mode < 0 || myTransferWriter.IsNull())
    myTransferWriter = new Transfer_FinderProcess;
  else
    myTransferWriter->Clear();
}

//=================================================================================================

void XSControl_TransferWriter::PrintStats(const int, const int) const
{
  Message_Messenger::StreamBuffer sout = myTransferWriter->Messenger()->SendInfo();
  //  A ameliorer ... !
  sout << "\n*******************************************************************\n";
  sout << "******        Statistics on Transfer (Write)                 ******" << std::endl;
  sout << "\n*******************************************************************\n";
  sout << "******        Transfer Mode = " << myTransferMode;
  const char* modehelp = myController->ModeWriteHelp(myTransferMode);
  if (modehelp && modehelp[0] != 0)
    sout << "  I.E.  " << modehelp;
  sout << "       ******" << std::endl;
}

//  ##########    LES ACTIONS    ##########

//=================================================================================================

bool XSControl_TransferWriter::RecognizeTransient(const occ::handle<Standard_Transient>& obj)
{
  if (myController.IsNull())
    return false;
  XSControl_Utils xu;
  TopoDS_Shape    sh = xu.BinderShape(obj);
  if (!sh.IsNull())
    return RecognizeShape(sh);
  return myController->RecognizeWriteTransient(obj, myTransferMode);
}

//=================================================================================================

IFSelect_ReturnStatus XSControl_TransferWriter::TransferWriteTransient(
  const occ::handle<Interface_InterfaceModel>& model,
  const occ::handle<Standard_Transient>&       obj,
  const Message_ProgressRange&                 theProgress)
{
  IFSelect_ReturnStatus status = IFSelect_RetVoid;
  if (myController.IsNull())
    return IFSelect_RetError;
  if (model.IsNull())
    return IFSelect_RetVoid;

  if (myTransferWriter.IsNull())
    myTransferWriter = new Transfer_FinderProcess;
  occ::handle<Transfer_ActorOfFinderProcess> nulact;
  myTransferWriter->SetActor(nulact);
  occ::handle<Standard_Transient> resultat;
  Message_Messenger::StreamBuffer sout = myTransferWriter->Messenger()->SendInfo();
  try
  {
    OCC_CATCH_SIGNALS
    PrintStats(myTransferMode);
    sout << "******        Transferring Transient, CDL Type = ";
    sout << obj->DynamicType()->Name() << "   ******" << std::endl;
    status = myController->TransferWriteTransient(obj,
                                                  myTransferWriter,
                                                  model,
                                                  myTransferMode,
                                                  theProgress);
  }
  catch (Standard_Failure const& anException)
  {
    sout << "****  ****  TransferWriteShape, EXCEPTION : ";
    sout << anException.GetMessageString();
    sout << std::endl;
    status = IFSelect_RetFail;
  }
  return status;
}

//=================================================================================================

bool XSControl_TransferWriter::RecognizeShape(const TopoDS_Shape& shape)
{
  if (myController.IsNull())
    return false;
  if (shape.IsNull())
    return false;
  return myController->RecognizeWriteShape(shape, myTransferMode);
}

//=================================================================================================

IFSelect_ReturnStatus XSControl_TransferWriter::TransferWriteShape(
  const occ::handle<Interface_InterfaceModel>& theModel,
  const TopoDS_Shape&                          theShape,
  const Message_ProgressRange&                 theProgress)
{
  IFSelect_ReturnStatus status = IFSelect_RetVoid;
  if (myController.IsNull())
    return IFSelect_RetError;
  if (theModel.IsNull())
    return IFSelect_RetVoid;

  TopoDS_Shape aShape = theShape;

  if (myTransferWriter.IsNull())
    myTransferWriter = new Transfer_FinderProcess;
  //  effacer l actor : Controller s en charge
  occ::handle<Transfer_ActorOfFinderProcess> nulact;
  myTransferWriter->SetActor(nulact);
  occ::handle<Standard_Transient> resultat;
  Message_Messenger::StreamBuffer sout = myTransferWriter->Messenger()->SendInfo();
  try
  {
    OCC_CATCH_SIGNALS
    PrintStats(myTransferMode);
    sout << "******        Transferring Shape, ShapeType = " << aShape.ShapeType();
    sout << "                      ******" << std::endl;
    status = myController->TransferWriteShape(aShape,
                                              myTransferWriter,
                                              theModel,
                                              myTransferMode,
                                              theProgress);
  }
  catch (Standard_Failure const& anException)
  {
    sout << "****  ****  TransferWriteShape, EXCEPTION : ";
    sout << anException.GetMessageString();
    sout << std::endl;
    status = IFSelect_RetFail;
  }
  return status;
}

//=================================================================================================

Interface_CheckIterator XSControl_TransferWriter::CheckList() const
{
  Interface_CheckIterator chl;
  if (myTransferWriter.IsNull())
    return chl;
  int i, nb = myTransferWriter->NbMapped();
  for (i = 1; i <= nb; i++)
  {
    DeclareAndCast(Transfer_SimpleBinderOfTransient, tb, myTransferWriter->MapItem(i));
    if (tb.IsNull())
      continue;
    occ::handle<Interface_Check> ach = tb->Check();
    if (ach->NbFails() == 0 || ach->NbWarnings() == 0)
      continue;
    DeclareAndCast(Transfer_TransientMapper, tm, myTransferWriter->Mapped(i));
    if (tm.IsNull())
      ach->GetEntity(myTransferWriter->Mapped(i));
    else
      ach->GetEntity(tm->Value());
    chl.Add(ach);
  }
  return chl;
}

//=================================================================================================

Interface_CheckIterator XSControl_TransferWriter::ResultCheckList(
  const occ::handle<Interface_InterfaceModel>& model) const
{
  Interface_CheckIterator chl;
  if (myTransferWriter.IsNull())
    return chl;
  int i, nb = myTransferWriter->NbMapped();
  for (i = 1; i <= nb; i++)
  {
    DeclareAndCast(Transfer_SimpleBinderOfTransient, tb, myTransferWriter->MapItem(i));
    if (tb.IsNull())
      continue;
    const occ::handle<Interface_Check> ach = tb->Check();
    if (ach->NbFails() == 0 || ach->NbWarnings() == 0)
      continue;
    occ::handle<Standard_Transient> ent = tb->Result();
    if (!ent.IsNull() && !model.IsNull())
      chl.Add(ach, model->Number(ent));
    else
      chl.Add(ach, 0);
  }
  return chl;
}
