// Copyright (c) 1995-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include <DrawTrSurf_Curve2d.hxx>

#include <Draw_Color.hxx>
#include <Draw_Display.hxx>
#include <Draw_Drawable3D.hxx>
#include <DrawTrSurf.hxx>
#include <DrawTrSurf_Params.hxx>
#include <Geom2d_Curve.hxx>
#include <Geom2d_Hyperbola.hxx>
#include <Geom2d_Line.hxx>
#include <Geom2dAdaptor_Curve.hxx>
#include <Geom2dLProp_CLProps2d.hxx>
#include <Geom_Plane.hxx>
#include <GeomTools_Curve2dSet.hxx>
#include <gp.hxx>
#include <Precision.hxx>
#include <NCollection_Array1.hxx>

IMPLEMENT_STANDARD_RTTIEXT(DrawTrSurf_Curve2d, DrawTrSurf_Drawable)

static double DrawTrSurf_CurveLimit = 400;
extern bool   Draw_Bounds;

//=================================================================================================

DrawTrSurf_Curve2d::DrawTrSurf_Curve2d(const occ::handle<Geom2d_Curve>& C, const bool DispOrigin)
    : DrawTrSurf_Drawable(50)
{
  curv           = C;
  look           = Draw_vert;
  disporigin     = DispOrigin;
  dispcurvradius = false;
  radiusmax      = 1.0e3;
  radiusratio    = 0.1;
}

//=================================================================================================

DrawTrSurf_Curve2d::DrawTrSurf_Curve2d(const occ::handle<Geom2d_Curve>& C,
                                       const Draw_Color&                aColor,
                                       const int                        Discret,
                                       const bool                       DispOrigin,
                                       const bool                       DispCurvRadius,
                                       const double                     RadiusMax,
                                       const double                     RadiusRatio)
    : DrawTrSurf_Drawable(Discret)
{
  curv           = C;
  look           = aColor;
  disporigin     = DispOrigin;
  dispcurvradius = DispCurvRadius;
  radiusmax      = RadiusMax;
  radiusratio    = RadiusRatio;
}

//=================================================================================================

void DrawTrSurf_Curve2d::DrawOn(Draw_Display& dis) const
{

  double First    = curv->FirstParameter();
  double Last     = curv->LastParameter();
  bool   firstInf = Precision::IsNegativeInfinite(First);
  bool   lastInf  = Precision::IsPositiveInfinite(Last);

  if (firstInf || lastInf)
  {
    gp_Pnt2d P1, P2;
    double   delta = 1;
    if (firstInf && lastInf)
    {
      do
      {
        delta *= 2;
        First = -delta;
        Last  = delta;
        curv->D0(First, P1);
        curv->D0(Last, P2);
      } while (P1.Distance(P2) < DrawTrSurf_CurveLimit);
    }
    else if (firstInf)
    {
      curv->D0(Last, P2);
      do
      {
        delta *= 2;
        First = Last - delta;
        curv->D0(First, P1);
      } while (P1.Distance(P2) < DrawTrSurf_CurveLimit);
    }
    else if (lastInf)
    {
      curv->D0(First, P1);
      do
      {
        delta *= 2;
        Last = First + delta;
        curv->D0(Last, P2);
      } while (P1.Distance(P2) < DrawTrSurf_CurveLimit);
    }
  }

  dis.SetColor(look);

  Geom2dAdaptor_Curve C2d(curv, First, Last);
  DrawCurve2dOn(C2d, dis);

  // mark the origin
  if (disporigin)
  {
    Draw_Bounds = false;
    gp_Pnt2d p1, p2;
    gp_Vec2d v;
    C2d.D1(Last, p1, v);
    if (v.Magnitude() > gp::Resolution())
    {
      double   L = 20 / dis.Zoom();
      double   H = 10 / dis.Zoom();
      gp_Dir2d d(v);
      p2.SetCoord(p1.X() - L * d.X() - H * d.Y(), p1.Y() - L * d.Y() + H * d.X());
      dis.MoveTo(p2);
      p2.SetCoord(p1.X() - L * d.X() + H * d.Y(), p1.Y() - L * d.Y() - H * d.X());
      dis.DrawTo(p1);
      dis.DrawTo(p2);
    }
    Draw_Bounds = true;
  }

  // Draw the curvature Radius
  if (dispcurvradius && (C2d.GetType() != GeomAbs_Line))
  {
    int                        ii;
    int                        intrv, nbintv = C2d.NbIntervals(GeomAbs_CN);
    NCollection_Array1<double> TI(1, nbintv + 1);
    C2d.Intervals(TI, GeomAbs_CN);
    double                Resolution = 1.0e-9, Curvature;
    Geom2dLProp_CLProps2d LProp(curv, 2, Resolution);
    gp_Pnt2d              P1, P2;

    for (intrv = 1; intrv <= nbintv; intrv++)
    {
      double t    = TI(intrv);
      double step = (TI(intrv + 1) - t) / GetDiscretisation();
      double LRad, ratio;
      for (ii = 1; ii <= GetDiscretisation(); ii++)
      {
        LProp.SetParameter(t);
        if (LProp.IsTangentDefined())
        {
          Curvature = std::abs(LProp.Curvature());
          if (Curvature > Resolution)
          {
            curv->D0(t, P1);
            dis.MoveTo(P1);
            LRad  = 1. / Curvature;
            ratio = ((LRad > radiusmax) ? radiusmax / LRad : 1);
            ratio *= radiusratio;
            LProp.CentreOfCurvature(P2);
            gp_Vec2d V(P1, P2);
            dis.DrawTo(P1.Translated(ratio * V));
          }
        }
        t += step;
      }
    }
  }
}

//=================================================================================================

occ::handle<Draw_Drawable3D> DrawTrSurf_Curve2d::Copy() const
{
  occ::handle<DrawTrSurf_Curve2d> DC =
    new DrawTrSurf_Curve2d(occ::down_cast<Geom2d_Curve>(curv->Copy()), look, GetDiscretisation());

  return DC;
}

//=================================================================================================

void DrawTrSurf_Curve2d::Dump(Standard_OStream& S) const
{
  GeomTools_Curve2dSet::PrintCurve2d(curv, S);
}

//=================================================================================================

void DrawTrSurf_Curve2d::Save(Standard_OStream& theStream) const
{
  GeomTools_Curve2dSet::PrintCurve2d(GetCurve(), theStream, true);
}

//=================================================================================================

occ::handle<Draw_Drawable3D> DrawTrSurf_Curve2d::Restore(Standard_IStream& theStream)
{
  const DrawTrSurf_Params&        aParams    = DrawTrSurf::Parameters();
  occ::handle<Geom2d_Curve>       aGeomCurve = GeomTools_Curve2dSet::ReadCurve2d(theStream);
  occ::handle<DrawTrSurf_Curve2d> aDrawCurve =
    new DrawTrSurf_Curve2d(aGeomCurve, aParams.CurvColor, aParams.Discret);
  return aDrawCurve;
}

//=================================================================================================

void DrawTrSurf_Curve2d::Whatis(Draw_Interpretor& S) const
{
  S << "2d curve";
}

//=================================================================================================

bool DrawTrSurf_Curve2d::Is3D() const
{
  return false;
}
