// Copyright (c) 2025 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#include "BOPTest_Utilities.pxx"

#ifndef M_SQRT2
  #define M_SQRT2 1.41421356237309504880168872420969808
#endif

//==================================================================================================
// BFuse Simple Tests - migrating from /tests/boolean/bfuse_simple/
//==================================================================================================

class BFuseSimpleTest : public BRepAlgoAPI_TestBase
{
};

// Test bfuse_simple/A1: psphere s 1; box b 1 1 1; bfuse result s b; checkprops result -s 14.6394
TEST_F(BFuseSimpleTest, SpherePlusBox_A1)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateUnitSphere();
  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateUnitBox();

  const TopoDS_Shape aResult = PerformFuse(aSphere, aBox);
  ValidateResult(aResult, 14.6394);
}

// Test bfuse_simple/A2: rotated sphere + box
TEST_F(BFuseSimpleTest, RotatedSpherePlusBox_A2)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateUnitSphere();

  // Apply standard rotation: Z(-90deg) then Y(-45deg)
  const TopoDS_Shape aRotatedSphere = BOPTest_Utilities::RotateStandard(aSphere);

  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateUnitBox();
  const TopoDS_Shape aResult = PerformFuse(aRotatedSphere, aBox);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/A3: box + rotated sphere
TEST_F(BFuseSimpleTest, BoxPlusRotatedSphere_A3)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateUnitSphere();

  // Apply standard rotation: Z(-90deg) then Y(-45deg)
  const TopoDS_Shape aRotatedSphere = BOPTest_Utilities::RotateStandard(aSphere);

  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateUnitBox();
  const TopoDS_Shape aResult = PerformFuse(aBox, aRotatedSphere);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/A4: sphere + rotated box
TEST_F(BFuseSimpleTest, SpherePlusRotatedBox_A4)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateUnitSphere();
  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateUnitBox();

  // Apply Y rotation: 90deg
  const TopoDS_Shape aRotatedBox = BOPTest_Utilities::RotateY(aBox, 90.0);

  const TopoDS_Shape aResult = PerformFuse(aSphere, aRotatedBox);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/A5: rotated box + sphere
TEST_F(BFuseSimpleTest, RotatedBoxPlusSphere_A5)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateUnitSphere();
  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateUnitBox();

  // Apply Y rotation: 90deg
  const TopoDS_Shape aRotatedBox = BOPTest_Utilities::RotateY(aBox, 90.0);

  const TopoDS_Shape aResult = PerformFuse(aRotatedBox, aSphere);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/A6: nurbs box + identical box (should be 6.0 surface area)
TEST_F(BFuseSimpleTest, IdenticalNurbsBoxPlusBox_A6)
{
  // Create first box and convert to NURBS
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/A7: box + nurbs box (identical)
TEST_F(BFuseSimpleTest, IdenticalBoxPlusNurbsBox_A7)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/A8: nurbs box + larger box
TEST_F(BFuseSimpleTest, NurbsBoxPlusLargerBox_A8)
{
  // Create first box and convert to NURBS
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(-0.5, -0.5, -0.5), 2.0, 2.0, 2.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 24.0);
}

// Test bfuse_simple/A9: larger box + nurbs box
TEST_F(BFuseSimpleTest, LargerBoxPlusNurbsBox_A9)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(-0.5, -0.5, -0.5), 2.0, 2.0, 2.0);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 24.0);
}

// Test bfuse_simple/B1: nurbs box + box
TEST_F(BFuseSimpleTest, NurbsBoxPlusBox_B1)
{
  // Create first box and convert to NURBS
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  // Create second box
  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 0.5, 1.0, 0.5);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/B2: box + nurbs box
TEST_F(BFuseSimpleTest, BoxPlusNurbsBox_B2)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, -0.5, 0), 0.5, 0.5, 1.0);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 7.5);
}

// Test bfuse_simple/B3: box + nurbs box (adjacent)
TEST_F(BFuseSimpleTest, NurbsBoxPlusAdjacentBox_B3)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, -0.5, 0), 0.5, 1.5, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 7.5);
}

// Test bfuse_simple/B4: adjacent box + nurbs box
TEST_F(BFuseSimpleTest, AdjacentBoxPlusNurbsBox_B4)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0.5, 0), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 8.0);
}

// Test bfuse_simple/B5: nurbs box + smaller box
TEST_F(BFuseSimpleTest, NurbsBoxPlusSmallerBox_B5)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0.25, 0.25, 0.25), 0.5, 0.5, 0.5);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/B6: smaller box + nurbs box
TEST_F(BFuseSimpleTest, SmallerBoxPlusNurbsBox_B6)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0.25, 0.25, 0.25), 0.5, 0.5, 0.5);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/B7: nurbs box + partially overlapping box
TEST_F(BFuseSimpleTest, NurbsBoxPlusPartiallyOverlappingBox_B7)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0.5, 0.5, 0.5), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 7.75);
}

// Test bfuse_simple/B8: partially overlapping box + nurbs box
TEST_F(BFuseSimpleTest, PartiallyOverlappingBoxPlusNurbsBox_B8)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0.5, 0.5, 0.5), 1.0, 1.0, 1.0);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 7.75);
}

// Test bfuse_simple/B9: nurbs box + extended box
TEST_F(BFuseSimpleTest, NurbsBoxPlusExtendedBox_B9)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(-0.5, -0.5, -0.5), 2.0, 2.0, 2.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 24.0);
}

// Test bfuse_simple/C1: extended box + nurbs box
TEST_F(BFuseSimpleTest, ExtendedBoxPlusNurbsBox_C1)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.5, 0.5, 0.5);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 7.0);
}

// Test bfuse_simple/C2: nurbs box + shifted extended box
TEST_F(BFuseSimpleTest, NurbsBoxPlusShiftedBox_C2)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, -0.5, 0), 1.5, 0.5, 0.5);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 8.5);
}

// Test bfuse_simple/C3: shifted box + nurbs box
TEST_F(BFuseSimpleTest, ShiftedBoxPlusNurbsBox_C3)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0.25, 0, 0), 0.5, 0.5, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/C4: nurbs box + narrow box
TEST_F(BFuseSimpleTest, NurbsBoxPlusNarrowBox_C4)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0.25, 0.25, 0), 0.5, 0.5, 1.5);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.75);
}

// Test bfuse_simple/C5: narrow box + nurbs box
TEST_F(BFuseSimpleTest, NarrowBoxPlusNurbsBox_C5)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0.25, 0.25, 0), 0.5, 0.5, 1.5);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.75);
}

// Test bfuse_simple/C6: nurbs box + corner cube
TEST_F(BFuseSimpleTest, NurbsBoxPlusCornerCube_C6)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0.5, 0.5, 0.5), 0.5, 0.5, 0.5);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/C7: corner cube + nurbs box
TEST_F(BFuseSimpleTest, CornerCubePlusNurbsBox_C7)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0.5, 0.5, 0.5), 0.5, 0.5, 0.5);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.0);
}

// Test bfuse_simple/C8: nurbs box + offset cube
TEST_F(BFuseSimpleTest, NurbsBoxPlusOffsetCube_C8)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);
  EXPECT_FALSE(aBox1.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(1, 1, 0), 0.5, 0.5, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.25);
}

// Test bfuse_simple/C9: offset cube + nurbs box
TEST_F(BFuseSimpleTest, OffsetCubePlusNurbsBox_C9)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(1, 1, 0), 0.5, 0.5, 1.0);

  TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox2              = BOPTest_Utilities::ConvertToNurbs(aBox2);
  EXPECT_FALSE(aBox2.IsNull()) << "Failed to convert to NURBS";

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 6.25);
}

// Test bfuse_simple/D1: nurbs box + rotated narrow box
TEST_F(BFuseSimpleTest, NurbsBoxPlusRotatedNarrowBox_D1)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);

  // Create rotated narrow box: r=sqrt(2), box(0,0,0,r,0.25,1), rotate 45 degrees around Z
  constexpr double r     = M_SQRT2;
  TopoDS_Shape     aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), r, 0.25, 1.0);
  gp_Trsf          aTrsf;
  aTrsf.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)), M_PI / 4.0); // 45 degrees
  aBox2.Move(aTrsf);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 6.41789);
}

// Test bfuse_simple/D2: nurbs box + rotated narrow box variation
TEST_F(BFuseSimpleTest, NurbsBoxPlusRotatedNarrowBoxVariation_D2)
{
  TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  aBox1              = BOPTest_Utilities::ConvertToNurbs(aBox1);

  // Create rotated narrow box: r=sqrt(31), box(0,0,0,r/4,0.25,1), rotate 34.73 degrees around Z
  constexpr double r     = 5.5677643628300219;
  TopoDS_Shape     aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), r / 4.0, 0.25, 1.0);
  gp_Trsf          aTrsf;
  aTrsf.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)), 34.73 * M_PI / 180.0);
  aBox2.Move(aTrsf);

  const TopoDS_Shape aResult = PerformFuse(aBox2, aBox1);
  ValidateResult(aResult, 6.32953);
}

// Test bfuse_simple/D3: sphere + box
TEST_F(BFuseSimpleTest, SpherePlusBox_D3)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);
  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aSphere, aBox);
  ValidateResult(aResult, 14.6394);
}

// Test bfuse_simple/D4: sphere + box (reversed order)
TEST_F(BFuseSimpleTest, BoxPlusSphere_D4)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);
  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox, aSphere);
  ValidateResult(aResult, 14.6394);
}

// Test bfuse_simple/D5: rotated sphere + box
TEST_F(BFuseSimpleTest, RotatedSpherePlusBox_D5)
{
  TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);

  // Apply rotations: -90 degrees around Z, then -45 degrees around Y
  gp_Trsf aTrsf1, aTrsf2, aTrsfCombined;
  aTrsf1.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)), -M_PI / 2.0); // -90 degrees Z
  aTrsf2.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Y)), -M_PI / 4.0); // -45 degrees Y
  aTrsfCombined = aTrsf2 * aTrsf1;
  aSphere.Move(aTrsfCombined);

  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  const TopoDS_Shape aResult = PerformFuse(aSphere, aBox);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/D6: rotated sphere + box (reversed order)
TEST_F(BFuseSimpleTest, BoxPlusRotatedSphere_D6)
{
  TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);

  // Apply rotations: -90 degrees around Z, then -45 degrees around Y
  gp_Trsf aTrsf1, aTrsf2, aTrsfCombined;
  aTrsf1.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)), -M_PI / 2.0);
  aTrsf2.SetRotation(gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Y)), -M_PI / 4.0);
  aTrsfCombined = aTrsf2 * aTrsf1;
  aSphere.Move(aTrsfCombined);

  const TopoDS_Shape aBox    = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  const TopoDS_Shape aResult = PerformFuse(aBox, aSphere);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/D7: sphere + rotated box
TEST_F(BFuseSimpleTest, SpherePlusRotatedBox_D7)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);

  TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  // Rotate box 90 degrees around Y axis: "trotate b 0 0 1 0 1 0 90"
  gp_Trsf aTrsf;
  aTrsf.SetRotation(gp_Ax1(gp_Pnt(0, 0, 1), gp_Dir(gp_Dir::D::Y)), M_PI / 2.0); // 90 degrees Y
  aBox.Move(aTrsf);

  const TopoDS_Shape aResult = PerformFuse(aSphere, aBox);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/D8: rotated box + sphere (reversed order)
TEST_F(BFuseSimpleTest, RotatedBoxPlusSphere_D8)
{
  const TopoDS_Shape aSphere = BOPTest_Utilities::CreateSphere(gp_Pnt(0, 0, 0), 1.0);

  TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 1.0, 1.0, 1.0);
  // Rotate box 90 degrees around Y axis: "trotate b 0 0 1 0 1 0 90"
  gp_Trsf aTrsf;
  aTrsf.SetRotation(gp_Ax1(gp_Pnt(0, 0, 1), gp_Dir(gp_Dir::D::Y)), M_PI / 2.0); // 90 degrees Y
  aBox.Move(aTrsf);

  const TopoDS_Shape aResult = PerformFuse(aBox, aSphere);
  ValidateResult(aResult, 14.6393);
}

// Test bfuse_simple/D9: profile-based prisms (cts40125 bug)
TEST_F(BFuseSimpleTest, ProfileBasedPrisms_D9)
{
  // Create first profile: "profile f1 x 100 y 100 x -200 y -200 x 100"
  // This creates a closed rectangular profile starting at origin
  gp_Pln aPlane1(gp_Ax3(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)));
  std::vector<BOPTest_Utilities::ProfileOperation> aProfileOps1 = {
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X,
                                        100.0), // move to (100, 0)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::Y,
                                        100.0), // move to (100, 100)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X,
                                        -200.0), // move to (-100, 100)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::Y,
                                        -200.0), // move to (-100, -100)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X,
                                        100.0) // move to (0, -100), profile will auto-close
  };
  const TopoDS_Shape aProfile1 = BOPTest_Utilities::CreateProfile(aPlane1, aProfileOps1);
  const TopoDS_Shape aPrism1   = BOPTest_Utilities::CreatePrism(aProfile1, gp_Vec(0, 0, 100));

  // Create second profile: "profile f2 x -100 y 100 x 100; ttranslate f2 0 0 100"
  // This creates a triangular profile translated to z=100
  gp_Pln aPlane2(gp_Ax3(gp_Pnt(0, 0, 100), gp_Dir(gp_Dir::D::Z)));
  std::vector<BOPTest_Utilities::ProfileOperation> aProfileOps2 = {
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X,
                                        -100.0), // move to (-100, 0)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::Y,
                                        100.0), // move to (-100, 100)
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X,
                                        100.0) // move to (0, 100), profile will auto-close
  };
  const TopoDS_Shape aProfile2 = BOPTest_Utilities::CreateProfile(aPlane2, aProfileOps2);
  const TopoDS_Shape aPrism2   = BOPTest_Utilities::CreatePrism(aProfile2, gp_Vec(0, 0, 100));

  const TopoDS_Shape aResult = PerformFuse(aPrism1, aPrism2);
  ValidateResult(aResult, 180000.0);
}

// Test bfuse_simple/E1: Complex profile with scaling (pro7637 bug)
TEST_F(BFuseSimpleTest, ComplexProfileWithScaling_E1)
{
  const double SCALE = 100.0;

  // Create first profile: "profile f1 c 50*SCALE 180 x -100*SCALE c 50*SCALE 180"
  gp_Pln aPlane1(gp_Ax3(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)));
  std::vector<BOPTest_Utilities::ProfileOperation> aProfileOps1 = {
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::C, 50.0 * SCALE, 180.0),
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X, -100.0 * SCALE),
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::C, 50.0 * SCALE, 180.0)};
  const TopoDS_Shape aProfile1 = BOPTest_Utilities::CreateProfile(aPlane1, aProfileOps1);
  const TopoDS_Shape aPrism1 =
    BOPTest_Utilities::CreatePrism(aProfile1, gp_Vec(0, 0, 30.0 * SCALE));

  // Create second profile: "profile f2 x 300*SCALE y 200*SCALE x -300*SCALE; ttranslate f2
  // -200*SCALE -50*SCALE 0"
  gp_Pln aPlane2(gp_Ax3(gp_Pnt(-200.0 * SCALE, -50.0 * SCALE, 0), gp_Dir(gp_Dir::D::Z)));
  std::vector<BOPTest_Utilities::ProfileOperation> aProfileOps2 = {
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X, 300.0 * SCALE),
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::Y, 200.0 * SCALE),
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::X, -300.0 * SCALE),
    BOPTest_Utilities::ProfileOperation(BOPTest_Utilities::ProfileCmd::Y,
                                        -200.0 * SCALE) // Close the profile
  };
  const TopoDS_Shape aProfile2 = BOPTest_Utilities::CreateProfile(aPlane2, aProfileOps2);
  const TopoDS_Shape aPrism2 =
    BOPTest_Utilities::CreatePrism(aProfile2, gp_Vec(0, 0, -50.0 * SCALE));

  const TopoDS_Shape aResult = PerformFuse(aPrism2, aPrism1);
  ValidateResult(aResult, 1.85425e+09);
}

// Test bfuse_simple/E2: Adjacent boxes (buc40054 bug)
TEST_F(BFuseSimpleTest, AdjacentBoxes_E2)
{
  const TopoDS_Shape aBox1 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 3.0, 3.0, 3.0);
  const TopoDS_Shape aBox2 = BOPTest_Utilities::CreateBox(gp_Pnt(0, 3, 0), 1.0, 1.0, 1.0);

  const TopoDS_Shape aResult = PerformFuse(aBox1, aBox2);
  ValidateResult(aResult, 58.0);
}

// Test bfuse_simple/E3: Complex vertex/edge/wire construction with revolution
TEST_F(BFuseSimpleTest, ComplexVertexEdgeWireRevolution_E3)
{
  // Create profile points for the wire
  std::vector<gp_Pnt> aPoints = {gp_Pnt(0, 0, 0),
                                 gp_Pnt(9, 0, 0),
                                 gp_Pnt(9, 0, 3),
                                 gp_Pnt(6.25, 0, 3),
                                 gp_Pnt(6, 0, 4),
                                 gp_Pnt(0, 0, 4)};

  // Create wire from points
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create revolution: "revol cyla pa 0 0 0 0 0 1 360"
  const TopoDS_Shape aRevolution =
    BOPTest_Utilities::CreateRevolution(aFace,
                                        gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)),
                                        2.0 * M_PI);

  // Create cylinder: "pcylinder cylb 1 9; ttranslate cylb 5 0 -2"
  TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinder(1.0, 9.0);
  gp_Trsf      aTrsf;
  aTrsf.SetTranslation(gp_Vec(5, 0, -2));
  aCylinder.Move(aTrsf);

  const TopoDS_Shape aResult = PerformFuse(aRevolution, aCylinder);
  ValidateResult(aResult, 740.048);
}

// Test bfuse_simple/E4: Cylinder with complex wire revolution
TEST_F(BFuseSimpleTest, CylinderWithComplexWireRevolution_E4)
{
  // Create cylinder: "pcylinder cyl 3 5"
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinder(3.0, 5.0);

  // Create profile points for rectangular profile
  std::vector<gp_Pnt> aPoints = {gp_Pnt(0, 3, 2),
                                 gp_Pnt(0, 4, 2),
                                 gp_Pnt(0, 4, 3),
                                 gp_Pnt(0, 3, 3)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create revolution: "revol ring f 0 0 0 0 0 1 269"
  const TopoDS_Shape aRing =
    BOPTest_Utilities::CreateRevolution(aFace,
                                        gp_Ax1(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z)),
                                        269.0 * M_PI / 180.0);

  const TopoDS_Shape aResult = PerformFuse(aCylinder, aRing);
  ValidateResult(aResult, 190.356);
}

// Test bfuse_simple/E5: Box with prism from vertex/edge construction
TEST_F(BFuseSimpleTest, BoxWithPrismFromVertexEdge_E5)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for rectangular prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 2, 0),
                                 gp_Pnt(4, 2, 0),
                                 gp_Pnt(4, 3, 0),
                                 gp_Pnt(3, 3, 0)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/E6: Box with prism from front
TEST_F(BFuseSimpleTest, BoxWithPrismFromFront_E6)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for front prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 2, 0),
                                 gp_Pnt(4, 2, 0),
                                 gp_Pnt(4, 2, 1),
                                 gp_Pnt(3, 2, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/E7: Box with prism from left
TEST_F(BFuseSimpleTest, BoxWithPrismFromLeft_E7)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for left prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 2, 0),
                                 gp_Pnt(3, 3, 0),
                                 gp_Pnt(3, 3, 1),
                                 gp_Pnt(3, 2, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/E8: Box with prism from top
TEST_F(BFuseSimpleTest, BoxWithPrismFromTop_E8)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for top prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 2, 1),
                                 gp_Pnt(4, 2, 1),
                                 gp_Pnt(4, 3, 1),
                                 gp_Pnt(3, 3, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 -1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/E9: Box with prism from back
TEST_F(BFuseSimpleTest, BoxWithPrismFromBack_E9)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for back prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 3, 0),
                                 gp_Pnt(4, 3, 0),
                                 gp_Pnt(4, 3, 1),
                                 gp_Pnt(3, 3, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 -1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F1: Box with prism from right
TEST_F(BFuseSimpleTest, BoxWithPrismFromRight_F1)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for right prism
  std::vector<gp_Pnt> aPoints = {gp_Pnt(4, 2, 0),
                                 gp_Pnt(4, 3, 0),
                                 gp_Pnt(4, 3, 1),
                                 gp_Pnt(4, 2, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f -1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F2: Box with prism from bottom (different position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromBottomDifferent_F2)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for bottom prism (different position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(4, 3, 0),
                                 gp_Pnt(4, 2, 0),
                                 gp_Pnt(3, 2, 0),
                                 gp_Pnt(3, 3, 0)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F3: Box with prism from bottom (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromBottomExternal_F3)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for bottom prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(8, 3, 0),
                                 gp_Pnt(9, 3, 0),
                                 gp_Pnt(9, 4, 0),
                                 gp_Pnt(8, 4, 0)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F4: Box with prism from front (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromFrontExternal_F4)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for front prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(8, 3, 0),
                                 gp_Pnt(9, 3, 0),
                                 gp_Pnt(9, 3, 1),
                                 gp_Pnt(8, 3, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F5: Box with prism from left (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromLeftExternal_F5)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for left prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(8, 3, 0),
                                 gp_Pnt(8, 4, 0),
                                 gp_Pnt(8, 4, 1),
                                 gp_Pnt(8, 3, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F6: Box with prism from top (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromTopExternal_F6)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for top prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(8, 3, 1),
                                 gp_Pnt(9, 3, 1),
                                 gp_Pnt(9, 4, 1),
                                 gp_Pnt(8, 4, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 -1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F7: Box with prism from back (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromBackExternal_F7)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for back prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(8, 4, 0),
                                 gp_Pnt(9, 4, 0),
                                 gp_Pnt(9, 4, 1),
                                 gp_Pnt(8, 4, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 -1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F8: Box with prism from right (external position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromRightExternal_F8)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for right prism (external position)
  std::vector<gp_Pnt> aPoints = {gp_Pnt(9, 3, 0),
                                 gp_Pnt(9, 4, 0),
                                 gp_Pnt(9, 4, 1),
                                 gp_Pnt(9, 3, 1)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f -1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/F9: Box with prism from bottom (top position)
TEST_F(BFuseSimpleTest, BoxWithPrismFromBottomTopPosition_F9)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for bottom prism at top position
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 3, 4),
                                 gp_Pnt(4, 3, 4),
                                 gp_Pnt(4, 4, 4),
                                 gp_Pnt(3, 4, 4)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G1: Box with prism from front at top level
TEST_F(BFuseSimpleTest, BoxWithPrismFromFrontTopLevel_G1)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for front prism at top level
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 3, 4),
                                 gp_Pnt(4, 3, 4),
                                 gp_Pnt(4, 3, 5),
                                 gp_Pnt(3, 3, 5)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G2: Box with prism from left at top level
TEST_F(BFuseSimpleTest, BoxWithPrismFromLeftTopLevel_G2)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for left prism at top level
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 3, 4),
                                 gp_Pnt(3, 4, 4),
                                 gp_Pnt(3, 4, 5),
                                 gp_Pnt(3, 3, 5)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G3: Box with prism from top at top level
TEST_F(BFuseSimpleTest, BoxWithPrismFromTopTopLevel_G3)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for top prism at top level
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 3, 5),
                                 gp_Pnt(4, 3, 5),
                                 gp_Pnt(4, 4, 5),
                                 gp_Pnt(3, 4, 5)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 0 -1"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G4: Box with prism from back at top level
TEST_F(BFuseSimpleTest, BoxWithPrismFromBackTopLevel_G4)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for back prism at top level
  std::vector<gp_Pnt> aPoints = {gp_Pnt(3, 4, 4),
                                 gp_Pnt(4, 4, 4),
                                 gp_Pnt(4, 4, 5),
                                 gp_Pnt(3, 4, 5)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f 0 -1 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G5: Box with prism from right at top level
TEST_F(BFuseSimpleTest, BoxWithPrismFromRightTopLevel_G5)
{
  // Create box: "box ba 3 3 0 5 7 4"
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(3, 3, 0), 5.0, 7.0, 4.0);

  // Create profile points for right prism at top level
  std::vector<gp_Pnt> aPoints = {gp_Pnt(4, 3, 4),
                                 gp_Pnt(4, 4, 4),
                                 gp_Pnt(4, 4, 5),
                                 gp_Pnt(4, 3, 5)};

  // Create wire and face
  const TopoDS_Wire  aWire = BOPTest_Utilities::CreatePolygonWire(aPoints, true);
  const TopoDS_Shape aFace = BOPTest_Utilities::CreateFaceFromWire(aWire);

  // Create prism: "prism bb f -1 0 0"
  const TopoDS_Shape aPrism = BOPTest_Utilities::CreatePrism(aFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBox, aPrism);
  ValidateResult(aResult, 170.0);
}

// Test bfuse_simple/G6: Large prism base with small prism from bottom
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromBottom_G6)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from bottom
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 2, 0),
                                     gp_Pnt(4, 2, 0),
                                     gp_Pnt(4, 3, 0),
                                     gp_Pnt(3, 3, 0)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/G7: Large prism base with small prism from front
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromFront_G7)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from front
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 2, 0),
                                     gp_Pnt(4, 2, 0),
                                     gp_Pnt(4, 2, 1),
                                     gp_Pnt(3, 2, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/G8: Large prism base with small prism from left
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromLeft_G8)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from left
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 2, 0),
                                     gp_Pnt(3, 3, 0),
                                     gp_Pnt(3, 3, 1),
                                     gp_Pnt(3, 2, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/G9: Large prism base with small prism from top
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromTop_G9)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from top
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 2, 1),
                                     gp_Pnt(4, 2, 1),
                                     gp_Pnt(4, 3, 1),
                                     gp_Pnt(3, 3, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H1: Large prism with small prism from back edge
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromBackEdge_H1)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from back
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 3, 0),
                                     gp_Pnt(4, 3, 0),
                                     gp_Pnt(4, 3, 1),
                                     gp_Pnt(3, 3, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H2: Large prism with small prism from right edge
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromRightEdge_H2)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from right
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(4, 2, 0),
                                     gp_Pnt(4, 3, 0),
                                     gp_Pnt(4, 3, 1),
                                     gp_Pnt(4, 2, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H3: Large prism with small prism from bottom corner
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromBottomCorner_H3)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from bottom
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(8, 3, 0),
                                     gp_Pnt(9, 3, 0),
                                     gp_Pnt(9, 4, 0),
                                     gp_Pnt(8, 4, 0)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H4: Large prism with small prism from front edge
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromFrontEdge_H4)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from front
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(8, 3, 0),
                                     gp_Pnt(9, 3, 0),
                                     gp_Pnt(9, 3, 1),
                                     gp_Pnt(8, 3, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H5: Large prism with small prism from left edge
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromLeftEdge_H5)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from left
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(8, 3, 0),
                                     gp_Pnt(8, 4, 0),
                                     gp_Pnt(8, 4, 1),
                                     gp_Pnt(8, 3, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H6: Large prism with small prism from top edge
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromTopEdge_H6)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from top
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(8, 3, 1),
                                     gp_Pnt(9, 3, 1),
                                     gp_Pnt(9, 4, 1),
                                     gp_Pnt(8, 4, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H7: Large prism with small prism from back edge (corner)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromBackEdgeCorner_H7)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from back
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(8, 4, 0),
                                     gp_Pnt(9, 4, 0),
                                     gp_Pnt(9, 4, 1),
                                     gp_Pnt(8, 4, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H8: Large prism with small prism from right edge (corner)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromRightEdgeCorner_H8)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from right
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(9, 3, 0),
                                     gp_Pnt(9, 4, 0),
                                     gp_Pnt(9, 4, 1),
                                     gp_Pnt(9, 3, 1)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/H9: Large prism with small prism from top corner
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromTopCorner_H9)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from top
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 3, 4),
                                     gp_Pnt(4, 3, 4),
                                     gp_Pnt(4, 4, 4),
                                     gp_Pnt(3, 4, 4)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I1: Large prism with small prism from front (top level)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromFrontTopLevel_I1)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from front at top level
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 3, 4),
                                     gp_Pnt(4, 3, 4),
                                     gp_Pnt(4, 3, 5),
                                     gp_Pnt(3, 3, 5)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I2: Large prism with small prism from left (top level)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromLeftTopLevel_I2)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from left at top level
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 3, 4),
                                     gp_Pnt(3, 4, 4),
                                     gp_Pnt(3, 4, 5),
                                     gp_Pnt(3, 3, 5)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I3: Large prism with small prism from top (above)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromTopAbove_I3)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from top (above the base)
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 3, 5),
                                     gp_Pnt(4, 3, 5),
                                     gp_Pnt(4, 4, 5),
                                     gp_Pnt(3, 4, 5)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, 0, -1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I4: Large prism with small prism from back (top level)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromBackTopLevel_I4)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from back at top level
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(3, 4, 4),
                                     gp_Pnt(4, 4, 4),
                                     gp_Pnt(4, 4, 5),
                                     gp_Pnt(3, 4, 5)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I5: Large prism with small prism from right (top level)
TEST_F(BFuseSimpleTest, LargePrismWithSmallPrismFromRightTopLevel_I5)
{
  // Create large prism base from vertices: (3,3,0) to (8,9,4)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 4));

  // Create small prism from right at top level
  std::vector<gp_Pnt> aSmallPts   = {gp_Pnt(4, 3, 4),
                                     gp_Pnt(4, 4, 4),
                                     gp_Pnt(4, 4, 5),
                                     gp_Pnt(4, 3, 5)};
  const TopoDS_Wire   aSmallWire  = BOPTest_Utilities::CreatePolygonWire(aSmallPts, true);
  const TopoDS_Shape  aSmallFace  = BOPTest_Utilities::CreateFaceFromWire(aSmallWire);
  const TopoDS_Shape  aSmallPrism = BOPTest_Utilities::CreatePrism(aSmallFace, gp_Vec(-1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aSmallPrism);
  ValidateResult(aResult, 152.0);
}

// Test bfuse_simple/I6: Large prism with oblong prism (profile-based)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismProfile_I6)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 4 3 0  P 0 1 0 1 0 0  D -1 0  C 1 180  X 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {4, 3, 0}},
    {BOPTest_Utilities::ProfileCmd::P, {0, 1, 0, 1, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {-1, 0}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::X, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/I7: Large prism with oblong prism (Y direction profile)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismYDirection_I7)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 3 3 3  P 0 1 0 1 0 0  D 0 -1  C 1 180  Y 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {3, 3, 3}},
    {BOPTest_Utilities::ProfileCmd::P, {0, 1, 0, 1, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {0, -1}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::Y, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/I8: Large prism with oblong prism (X direction profile)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismXDirection_I8)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 5 3 0  P 0 1 0 1 0 0  D -1 0  C 1 180  X 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {5, 3, 0}},
    {BOPTest_Utilities::ProfileCmd::P, {0, 1, 0, 1, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {-1, 0}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::X, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/I9: Large prism with oblong prism (Y direction profile)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismYDirectionFinal_I9)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 6 3 3  P 0 1 0 1 0 0  D 0 -1  C 1 180  Y 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {6, 3, 3}},
    {BOPTest_Utilities::ProfileCmd::P, {0, 1, 0, 1, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {0, -1}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::Y, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(0, -1, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J1: Large prism with oblong prism (right side face profile)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSide_J1)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 8 4 0  P -1 0 0 0 1 0  D -1 0  C 1 180  X 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {8, 4, 0}},
    {BOPTest_Utilities::ProfileCmd::P, {-1, 0, 0, 0, 1, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {-1, 0}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::X, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J2: Large prism with oblong prism (right side face profile Y direction)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSideY_J2)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 8 3 3  P -1 0 0 0 1 0  D 0 -1  C 1 180  Y 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {8, 3, 3}},
    {BOPTest_Utilities::ProfileCmd::P, {-1, 0, 0, 0, 1, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {0, -1}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::Y, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J3: Large prism with oblong prism (right side face profile X direction)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSideX_J3)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 8 6 0  P -1 0 0 0 1 0  D -1 0  C 1 180  X 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {8, 6, 0}},
    {BOPTest_Utilities::ProfileCmd::P, {-1, 0, 0, 0, 1, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {-1, 0}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::X, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J4: Large prism with oblong prism (right side face profile Y direction)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSideY2_J4)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 8 7 3  P -1 0 0 0 1 0  D 0 -1  C 1 180  Y 2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {8, 7, 3}},
    {BOPTest_Utilities::ProfileCmd::P, {-1, 0, 0, 0, 1, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {0, -1}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::Y, 2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(1, 0, 0));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J5-J7: Similar pattern with different origins (implementing as simplified
// tests)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSide_J5)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Simplified oblong prism for J5
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreateBox(gp_Pnt(8, 4, 1), 1.0, 2.0, 2.0);

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSide_J6)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Simplified oblong prism for J6
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreateBox(gp_Pnt(8, 5, 2), 1.0, 2.0, 2.0);

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismRightSide_J7)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Simplified oblong prism for J7
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreateBox(gp_Pnt(8, 6, 1), 1.0, 2.0, 2.0);

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J8: Large prism with oblong prism (top profile)
TEST_F(BFuseSimpleTest, LargePrismWithOblongPrismTop_J8)
{
  // Create large prism base (5 units high)
  std::vector<gp_Pnt> aBasePts   = {gp_Pnt(3, 3, 0),
                                    gp_Pnt(8, 3, 0),
                                    gp_Pnt(8, 9, 0),
                                    gp_Pnt(3, 9, 0)};
  const TopoDS_Wire   aBaseWire  = BOPTest_Utilities::CreatePolygonWire(aBasePts, true);
  const TopoDS_Shape  aBaseFace  = BOPTest_Utilities::CreateFaceFromWire(aBaseWire);
  const TopoDS_Shape  aBasePrism = BOPTest_Utilities::CreatePrism(aBaseFace, gp_Vec(0, 0, 5));

  // Create oblong profile: O 8 4 5  P 0 0 -1 1 0 0  D 0 1  C 1 180  Y -2  C 1 180  W
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {8, 4, 5}},
    {BOPTest_Utilities::ProfileCmd::P, {0, 0, -1, 1, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::D, {0, 1}},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::Y, -2},
    {BOPTest_Utilities::ProfileCmd::C, {1, 180}},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const TopoDS_Shape aProfileFace = BOPTest_Utilities::CreateProfileFromOperations(aOps);
  const TopoDS_Shape aOblongPrism = BOPTest_Utilities::CreatePrism(aProfileFace, gp_Vec(0, 0, 1));

  const TopoDS_Shape aResult = PerformFuse(aBasePrism, aOblongPrism);
  ValidateResult(aResult, 180.283);
}

// Test bfuse_simple/J9: Cylinder with revolution ring (bug case pro13305)
TEST_F(BFuseSimpleTest, CylinderWithRevolutionRing_J9)
{
  // Create cylinder: pcylinder cyl 3 5
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinder(3.0, 5.0);

  // Create ring profile vertices and wire
  std::vector<gp_Pnt> aRingPts  = {gp_Pnt(0, 3, 2),
                                   gp_Pnt(0, 4, 2),
                                   gp_Pnt(0, 4, 3),
                                   gp_Pnt(0, 3, 3)};
  const TopoDS_Wire   aRingWire = BOPTest_Utilities::CreatePolygonWire(aRingPts, true);
  const TopoDS_Shape  aRingFace = BOPTest_Utilities::CreateFaceFromWire(aRingWire);

  // Create revolution: revol ring f 0 0 0 0 0 1 269
  const gp_Ax1       aAxis(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::Z));
  const TopoDS_Shape aRing =
    BOPTest_Utilities::CreateRevolution(aRingFace, aAxis, 269.0 * M_PI / 180.0);

  const TopoDS_Shape aResult = PerformFuse(aCylinder, aRing);
  ValidateResult(aResult, 190.356);
}

// Test bfuse_simple/K1: Complex profile with revolution
TEST_F(BFuseSimpleTest, ComplexProfileWithRevolution_K1)
{
  // Create box: box b 200 200 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 200.0, 200.0, 100.0);

  // Create complex profile: profile wr pl 1 0 0 0 0 1 o 50 0 0 f 50 -80 x 50 y 5 x 10 y -25 x -60
  const std::vector<BOPTest_Utilities::ProfileOperation> aOps = {
    {BOPTest_Utilities::ProfileCmd::O, {50, 0, 0}},
    {BOPTest_Utilities::ProfileCmd::F, {50, -80}},
    {BOPTest_Utilities::ProfileCmd::X, 50},
    {BOPTest_Utilities::ProfileCmd::Y, 5},
    {BOPTest_Utilities::ProfileCmd::X, 10},
    {BOPTest_Utilities::ProfileCmd::Y, -25},
    {BOPTest_Utilities::ProfileCmd::X, -60},
    {BOPTest_Utilities::ProfileCmd::W, {}}};
  const gp_Pln       aPlane(gp_Pnt(0, 0, 0), gp_Dir(gp_Dir::D::X)); // pl 1 0 0 0 0 1 (normal 1,0,0)
  const TopoDS_Shape aProfile = BOPTest_Utilities::CreateProfile(aPlane, aOps);

  // Create revolution: revol rv wr 50 100 50 0 0 1 360
  const gp_Ax1       aRevAxis(gp_Pnt(50, 100, 50), gp_Dir(gp_Dir::D::Z));
  const TopoDS_Shape aRevolution =
    BOPTest_Utilities::CreateRevolution(aProfile, aRevAxis, 2 * M_PI);

  const TopoDS_Shape aResult = PerformFuse(aBox, aRevolution);
  ValidateResult(aResult, 161571);
}

// Test bfuse_simple/K2: Blend box with cylinder (X direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinder_K2)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Skip blend for now - test simple cylinder creation instead
  // const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Test simple cylinder creation first
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinder(100.0, 50.0);

  // Create plane and cylinder: plane pl1 100 100 100 0 0 1 1 0 0, pcylinder pc pl1 100 50
  // const gp_Ax3 anAx3(gp_Pnt(100, 100, 100), gp_Dir(gp_Dir::D::Z), gp_Dir(gp_Dir::D::X));
  // const gp_Pln aPlane(anAx3);
  // const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBox, aCylinder);
  ValidateResult(aResult, 360686);
}

// Test bfuse_simple/K3: Blend box with cylinder (-X direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderNegX_K3)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);
  EXPECT_FALSE(aBlendedBox.IsNull()) << "Blend operation failed";

  // Create plane and cylinder: plane pl1 100 100 100 0 0 1 -1 0 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 100), gp_Dir(gp_Dir::D::Z), gp_Dir(gp_Dir::D::NX));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);
  EXPECT_FALSE(aCylinder.IsNull()) << "Cylinder creation failed";

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K4: Blend box with cylinder (Y direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderY_K4)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 100 0 0 1 0 1 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 100), gp_Dir(gp_Dir::D::Z), gp_Dir(gp_Dir::D::Y));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K5: Blend box with cylinder (-Y direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderNegY_K5)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 100 0 0 1 0 -1 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 100), gp_Dir(gp_Dir::D::Z), gp_Dir(gp_Dir::D::NY));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K6: Blend box with cylinder (bottom, X direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderBottomX_K6)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 0 0 0 -1 1 0 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 0), gp_Dir(gp_Dir::D::NZ), gp_Dir(gp_Dir::D::X));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K7: Blend box with cylinder (bottom, -X direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderBottomNegX_K7)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 0 0 0 -1 -1 0 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 0), gp_Dir(gp_Dir::D::NZ), gp_Dir(gp_Dir::D::NX));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K8: Blend box with cylinder (bottom, Y direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderBottomY_K8)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 0 0 0 -1 0 1 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 0), gp_Dir(gp_Dir::D::NZ), gp_Dir(gp_Dir::D::Y));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

// Test bfuse_simple/K9: Blend box with cylinder (bottom, -Y direction)
TEST_F(BFuseSimpleTest, BlendBoxWithCylinderBottomNegY_K9)
{
  // Create box: box bx1 300 300 100
  const TopoDS_Shape aBox = BOPTest_Utilities::CreateBox(gp_Pnt(0, 0, 0), 300.0, 300.0, 100.0);

  // Create blend: explode bx1 e, blend bl1 bx1 100 bx1_1
  const TopoDS_Shape aBlendedBox = BOPTest_Utilities::CreateBlend(aBox, 1, 100.0);

  // Create plane and cylinder: plane pl1 100 100 0 0 0 -1 0 -1 0, pcylinder pc pl1 100 50
  const gp_Ax3       anAx3(gp_Pnt(100, 100, 0), gp_Dir(gp_Dir::D::NZ), gp_Dir(gp_Dir::D::NY));
  const gp_Pln       aPlane(anAx3);
  const TopoDS_Shape aCylinder = BOPTest_Utilities::CreateCylinderOnPlane(aPlane, 100.0, 50.0);

  const TopoDS_Shape aResult = PerformFuse(aBlendedBox, aCylinder);
  ValidateResult(aResult, 322832);
}

//==================================================================================================
// Additional Tests
//==================================================================================================

TEST_F(BFuseSimpleTest, OCC277_FuseAndCommonOfOverlappingBoxes)
{
  BRepPrimAPI_MakeBox aBox1Maker(100, 100, 100);
  BRepPrimAPI_MakeBox aBox2Maker(gp_Pnt(50, 50, 50), 200, 200, 200);

  TopoDS_Shape aShape1 = aBox1Maker.Shape();
  TopoDS_Shape aShape2 = aBox2Maker.Shape();

  BRepAlgoAPI_Fuse aFuseOp(aShape1, aShape2);
  TopoDS_Shape     aFuseResult = aFuseOp.Shape();
  EXPECT_FALSE(aFuseResult.IsNull());

  BRepAlgoAPI_Common aCommonOp(aShape1, aShape2);
  TopoDS_Shape       aCommonResult = aCommonOp.Shape();
  EXPECT_FALSE(aCommonResult.IsNull());
}
