// Created on: 1997-10-08
// Created by: Olga KOULECHOVA
// Copyright (c) 1997-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BRepFeat_RibSlot_HeaderFile
#define _BRepFeat_RibSlot_HeaderFile

#include <Standard.hxx>
#include <Standard_DefineAlloc.hxx>
#include <Standard_Handle.hxx>

#include <gp_Pnt.hxx>
#include <TopoDS_Shape.hxx>
#include <NCollection_List.hxx>
#include <TopTools_ShapeMapHasher.hxx>
#include <NCollection_DataMap.hxx>
#include <BRepFeat_PerfSelection.hxx>
#include <TopoDS_Wire.hxx>
#include <TopoDS_Face.hxx>
#include <BRepFeat_StatusError.hxx>
#include <BRepBuilderAPI_MakeShape.hxx>
#include <Standard_Integer.hxx>
class TopoDS_Edge;
class Geom_Plane;
class gp_Dir;
class Geom_Curve;
class TopoDS_Vertex;
class LocOpe_Gluer;
class BRepAlgoAPI_BooleanOperation;

//! Provides functions to build mechanical features.
//! Mechanical features include ribs - protrusions and grooves (or slots) - depressions along
//! planar (linear) surfaces or revolution surfaces. The semantics of mechanical features is built
//! around giving thickness to a contour. This thickness can either be unilateral - on one side
//! of the contour - or bilateral - on both sides.
//! As in the semantics of form features, the thickness is defined by construction of shapes
//! in specific contexts. The development contexts differ, however,in case of mechanical features.
//! Here they include extrusion:
//! -   to a limiting face of the basis shape
//! -   to or from a limiting plane
//! -   to a height.
class BRepFeat_RibSlot : public BRepBuilderAPI_MakeShape
{
public:
  DEFINE_STANDARD_ALLOC

  //! Returns true if F a TopoDS_Shape of type edge or face has been deleted.
  Standard_EXPORT bool IsDeleted(const TopoDS_Shape& F) override;

  //! Returns the list of generated Faces F. This list may be empty.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& Modified(const TopoDS_Shape& F) override;

  //! Returns a list NCollection_List<TopoDS_Shape> of the faces S created in the shape.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& Generated(const TopoDS_Shape& S) override;

  //! Returns the list of shapes created at the bottom of
  //! the created form. It may be an empty list.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& FirstShape() const;

  //! Returns the list of shapes created at the top of the
  //! created form. It may be an empty list.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& LastShape() const;

  //! Returns a list of the limiting and glueing faces
  //! generated by the feature. These faces did not originally exist in the basis shape.
  //! The list provides the information necessary for
  //! subsequent addition of a draft to a face. It may be an empty list.
  //! If a face has tangent edges, no draft is possible, and the tangent edges must
  //! subsequently be removed if you want to add a draft to the face.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& FacesForDraft() const;

  //! Returns a list of the limiting and glueing edges
  //! generated by the feature. These edges did not originally exist in the basis shape.
  //! The list provides the information necessary for
  //! subsequent addition of fillets. It may be an empty list.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& NewEdges() const;

  //! Returns a list of the tangent edges among the
  //! limiting and glueing edges generated by the
  //! feature. These edges did not originally exist in
  //! the basis shape and are tangent to the face
  //! against which the feature is built.
  //! The list provides the information necessary for
  //! subsequent addition of fillets. It may be an empty list.
  //! If an edge is tangent, no fillet is possible, and
  //! the edge must subsequently be removed if you want to add a fillet.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& TgtEdges() const;

  Standard_EXPORT static double IntPar(const occ::handle<Geom_Curve>& C, const gp_Pnt& P);

  Standard_EXPORT static TopoDS_Face ChoiceOfFaces(NCollection_List<TopoDS_Shape>& faces,
                                                   const occ::handle<Geom_Curve>&  cc,
                                                   const double                    par,
                                                   const double                    bnd,
                                                   const occ::handle<Geom_Plane>&  Pln);

  Standard_EXPORT BRepFeat_StatusError CurrentStatusError() const;

protected:
  //! Redefines the empty constructor.
  BRepFeat_RibSlot();

  //! General perform method...
  Standard_EXPORT void LFPerform();

  Standard_EXPORT gp_Pnt CheckPoint(const TopoDS_Edge&             e,
                                    const double                   bnd,
                                    const occ::handle<Geom_Plane>& Pln);

  Standard_EXPORT gp_Dir Normal(const TopoDS_Face& F, const gp_Pnt& P);

  Standard_EXPORT void EdgeExtention(TopoDS_Edge& e, const double bnd, const bool FirstLast);

  Standard_EXPORT double HeightMax(const TopoDS_Shape& theSbase,
                                   const TopoDS_Shape& theSUntil,
                                   gp_Pnt&             p1,
                                   gp_Pnt&             p2);

  Standard_EXPORT bool ExtremeFaces(const bool                     RevolRib,
                                    const double                   bnd,
                                    const occ::handle<Geom_Plane>& Pln,
                                    TopoDS_Edge&                   FirstEdge,
                                    TopoDS_Edge&                   LastEdge,
                                    TopoDS_Face&                   FirstFace,
                                    TopoDS_Face&                   LastFace,
                                    TopoDS_Vertex&                 FirstVertex,
                                    TopoDS_Vertex&                 LastVertex,
                                    bool&                          OnFirstFace,
                                    bool&                          OnLastFace,
                                    bool&                          PtOnFirstEdge,
                                    bool&                          PtOnLastEdge,
                                    TopoDS_Edge&                   OnFirstEdge,
                                    TopoDS_Edge&                   OnLastEdge);

  Standard_EXPORT void PtOnEdgeVertex(const bool           RevolRib,
                                      const TopoDS_Shape&  shape,
                                      const gp_Pnt&        point,
                                      const TopoDS_Vertex& FirstVertex,
                                      const TopoDS_Vertex& LastVertex,
                                      bool&                PtOnEdge,
                                      TopoDS_Edge&         OnEdge,
                                      bool&                PtOnVertex,
                                      TopoDS_Vertex&       OnVertex);

  Standard_EXPORT bool SlidingProfile(TopoDS_Face&                   Prof,
                                      const bool                     RevolRib,
                                      const double                   myTol,
                                      int&                           Concavite,
                                      const occ::handle<Geom_Plane>& myPln,
                                      const TopoDS_Face&             BndFace,
                                      const gp_Pnt&                  CheckPnt,
                                      const TopoDS_Face&             FirstFace,
                                      const TopoDS_Face&             LastFace,
                                      const TopoDS_Vertex&           FirstVertex,
                                      const TopoDS_Vertex&           LastVertex,
                                      const TopoDS_Edge&             FirstEdge,
                                      const TopoDS_Edge&             LastEdge);

  Standard_EXPORT bool NoSlidingProfile(TopoDS_Face&                   Prof,
                                        const bool                     RevolRib,
                                        const double                   myTol,
                                        int&                           Concavite,
                                        const occ::handle<Geom_Plane>& myPln,
                                        const double                   bnd,
                                        const TopoDS_Face&             BndFace,
                                        const gp_Pnt&                  CheckPnt,
                                        const TopoDS_Face&             FirstFace,
                                        const TopoDS_Face&             LastFace,
                                        const TopoDS_Vertex&           FirstVertex,
                                        const TopoDS_Vertex&           LastVertex,
                                        const TopoDS_Edge&             FirstEdge,
                                        const TopoDS_Edge&             LastEdge,
                                        const bool                     OnFirstFace,
                                        const bool                     OnLastFace);

  //! Updates the data structures of descendant
  //! shapes during the glueing operation.Returns the modified, generated
  //! and deleted faces during the course of the glueing operation.
  Standard_EXPORT void UpdateDescendants(const LocOpe_Gluer& G);

  Standard_EXPORT void UpdateDescendants(const BRepAlgoAPI_BooleanOperation& aBOP,
                                         const TopoDS_Shape&                 SResult,
                                         const bool                          SkipFace = false);

  gp_Pnt myFirstPnt;
  gp_Pnt myLastPnt;
  bool   myFuse;
  bool   mySliding;
  NCollection_DataMap<TopoDS_Shape, NCollection_List<TopoDS_Shape>, TopTools_ShapeMapHasher> myMap;
  NCollection_DataMap<TopoDS_Shape, NCollection_List<TopoDS_Shape>, TopTools_ShapeMapHasher>
                                                                           myLFMap;
  TopoDS_Shape                                                             myFShape;
  TopoDS_Shape                                                             myLShape;
  BRepFeat_PerfSelection                                                   myPerfSelection;
  TopoDS_Wire                                                              myWire;
  TopoDS_Shape                                                             mySbase;
  TopoDS_Face                                                              mySkface;
  TopoDS_Face                                                              myPbase;
  TopoDS_Shape                                                             myGShape;
  TopoDS_Shape                                                             mySUntil;
  NCollection_DataMap<TopoDS_Shape, TopoDS_Shape, TopTools_ShapeMapHasher> myGluedF;
  NCollection_List<TopoDS_Shape>                                           myNewEdges;
  NCollection_List<TopoDS_Shape>                                           myTgtEdges;
  NCollection_List<TopoDS_Shape>                                           myFacesForDraft;
  BRepFeat_StatusError                                                     myStatusError;
};

#include <BRepFeat_RibSlot.lxx>

#endif // _BRepFeat_RibSlot_HeaderFile
