// Copyright (c) 2013 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BRepMesh_DiscretFactory_HeaderFile
#define _BRepMesh_DiscretFactory_HeaderFile

#include <Standard.hxx>
#include <Standard_DefineAlloc.hxx>
#include <Standard_Macro.hxx>
#include <BRepMesh_PluginEntryType.hxx>
#include <BRepMesh_FactoryError.hxx>
#include <TCollection_AsciiString.hxx>
#include <NCollection_Map.hxx>
#include <OSD_Function.hxx>
#include <NCollection_DataMap.hxx>
#include <BRepMesh_DiscretRoot.hxx>

class TopoDS_Shape;

//! This class intended to setup / retrieve default triangulation algorithm.
//! Use BRepMesh_DiscretFactory::Get() static method to retrieve global Factory instance.
//! Use BRepMesh_DiscretFactory::Discret() method to retrieve meshing tool.
class BRepMesh_DiscretFactory
{
public:
  DEFINE_STANDARD_ALLOC

  //! Returns the global factory instance.
  Standard_EXPORT static BRepMesh_DiscretFactory& Get();

  //! Returns the list of registered meshing algorithms.
  const NCollection_Map<TCollection_AsciiString>& Names() const { return myNames; }

  //! Setup meshing algorithm by name.
  //! Returns TRUE if requested tool is available.
  //! On fail Factory will continue to use previous algo.
  bool SetDefaultName(const TCollection_AsciiString& theName)
  {
    return SetDefault(theName, myFunctionName);
  }

  //! Returns name for current meshing algorithm.
  const TCollection_AsciiString& DefaultName() const { return myDefaultName; }

  //! Advanced function. Changes function name to retrieve from plugin.
  //! Returns TRUE if requested tool is available.
  //! On fail Factory will continue to use previous algo.
  bool SetFunctionName(const TCollection_AsciiString& theFuncName)
  {
    return SetDefault(myDefaultName, theFuncName);
  }

  //! Returns function name that should be exported by plugin.
  const TCollection_AsciiString& FunctionName() const { return myFunctionName; }

  //! Returns error status for last meshing algorithm switch.
  BRepMesh_FactoryError ErrorStatus() const { return myErrorStatus; }

  //! Setup meshing algorithm that should be created by this Factory.
  //! Returns TRUE if requested tool is available.
  //! On fail Factory will continue to use previous algo.
  //! Call ::ErrorStatus() method to retrieve fault reason.
  Standard_EXPORT bool SetDefault(const TCollection_AsciiString& theName,
                                  const TCollection_AsciiString& theFuncName = "DISCRETALGO");

  //! Returns triangulation algorithm instance.
  //! @param theShape shape to be meshed.
  //! @param theLinDeflection linear deflection to be used for meshing.
  //! @param theAngDeflection angular deflection to be used for meshing.
  Standard_EXPORT occ::handle<BRepMesh_DiscretRoot> Discret(const TopoDS_Shape& theShape,
                                                            const double        theLinDeflection,
                                                            const double        theAngDeflection);

protected:
  //! Constructor
  Standard_EXPORT BRepMesh_DiscretFactory();

  //! Destructor
  Standard_EXPORT virtual ~BRepMesh_DiscretFactory();

  //! Clears factory data.
  Standard_EXPORT void clear();

  BRepMesh_PluginEntryType                                   myPluginEntry;
  BRepMesh_FactoryError                                      myErrorStatus;
  NCollection_Map<TCollection_AsciiString>                   myNames;
  TCollection_AsciiString                                    myDefaultName;
  TCollection_AsciiString                                    myFunctionName;
  NCollection_DataMap<TCollection_AsciiString, OSD_Function> myFactoryMethods;
};

#endif
