// Created on: 1995-09-18
// Created by: Bruno DUMORTIER
// Copyright (c) 1995-1999 Matra Datavision
// Copyright (c) 1999-2014 OPEN CASCADE SAS
//
// This file is part of Open CASCADE Technology software library.
//
// This library is free software; you can redistribute it and/or modify it under
// the terms of the GNU Lesser General Public License version 2.1 as published
// by the Free Software Foundation, with special exception defined in the file
// OCCT_LGPL_EXCEPTION.txt. Consult the file LICENSE_LGPL_21.txt included in OCCT
// distribution for complete text of the license and disclaimer of any warranty.
//
// Alternatively, this file may be used under the terms of Open CASCADE
// commercial license or contractual agreement.

#ifndef _BRepOffsetAPI_MakeOffset_HeaderFile
#define _BRepOffsetAPI_MakeOffset_HeaderFile

#include <Standard.hxx>
#include <Standard_DefineAlloc.hxx>

#include <TopoDS_Face.hxx>
#include <TopoDS_Shape.hxx>
#include <NCollection_List.hxx>
#include <BRepFill_OffsetWire.hxx>
#include <BRepBuilderAPI_MakeShape.hxx>
class TopoDS_Wire;
class TopoDS_Shape;

//! Describes algorithms for offsetting wires from a set of
//! wires contained in a planar face.
//! A MakeOffset object provides a framework for:
//! - defining the construction of an offset,
//! - implementing the construction algorithm, and
//! - consulting the result.
class BRepOffsetAPI_MakeOffset : public BRepBuilderAPI_MakeShape
{
public:
  DEFINE_STANDARD_ALLOC

  //! Constructs an algorithm for creating an empty offset
  Standard_EXPORT BRepOffsetAPI_MakeOffset();

  //! Constructs an algorithm for creating an algorithm
  //! to build parallels to the spine Spine
  Standard_EXPORT BRepOffsetAPI_MakeOffset(const TopoDS_Face&     Spine,
                                           const GeomAbs_JoinType Join         = GeomAbs_Arc,
                                           const bool             IsOpenResult = false);

  //! Initializes the algorithm to construct parallels to the spine Spine.
  //! Join defines the type of parallel generated by the
  //! salient vertices of the spine.
  //! The default type is GeomAbs_Arc where the vertices generate
  //! sections of a circle.
  //! If join type is GeomAbs_Intersection, the edges that
  //! intersect in a salient vertex generate the edges
  //! prolonged until intersection.
  Standard_EXPORT void Init(const TopoDS_Face&     Spine,
                            const GeomAbs_JoinType Join         = GeomAbs_Arc,
                            const bool             IsOpenResult = false);

  Standard_EXPORT BRepOffsetAPI_MakeOffset(const TopoDS_Wire&     Spine,
                                           const GeomAbs_JoinType Join         = GeomAbs_Arc,
                                           const bool             IsOpenResult = false);

  //! Initialize the evaluation of Offsetting.
  Standard_EXPORT void Init(const GeomAbs_JoinType Join         = GeomAbs_Arc,
                            const bool             IsOpenResult = false);

  //! Set approximation flag
  //! for conversion input contours into ones consisting of
  //! 2D circular arcs and 2D linear segments only.
  Standard_EXPORT void SetApprox(const bool ToApprox);

  //! Initializes the algorithm to construct parallels to the wire Spine.
  Standard_EXPORT void AddWire(const TopoDS_Wire& Spine);

  //! Computes a parallel to the spine at distance Offset and
  //! at an altitude Alt from the plane of the spine in relation
  //! to the normal to the spine.
  //! Exceptions: StdFail_NotDone if the offset is not built.
  Standard_EXPORT void Perform(const double Offset, const double Alt = 0.0);

  //! Builds the resulting shape (redefined from MakeShape).
  Standard_EXPORT void Build(
    const Message_ProgressRange& theRange = Message_ProgressRange()) override;

  //! returns a list of the created shapes
  //! from the shape <S>.
  Standard_EXPORT const NCollection_List<TopoDS_Shape>& Generated(const TopoDS_Shape& S) override;

  //! Converts each wire of the face into contour consisting only of
  //! arcs and segments. New 3D curves are built too.
  Standard_EXPORT static TopoDS_Face ConvertFace(const TopoDS_Face& theFace,
                                                 const double       theAngleTolerance);

private:
  bool                                  myIsInitialized;
  bool                                  myLastIsLeft;
  GeomAbs_JoinType                      myJoin;
  bool                                  myIsOpenResult;
  bool                                  myIsToApprox;
  TopoDS_Face                           myFace;
  NCollection_List<TopoDS_Shape>        myWires;
  NCollection_List<BRepFill_OffsetWire> myLeft;
  NCollection_List<BRepFill_OffsetWire> myRight;
};

#endif // _BRepOffsetAPI_MakeOffset_HeaderFile
