import streamlit as st
import re
import os
import json
import csv
import urllib.parse
import random
import time
import html
import uuid
import threading
from datetime import datetime, timezone, timedelta

# 引入格式处理库
try:
    import PyPDF2
    import docx
    from ebooklib import epub
    from bs4 import BeautifulSoup
except ImportError:
    pass 

# 尝试导入 ntplib
try:
    import ntplib
except ImportError:
    ntplib = None

# 从 utils 导入
from utils import (
    render_header, 
    generate_book_content, 
    ensure_log_file,
    save_file_locally,
    show_export_success_modal
)
from logic import FEATURE_MODELS, MODEL_MAPPING, OpenAI 

# 确保 DATA_DIR 可用
try: 
    from config import DATA_DIR
except ImportError:
    DATA_DIR = "data"

# ==============================================================================
# 🛡️ 严格审计日志系统 (Books 集成版)
# ==============================================================================

SYSTEM_LOG_PATH = os.path.join(DATA_DIR, "logs", "system_audit.csv")
_log_lock = threading.Lock()

def get_session_id():
    """获取或生成当前会话的唯一追踪ID"""
    if "session_trace_id" not in st.session_state:
        st.session_state.session_trace_id = str(uuid.uuid4())[:8]
    return st.session_state.session_trace_id

def log_audit_event(category, action, details, status="SUCCESS", module="BOOKS"):
    try:
        os.makedirs(os.path.dirname(SYSTEM_LOG_PATH), exist_ok=True)
        timestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        session_id = get_session_id()
        
        status_map = {"SUCCESS": "成功", "WARNING": "警告", "ERROR": "错误"}
        status_cn = status_map.get(status, status)
        
        module_map = {"BOOKS": "书籍管理", "WRITER": "写作终端", "SETTINGS": "系统设置"}
        module_cn = module_map.get(module, module)

        if isinstance(details, (dict, list)):
            try: details = json.dumps(details, ensure_ascii=False)
            except: details = str(details)
            
        row = [timestamp, session_id, module_cn, category, action, status_cn, details]
        
        with _log_lock:
            file_exists = os.path.exists(SYSTEM_LOG_PATH)
            with open(SYSTEM_LOG_PATH, mode='a', newline='', encoding='utf-8-sig') as f:
                writer = csv.writer(f)
                if not file_exists or os.path.getsize(SYSTEM_LOG_PATH) == 0:
                    writer.writerow(['时间', '会话ID', '模块', '类别', '操作', '状态', '详情']) 
                writer.writerow(row)
    except Exception as e:
        print(f"❌ 审计日志写入失败: {e}")

# ==============================================================================
# 1. 基础配置
# ==============================================================================
NOVEL_GENRES = {
    "玄幻奇幻": ["东方玄幻", "异世大陆", "王朝争霸", "高武世界", "西方奇幻", "领主种田", "魔法校园", "黑暗幻想"],
    "仙侠修真": ["凡人流", "古典仙侠", "修真文明", "幻想修仙", "洪荒封神", "无敌流", "家族修仙"],
    "都市现实": ["都市异能", "都市修仙", "神医赘婿", "文娱明星", "商战职场", "校花贴身", "鉴宝捡漏", "年代文"],
    "科幻末世": ["末世危机", "星际文明", "赛博朋克", "时空穿梭", "进化变异", "古武机甲", "无限流", "废土重建"],
    "历史军事": ["架空历史", "穿越重生", "秦汉三国", "两宋元明", "外国历史", "谍战特工", "军旅生涯", "大国崛起"],
    "游戏竞技": ["虚拟网游", "电子竞技", "游戏异界", "体育竞技", "卡牌游戏", "桌游棋牌", "全民领主"],
    "悬疑灵异": ["侦探推理", "诡异修仙", "盗墓探险", "风水秘术", "克苏鲁", "神秘复苏", "惊悚乐园"],
    "轻小说/二次元": ["原生幻想", "恋爱日常", "综漫同人", "变身入替", "搞笑吐槽", "系统流", "乙女向"],
    "诸天无限": ["诸天万界", "无限流", "综漫", "主神建设", "位面交易"],
    "脑洞创意": ["反套路", "迪化流", "聊天群", "幕后黑手", "灵气复苏"]
}

FLAT_GENRE_LIST = []
for main, subs in NOVEL_GENRES.items():
    for sub in subs:
        FLAT_GENRE_LIST.append(f"{main}-{sub}")

if hasattr(st, "dialog"):
    dialog_decorator = st.dialog
else:
    dialog_decorator = st.experimental_dialog

# ==============================================================================
# 2. 底层工具函数
# ==============================================================================

USAGE_LOG_PATH = os.path.join(DATA_DIR, "logs", "usage_log.csv")

def get_beijing_time():
    """获取北京时间 (UTC+8)"""
    try:
        utc_now = datetime.now(timezone.utc)
        beijing_time = utc_now.astimezone(timezone(timedelta(hours=8)))
        return beijing_time.strftime("%Y-%m-%d %H:%M:%S")
    except Exception:
        return (datetime.now() + timedelta(hours=8)).strftime("%Y-%m-%d %H:%M:%S")

def ensure_export_dir():
    export_dir = os.path.join(DATA_DIR, "exports")
    if not os.path.exists(export_dir): os.makedirs(export_dir)
    return export_dir

def get_cached_file_path(book_id, book_title):
    safe_title = re.sub(r'[\\/*?:"<>|]', "", str(book_title)).strip()
    return os.path.join(DATA_DIR, "exports", f"{book_id}_{safe_title}.txt")

def get_relation_dir():
    d = os.path.join(DATA_DIR, "relations")
    if not os.path.exists(d):
        try: os.makedirs(d)
        except: pass
    return d

def save_relations_to_disk(book_id, relations_data):
    rd = get_relation_dir()
    file_path = os.path.join(rd, f"book_{book_id}.json")
    try:
        with open(file_path, "w", encoding="utf-8") as f:
            json.dump(relations_data, f, ensure_ascii=False, indent=2)
        return True
    except Exception as e:
        print(f"Save Relation Error: {e}")
        return False

def load_relations_from_disk(book_id):
    file_path = os.path.join(get_relation_dir(), f"book_{book_id}.json")
    if os.path.exists(file_path):
        try:
            with open(file_path, "r", encoding="utf-8") as f: 
                return json.load(f)
        except Exception as e:
            print(f"Load Error: {e}")
            return []
    return []

def record_token_usage(provider, model, tokens, action_name, book_title=None):
    try:
        final_book_title = book_title
        if not final_book_title or final_book_title == "未知书籍":
            if 'current_book_id' in st.session_state and st.session_state.current_book_id:
                try:
                    if 'db' in st.session_state:
                        res = st.session_state.db.query("SELECT title FROM books WHERE id=?", (st.session_state.current_book_id,))
                        if res: final_book_title = res[0]['title']
                except: pass
        if not final_book_title: final_book_title = "未知书籍"

        price_per_1k = 0.03 
        model_str = str(model).lower()
        if "gpt-4" in model_str: price_per_1k = 0.2
        elif "mini" in model_str: price_per_1k = 0.01
        elif "deepseek" in model_str: price_per_1k = 0.005 
        
        cost = (tokens / 1000.0) * price_per_1k
        timestamp = get_beijing_time()
        log_dir = os.path.dirname(USAGE_LOG_PATH)
        if not os.path.exists(log_dir): os.makedirs(log_dir)
        
        file_exists = os.path.exists(USAGE_LOG_PATH)
        header = ['timestamp', 'provider', 'model', 'chars', 'cost', 'book_title']
        
        if file_exists:
            with open(USAGE_LOG_PATH, 'r', encoding='utf-8') as f:
                first_line = f.readline().strip()
            if 'book_title' not in first_line:
                with open(USAGE_LOG_PATH, 'r', encoding='utf-8') as old_f: lines = old_f.readlines()
                with open(USAGE_LOG_PATH, 'w', newline='', encoding='utf-8') as new_f:
                    writer = csv.writer(new_f)
                    writer.writerow(header)
                    for line in lines[1:]:
                        parts = line.strip().split(',')
                        if len(parts) >= 5: writer.writerow(parts[:5] + ["历史记录"])
        
        with open(USAGE_LOG_PATH, mode='a', newline='', encoding='utf-8') as f:
            writer = csv.writer(f)
            if not os.path.exists(USAGE_LOG_PATH) or os.path.getsize(USAGE_LOG_PATH) == 0:
                writer.writerow(header)
            writer.writerow([timestamp, provider, model, tokens, cost, final_book_title])
            
        print(f"💰 [计费] {action_name}: {tokens} tokens, ¥{cost:.4f} (书: {final_book_title})")
    except Exception as e:
        print(f"❌ 计费日志写入失败: {e}")

def process_and_save_tags(db_mgr, book_id, tags_list):
    if not tags_list: return []
    if isinstance(tags_list, str): tags_list = [tags_list]
    clean_tags = sorted(list(set([str(t).strip() for t in tags_list if str(t).strip()])))
    if not clean_tags: return []
    db_mgr.execute("DELETE FROM book_categories WHERE book_id=?", (book_id,))
    for tag in clean_tags:
        c_res = db_mgr.query("SELECT id FROM categories WHERE name=?", (tag,))
        if c_res: cid = c_res[0]['id']
        else: cid = db_mgr.execute("INSERT INTO categories (name) VALUES (?)", (tag,))
        db_mgr.execute("INSERT INTO book_categories (book_id, category_id) VALUES (?,?)", (book_id, cid))
    return clean_tags

def generate_bing_search_image(keyword):
    if not keyword: return ""
    encoded = urllib.parse.quote(keyword)
    return f"https://tse2.mm.bing.net/th?q={encoded}&w=300&h=300&c=7&rs=1&p=0"

def audit_download_callback(book_title, book_id):
    try:
        ensure_log_file()
        log_audit_event("数据导出", "下载书籍TXT", {"书籍": book_title, "ID": book_id})
    except Exception as e: print(f"Logging Error: {e}")

def robust_decode(data_bytes):
    encodings = ['utf-8', 'gb18030', 'gbk', 'big5', 'utf-16']
    for enc in encodings:
        try: return data_bytes.decode(enc)
        except UnicodeDecodeError: continue
    return data_bytes.decode('gb18030', errors='ignore')

def extract_text_from_file(uploaded_file):
    file_type = uploaded_file.name.split('.')[-1].lower()
    try:
        uploaded_file.seek(0) 
        if file_type == 'txt':
            return robust_decode(uploaded_file.getvalue())
        elif file_type == 'pdf':
            reader = PyPDF2.PdfReader(uploaded_file)
            text = []
            for page in reader.pages: text.append(page.extract_text() or "")
            return "\n".join(text)
        elif file_type == 'docx':
            doc = docx.Document(uploaded_file)
            return "\n".join([para.text for para in doc.paragraphs])
        elif file_type == 'epub':
            temp_path = f"temp_{int(time.time())}.epub"
            with open(temp_path, "wb") as f: f.write(uploaded_file.getvalue())
            book = epub.read_epub(temp_path)
            text = []
            for item in book.get_items():
                if item.get_type() == epub.EpubItemType.EBOOK_HTML:
                    soup = BeautifulSoup(item.get_content(), 'html.parser')
                    text.append(soup.get_text())
            if os.path.exists(temp_path): os.remove(temp_path)
            return "\n".join(text)
        else:
            return f"文件解析失败: 不支持的文件格式: {file_type}"
    except Exception as e:
        return f"文件解析失败 ({file_type}): {str(e)}"

def extract_and_parse_json(content):
    """
    🔥 超级增强版 JSON 解析器：自动修复截断、补全括号
    """
    try:
        content = re.sub(r'```json\s*', '', content, flags=re.IGNORECASE)
        content = re.sub(r'```\s*', '', content)
        content_clean = content.strip()

        # 1. 尝试直接解析
        try:
            return json.loads(content_clean, strict=False)
        except:
            pass

        # 2. 寻找边界
        start_brace = content.find('{')
        start_bracket = content.find('[')
        
        start_idx = -1
        if start_brace != -1 and start_bracket != -1:
            start_idx = min(start_brace, start_bracket)
        elif start_brace != -1:
            start_idx = start_brace
        elif start_bracket != -1:
            start_idx = start_bracket
            
        if start_idx == -1:
            return None

        json_candidate = content[start_idx:].strip()

        # 3. 智能修复函数
        def attempt_fix_and_load(raw_str):
            try:
                return json.loads(raw_str, strict=False)
            except json.JSONDecodeError:
                fixed = raw_str.rstrip()
                if fixed.endswith(','): fixed = fixed[:-1]
                if fixed.count('"') % 2 != 0: fixed += '"'
                
                open_braces = fixed.count('{') - fixed.count('}')
                open_brackets = fixed.count('[') - fixed.count(']')
                
                patches = []
                # 方案1：先补 } 再补 ]
                patch_str_1 = ('}' * open_braces) + (']' * open_brackets) 
                patches.append(patch_str_1)
                # 方案2：先补 ] 再补 }
                patch_str_2 = (']' * open_brackets) + ('}' * open_braces) 
                patches.append(patch_str_2)
                
                for p in patches:
                    try: return json.loads(fixed + p, strict=False)
                    except: continue
                return None

        parsed = attempt_fix_and_load(json_candidate)
        if parsed: return parsed
        
        # 4. 尝试截取到最后一个闭合符
        end_brace = json_candidate.rfind('}')
        end_bracket = json_candidate.rfind(']')
        end_idx = max(end_brace, end_bracket)
        
        if end_idx != -1:
            sub_candidate = json_candidate[:end_idx+1]
            try: return json.loads(sub_candidate, strict=False)
            except: pass

        return None
    except Exception:
        return None

def _resolve_ai_client(engine, assigned_key):
    """
    智能解析客户端：支持自定义模型(CUSTOM::) 和 原生模型
    添加调试日志以排查AI调用问题
    """
    print(f"🔍 解析AI客户端，分配key: {assigned_key}")
    
    # 1. 拦截自定义模型
    if assigned_key and str(assigned_key).startswith("CUSTOM::"):
        try:
            target_name = assigned_key.split("::", 1)[1]
            print(f"🔍 检测到自定义模型: {target_name}")
            settings = engine.get_config_db("ai_settings", {})
            custom_list = settings.get("custom_model_list", [])
            print(f"🔍 自定义模型列表: {len(custom_list)} 个")
            
            for m in custom_list:
                if m.get("name") == target_name:
                    api_key = m.get("key")
                    base_url = m.get("base")
                    model_id = m.get("api_model")
                    print(f"🔍 找到匹配模型: {model_id}, base_url: {base_url}")
                    
                    if not api_key or not base_url: 
                        print("❌ API Key 或 Base URL 缺失")
                        return None, None, None
                    
                    client = OpenAI(api_key=api_key, base_url=base_url)
                    return client, model_id, "custom"
            
            print(f"❌ 未找到自定义模型配置: {target_name}")
            return None, None, None
        except Exception as e:
            print(f"❌ 自定义模型解析失败: {e}")
            return None, None, None

    # 2. 原生模型
    print(f"🔍 使用原生模型: {assigned_key}")
    return engine.get_client(assigned_key)

# ==============================================================================
# 关系处理函数（按照characters.py的方法改造）
# ==============================================================================

def extract_simple_relation(full_label):
    """从完整关系描述中提取简单关系词"""
    if not full_label:
        return "相关"
    
    # 常见关系词映射
    relation_map = {
        "师徒": ["师徒", "师父", "徒弟", "传道", "授业"],
        "伴侣": ["夫妻", "道侣", "情侣", "恋人", "伴侣", "配偶"],
        "亲人": ["父子", "母女", "兄弟", "姐妹", "兄妹", "姐弟", "亲属", "亲戚"],
        "朋友": ["好友", "朋友", "挚友", "死党", "兄弟", "姐妹"],
        "敌对": ["仇敌", "敌人", "对手", "宿敌", "死对头"],
        "师徒": ["师父", "徒弟", "师徒", "传人", "弟子"],
        "上下级": ["主仆", "君臣", "上司", "下属", "领导"],
        "恩人": ["救命恩人", "恩人", "有恩"],
        "仇人": ["仇人", "血仇", "世仇", "死仇"]
    }
    
    full_label_lower = full_label.lower()
    
    for simple_rel, keywords in relation_map.items():
        for keyword in keywords:
            if keyword in full_label_lower:
                return simple_rel
    
    # 如果没匹配到，取前2-4个字符
    if len(full_label) <= 4:
        return full_label
    else:
        return full_label[:3] + ".."

def _write_ai_relations(db_mgr, book_id, relations, char_map):
    """按照characters.py格式保存关系数据"""
    if not relations: return 0
    n_rels = 0
    clean_relations = []
    
    # 构建更鲁棒的映射表：{ "林溯": id, "lin su": id }
    normalized_map = {}
    for name, cid in char_map.items():
        normalized_map[name.strip()] = cid
        normalized_map[name.strip().lower()] = cid
    
    for rel in relations:
        if not isinstance(rel, dict): continue
        
        # 🔥 核心修复：确保名字是干净的字符串
        char1 = _safe_clean_text(rel.get('char1', '')).strip()
        char2 = _safe_clean_text(rel.get('char2', '')).strip()
        desc = _safe_clean_text(rel.get('desc', '关联'))
        
        # 🔥 获取详细描述字段
        detail = _safe_clean_text(rel.get('detail', ''))
        reason = _safe_clean_text(rel.get('reason', ''))
        description = _safe_clean_text(rel.get('description', ''))
        
        # 🔥 构建详细描述：优先使用detail，其次是reason，最后是description
        detailed_desc = ""
        if detail:
            detailed_desc = detail
        elif reason:
            detailed_desc = reason
        elif description:
            detailed_desc = description
        else:
            detailed_desc = desc  # 默认使用简短描述
        
        if not char1 or not char2: continue

        c1_id = normalized_map.get(char1) or normalized_map.get(char1.lower())
        c2_id = normalized_map.get(char2) or normalized_map.get(char2.lower())
        
        # 模糊匹配尝试
        if not c1_id:
            for k, v in normalized_map.items(): 
                if char1 in k or k in char1: c1_id = v; break
        if not c2_id:
            for k, v in normalized_map.items():
                if char2 in k or k in char2: c2_id = v; break

        if c1_id and c2_id and c1_id != c2_id:
            # 🔥 按照characters.py格式保存关系数据
            clean_relations.append({
                "source": c1_id, 
                "target": c2_id, 
                "label": desc,  # 完整关系描述
                "description": detailed_desc,  # 详细描述
                "weight": 3
            })
            n_rels += 1
            
    save_relations_to_disk(book_id, clean_relations)
    return n_rels

# ==============================================================================
# 3. 核心逻辑
# ==============================================================================

# 🔥 新增：强力清洗函数，专门处理 AI 返回的字典或脏数据
def _safe_clean_text(val):
    """
    智能清洗数据：去除 JSON 格式，仅保留核心文本
    解决 {"title": "..."} 这种乱码写入数据库的问题
    """
    if val is None: return ""
    
    # 如果已经是字典或列表，进行拆包
    if isinstance(val, (dict, list)):
        try:
            # 如果是字典，优先提取 content/desc/description/value
            if isinstance(val, dict):
                # 尝试提取有意义的字段
                candidates = [
                    val.get('content'), 
                    val.get('desc'), 
                    val.get('description'), 
                    val.get('value'),
                    val.get('summary')
                ]
                # 过滤掉 None 和空字符串
                valid = [str(c).strip() for c in candidates if c and str(c).strip()]
                
                if valid:
                    # 如果有 content, 返回 content
                    return valid[0]
                
                # 如果没有常用键，且有 title，返回 title: content 格式（如果有的话）
                if 'title' in val:
                    # 尝试找其他剩余的 value
                    rest = [str(v) for k, v in val.items() if k != 'title' and v]
                    if rest:
                        return f"{val['title']}: {', '.join(rest)}"
                    return str(val['title'])
                    
                # 实在不行，拼接所有 value
                return " ".join([str(v) for v in val.values() if isinstance(v, (str, int, float))])
            
            # 如果是列表，递归清洗并拼接
            if isinstance(val, list):
                return ", ".join([_safe_clean_text(x) for x in val if x])
                
        except:
            return str(val)
            
    # 如果是字符串，尝试去除首尾引号
    s = str(val).strip()
    if s.startswith('"') and s.endswith('"'): s = s[1:-1]
    return s

def _write_detailed_world_settings(db_mgr, book_id, settings_list):
    if not settings_list: return 0
    saved_count = 0
    
    # 兼容: 如果传入的是单个字典而不是列表（修复AI返回单对象问题）
    if isinstance(settings_list, dict):
        settings_list = [settings_list]

    type_map = {
        "PowerSystem": "PowerSystem", 
        "Geography": "Geography",     
        "History": "TimeHistory",     
        "Culture": "CultureLife",     
        "RuleSystem": "RuleSystem",   
        "Organization": "Organization", 
        "Universe": "Universe",       
        "Other": "Other"
    }
    
    if isinstance(settings_list, list):
        for item in settings_list:
            if not isinstance(item, dict): continue
            
            # 🔥 修复：使用清洗函数
            title = _safe_clean_text(item.get("title", "未命名设定"))
            
            # 尝试获取内容
            content = item.get("content", "")
            
            # 如果没有content但有mechanism/description/desc，则拼凑
            if not content:
                parts = []
                if "mechanism" in item: parts.append(f"【机制】{_safe_clean_text(item['mechanism'])}")
                if "代价" in item: parts.append(f"【代价】{_safe_clean_text(item['代价'])}")
                if "desc" in item: parts.append(_safe_clean_text(item['desc']))
                if "description" in item: parts.append(_safe_clean_text(item['description']))
                content = "\n".join(parts)
            
            full_content = f"{title}\n{_safe_clean_text(content)}"
            
            category = item.get("category", "Other")
            mapped_cat = type_map.get(category, 'Other')
            db_status = f"Setting_{mapped_cat}"
            
            try:
                db_mgr.execute(
                    "INSERT INTO plots (book_id, content, status, importance) VALUES (?, ?, ?, 10)",
                    (book_id, full_content, db_status)
                )
                saved_count += 1
            except Exception as e:
                print(f"Setting insert error: {e}")
    return saved_count

def analyze_book_metadata_deep_ai(engine, book_title, full_text):
    # 🔥 修复核心：优先读取用户配置的 model_assignments，而不是写死 DSK_V3
    assignments = engine.get_config_db("model_assignments", {})
    assigned_key = assignments.get("import_char_analysis") 
    
    # 如果没配置，则回退到默认
    if not assigned_key:
        assigned_key = FEATURE_MODELS.get("import_char_analysis", {}).get('default', 'DSK_V3')
    
    client, model_name, _ = _resolve_ai_client(engine, assigned_key)
    
    if not client: 
        print(f"❌ Client resolution failed for key: {assigned_key}")
        return None, 0

    log_audit_event("书籍导入", "AI深度分析开始", {"书籍": book_title, "Model": assigned_key})
    
    sample = full_text[:25000]
    
    prompt = f"""
    你是一位资深网文主编。用户上传了小说《{book_title}》。
    
    请执行以下逻辑：
    1. **知识检索**：如果你确信自己阅览过《{book_title}》（且作者匹配），请直接根据你的知识库生成该书的详细数据。
    2. **文本分析**：如果你不认识这本书，必须基于我提供的【文本前2.5万字】进行分析。
    
    【任务目标】
    请返回一个**纯 JSON 对象**。
    ⚠️ **重要指令**：
    1. **所有value值必须且只能使用简体中文**（avatar_kw除外）。
    2. gender字段必须严格为 "男" 或 "女"。
    3. JSON格式必须合法，不要包含 ```json 标记。
    
    严格遵守以下结构：
    {{
        "synopsis": "剧情梗概(中文)",
        "tags": ["流派1", "关键词"],
        "characters": [
            {{
                "name": "姓名",
                "role": "主角/配角",
                "gender": "男/女",
                "desc": "人物简介",
                "origin": "出身",
                "profession": "职业",
                "cheat_ability": "金手指/能力",
                "power_level": "实力等级",
                "appearance_features": "外貌特征",
                "debts_and_feuds": "恩怨情仇",
                "avatar_kw": "English keywords for painting"
            }}
        ],
        "relations": [
            {{ 
                "char1": "角色A(必须与character.name完全一致)", 
                "char2": "角色B", 
                "desc": "关系描述",
                "detail": "详细起因缘由（如：在第3章中因为争夺宝物结仇）"
            }}
        ],
        "world_settings": [
            {{ "category": "PowerSystem", "title": "境界划分", "content": "详细内容" }}
        ]
    }}

    【文本前2.5万字】：
    {sample}
    """
    
    try:
        kwargs = {
            "model": model_name,
            "messages": [{"role": "user", "content": prompt}],
            "temperature": 0.5,
            "max_tokens": 4000
        }
        try:
            if "deepseek" in model_name.lower() or "gpt" in model_name.lower():
                kwargs["response_format"] = {"type": "json_object"}
        except: pass

        response = client.chat.completions.create(**kwargs)
        content = response.choices[0].message.content.strip()
        usage = response.usage.total_tokens if response.usage else 0
        
        print(f"DEBUG - AI Response Length: {len(content)}") 
        parsed_data = extract_and_parse_json(content)
        
        if not parsed_data:
            print(f"❌ JSON 解析失败，完整内容前500字符: {content[:500]}")
            # 这里不直接返回 None，而是尝试让外层处理空数据
            
        record_token_usage("OpenAI", model_name, usage, f"导入分析(智能版)-《{book_title}》", book_title)
        return parsed_data, usage
    except Exception as e:
        log_audit_event("书籍导入", "AI分析失败", {"错误信息": str(e)}, status="ERROR")
        print(f"AI Error: {e}")
        return None, 0

def generate_structure_via_ai_v2(engine, title, intro, genre_list, status_callback=None, target_chapter_count=50):
    if isinstance(genre_list, list): genre_str = ", ".join(genre_list)
    else: genre_str = str(genre_list)

    assignments = engine.get_config_db("model_assignments", {})
    assigned_key = assignments.get("novel_structure_gen") or assignments.get("books_arch_gen") or "GPT_4o"
    client, model_name, _ = _resolve_ai_client(engine, assigned_key)
    
    if not client: return False, f"⚠️ 未配置 AI 模型 (Key: {assigned_key})", {}, None

    base_max_tokens = 4000
    if model_name and "deepseek" in model_name.lower(): base_max_tokens = 8000

    final_data = { "characters": [], "relations": [], "world_settings": [], "structure": [] }
    total_tokens = 0

    try:
        # Step 1: 力量体系
        if status_callback: status_callback(f"⚔️ Step 1/6: 构建独创的力量/科技体系...", 5)
        
        prompt_power = f"""
        你是一位【{genre_str}】网文主编。请为《{title}》设计**核心升级体系**。
        简介："{intro}"
        要求：所有内容使用简体中文。
        
        返回 JSON：
        {{ "world_settings": [ {{ "category": "PowerSystem", "title": "...", "content": "..." }} ] }}
        """
        
        power_context_str = ""
        try:
            res_p = client.chat.completions.create(
                model=model_name, messages=[{"role": "user", "content": prompt_power}], temperature=0.7, max_tokens=2500
            )
            if hasattr(res_p, 'usage'): total_tokens += res_p.usage.total_tokens
            data_p = extract_and_parse_json(res_p.choices[0].message.content)
            
            # 🔥 容错：如果AI直接返回了 {title:..., content:...} 而不是 world_settings 列表
            settings = []
            if isinstance(data_p, dict):
                if "world_settings" in data_p:
                    settings = data_p.get("world_settings", [])
                else:
                    settings = [data_p]
            elif isinstance(data_p, list):
                settings = data_p
            
            if settings:
                final_data["world_settings"].extend(settings)
                power_context_str = json.dumps(settings, ensure_ascii=False)
        except Exception as e:
            print(f"Power step error: {e}")

        # Step 2: 地理与势力
        if status_callback: status_callback(f"🌍 Step 2/6: 完善世界格局与势力斗争...", 15)
        prompt_geo = f"""
        基于力量体系：{power_context_str[:1000]}
        简介："{intro}"
        请完善世界观：1. Geography(地理) 2. Organization(势力)。
        ⚠️ 要求：所有内容必须是简体中文。
        返回 JSON：{{ "world_settings": [ ... ] }}
        """
        world_context_str = power_context_str
        try:
            res_g = client.chat.completions.create(
                model=model_name, messages=[{"role": "user", "content": prompt_geo}], temperature=0.8, max_tokens=3000
            )
            if hasattr(res_g, 'usage'): total_tokens += res_g.usage.total_tokens
            data_g = extract_and_parse_json(res_g.choices[0].message.content)
            
            settings = []
            if isinstance(data_g, dict): settings = data_g.get("world_settings", [])
            elif isinstance(data_g, list): settings = data_g
            
            if settings:
                final_data["world_settings"].extend(settings)
                world_context_str = json.dumps(final_data["world_settings"], ensure_ascii=False)
        except Exception as e: print(f"World step error: {e}")

        # Step 3: 核心双雄
        if status_callback: status_callback(f"👤 Step 3/6: 深度刻画主角与宿敌...", 30)
        
        char_schema = """
        字段要求(必须简体中文)：
        - name: 姓名
        - role: "主角" 或 "反派"
        - gender: "男" 或 "女" (必填)
        - origin: 出身
        - profession: 身份
        - cheat_ability: 金手指
        - personality_flaw: 性格缺陷
        - debts_and_feuds: 恩怨
        - avatar_kw: English keywords only
        """
        
        prompt_core_chars = f"""
        基于世界观：{world_context_str[:1200]}
        简介："{intro}"
        请设计 2 名核心角色。
        ⚠️ **所有字段值必须为中文(avatar_kw除外)，gender必须为男或女**。
        {char_schema}
        返回 JSON：{{ "characters": [ ... ] }}
        """
        
        core_chars = []
        try:
            res_c1 = client.chat.completions.create(
                model=model_name, messages=[{"role": "user", "content": prompt_core_chars}], temperature=0.85, max_tokens=base_max_tokens
            )
            if hasattr(res_c1, 'usage'): total_tokens += res_c1.usage.total_tokens
            data_c1 = extract_and_parse_json(res_c1.choices[0].message.content)
            
            if isinstance(data_c1, dict): core_chars = data_c1.get("characters", [])
            elif isinstance(data_c1, list): core_chars = data_c1
            if core_chars: final_data["characters"].extend(core_chars)
        except Exception as e: print(f"Core Char error: {e}")

        # Step 4: 重要配角
        if status_callback: status_callback(f"👥 Step 4/6: 补充关键配角...", 50)
        existing_names = [c.get('name', 'unknown') for c in final_data["characters"] if isinstance(c, dict)]
        
        prompt_sub_chars = f"""
        基于主角与反派：{", ".join(existing_names)}
        请额外设计 4-6 名重要配角。
        ⚠️ **所有字段值必须为中文(avatar_kw除外)，gender必须为男或女**。
        返回 JSON：{{ "characters": [ ... ] }}
        """
        try:
            res_c2 = client.chat.completions.create(
                model=model_name, messages=[{"role": "user", "content": prompt_sub_chars}], temperature=0.85, max_tokens=base_max_tokens
            )
            if hasattr(res_c2, 'usage'): total_tokens += res_c2.usage.total_tokens
            data_c2 = extract_and_parse_json(res_c2.choices[0].message.content)
            sub_chars = []
            if isinstance(data_c2, dict): sub_chars = data_c2.get("characters", [])
            elif isinstance(data_c2, list): sub_chars = data_c2
            if sub_chars: final_data["characters"].extend(sub_chars)
        except Exception as e: print(f"Sub Char error: {e}")

        # Step 5: 人物关系网
        if status_callback: status_callback(f"🕸️ Step 5/6: 编织人物关系网...", 65)
        all_names = [c.get('name', '') for c in final_data["characters"] if isinstance(c, dict) and 'name' in c]
        
        prompt_rel = f"""
        角色列表：{", ".join(all_names)}
        请生成 10-15 组人物关系。
        ⚠️ **char1 和 char2 必须从上述角色列表中选择，使用完全一致的姓名**。
        ⚠️ **每个关系必须包含简短描述(desc)和详细描述(detail)**
        格式：{{ "relations": [ {{ "char1": "A", "char2": "B", "desc": "简短中文描述", "detail": "详细起因缘由，包括发生的事件、章节、原因等" }} ] }}
        """
        try:
            res_r = client.chat.completions.create(
                model=model_name, messages=[{"role": "user", "content": prompt_rel}], temperature=0.7, max_tokens=2000
            )
            if hasattr(res_r, 'usage'): total_tokens += res_r.usage.total_tokens
            data_r = extract_and_parse_json(res_r.choices[0].message.content)
            if isinstance(data_r, dict): final_data["relations"] = data_r.get("relations", [])
        except: pass

        # Step 6: 分批生成大纲
        all_chapters = []
        prev_summary = "故事开始。"
        lite_char_context = ", ".join(all_names[:6])
        num_batches = (target_chapter_count + 9) // 10
        
        for i in range(num_batches): 
            start_c = i * 10 + 1
            end_c = min((i + 1) * 10, target_chapter_count)
            if status_callback: 
                current_batch_progress = (i / num_batches) * 25
                status_callback(f"📝 Step 6/6: 正在构思第 {start_c}-{end_c} 章 (进度 {i+1}/{num_batches})...", 70 + int(current_batch_progress))
            
            prompt_outline = f"""
            小说：《{title}》
            核心角色：{lite_char_context}
            前情概要：{prev_summary}
            任务：生成第 {start_c} - {end_c} 章大纲。
            ⚠️ **所有内容必须为简体中文**。
            返回 JSON：{{ "chapters": [ {{ "title": "...", "summary": "..." }}, ... ] }}
            """
            try:
                res_out = client.chat.completions.create(
                    model=model_name, messages=[{"role": "user", "content": prompt_outline}], temperature=0.8, max_tokens=base_max_tokens
                )
                if hasattr(res_out, 'usage'): total_tokens += res_out.usage.total_tokens
                data_out = extract_and_parse_json(res_out.choices[0].message.content)
                new_chaps = []
                if isinstance(data_out, dict): new_chaps = data_out.get("chapters", [])
                elif isinstance(data_out, list): new_chaps = data_out
                if new_chaps:
                    all_chapters.extend(new_chaps)
                    last_sums = [c.get('summary','') for c in new_chaps[-2:] if isinstance(c, dict)]
                    prev_summary = f"第{end_c}章结束。近期剧情：{' '.join(last_sums)}"
                time.sleep(0.5)
            except Exception as e: print(f"Outline batch {i} error: {e}")

        if all_chapters:
            final_data["structure"] = [{ "part_name": "第一卷", "volumes": [{ "vol_name": "正文", "chapters": all_chapters }] }]
        
        if status_callback: status_callback("✅ 架构生成完毕，写入数据库...", 98)
        record_token_usage("OpenAI", model_name, total_tokens, f"架构生成-《{title}》", title)
        return True, final_data, {'total_tokens': total_tokens}, assigned_key

    except Exception as e:
        return False, f"AI 调用失败: {str(e)}", {}, None

def _write_ai_characters(db_mgr, book_id, chars):
    if not chars: return 0
    n_chars = 0
    # 🔥 修复：使用纯 URL 字符串
    dicebear_base = "https://api.dicebear.com/9.x/adventurer/svg?seed="
    
    for char in chars:
        if not isinstance(char, dict): continue
        
        # 🔥 名字清洗
        name = _safe_clean_text(char.get('name', '')).strip()
        if not name or name == "未命名": continue
        
        role = _safe_clean_text(char.get('role', '配角'))
        
        # 🔥 性别智能修正
        raw_gender = str(char.get('gender', ''))
        gender = '未知'
        if '男' in raw_gender: gender = '男'
        elif '女' in raw_gender: gender = '女'
        
        race = _safe_clean_text(char.get('race', '人族'))
        desc = _safe_clean_text(char.get('desc', ''))
        
        # 头像处理
        avatar_kw = char.get('avatar_kw', '') 
        # 如果 avatar_kw 是字典，只取值
        if isinstance(avatar_kw, dict): 
            avatar_kw = " ".join([str(v) for v in avatar_kw.values()])
            
        if not avatar_kw: avatar_kw = f"{name} fantasy style"
        
        if name in avatar_kw: search_query = avatar_kw
        else: search_query = f"{name} {avatar_kw}"
        
        avatar_url = generate_bing_search_image(search_query) 
        if not avatar_url: 
            avatar_url = f"{dicebear_base}{urllib.parse.quote(name)}&flip=true"

        # 🔥 使用 _safe_clean_text 清洗所有字段，防止写入 JSON 字符串
        origin = _safe_clean_text(char.get('origin', '未知'))
        profession = _safe_clean_text(char.get('profession', '无'))
        cheat_ability = _safe_clean_text(char.get('cheat_ability', '无'))
        power_level = _safe_clean_text(char.get('power_level', '未知'))
        ability_limitations = _safe_clean_text(char.get('ability_limitations', '无'))
        appearance_features = _safe_clean_text(char.get('appearance_features', ''))
        signature_sign = _safe_clean_text(char.get('signature_sign', ''))
        relationship_to_protagonist = _safe_clean_text(char.get('relationship_to_protagonist', '未知'))
        social_role = _safe_clean_text(char.get('social_role', ''))
        debts_and_feuds = _safe_clean_text(char.get('debts_and_feuds', ''))

        if not desc:
            desc = f"【身份】{origin}，{profession}。\n【外貌】{appearance_features}。\n【能力】{cheat_ability}（{power_level}）。"

        try:
            db_mgr.execute(
                """INSERT INTO characters (
                    book_id, name, role, gender, race, desc, is_major, avatar,
                    origin, profession, cheat_ability, power_level, ability_limitations,
                    appearance_features, signature_sign, relationship_to_protagonist, social_role, debts_and_feuds
                ) VALUES (?,?,?,?,?,?,?,?, ?,?,?,?,?, ?,?,?,?,?)""",
                (
                    book_id, name, role, gender, race, desc, True, avatar_url,
                    origin, profession, cheat_ability, power_level, ability_limitations,
                    appearance_features, signature_sign, relationship_to_protagonist, social_role, debts_and_feuds
                )
            )
            n_chars += 1
        except Exception as e:
            print(f"Insert char error: {e}")
            # 降级插入
            try:
                db_mgr.execute(
                    "INSERT INTO characters (book_id, name, role, gender, race, desc, is_major, avatar) VALUES (?,?,?,?,?,?,?,?)",
                    (book_id, name, role, gender, race, desc, True, avatar_url)
                )
                n_chars += 1
            except Exception as e2:
                print(f"Char insert failed completely: {e2}")

    return n_chars

def _write_ai_structure(db_mgr, book_id, structure):
    n_chaps = 0
    if not structure: return 0
    if not isinstance(structure, list): return 0

    for p_idx, part in enumerate(structure):
        if not isinstance(part, dict): continue
        p_name = _safe_clean_text(part.get('part_name', f'第{p_idx+1}篇'))
        part_id = db_mgr.execute("INSERT INTO parts (book_id, name, sort_order) VALUES (?,?,?)", (book_id, p_name, (p_idx+1)*100))
        
        volumes = part.get('volumes', [])
        for v_idx, vol in enumerate(volumes):
            if not isinstance(vol, dict): continue
            v_name = _safe_clean_text(vol.get('vol_name', f'第{v_idx+1}卷'))
            vol_id = db_mgr.execute("INSERT INTO volumes (book_id, part_id, name, sort_order) VALUES (?,?,?,?)", (book_id, part_id, v_name, (v_idx+1)*100))
            raw_chapters = vol.get('chapters', [])
            for c_idx, chap in enumerate(raw_chapters):
                if isinstance(chap, dict):
                    c_title = _safe_clean_text(chap.get('title', f"第{c_idx+1}章"))
                    c_summary = _safe_clean_text(chap.get('summary', ""))
                else:
                    c_title = str(chap)
                    c_summary = ""
                
                db_mgr.execute("INSERT INTO chapters (volume_id, title, content, summary, sort_order) VALUES (?,?,?,?,?)",
                    (vol_id, c_title, "", c_summary, c_idx+1))
                n_chaps += 1
    return n_chaps

def _process_ai_generated_data(db_mgr, engine, book_id, res_data, book_title, book_category):
    # 🛡️ 容错核心：确保 res_data 是字典
    if isinstance(res_data, list):
        if len(res_data) > 0 and isinstance(res_data[0], dict):
             merged = {}
             for item in res_data:
                 if isinstance(item, dict): merged.update(item)
             res_data = merged
        else: res_data = {}
    
    if not isinstance(res_data, dict): res_data = {}
    
    def get_robust_list(data, possible_keys):
        for k in possible_keys:
            if k in data and isinstance(data[k], list): return data[k]
        return []

    char_keys = ['characters', 'Characters', 'roles', 'chars', '角色列表', '人物']
    characters = get_robust_list(res_data, char_keys)
    n_chars = _write_ai_characters(db_mgr, book_id, characters)
    
    # 重新查询 ID 映射，确保能够正确连线
    char_map_res = db_mgr.query("SELECT name, id FROM characters WHERE book_id=?", (book_id,))
    char_map = {r['name']: r['id'] for r in char_map_res}
    
    rel_keys = ['relations', 'Relations', 'relationships', 'relationship', '人物关系', '关系']
    relations = get_robust_list(res_data, rel_keys)
    n_rels = _write_ai_relations(db_mgr, book_id, relations, char_map)
    
    setting_keys = ['world_settings', 'WorldSettings', 'settings', 'world', '世界观', '设定']
    world_settings = get_robust_list(res_data, setting_keys)
    n_settings = _write_detailed_world_settings(db_mgr, book_id, world_settings)
    
    return n_chars, n_rels, n_settings

# ==============================================================================
# 解析逻辑
# ==============================================================================
CN_NUM = {'零':0, '一':1, '二':2, '三':3, '四':4, '五':5, '六':6, '七':7, '八':8, '九':9, '十':10, '百':100, '千':1000, '万':10000}
def parse_volume_number(num_str):
    if not num_str: return 0
    if num_str.isdigit(): return int(num_str)
    try:
        val = 0; current_val = 0
        if num_str.startswith('十'): num_str = '一' + num_str 
        for char in num_str:
            if char in CN_NUM:
                digit = CN_NUM[char]
                if digit >= 10:
                    if current_val == 0: current_val = 1; current_val *= digit; val += current_val; current_val = 0
                else: current_val = digit
        val += current_val
        return val if val > 0 else 0
    except: return 0

def _parse_book_structure(full_text):
    lines = full_text.splitlines()
    parts_list = []
    part_index_map = {} 
    
    default_part = {'idx': 1, 'name': '正文', 'vol_map': {}, 'vol_list': []}
    parts_list.append(default_part)
    part_index_map[1] = default_part
    
    default_vol = {'idx': 1, 'name': '默认卷', 'chapters': []}
    default_part['vol_list'].append(default_vol)
    default_part['vol_map'][1] = default_vol
    
    current_part = default_part
    current_vol = default_vol
    current_chap_title = None
    current_chap_content = []

    combined_pattern = re.compile(r'^\s*(?:第\s*([0-9零一二三四五六七八九十百千万]+)\s*[卷部篇集]|Volume\s*(\d+))\s*(.*?)\s*(第\s*|)([0-9零一二三四五六七八九十百千万]+[章节回]|Chapter\s*\d+|序章|楔子|前言|尾声|后记)(.*)$', re.IGNORECASE)
    part_pattern = re.compile(r'^\s*(?:第\s*([0-9零一二三四五六七八九十百千万]+)\s*[篇部]|Part\s*(\d+))(.*)$', re.IGNORECASE)
    vol_pattern = re.compile(r'^\s*(?:第\s*([0-9零一二三四五六七八九十百千万]+)\s*[卷集]|Volume\s*(\d+))(.*)$', re.IGNORECASE)
    chap_pattern = re.compile(r'^\s*(?:正文\s*)?(第\s*|)([0-9零一二三四五六七八九十百千万]+[章节回]|Chapter\s*\d+|序章|楔子|前言|尾声|后记)(.*)$', re.IGNORECASE)

    def save_current_chapter():
        if current_chap_title and current_chap_content:
            content_str = "\n".join(current_chap_content).strip()
            if content_str: current_vol['chapters'].append({'title': current_chap_title, 'content': content_str})

    def process_volume_switch(num_str, title_str):
        nonlocal current_vol
        v_idx = parse_volume_number(num_str)
        if v_idx == 0: v_idx = len(current_part['vol_list']) + 1
        prefix = f"第{num_str}卷" if not num_str.isdigit() else f"Volume {num_str}"
        full_v_name = title_str if title_str and title_str.startswith(prefix) else f"{prefix} {title_str}".strip()

        if v_idx in current_part['vol_map']:
            current_vol = current_part['vol_map'][v_idx]
            if len(full_v_name) > len(current_vol['name']): current_vol['name'] = full_v_name
        else:
            is_curr_vol_empty = (len(current_vol['chapters']) == 0 and current_vol['name'] == '默认卷')
            if is_curr_vol_empty and v_idx == 1:
                current_vol['name'] = full_v_name; current_part['vol_map'][1] = current_vol 
            else:
                new_vol = {'idx': v_idx, 'name': full_v_name, 'chapters': []}
                current_part['vol_list'].append(new_vol); current_part['vol_map'][v_idx] = new_vol; current_vol = new_vol

    for line in lines:
        stripped = line.strip()
        if not stripped:
            if current_chap_title: current_chap_content.append(line)
            continue

        combined_match = combined_pattern.match(stripped)
        part_match = part_pattern.match(stripped) if not combined_match else None
        vol_match = vol_pattern.match(stripped) if not combined_match else None
        chap_match = chap_pattern.match(stripped) if not combined_match and not part_match and not vol_match else None

        if combined_match:
            save_current_chapter() 
            v_num = combined_match.group(1) or combined_match.group(2)
            v_title_part = combined_match.group(3).strip()
            process_volume_switch(v_num, v_title_part)
            c_marker = combined_match.group(4).strip() + combined_match.group(5).strip()
            c_title_part = combined_match.group(6).strip()
            current_chap_title = f"{c_marker} {c_title_part}".strip()
            current_chap_content = []
            continue

        if part_match:
            save_current_chapter()
            num_str = part_match.group(1) or part_match.group(2)
            p_idx = parse_volume_number(num_str)
            if p_idx == 0: p_idx = len(parts_list) + 1
            p_title = part_match.group(3).strip()
            
            new_part = {'idx': p_idx, 'name': f"第{num_str}篇 {p_title}".strip(), 'vol_map': {}, 'vol_list': []}
            parts_list.append(new_part); part_index_map[p_idx] = new_part; current_part = new_part
            vol = { 'idx': 1, 'name': '默认卷', 'chapters': [] }; current_part['vol_list'].append(vol); current_part['vol_map'][1] = vol; current_vol = vol
            current_chap_title = None; current_chap_content = []
            continue

        if vol_match:
            save_current_chapter()
            num = vol_match.group(1) or vol_match.group(2)
            title = vol_match.group(3).strip()
            process_volume_switch(num, title)
            current_chap_title = None; current_chap_content = []
            continue

        if chap_match:
            save_current_chapter()
            raw_title = f"{chap_match.group(1).strip()}{chap_match.group(2).strip()} {chap_match.group(3).strip() or ''}".strip()
            current_chap_title = raw_title
            current_chap_content = []
            continue

        if current_chap_title: current_chap_content.append(line)
        else:
            if stripped and len(stripped) < 50: 
                 if not current_chap_content: 
                     current_chap_title = "序言/引子"; current_chap_content.append(line)
                 else: current_chap_content.append(line)
            elif stripped:
                 current_chap_content.append(line)
    
    save_current_chapter()
    
    total_chaps = sum(len(v['chapters']) for p in parts_list for v in p['vol_list'])

    if total_chaps < 5 and len(full_text) > 50000:
        print("⚠️ 触发暴力分章模式...")
        log_audit_event("书籍导入", "触发暴力分章", "检测到章节数过少，自动切换模式", status="WARNING")
        
        fallback_structure = [{'part_name': '正文', 'volumes': [{'vol_name': '全书', 'chapters': []}]}]
        target_vol = fallback_structure[0]['volumes'][0]
        
        force_chap_pattern = re.compile(r'^\s*(?:正文\s*)?(第\s*[0-9零一二三四五六七八九十百千万]+\s*[章节回]|Chapter\s*\d+)(.*)$', re.MULTILINE)
        split_res = force_chap_pattern.split(full_text)
        
        if split_res[0].strip():
            target_vol['chapters'].append({'title': '序章', 'content': split_res[0].strip()})
            
        i = 1
        while i < len(split_res) - 1:
            title_key = split_res[i] 
            title_suffix = split_res[i+1] 
            content = split_res[i+2] if i+2 < len(split_res) else ""
            full_title = f"{title_key} {title_suffix}".strip()
            target_vol['chapters'].append({'title': full_title, 'content': content.strip()})
            i += 3
            
        return fallback_structure

    final_structure = []
    for p in parts_list:
        valid_vols = []
        for v in p['vol_list']:
            if v['chapters']: valid_vols.append({'vol_name': v['name'], 'chapters': v['chapters']})
        if valid_vols: final_structure.append({'part_name': p['name'], 'volumes': valid_vols})
        
    return final_structure

def _import_book_process(db_mgr, engine, uploaded_file, book_id, book_title, book_author, genre_hint=""):
    progress_bar = st.progress(0)
    n_chaps, n_chars, n_rels, n_settings, detected_tags, usage_info = 0, 0, 0, 0, [], {}
    
    try:
        with st.status("🚀 正在导入书籍...", expanded=True) as status:
            status.write(f"📂 正在读取 {uploaded_file.name} ...")
            uploaded_file.seek(0) 
            full_text = extract_text_from_file(uploaded_file)
            progress_bar.progress(10)
            if "文件解析失败" in full_text: st.error(full_text); return 0, 0, 0, 0, 0, [], {}

            if "-" in uploaded_file.name:
                parts = uploaded_file.name.rsplit('.', 1)[0].split('-')
                if len(parts) >= 2:
                    book_title = parts[0]
                    book_author = parts[1]
                    db_mgr.execute("UPDATE books SET title=?, author=? WHERE id=?", (book_title, book_author, book_id))
            
            status.write("📂 正在解析章节结构 (双模式扫描)...")
            structure = _parse_book_structure(full_text)
            progress_bar.progress(25)
            
            total_chaps = sum(sum(len(v['chapters']) for v in p['volumes']) for p in structure)
            if total_chaps == 0:
                 structure = [{'part_name': '正文', 'volumes': [{'vol_name': '全书', 'chapters': [{'title': '全文内容', 'content': full_text}]}]}]
                 total_chaps = 1
            status.write(f"✅ 解析完成：共 {len(full_text)} 字，{total_chaps} 章")

            status.write("🧠 正在 AI 深度分析 (知识检索 + 文本分析)...")
            progress_bar.progress(40)
            
            ai_data, tokens = analyze_book_metadata_deep_ai(engine, book_title, full_text)
            usage_info = {'total_tokens': tokens}
            
            if ai_data and "error" not in ai_data:
                synopsis = ai_data.get('synopsis', '暂无简介')
                detected_tags = ai_data.get('tags', [genre_hint or "未分类"])
                
                db_mgr.execute("UPDATE books SET intro=? WHERE id=?", (synopsis, book_id))
                
                n_chars, n_rels, n_settings = _process_ai_generated_data(db_mgr, engine, book_id, ai_data, book_title, detected_tags[0])
                process_and_save_tags(db_mgr, book_id, detected_tags)
                status.write(f"✅ 档案建立：{n_chars} 名角色，{n_rels} 条关系")
            else:
                detected_tags = [genre_hint or "未分类"]

            progress_bar.progress(60)

            status.write("💾 正在写入数据库...")
            curr = 0; current_prog = 60.0
            for p_idx, part in enumerate(structure):
                part_id = db_mgr.execute("INSERT INTO parts (book_id, name, sort_order) VALUES (?,?,?)", (book_id, part['part_name'], (p_idx+1)*100))
                for v_idx, vol in enumerate(part['volumes']):
                    vol_id = db_mgr.execute("INSERT INTO volumes (book_id, part_id, name, sort_order) VALUES (?,?,?,?)", (book_id, part_id, vol['vol_name'], (v_idx+1)*100))
                    for c_idx, chap in enumerate(vol['chapters']):
                        db_mgr.execute("INSERT INTO chapters (volume_id, title, content, summary, sort_order) VALUES (?,?,?,?,?)",
                            (vol_id, chap['title'], chap['content'], f"字数:{len(chap['content'])}", c_idx+1))
                        curr += 1
                        if curr % 50 == 0:
                            current_prog = min(99, current_prog + 0.5)
                            progress_bar.progress(int(current_prog))
            
            n_chaps = total_chaps
            progress_bar.progress(100)
            db_mgr.execute("UPDATE books SET updated_at=? WHERE id=?", (get_beijing_time(), book_id))
            
            status.update(label=f"🎉 导入成功！共 {n_chaps} 章", state="complete", expanded=False)
            
            log_audit_event("书籍导入", "导入成功", {
                "书籍": book_title, 
                "作者": book_author, 
                "总字数": len(full_text), 
                "章节数": n_chaps
            })
            
            return n_chaps, n_chars, n_rels, n_settings, detected_tags, usage_info

    except Exception as e:
        log_audit_event("书籍导入", "导入致命错误", {"错误信息": str(e)}, status="ERROR")
        st.error(f"导入出错: {e}")
        return 0, 0, 0, 0, [], {}

# ==============================================================================
# 4. 弹窗组件
# ==============================================================================

@dialog_decorator("➕ 添加自定义流派")
def dialog_add_custom_genre():
    st.markdown("请输入新的流派名称：")
    db_mgr = st.session_state.db
    new_genre = st.text_input("流派名称", key="input_custom_genre_modal")
    
    if st.button("确认添加", type="primary", width="stretch"):
        val = new_genre.strip()
        if val:
            existing = db_mgr.query("SELECT id FROM categories WHERE name=?", (val,))
            if not existing:
                try:
                    db_mgr.execute("INSERT INTO categories (name) VALUES (?)", (val,))
                    log_audit_event("基础配置", "添加流派", {"流派名称": val})
                    
                    ms_key = "new_book_selected_genres"
                    current_selection = st.session_state.get(ms_key, [])
                    if not isinstance(current_selection, list): current_selection = []
                    
                    if val not in current_selection:
                        st.session_state[ms_key] = current_selection + [val]
                    
                    st.success(f"✅ 已添加并选中：{val}")
                    time.sleep(0.5)
                    st.rerun()
                except Exception as e:
                    st.error(f"添加失败: {e}")
            else:
                st.warning("⚠️ 该流派已存在")
        else:
            st.warning("名称不能为空")

@dialog_decorator("🎉 AI 创作完成")
def show_import_report_modal(data):
    db_mgr = st.session_state.db
    book_title = data.get('title', '新书')
    book_id = data.get('book_id')
    
    st.success(f"《{book_title}》架构已生成！")
    c1, c2, c3, c4 = st.columns(4)
    c1.metric("章节", data.get('chapters', 0))
    c2.metric("角色", data.get('chars', 0))
    c3.metric("关系", data.get('relations', 0))
    c4.metric("设定", data.get('settings', 0))
    st.divider()
    usage_info = data.get('usage', {})
    if usage_info and 'total_tokens' in usage_info:
        st.caption(f"Tokens 消耗: {usage_info.get('total_tokens', 0)}")
    
    if st.button("开始写作", type="primary", width="stretch"):
        if 'import_report_data' in st.session_state: del st.session_state['import_report_data']
        st.session_state.current_book_id = book_id
        st.session_state.current_menu = "write"
        st.rerun()
    if st.button("关闭", type="secondary", width="stretch"):
        if 'import_report_data' in st.session_state: del st.session_state['import_report_data']
        st.rerun()

# ==============================================================================
# 5. UI 渲染函数
# ==============================================================================

def render_import_section(engine):
    """书籍导入文件拖拽区"""
    db_mgr = st.session_state.db
    
    st.markdown("""
    <style>
    [data-testid="stFileUploaderDropzone"] {
        position: relative;
        padding: 30px 10px;
        border: 2px dashed #4CAF50;
        background-color: #f9f9f9;
        min-height: 120px;
    }
    [data-testid="stFileUploaderDropzone"] div div::before { display: none; }
    [data-testid="stFileUploaderDropzone"] div div span { display: none; }
    [data-testid="stFileUploaderDropzone"] div div small { display: none; }
    [data-testid="stFileUploaderDropzone"] button { display: none; }
    
    [data-testid="stFileUploaderDropzone"]::after { 
        content: "点击或将文件拖拽至此上传"; 
        visibility: visible; 
        display: block;
        position: absolute; 
        top: 40%; 
        left: 50%; 
        transform: translate(-50%, -50%);
        color: #333; 
        font-weight: bold; 
        font-size: 16px; 
        pointer-events: none;
    }
    [data-testid="stFileUploaderDropzone"]::before { 
        content: "支持 TXT / PDF / EPUB / DOCX (最大 200MB)"; 
        visibility: visible; 
        display: block;
        position: absolute; 
        top: 60%; 
        left: 50%; 
        transform: translate(-50%, -50%);
        color: #666; 
        font-size: 12px; 
        pointer-events: none;
    }
    </style>
    """, unsafe_allow_html=True)
    
    with st.expander("📥 导入书籍", expanded=False):
        uploaded_file = st.file_uploader("文件上传区", type=["txt", "pdf", "epub", "docx"], key="import_file_real", label_visibility="collapsed")
        
        if uploaded_file:
            file_id = f"{uploaded_file.name}_{uploaded_file.size}"
            if st.session_state.get('last_loaded_file') != file_id:
                new_title = os.path.splitext(uploaded_file.name)[0]
                if "-" in new_title: new_title = new_title.split('-')[0]
                st.session_state['import_book_title_ui'] = new_title
                st.session_state['last_loaded_file'] = file_id
                st.rerun()

        with st.form("form_import_action"):
            c1, c2 = st.columns(2)
            default_title = st.session_state.get('import_book_title_ui', "")
            title_input = c1.text_input("书名", value=default_title)
            author_input = c2.text_input("作者", value="未知")
            genre_hint = st.text_input("📚 辅助关键词 (让 AI 更懂这本小说)", placeholder="例如：玄幻, 退婚流, 斗气")
            submitted = st.form_submit_button("🚀 开始导入", type="primary", width="stretch")
        
        if submitted:
            if not uploaded_file: st.error("请先上传文件")
            elif not title_input.strip(): st.error("书名不能为空")
            else:
                bid = None 
                try:
                    now_str = get_beijing_time()
                    bid = db_mgr.execute("INSERT INTO books (title, author, intro, created_at, updated_at) VALUES (?,?,?,?,?)", 
                                         (title_input, author_input, "导入中...", now_str, now_str))
                    
                    n_chaps, n_chars, n_rels, n_settings, detected_tags, usage = _import_book_process(db_mgr, engine, uploaded_file, bid, title_input, author_input, genre_hint)
                    
                    st.session_state['import_report_data'] = {
                        'title': title_input, 'chapters': n_chaps, 'chars': n_chars, 'relations': n_rels, 'settings': n_settings, 'tags': detected_tags, 'book_id': bid, 'usage': usage
                    }
                    st.rerun()
                except Exception as e:
                    if bid: db_mgr.execute("DELETE FROM books WHERE id=?", (bid,))
                    st.error(f"导入错误: {e}")

def render_book_card(book):
    book = dict(book)
    bid = book['id']
    book_title = book['title']
    db_mgr = st.session_state.db
    
    book_categories = db_mgr.query("SELECT c.name FROM book_categories bc JOIN categories c ON bc.category_id = c.id WHERE bc.book_id = ?", (bid,))
    genre_list = [c['name'] for c in book_categories]
    genre_value = " / ".join(genre_list) if genre_list else '未分类'
    
    raw_c_time = str(book.get('created_at', '')).replace('T', ' ').split('.')[0]
    raw_u_time = str(book.get('updated_at', '')).replace('T', ' ').split('.')[0]
    
    file_path = get_cached_file_path(bid, book_title)
    size_label = None
    if os.path.exists(file_path):
        f_size = os.path.getsize(file_path) / 1024 
        size_label = f"{f_size:.1f}KB" if f_size < 1024 else f"{f_size/1024:.1f}MB"

    intro_full = html.escape(book.get('intro') or "暂无简介")

    with st.container(border=True):
        c_head_L, c_head_R = st.columns([3, 1])
        with c_head_L: 
            st.markdown(f"""
            <div title="{intro_full}" style='height: 48px; overflow: hidden; display: -webkit-box; -webkit-line-clamp: 2; -webkit-box-orient: vertical; cursor: help;'>
                <h4 style='margin:0; padding:0; line-height: 1.2;'>📖 {book['title']}</h4>
            </div>
            """, unsafe_allow_html=True)
        with c_head_R:
            if size_label: st.markdown(f"<div style='text-align: right; color: #888; font-size: 12px; margin-top: 5px;'>📦 {size_label}</div>", unsafe_allow_html=True)
        
        st.markdown(f"""
        <div style='height: 90px; overflow-y: hidden; font-size: 13px; color: #555; margin-bottom: 10px; border-bottom: 1px dashed #eee;'>
            <div style='margin-bottom: 2px;'><b>作者:</b> {book['author']}</div>
            <div style='margin-bottom: 2px;'><b>分类:</b> {genre_value}</div>
            <div style='margin-bottom: 2px; color:#888; font-size:12px;'>📅 创建: {raw_c_time}</div>
            <div style='margin-bottom: 2px; color:#2e7d32; font-size:12px;'>⏱️ 更新: <b>{raw_u_time}</b></div>
        </div>
        """, unsafe_allow_html=True)

        st.markdown("<div style='height: 4px'></div>", unsafe_allow_html=True)

        c1, c2, c3, c4 = st.columns(4)
        if c1.button("✍️ 写作", key=f"ent_{bid}", type="primary", width="stretch", help="进入写作模式"):
            db_mgr.execute("UPDATE books SET updated_at=? WHERE id=?", (get_beijing_time(), bid))
            log_audit_event("写作模式", "进入写作", {"书籍": book['title'], "ID": bid})
            st.session_state.current_book_id = bid
            st.session_state.current_menu = "write"
            st.session_state.trigger_scroll_to_top = True 
            st.rerun()
            
        if c2.button("📑 预览", key=f"view_{bid}", width="stretch", help="查看章节列表"):
             st.session_state.current_book_id = bid; st.session_state.current_menu = "chapters"; st.rerun()
             
        with c3:
            if st.button("📥 导出", key=f"dl_{bid}", width="stretch", help="导出最新内容到本地文件夹"):
                try:
                    content = generate_book_content(db_mgr, bid)
                    with open(file_path, "w", encoding='utf-8') as f: f.write(content)
                    success, saved_path = save_file_locally(f"{book_title}.txt", content)
                    if success:
                        audit_download_callback(book_title, bid)
                        show_export_success_modal(saved_path)
                except Exception as e:
                    st.error(f"导出失败: {e}")
                    
        if c4.button("🗑️ 删除", key=f"del_{bid}", width="stretch", help="永久删除"):
            db_mgr.execute("DELETE FROM books WHERE id=?", (bid,))
            if os.path.exists(file_path): os.remove(file_path)
            log_audit_event("书籍管理", "删除书籍", {"书籍": book['title'], "ID": bid}, status="WARNING")
            st.rerun()

def render_books(engine):
    """书籍管理主界面"""
    db_mgr = st.session_state.db
    ensure_export_dir() 
    
    if 'import_report_data' in st.session_state:
        show_import_report_modal(st.session_state['import_report_data'])

    render_header("📚", "书籍管理")
    render_import_section(engine) 
    
    with st.expander("✨ AI 架构向导 (从零开始)", expanded=False):
        with st.form("form_new_book"):
            c_title, c_auth, c_num = st.columns([2, 1, 1])
            b_title = c_title.text_input("书名", placeholder="例如：诡秘之主")
            b_author = c_auth.text_input("作者", "我")
            b_target_chaps = c_num.number_input("生成章节数量", min_value=10, max_value=200, value=50, step=10, help="每10章为一个生成批次")
            
            existing_cats_db = db_mgr.query("SELECT name FROM categories")
            db_cats = [c['name'] for c in existing_cats_db] if existing_cats_db else []
            all_options = sorted(list(set(FLAT_GENRE_LIST + db_cats)))
            
            ms_key = "new_book_selected_genres"
            if ms_key not in st.session_state: st.session_state[ms_key] = []
            
            b_category = st.multiselect("流派", all_options, placeholder="选择流派...", key=ms_key)
            if st.form_submit_button("➕ 添加流派", type="secondary"): dialog_add_custom_genre()

            b_intro = st.text_area("简介 / 核心脑洞 (AI 生成依据)", height=150, placeholder="例如：穿越到异界，开局被退婚...")
            
            c_sub1, c_sub2 = st.columns([1, 4])
            btn_create = c_sub1.form_submit_button("仅创建", width="stretch")
            btn_create_gen = c_sub2.form_submit_button(f"🚀 启动 AI 架构师 (生成角色 + {b_target_chaps}章大纲)", type="primary", width="stretch")
            
            if btn_create or btn_create_gen:
                if not b_title.strip():
                    st.error("书名不能为空")
                elif btn_create_gen and not b_intro.strip():
                    st.error("AI 模式必须填写简介")
                else:
                    bid = None
                    try:
                        now_str = get_beijing_time()
                        bid = db_mgr.execute("INSERT INTO books (title, author, intro, created_at, updated_at) VALUES (?,?,?,?,?)", 
                                             (b_title, b_author, b_intro, now_str, now_str))
                        if b_category: process_and_save_tags(db_mgr, bid, b_category)

                        if btn_create_gen:
                            log_audit_event("AI架构", "启动架构生成", {"书名": b_title, "目标章节": b_target_chaps})
                            
                            status_container = st.status("🧠 AI 正在构思...", expanded=True)
                            progress_bar = status_container.progress(5)
                            def update_status(msg, p):
                                status_container.write(msg)
                                progress_bar.progress(p)

                            ok, res_data, usage, _ = generate_structure_via_ai_v2(engine, b_title, b_intro, b_category, update_status, target_chapter_count=b_target_chaps)
                            
                            if ok:
                                progress_bar.progress(95)
                                status_container.write("💾 正在写入数据库...")
                                
                                # 🛡️ 容错处理
                                if isinstance(res_data, list):
                                     if len(res_data) > 0 and isinstance(res_data[0], dict):
                                          merged_data = {}
                                          for item in res_data:
                                              if isinstance(item, dict): merged_data.update(item)
                                          res_data = merged_data
                                     else: res_data = {}
                                if not isinstance(res_data, dict): res_data = {}

                                n_chars, n_rels, n_settings = _process_ai_generated_data(db_mgr, engine, bid, res_data, b_title, b_category)
                                struct_data = res_data.get('structure', [])
                                n_chaps = _write_ai_structure(db_mgr, bid, struct_data)
                                
                                db_mgr.execute("UPDATE books SET updated_at=? WHERE id=?", (get_beijing_time(), bid))
                                status_container.update(label="✅ 完成！", state="complete")
                                
                                log_audit_event("AI架构", "架构生成完成", {"书名": b_title, "ID": bid})
                                
                                st.session_state['import_report_data'] = {
                                    'title': b_title, 'chapters': n_chaps, 'chars': n_chars, 'relations': n_rels, 'settings': n_settings, 'tags': b_category, 'book_id': bid, 'usage': usage
                                }
                                time.sleep(1); st.rerun()
                            else:
                                log_audit_event("AI架构", "架构生成失败", {"错误": str(res_data)}, status="ERROR")
                                st.error(f"AI 生成失败: {res_data}")
                        else:
                            log_audit_event("书籍管理", "新建书籍(手动)", {"书名": b_title})
                            st.toast("✅ 书籍已创建"); time.sleep(0.5); st.rerun()
                    except Exception as e:
                        st.error(f"操作失败: {e}")
                        if bid: db_mgr.execute("DELETE FROM books WHERE id=?", (bid,))

    books = db_mgr.query("SELECT * FROM books ORDER BY updated_at DESC")
    if books:
        for i in range(0, len(books), 2):
            cols = st.columns(2)
            with cols[0]: render_book_card(books[i])
            if i+1 < len(books):
                with cols[1]: render_book_card(books[i+1])
    else:
        st.info("暂无书籍。")