/*
 * SPDX-FileCopyrightText: 2025 Espressif Systems (Shanghai) CO LTD
 *
 * SPDX-License-Identifier: Apache-2.0
 */

#pragma once

#include <stdint.h>
#include <stdbool.h>
#include "sdkconfig.h"
#include "esp_ieee802154_types.h"
#ifdef __cplusplus
extern "C" {
#endif

/**
 * @brief  Register the process callbacks for the event generated by IEEE 802.15.4 subsystem.
 *
 * @param[in]  cb_list The event process callback list, please refer to `esp_ieee802154_event_cb_list_t`.
 *
 * @return
 *      - ESP_OK on success.
 */
esp_err_t ieee802154_event_callback_list_register(esp_ieee802154_event_cb_list_t cb_list);

/**
 * @brief  Unregister the process callbacks for the event generated by IEEE 802.15.4 subsystem.
 *
 * @return
 *      - ESP_OK on success.
 */
esp_err_t ieee802154_event_callback_list_unregister(void);

/**
 * @brief  The inner handler when reception successfully.
 *
 */
void ieee802154_inner_receive_done(uint8_t *data, esp_ieee802154_frame_info_t *frame_info);

/**
 * @brief  The inner handler when the SFD field of the frame was received.
 *
 */
void ieee802154_inner_receive_sfd_done(void);

/**
 * @brief  The inner handler when transmission finished successfully.
 *
 */
void ieee802154_inner_transmit_done(const uint8_t *frame, const uint8_t *ack, esp_ieee802154_frame_info_t *ack_frame_info);

/**
 * @brief  The inner handler when transmission failed.
 *
 */
void ieee802154_inner_transmit_failed(const uint8_t *frame, esp_ieee802154_tx_error_t error);

/**
 * @brief  The inner handler when the SFD field of the frame was transmitted.
 *
 */
void ieee802154_inner_transmit_sfd_done(uint8_t *frame);

/**
 * @brief  The inner handler when ED processed done.
 *
 */
void ieee802154_inner_energy_detect_done(int8_t power);

/**
 * @brief  The inner wrapper for enh-ack generation.
 *
 */
esp_err_t ieee802154_inner_enh_ack_generator(uint8_t *frame, esp_ieee802154_frame_info_t *frame_info, uint8_t* enhack_frame);

#ifdef __cplusplus
}
#endif
