import React, {type ReactNode, useState, useEffect} from 'react';
import clsx from 'clsx';
import NavbarNavLink from '@theme/NavbarItem/NavbarNavLink';
import type {Props} from '@theme/NavbarItem/DefaultNavbarItem/Desktop';
import styles from './styles.module.css';

// GitHub Stars Hook
function useGitHubStars() {
  const [stars, setStars] = useState<number | null>(null);

  useEffect(() => {
    fetch('https://api.github.com/repos/oceanbase/powermem')
      .then((res) => res.json())
      .then((data) => {
        if (data.stargazers_count && data.stargazers_count > 1000) {
          setStars(data.stargazers_count);
        }
      })
      .catch(() => {
        // Silently fail
      });
  }, []);

  return stars;
}

export default function DefaultNavbarItemDesktop({
  className,
  isDropdownItem = false,
  href,
  label,
  ...props
}: Props): ReactNode {
  // Check if this is an icon link
  const isGitHub = href?.includes('github.com') || label === 'GitHub';
  const isDiscord = href?.includes('discord.com') || label === 'Discord';
  const isX = href?.includes('x.com') || label === 'X';
  const isIconLink = isGitHub || isDiscord || isX;
  
  // Get GitHub stars if this is GitHub link
  const stars = isGitHub ? useGitHubStars() : null;

  // For icon links, use custom className and render icon instead of label
  // GitHub with stars needs more width
  const iconClassName = isIconLink 
    ? clsx(
        isDropdownItem ? 'dropdown__link' : 'navbar__item navbar__link',
        className,
        styles.iconNavbarLink,
        isGitHub && stars && stars > 100 && styles.iconNavbarLinkWithStars
      )
    : clsx(
        isDropdownItem ? 'dropdown__link' : 'navbar__item navbar__link',
        className,
      );

  const element = (
    <NavbarNavLink
      className={iconClassName}
      isDropdownLink={isDropdownItem}
      href={href}
      label={isIconLink ? undefined : label}
      html={isIconLink ? undefined : undefined}
      {...props}
    >
      {isIconLink && (
        <>
          {isGitHub && (
            <>
              <svg
                className={styles.icon}
                viewBox="0 0 24 24"
                fill="none"
                xmlns="http://www.w3.org/2000/svg"
              >
                <path
                  d="M12 1C5.923 1 1 5.923 1 12c0 4.867 3.149 8.979 7.521 10.436.55.096.756-.233.756-.522 0-.262-.013-1.128-.013-2.049-2.764.509-3.479-.674-3.699-1.292-.124-.317-.66-1.293-1.127-1.554-.385-.207-.936-.715-.014-.729.866-.014 1.485.797 1.691 1.128.99 1.663 2.571 1.196 3.204.907.096-.715.385-1.196.701-1.471-2.448-.275-5.005-1.224-5.005-5.432 0-1.196.426-2.186 1.128-2.956-.111-.275-.496-1.402.11-2.915 0 0 .921-.288 3.024 1.128a10.193 10.193 0 0 1 2.75-.371c.936 0 1.871.123 2.75.371 2.104-1.43 3.025-1.128 3.025-1.128.605 1.513.221 2.64.111 2.915.701.77 1.127 1.747 1.127 2.956 0 4.222-2.571 5.157-5.019 5.432.399.344.743 1.004.743 2.035 0 1.471-.014 2.654-.014 3.025 0 .289.206.632.756.522C19.851 20.979 23 16.854 23 12c0-6.077-4.922-11-11-11Z"
                  fill="currentColor"
                  fillRule="evenodd"
                  clipRule="evenodd"
                />
              </svg>
              {stars !== null && stars > 100 && (
                <span className={styles.stars}>
                  {stars.toLocaleString()}
                </span>
              )}
            </>
          )}
          {isDiscord && (
            <svg className={styles.icon} viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <use href="#social-discord" />
            </svg>
          )}
          {isX && (
            <svg className={styles.icon} viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
              <use href="#social-twitter" />
            </svg>
          )}
        </>
      )}
    </NavbarNavLink>
  );

  if (isDropdownItem) {
    return <li>{element}</li>;
  }

  return element;
}
