---
id: 67d301a5f536d1a9e5df1a8c
title: What Is Type Narrowing, and How Does It Work?
challengeType: 19
dashedName: what-is-type-narrowing-and-how-does-it-work
---

# --description--

There are going to be times where you have a value with a broad type and you need to narrow it down to a more specific type. For example, maybe you need to ensure an object matches an interface you defined. Or a string is within a specific list of values. There are quite a few ways to achieve this.

The first is narrowing by truthiness. Consider our example from the last lesson:

```js
const email = document.querySelector<HTMLInputElement>("#email");
console.log(email.value);
```

We get a compiler error trying to access the `value` property of `email`, because `email` might be `null`. However, we can use a conditional statement to confirm `email` is truthy before accessing the property:

```js
const email = document.querySelector<HTMLInputElement>("#email");
if (email) {
  console.log(email.value);
}
```

In this updated example, because `null` is not a truthy value, TypeScript is able to infer that `email` MUST be an `input` element within the conditional block. So it no longer throws a compiler error.

Truthiness checks can also work in the reverse direction:

```js
const email = document.querySelector<HTMLInputElement>("#email");
if (!email) {
  throw new ReferenceError("Could not find email element!")
}
console.log(email.value);
```

With this approach, we throw an error if `email` is falsy. `null` is a falsy value. Throwing an error ends the logical execution of this code, which means when we reach the `console.log()` call TypeScript knows `email` cannot be `null`.

Optional chaining is also a form of type narrowing, under the same premise that the property access can't happen if the `email` value is `null`.

```js
const email = document.querySelector<HTMLInputElement>("#email");
console.log(email?.value);
```

But what about other types? Well, you can also narrow types using the `typeof` operator. Let's look at an example of a variable we have indicated might be a string OR a number:

```js
const myVal = Math.random() > 0.5 ? 222 : "222";
console.log(myVal / 10)
```

In this example, we see a compiler error because we cannot perform arithmetic on a string value. But we can use a conditional to check the `typeof` the `myVal` variable:

```js
const myVal = Math.random() > 0.5 ? 222 : "222";
if (typeof myVal === "number") {
  console.log(myVal / 10);
}
```

Because we have used the `typeof` keyword, TypeScript now knows that `myVal` has to be a number, and we can safely perform arithmetic with it.

But what about more complex object types? If the object in question happens to come from a class, you can actually use the `instanceof` keyword to narrow the type. Going back to our `querySelector()` example:

```js
const email = document.querySelector("#email");
```

Rather than passing a generic type and telling TypeScript what the element is, we can use `instanceof` to narrow the type and write safer code:

```js
const email = document.querySelector("#email");

if (email instanceof HTMLInputElement) {
    console.log(email.value);
}
```

This approach may seem the same as our previous one, but `instanceof` is a runtime validation - which means, if we somehow got the TypeScript type wrong, our JavaScript code will still confirm that `email` is an `input` element.

Next, let's look at an example where we fetch a `User` object from an API and try to print the information:

```js
interface User {
    name: string;
    age: number;
}

const printAge = (user: User) => 
  console.log(`${user.name} is ${user.age} years old!`)

const request = await fetch("url")
const myUser = await request.json();
printAge(myUser);
```

We'll get a compiler error trying to pass `myUser` into the function because, even though we know the API returns the correct object, TypeScript does not. And the `.json()` method does not take a generic type.

The "easy" way to resolve this issue would be to cast the type:

```js
interface User {
    name: string;
    age: number;
}

const printAge = (user: User) => 
  console.log(`${user.name} is ${user.age} years old!`)

const request = await fetch("url")
const myUser = await request.json() as User;
printAge(myUser);
```

But whenever you cast the type, you're essentially weakening TypeScript's ability to catch potential errors. So instead of casting the type, you can write a type guard:

```js
interface User {
    name: string;
    age: number;
}

const isValidUser = (user: unknown): user is User => {
  return !!user && 
    typeof user === "object" &&
    "name" in user &&
    "age" in user;
}
```

The return type here is the key component of this function definition. The `user is User` syntax indicates that our function returns a boolean value, which when `true` means the `user` value satisfies the `User` interface. We then do some basic checks to ensure that the structure of the `user` object matches - note the use of a truthiness (`!!user`) narrowing and a `typeof` narrowing. We must do this because `typeof null` returns `"object"`:

```js
interface User {
    name: string;
    age: number;
}

const isValidUser = (user: unknown): user is User => {
  return !!user && 
    typeof user === "object" &&
    "name" in user &&
    "age" in user;
}

const printAge = (user: User) => 
  console.log(`${user.name} is ${user.age} years old!`)

const request = await fetch("url")
const myUser = await request.json() as User;
if (isValidUser(myUser)) {
    printAge(myUser);
}
```

Now, if we combine all of our logic together, we no longer get compiler errors and can successfully build our code.

Type narrowing is a powerful feature that helps you write safer, less error-prone code - but remember that TypeScript types are not completely rigid, so avoid practices like casting the type of a value without narrowing it.

# --questions--

## --text--

Which of the following is NOT a method of type narrowing discussed in the lesson?

## --answers--

Truthiness checks.

### --feedback--

While all of these can be used to narrow a type, one of them wasn't mentioned.

---

Optional chaining.

### --feedback--

While all of these can be used to narrow a type, one of them wasn't mentioned.

---

`typeof` operator.

### --feedback--

While all of these can be used to narrow a type, one of them wasn't mentioned.

---

Switch statements.

## --video-solution--

4

## --text--

What is the purpose of the `user is User` syntax in the type guard function?

## --answers--

It casts the `user` parameter to the `User` type.

### --feedback--

Think about the word `is`, and how the phrase `user is User` works.

---

It indicates that the function returns a boolean value.

### --feedback--

Think about the word `is`, and how the phrase `user is` User works.

---

It indicated that `user` satisfies the `User` interface.

---

It creates a new instance of the `User` interface.

### --feedback--

Think about the word `is`, and how the phrase `user is` User works.

## --video-solution--

3

## --text--

Which of the following is true about type casting in TypeScript?

## --answers--

It's the recommended way to resolve type issues.

### --feedback--

Type casting should be done as a last resort.

---

It strengthens TypeScript's ability to catch errors.

### --feedback--

Type casting should be done as a last resort.

---

It's done using the `as` keyword.

---

It should always be used instead of type guards.

### --feedback--

Type casting should be done as a last resort.

## --video-solution--

3
