import { test, expect } from '@playwright/test';
import translations from '../client/i18n/locales/english/translations.json';
import { addGrowthbookCookie } from './utils/add-growthbook-cookie';

const pageElements = {
  mainHeading: 'main-head',
  donateText1: 'donate-text-1',
  donateText2: 'donate-text-2',
  donateText3: 'donate-text-3',
  faqHeading: 'faq-head',
  campersImage: 'landing-page-figure'
};

const donationStringReplacements = {
  usdPlaceHolder: '{{usd}}',
  hoursPlaceHolder: '{{hours}}'
};

const donationFormStrings = {
  confirmTwentyDollar: translations.donate['confirm-monthly'].replace(
    donationStringReplacements.usdPlaceHolder,
    '20'
  ),
  confirmTwentyFiveDollar: translations.donate['confirm-monthly'].replace(
    donationStringReplacements.usdPlaceHolder,
    '25'
  ),
  confirmFiveDollars: translations.donate['confirm-monthly'].replace(
    donationStringReplacements.usdPlaceHolder,
    '5'
  ),
  twentyDollarsLearningContribution: translations.donate['your-donation-2']
    .replace(donationStringReplacements.usdPlaceHolder, '20')
    .replace(donationStringReplacements.hoursPlaceHolder, '1,000'),
  twentyFiveDollarsLearningContribution: translations.donate['your-donation-2']
    .replace(donationStringReplacements.usdPlaceHolder, '25')
    .replace(donationStringReplacements.hoursPlaceHolder, '1,250'),
  fiveDollarsLearningContribution: translations.donate['your-donation-2']
    .replace(donationStringReplacements.usdPlaceHolder, '5')
    .replace(donationStringReplacements.hoursPlaceHolder, '250'),
  editAmount: translations.donate['edit-amount'],
  donate: translations.buttons.donate
};

function donatePageTests() {
  test('should render correctly', async ({ page }) => {
    await expect(page).toHaveTitle(
      `${translations.donate.title} | freeCodeCamp.org`
    );

    const mainHeading = page.getByTestId(pageElements.mainHeading);
    await expect(mainHeading).toHaveText(translations.donate['help-more']);

    const donateText1 = page.getByTestId(pageElements.donateText1);
    await expect(donateText1).toHaveText(translations.donate.efficiency);

    const donateText2 = page.getByTestId(pageElements.donateText2);
    await expect(donateText2).toHaveText(translations.donate['why-donate-1']);

    const donateText3 = page.getByTestId(pageElements.donateText3);
    await expect(donateText3).toHaveText(translations.donate['why-donate-2']);

    const faqHead = page.getByTestId(pageElements.faqHeading);
    await expect(faqHead).toHaveText(translations.donate.faq);
  });

  test('should display the faq list with buttons', async ({ page }) => {
    const faq1 = page.getByRole('button', {
      name: translations.donate['get-help']
    });
    await expect(faq1).toBeVisible();
    await faq1.click();
    await expect(
      page.getByText(translations.donate['forward-receipt'])
    ).toBeVisible();
    await faq1.click();

    const faq2 = page.getByRole('button', {
      name: translations.donate['how-transparent']
    });
    await expect(faq2).toBeVisible();
    await faq2.click();
    await expect(
      page.getByText(translations.donate['very-transparent'])
    ).toBeVisible();
    await expect(
      page.getByText('You can download our IRS Determination Letter here.')
    ).toBeVisible();
    await expect(
      page.getByText(
        'You can download our most recent 990 (annual tax report) here.'
      )
    ).toBeVisible();
    await faq2.click();

    const faq3 = page.getByRole('button', {
      name: translations.donate['how-efficient']
    });
    await expect(faq3).toBeVisible();
    await faq3.click();
    await expect(
      page.getByText(translations.donate['fcc-budget'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['help-millions'])
    ).toBeVisible();
    await faq3.click();

    const faq4 = page.getByRole('button', {
      name: translations.donate['how-one-time']
    });
    await expect(faq4).toBeVisible();
    await faq4.click();
    await expect(
      page.getByText(
        "If you'd prefer to make one-time donations, you can support freeCodeCamp's mission whenever you have cash to spare. You can use this link to donate whatever amount feels right through PayPal."
      )
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['wire-transfer'])
    ).toBeVisible();
    await faq4.click();

    const faq5 = page.getByRole('button', {
      name: translations.donate['does-crypto']
    });
    await expect(faq5).toBeVisible();
    await faq5.click();
    await expect(
      page.getByText(translations.donate['yes-cryptocurrency'])
    ).toBeVisible();
    await faq5.click();

    const faq6 = page.getByRole('button', {
      name: translations.donate['can-check']
    });
    await expect(faq6).toBeVisible();
    await faq6.click();
    await expect(
      page.getByText(translations.donate['yes-check'])
    ).toBeVisible();
    await expect(page.getByText('Free Code Camp, Inc.')).toBeVisible();
    await expect(page.getByText('3905 Hedgcoxe Rd')).toBeVisible();
    await expect(page.getByText('PO Box 250352')).toBeVisible();
    await expect(page.getByText('Plano, TX 75025')).toBeVisible();
    await faq6.click();

    const faq7 = page.getByRole('button', {
      name: translations.donate['how-matching-gift']
    });
    await expect(faq7).toBeVisible();
    await faq7.click();
    await expect(
      page.getByText(translations.donate['employers-vary'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['some-volunteer'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['help-matching-gift'])
    ).toBeVisible();
    await faq7.click();

    const faq8 = page.getByRole('button', {
      name: translations.donate['how-endowment']
    });
    await expect(faq8).toBeVisible();
    await faq8.click();
    await expect(
      page.getByText(translations.donate['endowment'])
    ).toBeVisible();
    await faq8.click();

    const faq9 = page.getByRole('button', {
      name: translations.donate['how-legacy']
    });
    await expect(faq9).toBeVisible();
    await faq9.click();
    await expect(
      page.getByText(translations.donate['we-honored'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['legacy-gift-message'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['thank-wikimedia'])
    ).toBeVisible();
    await expect(
      page.getByText(translations.donate['legacy-gift-questions'])
    ).toBeVisible();
    await faq9.click();

    const faq10 = page.getByRole('button', {
      name: translations.donate['how-stock']
    });
    await expect(faq10).toBeVisible();
    await faq10.click();
    await expect(
      page.getByText(translations.donate['welcome-stock'])
    ).toBeVisible();
    await faq10.click();

    const faq11 = page.getByRole('button', {
      name: translations.donate['how-update']
    });
    await expect(faq11).toBeVisible();
    await faq11.click();
    await expect(
      page.getByText(translations.donate['forward-receipt'])
    ).toBeVisible();
    await faq11.click();

    const faq12 = page.getByRole('button', {
      name: translations.donate['anything-else']
    });
    await expect(faq12).toBeVisible();
    await faq12.click();
    await expect(
      page.getByText(translations.donate['other-support'])
    ).toBeVisible();
    await faq12.click();
  });

  test('should make $5 tier selectable', async ({ page }) => {
    await page.click('[role="tab"]:has-text("$5")');

    await expect(
      page.getByText(donationFormStrings.confirmFiveDollars)
    ).toBeVisible();

    await expect(
      page.getByText(donationFormStrings.fiveDollarsLearningContribution)
    ).toBeVisible();
  });

  test('should switch between tier selection and payment options', async ({
    page
  }) => {
    // Tier selection
    await page.click('[role="tab"]:has-text("$5")');
    await expect(
      page.getByText(donationFormStrings.confirmFiveDollars)
    ).toBeVisible();
    await expect(
      page.getByText(donationFormStrings.fiveDollarsLearningContribution)
    ).toBeVisible();
    await page.click(`button:has-text("${donationFormStrings.donate}")`);

    // Donation form
    const isEditButtonVisible = await page.isVisible(
      `button:has-text("${donationFormStrings.editAmount}")`
    );
    expect(isEditButtonVisible).toBeTruthy();
    await expect(page.getByTestId('donation-form')).toBeVisible();
    await page.click(`button:has-text("${donationFormStrings.editAmount}")`);

    // Tier selection
    await expect(
      page.getByText(donationFormStrings.confirmFiveDollars)
    ).toBeVisible();
    await expect(
      page.getByText(donationFormStrings.fiveDollarsLearningContribution)
    ).toBeVisible();
  });
}

interface DefaultTierTestConfig {
  defaultTier: string;
  tiers: string[];
  confirmationText: string;
  contributionText: string;
}

function donatePageDefault(config: DefaultTierTestConfig) {
  const { defaultTier, tiers, confirmationText, contributionText } = config;
  test(`should select ${defaultTier} tier by default`, async ({ page }) => {
    await expect(page.getByText(confirmationText)).toBeVisible();
    const tabs = await page.$$('[role="tab"]');
    expect(tabs.length).toBe(tiers.length);
    for (const tab of tabs) {
      const tabText = await tab.innerText();
      expect(tiers).toContain(tabText);
      const isActive = await tab.getAttribute('data-state');
      if (tabText === defaultTier) {
        expect(isActive).toBe('active');
      } else {
        expect(isActive).not.toBe('active');
      }
    }
    await expect(page.getByText(contributionText)).toBeVisible();
  });
}

test.describe('Authenticated User', () => {
  test.beforeEach(async ({ page }) => {
    await page.goto('/donate');
  });
  donatePageTests();
});

test.describe('Authenticated User Page Defaults - Variation A', () => {
  test.beforeEach(async ({ context, page }) => {
    await addGrowthbookCookie({ context, variation: 'A243' });
    await page.goto('/donate');
  });
  donatePageDefault({
    defaultTier: '$20',
    tiers: ['$5', '$10', '$20', '$40'],
    confirmationText: donationFormStrings.confirmTwentyDollar,
    contributionText: donationFormStrings.twentyDollarsLearningContribution
  });
});

test.describe('Unauthenticated User', () => {
  test.use({ storageState: { cookies: [], origins: [] } });
  test.beforeEach(async ({ page }) => {
    await page.goto('/donate');
  });
  donatePageTests();
});

test.describe('Unauthenticated User Page Default - Variation B', () => {
  test.use({ storageState: { cookies: [], origins: [] } });
  test.beforeEach(async ({ context, page }) => {
    await addGrowthbookCookie({ context, variation: 'B145' });
    await page.goto('/donate');
  });
  donatePageDefault({
    defaultTier: '$25',
    tiers: ['$5', '$10', '$25', '$40'],
    confirmationText: donationFormStrings.confirmTwentyFiveDollar,
    contributionText: donationFormStrings.twentyFiveDollarsLearningContribution
  });
});
