import { SsoProviderName, SsoProviderType } from '@logto/schemas';
import { conditional, trySafe } from '@silverhand/essentials';

import RequestError from '#src/errors/RequestError/index.js';
import assertThat from '#src/utils/assert-that.js';

import SamlConnector from '../SamlConnector/index.js';
import { type SingleSignOnFactory } from '../index.js';
import { type SingleSignOn, type SingleSignOnConnectorData } from '../types/connector.js';
import {
  defaultAttributeMapping,
  samlConnectorConfigGuard,
  type SamlMetadata,
} from '../types/saml.js';
import {
  type SingleSignOnConnectorSession,
  type CreateSingleSignOnSession,
} from '../types/session.js';

/**
 * SAML SSO connector
 *
 * This class extends the basic SAML connector class and add some business related utils methods.
 *
 * @property data The SAML connector data from the database
 * @property inValidConnectorConfig Whether the connector config is invalid
 *
 * @method getConfig Get the SP and IdP metadata
 * @method getUserInfo Get user info from the SAML assertion.
 * @method getAuthorizationUrl Get the SAML SSO URL.
 */
export class SamlSsoConnector extends SamlConnector implements SingleSignOn {
  constructor(
    readonly data: SingleSignOnConnectorData,
    endpoint: URL
  ) {
    const parseConfigResult = samlConnectorConfigGuard.safeParse(data.config);

    // Fallback to undefined if config is invalid
    super(endpoint, data.id, conditional(parseConfigResult.success && parseConfigResult.data));
  }

  async getIssuer() {
    const { entityId } = await this.getSamlIdpMetadata();

    return entityId;
  }

  /**
   * `defaultAttributeMapping`: Default attribute mapping
   * `serviceProvider`: SP metadata
   * `identityProvider`: IdP metadata. Returns undefined if the idp config is invalid.
   */
  async getConfig(): Promise<SamlMetadata> {
    const serviceProvider = this.serviceProviderMetadata;
    const identityProvider = await trySafe(async () => this.getSamlIdpMetadata());

    return {
      defaultAttributeMapping,
      serviceProvider,
      identityProvider,
    };
  }

  /**
   * Get SAML SSO URL.
   * This URL will be used to redirect to the SAML identity provider.
   *
   * @param jti The unique identifier for the connector session.
   * @param redirectUri The redirect uri for the identity provider.
   * @param state The state generated by Logto experience client.
   * @param setSession Set the connector session data to the oidc provider session storage. @see @logto/connector-kit
   */
  async getAuthorizationUrl(
    {
      jti,
      redirectUri,
      state,
      connectorId,
    }: { jti: string; redirectUri: string; state: string; connectorId: string },
    setSession: CreateSingleSignOnSession
  ) {
    // We use jti as the value of the RelayState in the SAML request. So we can get it back from the SAML response and retrieve the connector session.
    const singleSignOnUrl = await this.getSingleSignOnUrl(jti);
    await setSession({ connectorId, redirectUri, state });

    return singleSignOnUrl;
  }

  /**
   * Get social user info.
   *
   * @param connectorSession The connector session data from interaction session storage.
   * @returns The social user info extracted from SAML assertion.
   *
   * @remarks For SAML connector, userInfo will be extracted from the SAML assertion by ACS callback endpoint.
   * This method only asserts the userInfo is not null and directly return it.
   */
  async getUserInfo({ userInfo }: SingleSignOnConnectorSession) {
    assertThat(userInfo, new RequestError('session.connector_session_not_found'));

    return { userInfo };
  }
}

export const samlSsoConnectorFactory: SingleSignOnFactory<SsoProviderName.SAML> = {
  providerName: SsoProviderName.SAML,
  providerType: SsoProviderType.SAML,
  logo: 'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTAgMjAuMDk4OEMwIDE3LjQ2MzIgMi41MTQzNCA3LjYzMDE1IDcuNzEyOTEgMS4wNDEwMkM0LjA0MzMzIDcuOTY4MDYgMS45MDg3MSAxMi42NTgzIDMuNDY1NzEgMTYuMzE0M0M0LjY4ODkxIDE5LjE4NjUgMTAuOTc0OCAxOC4yNzQxIDE0LjM3MjUgMTcuMTI1M0M4LjMyNDUxIDE5Ljg2MjMgMi4xMTc5NCAyMC4yMzQgMCAyMC4wOTg4WiIgZmlsbD0iI0MxMjcyRCIvPgo8cGF0aCBkPSJNMTEuMjYwNSAwLjc1QzEzLjU1NTcgMi4wNjc4MyAyMC44NjEzIDkuMTQ5ODMgMjQgMTYuOTIxN0MxOS44MDI2IDEwLjI5NzcgMTYuNzg1NSA2LjExNDE2IDEyLjgyMzMgNS42MjcxM0M5LjcxMDU1IDUuMjQ0NTIgNy4zNjIxMSAxMS4xMTQ0IDYuNjYzNyAxNC42MTUyQzcuMzA0MjQgOC4wMzc4IDEwLjA4MzggMi41MDY1IDExLjI2MDUgMC43NVoiIGZpbGw9IiNDMTI3MkQiLz4KPHBhdGggZD0iTTIyLjUzMiAyMC4wNzUyQzIwLjIzNjggMjEuMzkzMSAxMC40MTY4IDI0LjE0NDEgMi4wNzk1OSAyMi45NjE0QzkuOTQ2NiAyMi42NTgzIDE1LjA5ODMgMjIuMTUxNiAxNy41MDM1IDE4Ljk4MjZDMTkuMzkzMSAxNi40OTMxIDE1LjQ1NTYgMTEuNTM1NSAxMi43NTYzIDkuMTgzNTlDMTguMTYzOCAxMy4wMjQgMjEuNTkwNyAxOC4xODM2IDIyLjUzMiAyMC4wNzUyWiIgZmlsbD0iI0MxMjcyRCIvPgo8L3N2Zz4K',
  logoDark:
    'data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMjQiIGhlaWdodD0iMjQiIHZpZXdCb3g9IjAgMCAyNCAyNCIgZmlsbD0ibm9uZSIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj4KPHBhdGggZD0iTTAgMjAuMDk4OEMwIDE3LjQ2MzIgMi41MTQzNCA3LjYzMDE1IDcuNzEyOTEgMS4wNDEwMkM0LjA0MzMzIDcuOTY4MDYgMS45MDg3MSAxMi42NTgzIDMuNDY1NzEgMTYuMzE0M0M0LjY4ODkxIDE5LjE4NjUgMTAuOTc0OCAxOC4yNzQxIDE0LjM3MjUgMTcuMTI1M0M4LjMyNDUxIDE5Ljg2MjMgMi4xMTc5NCAyMC4yMzQgMCAyMC4wOTg4WiIgZmlsbD0iI0MxMjcyRCIvPgo8cGF0aCBkPSJNMTEuMjYwNSAwLjc1QzEzLjU1NTcgMi4wNjc4MyAyMC44NjEzIDkuMTQ5ODMgMjQgMTYuOTIxN0MxOS44MDI2IDEwLjI5NzcgMTYuNzg1NSA2LjExNDE2IDEyLjgyMzMgNS42MjcxM0M5LjcxMDU1IDUuMjQ0NTIgNy4zNjIxMSAxMS4xMTQ0IDYuNjYzNyAxNC42MTUyQzcuMzA0MjQgOC4wMzc4IDEwLjA4MzggMi41MDY1IDExLjI2MDUgMC43NVoiIGZpbGw9IiNDMTI3MkQiLz4KPHBhdGggZD0iTTIyLjUzMTkgMjAuMDc1MkMyMC4yMzY3IDIxLjM5MzEgMTAuNDE2NyAyNC4xNDQxIDIuMDc5NDcgMjIuOTYxNEM5Ljk0NjQ4IDIyLjY1ODMgMTUuMDk4MSAyMi4xNTE2IDE3LjUwMzQgMTguOTgyNkMxOS4zOTI5IDE2LjQ5MzEgMTUuNDU1NSAxMS41MzU1IDEyLjc1NjIgOS4xODM1OUMxOC4xNjM2IDEzLjAyNCAyMS41OTA2IDE4LjE4MzYgMjIuNTMxOSAyMC4wNzUyWiIgZmlsbD0iI0MxMjcyRCIvPgo8L3N2Zz4K',
  description: {
    de: 'XML-basierter Open Standard für das Single Sign-On im Web und die Identitätsföderation.',
    en: 'XML-based open standard for web single sign-on and identity federation.',
    es: 'Estándar abierto basado en XML para la autenticación única en la web y la federación de identidades.',
    fr: "Norme ouverte basée sur XML pour la connexion unique sur le web et la fédération d'identité.",
    it: 'Standard aperto basato su XML per il single sign-on e la federazione delle identità web.',
    ja: 'ウェブのシングルサインオンとアイデンティティフェデレーションのためのXMLベースのオープンスタンダード。',
    ko: '웹 단일 로그인 및 ID 페더레이션을 위한 XML 기반 오픈 표준.',
    'pl-PL':
      'Otwarty standard oparty na XML do jednolitej autoryzacji w sieci i federacji tożsamości.',
    'pt-BR':
      'Padrão aberto baseado em XML para autenticação única na web e federação de identidade.',
    'pt-PT':
      'Padrão aberto baseado em XML para autenticação única na web e federação de identidade.',
    ru: 'XML-основаный открытый стандарт для единого входа в систему и федерации идентификации в сети.',
    'tr-TR': 'Web tek oturum açma ve kimlik federasyonu için XML tabanlı açık standart.',
    'zh-CN': '基于 XML 的 Web 单点登录和身份联合的开放标准。',
    'zh-HK': '基於 XML 的 Web 單一登錄和身份聯合的開放標準。',
    'zh-TW': '基於 XML 的 Web 單一登入和身分聯合的開放標準。',
  },
  name: {
    en: 'SAML',
  },
  configGuard: samlConnectorConfigGuard,
  constructor: SamlSsoConnector,
};
