import { type ToZodObject } from '@logto/connector-kit';
import { z } from 'zod';

import { extendedSocialUserInfoGuard, type ExtendedSocialUserInfo } from './saml.js';

/**
 * Single sign on connector session
 *
 * @remark this is a forked version of @logto/connector-kit
 * Simplified the type definition to only include the properties we need.
 * Create additional type guard to validate the session data.
 * @see @logto/connector-kit/types/social.ts
 */
export type SingleSignOnConnectorSession = {
  /** The state generated by Logto experience client. */
  state: string;
  /** The redirect uri for the identity provider. */
  redirectUri: string;
  connectorId: string;
  /**  OIDC only properties, generated by OIDC connector factory, used to verify the identity provider response. */
  nonce?: string;
  /**  User info returned by the identity provider.
   * SAML only properties, parsed from the SAML assertion.
   * We store the assertion in the session storage after receiving it from the identity provider.
   * So the client authentication handler can get it later.
   */
  userInfo?: ExtendedSocialUserInfo;
};

export const singleSignOnConnectorSessionGuard = z.object({
  state: z.string(),
  redirectUri: z.string(),
  connectorId: z.string(),
  nonce: z.string().optional(),
  userInfo: extendedSocialUserInfoGuard.optional(),
}) satisfies ToZodObject<SingleSignOnConnectorSession>;

export type CreateSingleSignOnSession = (storage: SingleSignOnConnectorSession) => Promise<void>;

/**
 * Single sign on interaction identifier result
 *
 * @remark this session data  is used to store the authentication result from the identity provider. {@link /packages/core/src/routes/interaction/utils/single-sign-on.ts}
 * This is needed because we need to split the authentication process into sign in and sign up two parts.
 * If the SSO identity is found in DB we will directly sign in the user.
 * If the SSO identity is not found in DB we will throw an error and let the client to create a new user.
 * In the SSO registration endpoint, we will validate this session data and create a new user accordingly.
 *
 * Deprecated in the experience APIs. We will reply on the SSO verification record to store the authentication result.
 */
export type SingleSignOnInteractionIdentifierResult = {
  connectorId: string;
  issuer: string;
  userInfo: ExtendedSocialUserInfo;
};
export const singleSignOnInteractionIdentifierResultGuard = z.object({
  singleSignOnIdentifier: z.object({
    connectorId: z.string(),
    issuer: z.string(),
    userInfo: extendedSocialUserInfoGuard,
  }),
}) satisfies ToZodObject<{
  singleSignOnIdentifier: SingleSignOnInteractionIdentifierResult;
}>;
