/* eslint-disable max-lines */
import { type DataHookEvent, type GeneratedSchema, type SchemaLike } from '@logto/schemas';
import { generateStandardId } from '@logto/shared';
import { condArray, type DeepPartial, isPlainObject } from '@silverhand/essentials';
import camelcase from 'camelcase';
import deepmerge from 'deepmerge';
import { type Context, type Middleware } from 'koa';
import compose from 'koa-compose';
import Router, { type IRouterParamContext } from 'koa-router';
import { z } from 'zod';

import { type SearchOptions } from '#src/database/utils.js';
import { buildManagementApiContext } from '#src/libraries/hook/utils.js';
import koaGuard from '#src/middleware/koa-guard.js';
import { type WithHookContext } from '#src/middleware/koa-management-api-hooks.js';
import koaPagination from '#src/middleware/koa-pagination.js';

import { type TwoRelationsQueries } from './RelationQueries.js';
import type SchemaQueries from './SchemaQueries.js';
import { parseSearchOptions } from './search.js';

const defaultConfig = Object.freeze({
  disabled: {
    get: false,
    post: false,
    getById: false,
    patchById: false,
    deleteById: false,
  },
  searchFields: [],
  middlewares: [],
});

type RouteMethod = 'get' | 'post' | 'put' | 'delete' | 'patch';

/**
 * Defines the scope where a middleware can be applied in SchemaRouter.
 *
 * - `native`: Apply to native CRUD routes of the schema (e.g., `GET /`, `POST /`, `GET /:id`, `PATCH /:id`, `DELETE /:id`)
 * - `relation`: Apply to relation routes between schemas (e.g., `GET /:id/relations`, `POST /:id/relations`)
 */
type MiddlewareScope = 'native' | 'relation';

type SchemaMiddleware<
  StateT = unknown,
  ContextT extends IRouterParamContext = IRouterParamContext,
  ResponseBodyT = unknown,
> = Middleware<StateT, ContextT, ResponseBodyT>;

type MiddlewareConfig = {
  /** The middleware to apply */
  middleware: SchemaMiddleware;
  /** Define the scope where the middleware will be applied.
   * If not provided, applies to both native and relation routes.
   */
  scope?: MiddlewareScope;
  /**
   * The HTTP methods this middleware applies to.
   * If not provided, applies to all methods.
   */
  method?: RouteMethod[];
  /**
   * Status codes that may be returned by this middleware.
   * These codes will be accepted by the route guard's response validation.
   * If not provided, no additional status codes will be added to the route guard.
   */
  status?: number[];
};

/**
 * Generate the pathname for from a table name.
 *
 * @example
 * ```ts
 * tableToPathname('organization_role') // => 'organization-role'
 * ```
 */
export const tableToPathname = (tableName: string) => tableName.replaceAll('_', '-');

/**
 * Generate the camel case schema ID column name.
 *
 * @example
 * ```ts
 * camelCaseSchemaId({ tableSingular: 'organization' as const }) // => 'organizationId'
 * ```
 *
 */
const camelCaseSchemaId = <T extends { tableSingular: Table }, Table extends string>(schema: T) =>
  `${camelcase(schema.tableSingular)}Id` as const;

type SchemaRouterConfig<Key extends string> = {
  /** Disable certain routes for the router. */
  disabled: {
    /** Disable `GET /` route. */
    get: boolean;
    /** Disable `POST /` route. */
    post: boolean;
    /** Disable `GET /:id` route. */
    getById: boolean;
    /** Disable `PATCH /:id` route. */
    patchById: boolean;
    /** Disable `DELETE /:id` route. */
    deleteById: boolean;
  };
  /** Lifecycle hooks for certain actions. */
  hooks?: {
    /** Triggered after an entity is inserted. */
    afterInsert?: (ctx: Context) => void;
    /** Triggered after an entity is deleted. */
    afterDelete?: (ctx: Context) => void;
  };
  /** Middlewares that are used before creating API routes */
  middlewares?: MiddlewareConfig[];
  /** A custom error handler for the router before throwing the error. */
  errorHandler?: (error: unknown) => void;
  /** The fields that can be searched for the `GET /` route. */
  searchFields: SearchOptions<Key>['fields'];
  /**
   * The length of the ID generated by `generateStandardId()`.
   *
   * @see {@link generateStandardId} for the default length.
   */
  idLength?: number;
  /**
   * The guard for the entity returned by the following routes:
   *
   * - `GET /:id`
   * - `POST /`
   * - `PATCH /:id`
   *
   * If not provided, the `schema.guard` will be used.
   */
  entityGuard?: z.ZodTypeAny;
  /**
   * If the GET route's pagination is optional.
   * @default false
   */
  isPaginationOptional?: boolean;
};

type RelationRoutesConfig = {
  /** The event that should be triggered when the relation is modified. */
  hookEvent?: DataHookEvent;
  /** Disable certain routes for the relation. */
  disabled: {
    /** Disable `GET /:id/[pathname]` route. */
    get: boolean;
  };
  /**
   * If the GET route's pagination is optional.
   * @default false
   */
  isPaginationOptional?: boolean;
};

/**
 * A standard RESTful router for a schema.
 *
 * It provides the following routes by configuring the `actions`:
 *
 * - `GET /`: Get a list of entities.
 * - `POST /`: Create an entity.
 * - `GET /:id`: Get an entity by ID.
 * - `PATCH /:id`: Update an entity by ID.
 * - `DELETE /:id`: Delete an entity by ID.
 *
 * Browse the source code for more details about request/response validation.
 */
export default class SchemaRouter<
  Key extends string,
  CreateSchema extends Partial<SchemaLike<Key> & { id: string }>,
  Schema extends SchemaLike<Key> & { id: string },
  StateT = unknown,
  CustomT extends WithHookContext = WithHookContext,
> extends Router<StateT, CustomT> {
  public readonly config: SchemaRouterConfig<Key>;

  constructor(
    public readonly schema: GeneratedSchema<Key, CreateSchema, Schema>,
    public readonly queries: SchemaQueries<Key, CreateSchema, Schema>,
    config: DeepPartial<SchemaRouterConfig<Key>> = {}
  ) {
    super({ prefix: '/' + tableToPathname(schema.table) });

    this.config = deepmerge(defaultConfig, config, { isMergeableObject: isPlainObject });

    // Apply global middlewares (those without specific scope)
    if (this.config.middlewares?.length) {
      for (const { middleware, scope } of this.config.middlewares) {
        if (!scope) {
          this.use(middleware);
        }
      }
    }

    if (this.config.errorHandler) {
      this.use(async (_, next) => {
        try {
          await next();
        } catch (error: unknown) {
          this.config.errorHandler?.(error);
          throw error;
        }
      });
    }

    this.#addRoutes();
  }

  /**
   * Add routes for relations between the current schema and another schema.
   *
   * The routes are:
   *
   * - `GET /:id/[pathname]`: Get the entities of the relation with pagination.
   * - `POST /:id/[pathname]`: Add entities to the relation.
   * - `PUT /:id/[pathname]`: Replace the entities in the relation.
   * - `DELETE /:id/[pathname]/:relationSchemaId`: Remove an entity from the relation set.
   * The `:relationSchemaId` is the entity ID in the relation schema.
   *
   * The `[pathname]` is determined by the `pathname` parameter.
   *
   * @remarks
   * The `POST /:id/[pathname]` route accepts a JSON body with the following format:
   *
   * ```json
   * { "[relationSchemaIds]": ["id1", "id2", "id3"] }
   * ```
   *
   * The `[relationSchemaIds]` is the camel case of the relation schema's table name in
   * singular form with `Ids` suffix. For example, if the relation schema's table name is
   * `organization_roles`, the `[relationSchemaIds]` will be `organizationRoleIds`.
   *
   * @param relationQueries The queries class for the relation.
   * @param pathname The pathname of the relation. If not provided, it will be
   * the camel case of the relation schema's table name.
   * @see {@link TwoRelationsQueries} for the `relationQueries` configuration.
   */
  addRelationRoutes<
    RelationCreateSchema extends Partial<SchemaLike<string> & { id: string }>,
    RelationSchema extends SchemaLike<string> & { id: string },
  >(
    relationQueries: TwoRelationsQueries<
      typeof this.schema,
      GeneratedSchema<string, RelationCreateSchema, RelationSchema>
    >,
    pathname = tableToPathname(relationQueries.schemas[1].table),
    { disabled, hookEvent, isPaginationOptional }: Partial<RelationRoutesConfig> = {}
  ) {
    const relationSchema = relationQueries.schemas[1];
    const relationSchemaId = camelCaseSchemaId(relationSchema);
    const columns = {
      schemaId: camelCaseSchemaId(this.schema),
      relationSchemaId,
      relationSchemaIds: relationSchemaId + 's',
    };
    const appendHookContext = (ctx: WithHookContext<IRouterParamContext & Context>, id: string) => {
      if (hookEvent) {
        ctx.appendDataHookContext(hookEvent, {
          ...buildManagementApiContext(ctx),
          [columns.schemaId]: id,
        });
      }
    };

    if (!disabled?.get) {
      this.get(
        `/:id/${pathname}`,
        koaPagination({ isOptional: isPaginationOptional }),
        koaGuard({
          params: z.object({ id: z.string().min(1) }),
          response: relationSchema.guard.array(),
          status: this.#collectRouteStatuses('get', [200, 404], 'relation'),
        }),
        this.#assembleQualifiedMiddlewares('get', 'relation'),
        async (ctx, next) => {
          const { id } = ctx.guard.params;

          // Ensure that the main entry exists
          await this.queries.findById(id);

          const [totalCount, entities] = await relationQueries.getEntities(
            relationSchema,
            { [columns.schemaId]: id },
            ctx.pagination.disabled ? undefined : ctx.pagination
          );

          if (!ctx.pagination.disabled) {
            ctx.pagination.totalCount = totalCount;
          }
          ctx.body = entities;
          return next();
        }
      );
    }

    this.post(
      `/:id/${pathname}`,
      koaGuard({
        params: z.object({ id: z.string().min(1) }),
        body: z.object({ [columns.relationSchemaIds]: z.string().min(1).array().nonempty() }),
        status: this.#collectRouteStatuses('post', [201, 422], 'relation'),
      }),
      this.#assembleQualifiedMiddlewares('post', 'relation'),
      async (ctx, next) => {
        const {
          params: { id },
          body: { [columns.relationSchemaIds]: relationIds },
        } = ctx.guard;

        await relationQueries.insert(
          ...(relationIds?.map((relationId) => ({
            [columns.schemaId]: id,
            [columns.relationSchemaId]: relationId,
          })) ?? [])
        );
        ctx.status = 201;
        appendHookContext(ctx, id);
        return next();
      }
    );

    this.put(
      `/:id/${pathname}`,
      koaGuard({
        params: z.object({ id: z.string().min(1) }),
        body: z.object({ [columns.relationSchemaIds]: z.string().min(1).array() }),
        status: this.#collectRouteStatuses('put', [204, 422], 'relation'),
      }),
      this.#assembleQualifiedMiddlewares('put', 'relation'),
      async (ctx, next) => {
        const {
          params: { id },
          body: { [columns.relationSchemaIds]: relationIds },
        } = ctx.guard;

        await relationQueries.replace(id, relationIds ?? []);
        ctx.status = 204;
        appendHookContext(ctx, id);
        return next();
      }
    );

    this.delete(
      `/:id/${pathname}/:${camelCaseSchemaId(relationSchema)}`,
      koaGuard({
        params: z
          .object({ id: z.string().min(1) })
          .extend({ [relationSchemaId]: z.string().min(1) }),
        status: this.#collectRouteStatuses('delete', [204, 422], 'relation'),
      }),
      this.#assembleQualifiedMiddlewares('delete', 'relation'),
      async (ctx, next) => {
        const {
          params: { id, [relationSchemaId]: relationId },
        } = ctx.guard;

        await relationQueries.delete({
          // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- `koaGuard()` ensures the value is not `undefined`
          [columns.schemaId]: id!,
          // eslint-disable-next-line @typescript-eslint/no-non-null-assertion -- `koaGuard()` ensures the value is not `undefined`
          [columns.relationSchemaId]: relationId!,
        });

        ctx.status = 204;
        // eslint-disable-next-line @typescript-eslint/no-non-null-assertion
        appendHookContext(ctx, id!);
        return next();
      }
    );
  }

  #addRoutes() {
    const { queries, schema, config } = this;
    const { disabled, searchFields, idLength, entityGuard, isPaginationOptional } = config;

    if (!disabled.get) {
      this.get(
        '/',
        koaPagination({ isOptional: isPaginationOptional }),
        koaGuard({
          query: z.object({ q: z.string().optional() }),
          response: (entityGuard ?? schema.guard).array(),
          status: this.#collectRouteStatuses('get', [200]),
        }),
        this.#assembleQualifiedMiddlewares('get'),
        async (ctx, next) => {
          const search = parseSearchOptions(searchFields, ctx.guard.query);
          const { limit, offset } = ctx.pagination;
          const [count, entities] = await queries.findAll(limit, offset, search);

          ctx.pagination.totalCount = count;
          ctx.body = entities;
          return next();
        }
      );
    }

    if (!disabled.post) {
      this.post(
        '/',
        koaGuard({
          // @ts-expect-error -- `.omit()` doesn't play well with generics
          body: schema.createGuard.omit({ id: true }),
          response: entityGuard ?? schema.guard,
          status: this.#collectRouteStatuses('post', [201]), // TODO: 409/422 for conflict?
        }),
        this.#assembleQualifiedMiddlewares('post'),
        async (ctx, next) => {
          // eslint-disable-next-line no-restricted-syntax -- `.omit()` doesn't play well with generics
          ctx.body = await queries.insert({
            id: generateStandardId(idLength),
            ...ctx.guard.body,
          } as CreateSchema);
          this.config.hooks?.afterInsert?.(ctx);
          ctx.status = 201;
          return next();
        }
      );
    }

    if (!disabled.getById) {
      this.get(
        '/:id',
        koaGuard({
          params: z.object({ id: z.string().min(1) }),
          response: entityGuard ?? schema.guard,
          status: this.#collectRouteStatuses('get', [200, 404]),
        }),
        this.#assembleQualifiedMiddlewares('get'),
        async (ctx, next) => {
          ctx.body = await queries.findById(ctx.guard.params.id);
          return next();
        }
      );
    }

    if (!disabled.patchById) {
      this.patch(
        '/:id',
        koaGuard({
          params: z.object({ id: z.string().min(1) }),
          body: schema.updateGuard,
          response: entityGuard ?? schema.guard,
          status: this.#collectRouteStatuses('patch', [200, 404]), // TODO: 409/422 for conflict?
        }),
        this.#assembleQualifiedMiddlewares('patch'),
        async (ctx, next) => {
          ctx.body = await queries.updateById(ctx.guard.params.id, ctx.guard.body);
          return next();
        }
      );
    }

    if (!disabled.deleteById) {
      this.delete(
        '/:id',
        koaGuard({
          params: z.object({ id: z.string().min(1) }),
          status: this.#collectRouteStatuses('delete', [204, 404]),
        }),
        this.#assembleQualifiedMiddlewares('delete'),
        async (ctx, next) => {
          await queries.deleteById(ctx.guard.params.id);
          this.config.hooks?.afterDelete?.(ctx);
          ctx.status = 204;
          return next();
        }
      );
    }
  }

  #assembleQualifiedMiddlewares<StateT, ContextT extends IRouterParamContext, ResponseBodyT>(
    method: RouteMethod,
    currentScope: MiddlewareScope = 'native'
  ): Middleware<StateT, ContextT, ResponseBodyT> {
    const pickedMiddlewares: Array<Middleware<StateT, ContextT, ResponseBodyT>> = [];

    for (const middlewareConfig of this.config.middlewares ?? []) {
      // We have dealt with the global middlewares.
      if (
        !middlewareConfig.scope ||
        !this.#shouldUseScopedMiddleware(middlewareConfig, method, currentScope)
      ) {
        continue;
      }

      const typedMiddleware: Middleware<StateT, ContextT, ResponseBodyT> = async (context, next) =>
        // eslint-disable-next-line @typescript-eslint/no-unsafe-return
        middlewareConfig.middleware(context, next);

      // eslint-disable-next-line @silverhand/fp/no-mutating-methods
      pickedMiddlewares.push(typedMiddleware);
    }

    return compose(pickedMiddlewares);
  }

  #collectRouteStatuses(
    method: RouteMethod,
    baseStatuses: number[],
    currentScope: MiddlewareScope = 'native'
  ): number[] | undefined {
    const statusSet = new Set<number>(baseStatuses);

    for (const middlewareConfig of this.config.middlewares ?? []) {
      // Skip middlewares that will not run for the current route
      if (!this.#shouldUseScopedMiddleware(middlewareConfig, method, currentScope)) {
        continue;
      }

      for (const code of condArray(middlewareConfig.status)) {
        statusSet.add(code);
      }
    }

    return statusSet.size > 0 ? [...statusSet] : undefined;
  }

  #shouldUseScopedMiddleware(
    config: Pick<MiddlewareConfig, 'scope' | 'method'>,
    currentMethod: RouteMethod,
    currentScope: MiddlewareScope
  ): boolean {
    const { scope, method } = config;

    // If scope is specified, check if it matches the current scope
    if (scope !== undefined && scope !== currentScope) {
      return false;
    }

    // If method is specified, check if it includes the current method
    if (method !== undefined && !method.includes(currentMethod)) {
      return false;
    }

    // If we reach here, the middleware should be used
    return true;
  }
}
/* eslint-enable max-lines */
