// LOG-88: Refactor '@logto/schemas' type gen

import { condArray, condString, conditionalString } from '@silverhand/essentials';
import camelcase from 'camelcase';
import pluralize from 'pluralize';

import type { TableWithType } from './types.js';

const createTypeRemark = (originalType: string) => [
  '',
  '@remarks This is a type for database creation.',
  '@see {@link ' + originalType + '} for the original type.',
];

// Tenant ID should be optional for create types since it'll be generated by the database trigger
const tenantId = 'tenant_id';

type PrintCommentsOptions = {
  tabSize?: number;
  newLine?: boolean;
};
const printComments = (
  comments?: string | string[],
  { tabSize = 2, newLine = true }: PrintCommentsOptions = {}
) =>
  condString(
    comments &&
      condArray<string>(
        ' '.repeat(tabSize),
        Array.isArray(comments)
          ? ['/**', ...comments.map((comment) => ` * ${comment}`), ' */'].join(
              '\n' + ' '.repeat(tabSize)
            )
          : `/** ${comments} */`,
        newLine && '\n'
      ).join('')
  );

export const generateSchema = ({ name, comments, fields }: TableWithType) => {
  const modelName = pluralize(camelcase(name, { pascalCase: true }), 1);
  const databaseEntryType = `Create${modelName}`;

  return [
    printComments([...condArray(comments), ...createTypeRemark(modelName)], {
      tabSize: 0,
      newLine: false,
    }),
    `export type ${databaseEntryType} = {`,
    ...fields.map(
      ({ name, comments, type, isArray, nullable, hasDefaultValue }) =>
        printComments(comments) +
        `  ${camelcase(name)}${conditionalString(
          (nullable || hasDefaultValue || name === tenantId) && '?'
        )}: ${type}${conditionalString(isArray && '[]')}${conditionalString(
          nullable && ' | null'
        )};`
    ),
    '};',
    '',
    printComments(comments, { tabSize: 0, newLine: false }),
    `export type ${modelName} = {`,
    ...fields.map(
      ({ name, comments, type, isArray, nullable, hasDefaultValue }) =>
        printComments(comments) +
        `  ${camelcase(name)}: ${type}${conditionalString(isArray && '[]')}${
          nullable ? ' | null' : ''
        };`
    ),
    '};',
    '',
    `export type ${modelName}Keys = ${fields
      .map(({ name }) => `'${camelcase(name)}'`)
      .join(' | ')};`,
    '',
    `const createGuard: Guard<${databaseEntryType}> = z.object({`,

    ...fields.map(
      // eslint-disable-next-line complexity
      ({ name, type, isArray, isEnum, nullable, hasDefaultValue, tsType, isString, maxLength }) => {
        if (tsType) {
          return `  ${camelcase(name)}: ${camelcase(tsType)}Guard${conditionalString(
            nullable && '.nullable()'
          )}${conditionalString((nullable || hasDefaultValue) && '.optional()')},`;
        }

        return `  ${camelcase(name)}: z.${isEnum ? `nativeEnum(${type})` : `${type}()`}${
          // Non-nullable strings should have a min length of 1
          conditionalString(isString && !(nullable || name === tenantId) && `.min(1)`)
        }${
          // String types value in DB should have a max length
          conditionalString(isString && maxLength && `.max(${maxLength})`)
        }${conditionalString(isArray && '.array()')}${conditionalString(
          nullable && '.nullable()'
        )}${conditionalString(
          (nullable || hasDefaultValue || name === tenantId) && '.optional()'
        )},`;
      }
    ),
    '});',
    '',
    `const guard: Guard<${modelName}> = z.object({`,

    ...fields.map(
      // eslint-disable-next-line complexity
      ({ name, type, isArray, isEnum, nullable, tsType, isString, maxLength, hasDefaultValue }) => {
        if (tsType) {
          return `  ${camelcase(name)}: ${camelcase(tsType)}Guard${conditionalString(
            nullable && '.nullable()'
          )},`;
        }

        return `  ${camelcase(name)}: z.${isEnum ? `nativeEnum(${type})` : `${type}()`}${
          // Non-nullable strings should have a min length of 1
          conditionalString(
            isString && !(nullable || hasDefaultValue || name === tenantId) && `.min(1)`
          )
        }${
          // String types value in DB should have a max length
          conditionalString(isString && maxLength && `.max(${maxLength})`)
        }${conditionalString(isArray && '.array()')}${conditionalString(
          nullable && '.nullable()'
        )},`;
      }
    ),
    '  });',
    '',
    `export const ${camelcase(name, {
      pascalCase: true,
    })}: GeneratedSchema<
      ${modelName}Keys,
      ${databaseEntryType},
      ${modelName},
      '${name}',
      '${pluralize(name, 1)}'
    > = Object.freeze({`,
    `  table: '${name}',`,
    `  tableSingular: '${pluralize(name, 1)}',`,
    '  fields: {',
    ...fields.map(({ name }) => `    ${camelcase(name)}: '${name}',`),
    '  },',
    '  fieldKeys: [',
    ...fields.map(({ name }) => `    '${camelcase(name)}',`),
    '  ] as const,',
    '  createGuard,',
    '  guard,',
    '  updateGuard: guard.partial(),',
    '});',
  ].join('\n');
};
