import { assertEnv, getEnv, getEnvAsStringArray, tryThat, yes } from '@silverhand/essentials';

import UrlSet from './UrlSet.js';
import { throwErrorWithDsnMessage } from './throw-errors.js';

export default class GlobalValues {
  public readonly isProduction = getEnv('NODE_ENV') === 'production';
  public readonly isIntegrationTest = yes(getEnv('INTEGRATION_TEST'));
  public readonly isUnitTest = getEnv('NODE_ENV') === 'test';
  public readonly isDevFeaturesEnabled = !this.isProduction || yes(getEnv('DEV_FEATURES_ENABLED'));

  public readonly httpsCert = process.env.HTTPS_CERT_PATH;
  public readonly httpsKey = process.env.HTTPS_KEY_PATH;
  public readonly isHttpsEnabled = Boolean(this.httpsCert && this.httpsKey);

  /**
   * The UrlSet with no prefix for Logto core service. It serves requests to the OIDC Provider and Management APIs
   * from all tenants.
   *
   * Especially, a glob (`*`) is allowed for the hostname of its property `endpoint` to indicate if the domain-based multi-tenancy (DBMT)
   * is enabled which affects some critical behaviors of Logto.
   *
   * **When DBMT is enabled**
   *
   * - For non-admin tenants, tenant endpoint will be generated by replacing the glob in the `urlSet.endpoint`.
   * - For admin tenant, if `adminUrlSet` has no endpoint available, tenant endpoint will be generated by replacing the glob in the `urlSet.endpoint`.
   * - Admin Console will NOT be served under admin tenant since the cloud service will do.
   * - Incoming requests will use glob matching to parse the tenant ID from the request URL.
   *
   * ```ts
   * // ENDPOINT='https://*.domain.com'
   * getTenantEndpoint('foo') => 'https://foo.domain.com'
   * ```
   *
   * **When DBMT is disabled**
   *
   * - For non-admin tenants, tenant endpoint will always be `urlSet.endpoint`.
   * - For admin tenant, tenant endpoint will always be `adminUrlSet.endpoint`.
   * - Admin Console will be served under admin tenant.
   * - Incoming requests will check whether the URL matches adminUrlSet.endpoint, which indicates the admin tenant ID. If there is no match, the default tenant ID will be used.
   */
  public readonly urlSet = new UrlSet(this.isHttpsEnabled, 3001);
  /**
   * The UrlSet with prefix `ADMIN_` for Logto admin tenant. To completely disable it, set `ADMIN_DISABLE_LOCALHOST` to a truthy value and leave `ADMIN_ENDPOINT` unset.
   *
   * Should be disabled on the cloud.
   *
   * @see urlSet For mutual effects between these two sets.
   */
  public readonly adminUrlSet = new UrlSet(this.isHttpsEnabled, 3002, 'ADMIN_');
  /**
   * The UrlSet with prefix `CLOUD_` for Logto cloud service. It affects Admin Console Redirect URIs and some CORS configuration.
   */
  public readonly cloudUrlSet = new UrlSet(this.isHttpsEnabled, 3003, 'CLOUD_');

  /** @see urlSet For detailed explanation. */
  public readonly isDomainBasedMultiTenancy = this.urlSet.endpoint.hostname.includes('*');

  /**
   * **NOTE: This is an internal dev-only feature.**
   *
   * This value indicates path-based multi-tenancy (PBMT) is enabled by setting env variable `PATH_BASED_MULTI_TENANCY` to a truthy value.
   *
   * Note the value will always be `false` if domain-based multi-tenancy is enabled.
   *
   * **When PBMT is enabled**
   *
   * - For non-admin tenants, tenant endpoint will be generated by appending the tenant ID to `urlSet.endpoint`.
   * - For admin tenant, if `adminUrlSet` has no endpoint available, tenant endpoint will be generated by appending the tenant ID to `urlSet.endpoint`.
   * - Admin Console will NOT be served under admin tenant since the cloud service will do.
   * - Incoming requests will try to match the position of pathname segments of the URLs in `urlSet.deduplicated()` to parse the tenant ID from the request URL.
   *
   * ```ts
   * // ENDPOINT='https://domain.com/foo'
   * getTenantEndpoint('bar') => 'https://domain.com/foo/bar'
   * matchTenantId('https://domain.com/foo/bar') => 'bar'
   * matchTenantId('http://localhost:3001/foo/bar') => 'foo'
   * ```
   *
   * @see urlSet
   */
  public readonly isPathBasedMultiTenancy =
    !this.isDomainBasedMultiTenancy && yes(getEnv('PATH_BASED_MULTI_TENANCY'));

  /** Alias for `isDomainBasedMultiTenancy || isPathBasedMultiTenancy`. */
  public get isMultiTenancy(): boolean {
    return this.isDomainBasedMultiTenancy || this.isPathBasedMultiTenancy;
  }

  /** If the env explicitly indicates it's in the cloud environment. */
  public readonly isCloud = yes(getEnv('IS_CLOUD'));

  /**
   * Indicates whether this Logto instance supports multiple custom domains.
   *
   * **NOTE: Only available to enterprise customers running private instances that need this feature.**
   *
   * Controlled by the `MULTIPLE_CUSTOM_DOMAINS_ENABLED` environment variable. When enabled, the instance
   * can operate with multiple custom domains across both development and production tenants.
   */
  public readonly isMultipleCustomDomainsEnabled = yes(getEnv('MULTIPLE_CUSTOM_DOMAINS_ENABLED'));

  // eslint-disable-next-line unicorn/consistent-function-scoping
  public readonly databaseUrl = tryThat(() => assertEnv('DB_URL'), throwErrorWithDsnMessage);
  public readonly developmentTenantId = getEnv('DEVELOPMENT_TENANT_ID');
  /** @deprecated Use the built-in user default role configuration (`Roles.isDefault`) instead. */
  public readonly userDefaultRoleNames = getEnvAsStringArray('USER_DEFAULT_ROLE_NAMES');
  public readonly developmentUserId = getEnv('DEVELOPMENT_USER_ID');
  public readonly trustProxyHeader = yes(getEnv('TRUST_PROXY_HEADER'));
  public readonly ignoreConnectorVersionCheck = yes(getEnv('IGNORE_CONNECTOR_VERSION_CHECK'));

  /** Maximum number of tenants to keep in the tenant pool. */
  public readonly tenantPoolSize = Number(getEnv('TENANT_POOL_SIZE', '100'));
  /** Maximum number of clients to keep in a single database pool (i.e. per `Tenant` class). */
  public readonly databasePoolSize = Number(getEnv('DATABASE_POOL_SIZE', '20'));

  public readonly databaseConnectionTimeout = Number(getEnv('DATABASE_CONNECTION_TIMEOUT', '5000'));

  /** Global switch for enabling/disabling case-sensitive usernames. */
  public readonly isCaseSensitiveUsername = yes(getEnv('CASE_SENSITIVE_USERNAME', 'true'));

  /**
   * The API key for status endpoint protection. If it's set, requests to the status endpoint may
   * supply the key in the header for receiving response with additional details.
   *
   * @optional
   */
  public readonly statusApiKey = getEnv('STATUS_API_KEY');

  /** The write-only key for PostHog integration. */
  public readonly posthogPublicKey = process.env.POSTHOG_PUBLIC_KEY;
  /** The PostHog host URL for SDK to send events to. */
  public readonly posthogPublicHost = process.env.POSTHOG_PUBLIC_HOST;

  /**
   * The Redis endpoint (optional). If it's set, the central cache mechanism will be automatically enabled.
   *
   * You can set it to a truthy value like `true` or `1` to enable cache with the default Redis URL.
   */
  public readonly redisUrl = getEnv('REDIS_URL');

  public get dbUrl(): string {
    return this.databaseUrl;
  }

  public get endpoint(): URL {
    return this.urlSet.endpoint;
  }

  /**
   * For cloud use only.
   * Define regional Azure function app endpoint and key to enable the Logto Azure Functions integration.
   * This is the prerequisite of the calling on `@logto/azure-functions`.
   */
  public get azureFunctionAppEndpoint() {
    return getEnv('AZURE_FUNCTION_APP_ENDPOINT');
  }

  public get azureFunctionAppKey() {
    return getEnv('AZURE_FUNCTION_APP_KEY');
  }

  /**
   * The key encryption key (KEK) for the secret vault.
   * It is used to encrypt and decrypt secret DEKs (data encryption keys) in the secret vault.
   */
  public get secretVaultKek() {
    return getEnv('SECRET_VAULT_KEK');
  }

  constructor() {
    if (this.isPathBasedMultiTenancy) {
      console.warn(
        '\n****** LOGTO WARNING ******\n\n' +
          'Path-based multi-tenancy is an internal dev-only feature. It is unstable and not intended for production use.\n\n' +
          'Known issues:\n\n' +
          '- Sign-in experience is unavailable in user tenants.\n' +
          '- The Admin Console may display incorrect user endpoints on multiple pages, such as guide, config, etc.' +
          ' This issue is caused by the native URL constructor new URL(), which overrides the base pathname.\n\n' +
          '****** END LOGTO WARNING ******\n'
      );
    }
  }
}
