package com.luohuo.flex.ai.controller.model;

import cn.hutool.core.util.ObjUtil;
import com.luohuo.flex.ai.common.pojo.PageResult;
import com.luohuo.flex.ai.controller.model.vo.model.AiModelPageReqVO;
import com.luohuo.flex.ai.controller.model.vo.model.AiModelRespVO;
import com.luohuo.flex.ai.controller.model.vo.model.AiModelSaveMyReqVO;
import com.luohuo.flex.ai.controller.model.vo.model.AiModelSaveReqVO;
import com.luohuo.flex.ai.controller.model.vo.model.AiModelSimpleRespVO;
import com.luohuo.flex.ai.dal.model.AiModelDO;
import com.luohuo.flex.ai.enums.CommonStatusEnum;
import com.luohuo.flex.ai.service.model.AiModelService;
import com.luohuo.flex.ai.service.model.AiModelUsageService;
import com.luohuo.flex.ai.utils.BeanUtils;
import com.luohuo.basic.base.R;
import com.luohuo.basic.context.ContextUtil;
import io.swagger.v3.oas.annotations.Operation;
import io.swagger.v3.oas.annotations.Parameter;
import io.swagger.v3.oas.annotations.tags.Tag;
import jakarta.annotation.Resource;
import jakarta.validation.Valid;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.*;

import java.util.List;

import static com.luohuo.basic.base.R.success;
import static com.luohuo.basic.utils.collection.CollectionUtils.convertList;

@Tag(name = "管理后台 - AI 模型")
@RestController
@RequestMapping("/model")
@Validated
public class AiModelController {

	@Resource
	private AiModelService modelService;

	@Resource
	private AiModelUsageService modelUsageService;

	@PostMapping("/create")
	@Operation(summary = "创建模型")
	public R<Long> createModel(@Valid @RequestBody AiModelSaveReqVO createReqVO) {
		if (createReqVO.getPublicStatus() == null || Integer.valueOf(1).equals(createReqVO.getPublicStatus())) {
			AiModelSaveMyReqVO myReqVO = BeanUtils.toBean(createReqVO, AiModelSaveMyReqVO.class);
			return success(modelService.createModelMy(myReqVO, ContextUtil.getUid()));
		}
		return success(modelService.createModel(createReqVO));
	}

	@PutMapping("/update")
	@Operation(summary = "更新模型")
	public R<Boolean> updateModel(@Valid @RequestBody AiModelSaveReqVO updateReqVO) {
		AiModelDO model = modelService.getModel(updateReqVO.getId());
		if (model == null) {
			return success(false);
		}

		Long uid = ContextUtil.getUid();
		if (ObjUtil.notEqual(model.getUserId(), uid)) {
			return success(false);
		}

		AiModelSaveMyReqVO myReqVO = BeanUtils.toBean(updateReqVO, AiModelSaveMyReqVO.class);
		modelService.updateModelMy(myReqVO, ContextUtil.getUid());
		return success(true);
	}

	@DeleteMapping("/delete")
	@Operation(summary = "删除模型")
	@Parameter(name = "id", description = "编号", required = true)
	public R<Boolean> deleteModel(@RequestParam("id") String id) {
		Long modelId = Long.parseLong(id);
		AiModelDO model = modelService.getModel(modelId);
		if (model == null) {
			return success(false);
		}

		Long uid = ContextUtil.getUid();
		if (ObjUtil.notEqual(model.getUserId(), uid)) {
			return success(false);
		}

		modelService.deleteModelMy(modelId, uid);
		return success(true);
	}

	@GetMapping("/get")
	@Operation(summary = "获得模型")
	@Parameter(name = "id", description = "编号", required = true, example = "1024")
	public R<AiModelRespVO> getModel(@RequestParam("id") Long id) {
		AiModelDO model = modelService.getModel(id);
		return success(BeanUtils.toBean(model, AiModelRespVO.class));
	}

	@GetMapping("/get-remaining-usage")
	@Operation(summary = "获得模型剩余使用次数")
	@Parameter(name = "id", description = "模型编号", required = true, example = "1024")
	public R<Integer> getModelRemainingUsage(@RequestParam("id") Long id) {
		return success(modelUsageService.getRemainingUsageCount(ContextUtil.getUid(), id));
	}

	@GetMapping("/page")
	@Operation(summary = "获得模型分页（包含系统公开模型和用户私有模型）")
	public R<PageResult<AiModelRespVO>> getModelPage(@Valid AiModelPageReqVO pageReqVO) {
		PageResult<AiModelDO> pageResult = modelService.getModelMyPage(pageReqVO, ContextUtil.getUid());
		return success(BeanUtils.toBean(pageResult, AiModelRespVO.class));
	}

	@GetMapping("/simple-list")
	@Operation(summary = "获得模型列表（包含系统公开模型和用户私有模型）")
	@Parameter(name = "type", description = "类型", required = true, example = "1")
	@Parameter(name = "platform", description = "平台", example = "midjourney")
	public R<List<AiModelRespVO>> getModelSimpleList(@RequestParam("type") Integer type, @RequestParam(value = "platform", required = false) String platform) {
		List<AiModelDO> list = modelService.getModelListByStatusAndTypeAndUserId(CommonStatusEnum.ENABLE.getStatus(), type, platform, ContextUtil.getUid());
		return success(convertList(list, model -> new AiModelRespVO().setId(model.getId())
				.setName(model.getName()).setModel(model.getModel()).setUserId(model.getUserId()).setPlatform(model.getPlatform())));
	}

	@GetMapping("/admin/all-list")
	@Operation(summary = "获得所有模型列表（后台管理专用）")
	public R<List<AiModelSimpleRespVO>> getAllModelList() {
		List<AiModelDO> list = modelService.getAllModelList();
		return success(convertList(list, model -> {
			AiModelSimpleRespVO vo = new AiModelSimpleRespVO();
			vo.setId(model.getId());
			vo.setName(model.getName());
			vo.setModel(model.getModel());
			vo.setPlatform(model.getPlatform());
			return vo;
		}));
	}

	@GetMapping("/admin/page")
	@Operation(summary = "获得所有模型分页（后台管理专用）")
	public R<PageResult<AiModelRespVO>> getAdminModelPage(@Valid AiModelPageReqVO pageReqVO) {
		PageResult<AiModelDO> pageResult = modelService.getModelPage(pageReqVO);
		return success(BeanUtils.toBean(pageResult, AiModelRespVO.class));
	}

	@PutMapping("/admin/update")
	@Operation(summary = "管理员更新模型")
	public R<Boolean> updateModelAdmin(@Valid @RequestBody AiModelSaveReqVO updateReqVO) {
		modelService.updateModelAdmin(updateReqVO);
		return success(true);
	}

	@DeleteMapping("/admin/delete")
	@Operation(summary = "管理员删除模型")
	@Parameter(name = "id", description = "编号", required = true)
	public R<Boolean> deleteModelAdmin(@RequestParam("id") Long id) {
		modelService.deleteModelAdmin(id);
		return success(true);
	}

}