# 6.BeanFactory后置处理器

作用：为BeanFactory提供一些扩展，比如补充一些bean的定义。

其他相关类等详情可以查看

https://github.com/geekibli/demo-spring/blob/bef20693e8bf27466c4347ec61f0ae95cf5f1ed1/src/main/java/com/example/testspring/a05

## 6.1 从一个例子开始

首先看一段代码：

```java
public class A05Application {
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);

        context.refresh();

        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }
        
        context.close();
    }
}
```

查看运行结果：

```
20:33:17.674 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Refreshing org.springframework.context.support.GenericApplicationContext@21588809
20:33:17.735 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'config'
name : {}config
20:33:17.812 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Closing org.springframework.context.support.GenericApplicationContext@21588809, started on Wed Jan 18 20:33:17 CST 2023
```

发现只有一个我们手动注册的一个bean。

config类如下：

```java
@Configuration
@ComponentScan("com.example.testspring.a05.Component")
public class Config {

    @Bean
    public Bean1 bean1(){
        return new Bean1();
    }
    
}
```

下面向beanFactory中添加一个处理器，再次运行：

```java
context.registerBean(ConfigurationClassPostProcessor.class);
```

 运行结果：

```
name : config
name : org.springframework.context.annotation.ConfigurationClassPostProcessor
name : bean2
name : bean1
```

## 6.2 ConfigurationClassPostProcessor作用

ConfigurationClassPostProcessor可以识别@ComponentScan， @Bean , @Import , @ImportResource注解。

## 6.3 MapperScaner

如果是SpringBoot可以使用@MapperScaner注解，通过basePackage属性，执行要扫描的mapper文件的路径。

下面是以后置处理器的方式，可以手动设置一些参数，也可以实现mapper文件扫描。

![](https://oscimg.oschina.net/oscnet/up-d28e1f82285d333e8c372632a869abfbccc.png)



## 6.4 ConfigurationClassPostProcessor的实现逻辑

下面是模拟的方式实现@ComponentScaner的实现逻辑：

首先是扫描逻辑：

```java
public class A05Application {
    @SneakyThrows
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);
//        context.registerBean(ConfigurationClassPostProcessor.class);

        ComponentScan componentScan = AnnotationUtils.findAnnotation(Config.class, ComponentScan.class);
        if (componentScan != null) {
            String[] strings = componentScan.basePackages();
            for (String name : strings ) {
                System.out.println("name " + name);
                // com.example.testspring.a05.Component ---> classpath*:com/example/testspring/a05/Component/**/*.class
                String replace = name.replace(".", "/");

                String path = "classpath*:" + replace + "/**/*.class";
                System.out.println("path : " + path);

                CachingMetadataReaderFactory factory = new CachingMetadataReaderFactory();

                // 这块注意的是，扫描的路径是 /target/classes 路径下的，二进制字节码文件
                // 比如： /Users/lei.gao/Documents/demos/test-spring/target/classes/com/example/testspring/a05/Component/Bean3.class
                Resource[] resources = context.getResources(path);
                for (Resource resource : resources) {
                    System.out.println("resource " + resource);
                    // 能扫描到路径下所有的类，不管有没有添加@Component注解

                    MetadataReader reader = factory.getMetadataReader(resource);
                    System.out.println("classname : "  + reader.getClassMetadata().getClassName());

                    // 如果bean添加的是Component的派生注解，比如controller，下面这种方式还是false
                    System.out.println("是否加了Component : "  + reader.getAnnotationMetadata().hasAnnotation(Component.class.getName()));

                    // 如果是@Component的派生注解，比如controller，是可以扫描到的。
                    System.out.println("是否加了Component派生 : "  + reader.getAnnotationMetadata().hasMetaAnnotation(Component.class.getName()));
                }
            }
        }

        context.refresh();
        context.close();
    }
}
```

**执行结果：**

```
name com.example.testspring.a05.Component
path : classpath*:com/example/testspring/a05/Component/**/*.class
resource file [/Users/lei.gao/Documents/demos/test-spring/target/classes/com/example/testspring/a05/Component/Bean2.class]
classname : com.example.testspring.a05.Component.Bean2
是否加了Component : true
是否加了Component派生 : false
resource file [/Users/lei.gao/Documents/demos/test-spring/target/classes/com/example/testspring/a05/Component/Bean3.class]
classname : com.example.testspring.a05.Component.Bean3
是否加了Component : false
是否加了Component派生 : true
resource file [/Users/lei.gao/Documents/demos/test-spring/target/classes/com/example/testspring/a05/Component/Bean4.class]
classname : com.example.testspring.a05.Component.Bean4
是否加了Component : false
是否加了Component派生 : false
21:12:39.806 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Refreshing org.springframework.context.support.GenericApplicationContext@21588809
21:12:39.852 [main] DEBUG org.springframework.beans.factory.support.DefaultListableBeanFactory - Creating shared instance of singleton bean 'config'
21:12:39.914 [main] DEBUG org.springframework.context.support.GenericApplicationContext - Closing org.springframework.context.support.GenericApplicationContext@21588809, started on Wed Jan 18 21:12:39 CST 2023
```



扫描到对应的bean之后就是生成对应的BeanDefinition，然后添加到beanFactory中，

[**视频链接**](https://www.bilibili.com/video/BV1P44y1N7QG?p=22&spm_id_from=pageDriver&vd_source=3ff1db20d26ee8426355e893ae553d51)

大致代码如下：

```java
 // 如果bean添加的是Component的派生注解，比如controller，下面这种方式还是false
 boolean isComponent = reader.getAnnotationMetadata().hasAnnotation(Component.class.getName());
 System.out.println("是否加了Component : "  + isComponent);

// 如果是@Component的派生注解，比如controller，是可以扫描到的。
boolean isMetaComponent = reader.getAnnotationMetadata().hasMetaAnnotation(Component.class.getName());
System.out.println("是否加了Component派生 : "  + isMetaComponent);

if (isComponent || isMetaComponent) {

AbstractBeanDefinition beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(reader.getClassMetadata().getClassName()).getBeanDefinition();

String beanName = generator.generateBeanName(beanDefinition, context.getDefaultListableBeanFactory());

context.getDefaultListableBeanFactory().registerBeanDefinition(beanName, beanDefinition);

}
```

看一下打印结果：

```
name : config
name : bean2
name : bean3
```



### 6.5 封装后置处理器功能

```java
public class ComponentScanPostProcessor implements BeanFactoryPostProcessor {

    @Override
    public void postProcessBeanFactory(ConfigurableListableBeanFactory configurableListableBeanFactory) throws BeansException {
        try {
            ComponentScan componentScan = AnnotationUtils.findAnnotation(Config.class, ComponentScan.class);
            if (componentScan != null) {
                String[] strings = componentScan.basePackages();
                for (String name : strings) {
                    System.out.println("name " + name);
                    // com.example.testspring.a05.Component ---> classpath*:com/example/testspring/a05/Component/**/*.class
                    String replace = name.replace(".", "/");

                    String path = "classpath*:" + replace + "/**/*.class";
                    System.out.println("path : " + path);

                    CachingMetadataReaderFactory factory = new CachingMetadataReaderFactory();

                    AnnotationBeanNameGenerator generator = new AnnotationBeanNameGenerator();

                    // 这块注意的是，扫描的路径是 /target/classes 路径下的，二进制字节码文件
                    // 比如： /Users/lei.gao/Documents/demos/test-spring/target/classes/com/example/testspring/a05/Component/Bean3.class
                    Resource[] resources = new PathMatchingResourcePatternResolver().getResources(path);
                    for (Resource resource : resources) {
                        System.out.println("resource " + resource);
                        // 能扫描到路径下所有的类，不管有没有添加@Component注解

                        MetadataReader reader = factory.getMetadataReader(resource);
                        System.out.println("classname : " + reader.getClassMetadata().getClassName());

                        // 如果bean添加的是Component的派生注解，比如controller，下面这种方式还是false
                        boolean isComponent = reader.getAnnotationMetadata().hasAnnotation(Component.class.getName());
                        System.out.println("是否加了Component : " + isComponent);

                        // 如果是@Component的派生注解，比如controller，是可以扫描到的。
                        boolean isMetaComponent = reader.getAnnotationMetadata().hasMetaAnnotation(Component.class.getName());
                        System.out.println("是否加了Component派生 : " + isMetaComponent);

                        if (isComponent || isMetaComponent) {

                            AbstractBeanDefinition beanDefinition = BeanDefinitionBuilder.genericBeanDefinition(reader.getClassMetadata().getClassName()).getBeanDefinition();

                            if (configurableListableBeanFactory instanceof DefaultListableBeanFactory) {
                                BeanDefinitionRegistry beanFactory = (BeanDefinitionRegistry) configurableListableBeanFactory;
                                String beanName = generator.generateBeanName(beanDefinition, beanFactory);
                                beanFactory.registerBeanDefinition(beanName, beanDefinition);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            System.err.println(e);
        }
    }
}
```

上面是自定义的postProcessor，那么如何使用呢，和之前的方式是一样的：

```java
public class A05Application {
    @SneakyThrows
    public static void main(String[] args) {
        GenericApplicationContext context = new GenericApplicationContext();
        context.registerBean("config", Config.class);
//        context.registerBean(ConfigurationClassPostProcessor.class);
      
       // 上面自己封装PostProcessor
        context.registerBean(ComponentScanPostProcessor.class);

        context.refresh();

        for (String name : context.getBeanDefinitionNames()) {
            System.out.println("name : " + name);
        }

        context.close();
    }
}
```

