/* SPDX-License-Identifier: GPL-2.0 */
/**
* \file adi_adrv9025_dynamic_slicer.h
* \brief Contains ADRV9025 function prototypes for adi_adrv9025_dynamic_slicer.c
*
* ADRV9025 API Version: 7.0.0.14
*/

/**
* Copyright 2015 - 2020 Analog Devices Inc.
* Released under the ADRV9025 API license, for more information
* see the "LICENSE.pdf" file in this zip file.
*/

#ifndef _ADI_ADRV9025_DYNAMIC_SLICER_H_
#define _ADI_ADRV9025_DYNAMIC_SLICER_H_

#include "adi_adrv9025_dynamic_slicer_types.h"

#ifdef __cplusplus
extern "C" {
#endif

/**
* \brief Configures the dynamic slicer mode of operation for Rx data
*
* The slicer works in conjunction with the gain compensation module.
* The digital gain compensation module compensates for the front end attenuation
* thereby increasing the bit resolution of the Rx data word. The slicer is then
* applied to attenuate the data such that it can fit into the JESD data path resolution.
*
* Dynamic slicer configuration allows the receiver data to be sent on all bits of the 12-bit/16-bit
* JESD data unless there is a gain change. At every gain change, the new slicer configuration is
* transmitted. The dynamic slicer data is 56 bits long transmitted across 28 samples of I and Q JESD data
* which includes 32 bits of user programmable sync header for identification,
* 16 bits consisting of an AGC sync bit and the slicer control word followed by 8 CRC bits.
* The dynamic slicer data transmitted over the JESD I and Q lines are shown below
*
*  I Data (consisting of odd bits of the 56 bit dynamic slicer data):
*  S31,S29,S27,S25,S23,S21,S19,S17,S15,S13,S11,S9,S7,S5,S3,S1,SYNC,0,0,0,0,0,SL3,SL1,C7,C5,C3,C1 (LSB - Transmitted first)
*
*  Q Data (consisting of the even bits of the 56 bit dynamic slicer data):
*  S30,S28,S26,S24,S22,S20,S18,S16,S14,S12,S10,S8,S6,S4,S2,S0,0,0,0,0,0,0,SL2,SL0,C6,C4,C2,C0 (LSB - Transmitted first)
*
* Where S31-S0 are the user programmable sync header bits, SYNC - AGC sync bit, SL3-SL0 is the slicer control word
* and C7-C0 is the CRC generated by applying the the CRC generator polynomial x^8 + x^2 + x + 1 on the LSB 16 bits
* of the 56 bit dynamic slicer data which includes {SYNC,0,0,0,0,0,0,0,0,0,0,0,SL3,SL2,SL1,SL0}.
*
* Please note that enabling the dynamic slicer mode of operation enables the gain compensation on the device.
*
* Please note that while embedding 1 dynamic slicer data bit at LSB or MSB-1 posn is allowed for signed magnitude data,
* embedding of 1 dynamic slicer data bit at LSB is the only allowed option for 2s complement data.
*
* \dep_begin
* \dep{device->halDevInfo}
* \dep_end
*
* \param device Pointer to the device settings structure
* \param rxDynamicSlicerCfg Array of dynamic slicer configuration structures
* \param arraySize No. of elements in the rxDynamicSlicerCfg array
*
* \retval ADI_ADRV9025_WARN_RESET_LOG Recovery action for log reset
* \retval ADI_ADRV9025_ERR_CHECK_PARAM Recovery action for bad parameter check
* \retval ADI_ADRV9025_ERR_RESET_SPI Recovery action for SPI reset required
* \retval ADI_ADRV9025_NO_ACTION Function completed successfully, no action required
*/
int32_t adi_adrv9025_RxDynamicSlicerConfigSet(adi_adrv9025_Device_t*               device,
                                              adi_adrv9025_RxDynamicSlicerConfig_t rxDynamicSlicerCfg[],
                                              uint8_t                              arraySize);

/**
* \brief Retrieves the dynamic slicer configuration settings from the device
*
* Reads the current status of the dynamic slicer configuration from the device
*
* \dep_begin
* \dep{device->halDevInfo}
* \dep_end
*
* \param device Pointer to the device settings structure
* \param rxChannel Target Rx channel whose dynamic slicer configuration is requested
* \param rxDynamicSlicerCfg Pointer to the dynamic slicer config which will be updated with the
*
* \retval ADI_ADRV9025_WARN_RESET_LOG Recovery action for log reset
* \retval ADI_ADRV9025_ERR_CHECK_PARAM Recovery action for bad parameter check
* \retval ADI_ADRV9025_ERR_RESET_SPI Recovery action for SPI reset required
* \retval ADI_ADRV9025_NO_ACTION Function completed successfully, no action required
*/
int32_t adi_adrv9025_RxDynamicSlicerConfigGet(adi_adrv9025_Device_t*                device,
                                              adi_adrv9025_RxChannels_e             rxChannel,
                                              adi_adrv9025_RxDynamicSlicerConfig_t* rxDynamicSlicerCfg);

/**
* \brief Commands the ADRV9025 to resend dynamic slicer data
*
* The 56 bit dynamic slicer data is transmitted via JESD data automatically when the Rx gain changes.
* The user can request the slicer to resend dynamic slicer data on demand through this API.
*
* \pre This function can be called only if the dynamic slicer config is enabled previously via
*      adi_adrv9025_RxDynamicSlicerConfigSet() API.
*
* \dep_begin
* \dep{device->halDevInfo}
* \dep_end
*
* \param device Pointer to the device settings structure
* \param rxChannelMask A mask consisting of the 'OR'ed Rx channels for which the
*                      dynamic slicer data is required to be resent
*
* \retval ADI_ADRV9025_WARN_RESET_LOG Recovery action for log reset
* \retval ADI_ADRV9025_ERR_CHECK_PARAM Recovery action for bad parameter check
* \retval ADI_ADRV9025_ERR_RESET_SPI Recovery action for SPI reset required
* \retval ADI_ADRV9025_NO_ACTION Function completed successfully, no action required
*/
int32_t adi_adrv9025_RxDynamicSlicerDataResend(adi_adrv9025_Device_t* device,
                                               uint32_t               rxChannelMask);

#ifdef __cplusplus
}
#endif

#endif
