// SPDX-License-Identifier: GPL-2.0-only
/*
 * Xilinx VPSS Scaler
 *
 * Copyright (C) 2017 Xilinx, Inc.
 * Copyright (C) 2025, Advanced Micro Devices, Inc. All rights reserved.
 */

#include <linux/clk.h>
#include <linux/delay.h>
#include <linux/device.h>
#include <linux/gpio/consumer.h>
#include <linux/module.h>
#include <linux/of.h>
#include <linux/platform_device.h>

#include <media/v4l2-async.h>
#include <media/v4l2-subdev.h>
#include "xilinx-vip.h"

#define XSCALER_MIN_WIDTH		(64)
#define XSCALER_MAX_WIDTH		(8192)
#define XSCALER_MIN_HEIGHT		(64)
#define XSCALER_MAX_HEIGHT		(4320)
#define XSCALER_MAX_PHASES		(64)

/* Modify to defaults incase it is not configured from application */
#define XSCALER_DEF_IN_HEIGHT		(720)
#define XSCALER_DEF_IN_WIDTH		(1280)
#define XSCALER_DEF_OUT_HEIGHT		(1080)
#define XSCALER_DEF_OUT_WIDTH		(1920)

#define XSCALER_HSF			(0x0100)
#define XSCALER_VSF			(0x0104)
#define XSCALER_SF_SHIFT		(20)
#define XSCALER_SF_MASK			(0xffffff)
#define XSCALER_SOURCE_SIZE		(0x0108)
#define XSCALER_SIZE_HORZ_SHIFT		(0)
#define XSCALER_SIZE_VERT_SHIFT		(16)
#define XSCALER_SIZE_MASK		(0xfff)
#define XSCALER_HAPERTURE		(0x010c)
#define XSCALER_VAPERTURE		(0x0110)
#define XSCALER_APERTURE_START_SHIFT	(0)
#define XSCALER_APERTURE_END_SHIFT	(16)
#define XSCALER_OUTPUT_SIZE		(0x0114)
#define XSCALER_COEF_DATA_IN		(0x0134)
#define XSCALER_BITSHIFT_16		(16)
#define XSCALER_PHDATA_SHIFT		(12)
#define XSCALER_PHDATA_H_SHIFT		(20)

/* Video subsytems block offset */
#define S_AXIS_RESET_OFF	(0x00010000)
#define V_HSCALER_OFF		(0x00000000)
#define V_VSCALER_OFF		(0x00020000)

/* HW Reset Network GPIO Channel */
#define XGPIO_CH_RESET_SEL		(1)
#define XGPIO_RESET_MASK_VIDEO_IN	BIT(0)
#define XGPIO_RESET_MASK_IP_AXIS	BIT(1)
#define XGPIO_RESET_MASK_IP_AXIMM	BIT(0)
#define XGPIO_RESET_MASK_ALL_BLOCKS	(XGPIO_RESET_MASK_VIDEO_IN  | \
						XGPIO_RESET_MASK_IP_AXIS)
#define XGPIO_DATA_OFFSET		(0x0)
#define XGPIO_TRI_OFFSET		(0x4)
#define XGPIO_DATA2_OFFSET		(0x8)
#define XGPIO_TRI2_OFFSET		(0xc)

#define XGPIO_GIE_OFFSET		(0x11c)
#define XGPIO_ISR_OFFSET		(0x120)
#define XGPIO_IER_OFFSET		(0x128)
#define XGPIO_CHAN_OFFSET		(8)
#define STEP_PRECISION			(65536)

/* Video IP Formats */
enum xscaler_vid_reg_fmts {
	XVIDC_CSF_RGB = 0,
	XVIDC_CSF_YCRCB_444,
	XVIDC_CSF_YCRCB_422,
	XVIDC_CSF_YCRCB_420,
};

/* Video IP PPC */
#define XSCALER_PPC_1			(1)
#define XSCALER_PPC_2			(2)
#define XSCALER_PPC_4			(4)
#define XSCALER_PPC_8			(8)

#define XV_HSCALER_MAX_H_TAPS           (12)
#define XV_HSCALER_MAX_H_PHASES         (64)
#define XV_HSCALER_MAX_LINE_WIDTH	(8192)
#define XV_VSCALER_MAX_V_TAPS           (12)
#define XV_VSCALER_MAX_V_PHASES         (64)

#define XV_HSCALER_TAPS_2		(2)
#define XV_HSCALER_TAPS_4		(4)
#define XV_HSCALER_TAPS_6		(6)
#define XV_HSCALER_TAPS_8		(8)
#define XV_HSCALER_TAPS_10		(10)
#define XV_HSCALER_TAPS_12		(12)
#define XV_VSCALER_TAPS_2		(2)
#define XV_VSCALER_TAPS_4		(4)
#define XV_VSCALER_TAPS_6		(6)
#define XV_VSCALER_TAPS_8		(8)
#define XV_VSCALER_TAPS_10		(10)
#define XV_VSCALER_TAPS_12		(12)

/* Mask definitions for Low and high 16 bits in a 32 bit number */
#define XHSC_MASK_LOW_16BITS		GENMASK(15, 0)
#define XHSC_MASK_HIGH_16BITS		GENMASK(31, 16)
#define XHSC_MASK_LOW_32BITS		GENMASK(31, 0)
#define XHSC_MASK_LOW_20BITS		GENMASK(19, 0)
#define XHSC_MASK_LOW_12BITS		GENMASK(11, 0)
#define XHSC_STEP_PRECISION_SHIFT	(16)
#define XHSC_HPHASE_STEP_4		(4)
#define XHSC_HPHASE_SHIFT_BY_6		(6)
#define XHSC_HPHASE_MULTIPLIER		(9)
#define XHSC_HPHASE_MULTIPLIER_8PPC	(10)
#define XHSC_HPHASE_MUL_4PPC		(10)
#define XHSC_HPHASE_MUL_8PPC		(11)

/* Mask definitions for Low and high 16 bits in a 32 bit number */
#define XVSC_MASK_LOW_16BITS            GENMASK(15, 0)
#define XVSC_MASK_HIGH_16BITS           GENMASK(31, 16)

/* XSCALER POWER MACROS */
#define XSCALER_RESET_ASSERT	(0x1)
#define XSCALER_RESET_DEASSERT	(0x0)

/* Scaler AP Control Registers */
#define XSCALER_START		BIT(0)
#define XSCALER_AUTO_RESTART	BIT(7)
#define XSCALER_STREAM_ON	(XSCALER_START | XSCALER_AUTO_RESTART)

/* H-scaler registers */
#define XV_HSCALER_CTRL_ADDR_AP_CTRL				(0x0000)
#define XV_HSCALER_CTRL_ADDR_GIE				(0x0004)
#define XV_HSCALER_CTRL_ADDR_IER				(0x0008)
#define XV_HSCALER_CTRL_ADDR_ISR				(0x000c)
#define XV_HSCALER_CTRL_ADDR_HWREG_HEIGHT_DATA			(0x0010)
#define XV_HSCALER_CTRL_ADDR_HWREG_WIDTHIN_DATA			(0x0018)
#define XV_HSCALER_CTRL_ADDR_HWREG_WIDTHOUT_DATA		(0x0020)
#define XV_HSCALER_CTRL_ADDR_HWREG_COLORMODE_DATA		(0x0028)
#define XV_HSCALER_CTRL_ADDR_HWREG_PIXELRATE_DATA		(0x0030)
#define XV_HSCALER_CTRL_ADDR_HWREG_COLORMODEOUT_DATA		(0X0038)
#define XV_HSCALER_CTRL_ADDR_HWREG_HFLTCOEFF_BASE		(0x0800)
#define XV_HSCALER_CTRL_ADDR_HWREG_HFLTCOEFF_HIGH		(0x0bff)

/* Coefficients for 6, 8, 10 and 12 tap filters */

static const s16 XV_lanczos2_taps6[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_6] = {
	{   0,    0, 4096,    0,    0,   0, },
	{   0,  -40, 4099,   42,    0,  -5, },
	{  -1,  -77, 4097,   87,   -1,  -9, },
	{  -2, -111, 4092,  134,   -2, -15, },
	{  -4, -143, 4082,  184,   -4, -19, },
	{  -6, -173, 4068,  237,   -7, -23, },
	{  -8, -201, 4051,  292,  -10, -28, },
	{ -11, -226, 4029,  350,  -13, -33, },
	{ -14, -248, 4003,  411,  -18, -38, },
	{ -17, -269, 3974,  474,  -23, -43, },
	{ -21, -287, 3940,  539,  -28, -47, },
	{ -24, -303, 3903,  608,  -34, -54, },
	{ -28, -317, 3862,  678,  -41, -58, },
	{ -32, -329, 3817,  751,  -49, -62, },
	{ -37, -339, 3768,  826,  -57, -65, },
	{ -41, -347, 3716,  903,  -65, -70, },
	{ -45, -353, 3661,  982,  -75, -74, },
	{ -50, -358, 3602, 1063,  -84, -77, },
	{ -54, -361, 3539, 1146,  -95, -79, },
	{ -58, -362, 3474, 1230, -106, -82, },
	{ -62, -361, 3406, 1317, -117, -87, },
	{ -66, -359, 3335, 1404, -128, -90, },
	{ -70, -356, 3261, 1493, -140, -92, },
	{ -74, -351, 3185, 1583, -153, -94, },
	{ -77, -346, 3106, 1673, -165, -95, },
	{ -81, -339, 3025, 1765, -178, -96, },
	{ -84, -331, 2942, 1857, -191, -97, },
	{ -87, -322, 2858, 1950, -204, -99, },
	{ -89, -313, 2771, 2043, -217, -99, },
	{ -92, -302, 2683, 2136, -230, -99, },
	{ -94, -292, 2594, 2228, -243, -97, },
	{ -95, -280, 2504, 2321, -256, -98, },
	{ -97, -268, 2413, 2413, -268, -97, },
	{ -97, -256, 2321, 2504, -280, -96, },
	{ -98, -243, 2228, 2594, -292, -93, },
	{ -98, -230, 2136, 2683, -302, -93, },
	{ -98, -217, 2043, 2771, -313, -90, },
	{ -98, -204, 1950, 2858, -322, -88, },
	{ -97, -191, 1857, 2942, -331, -84, },
	{ -96, -178, 1765, 3025, -339, -81, },
	{ -95, -165, 1673, 3106, -346, -77, },
	{ -93, -153, 1583, 3185, -351, -75, },
	{ -91, -140, 1493, 3261, -356, -71, },
	{ -89, -128, 1404, 3335, -359, -67, },
	{ -86, -117, 1317, 3406, -361, -63, },
	{ -83, -106, 1230, 3474, -362, -57, },
	{ -80,  -95, 1146, 3539, -361, -53, },
	{ -77,  -84, 1063, 3602, -358, -50, },
	{ -73,  -75,  982, 3661, -353, -46, },
	{ -69,  -65,  903, 3716, -347, -42, },
	{ -65,  -57,  826, 3768, -339, -37, },
	{ -61,  -49,  751, 3817, -329, -33, },
	{ -57,  -41,  678, 3862, -317, -29, },
	{ -52,  -34,  608, 3903, -303, -26, },
	{ -47,  -28,  539, 3940, -287, -21, },
	{ -43,  -23,  474, 3974, -269, -17, },
	{ -38,  -18,  411, 4003, -248, -14, },
	{ -33,  -13,  350, 4029, -226, -11, },
	{ -28,  -10,  292, 4051, -201,  -8, },
	{ -24,   -7,  237, 4068, -173,  -5, },
	{ -19,   -4,  184, 4082, -143,  -4, },
	{ -14,   -2,  134, 4092, -111,  -3, },
	{  -9,   -1,   87, 4097,  -77,  -1, },
	{  -5,    0,   42, 4099,  -40,   0, }
};

/* ScalingRatio = 1.25 */
static const s16 XV_fixedcoeff_taps6_SR1p2[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_6] = {
	{ -102,  512, 3208,  512, -102,  68, },
	{  -97,  471, 3209,  555, -107,  65, },
	{  -92,  431, 3208,  599, -113,  63, },
	{  -87,  392, 3205,  645, -118,  59, },
	{  -82,  354, 3199,  691, -124,  58, },
	{  -77,  318, 3191,  739, -130,  55, },
	{  -72,  282, 3181,  788, -136,  53, },
	{  -68,  248, 3169,  838, -141,  50, },
	{  -64,  216, 3155,  889, -147,  47, },
	{  -59,  184, 3139,  941, -153,  44, },
	{  -55,  154, 3120,  993, -158,  42, },
	{  -52,  125, 3100, 1047, -164,  40, },
	{  -48,   98, 3077, 1101, -169,  37, },
	{  -44,   71, 3052, 1157, -174,  34, },
	{  -41,   46, 3025, 1212, -180,  34, },
	{  -38,   23, 2996, 1269, -184,  30, },
	{  -35,    0, 2965, 1326, -189,  29, },
	{  -32,  -21, 2933, 1383, -193,  26, },
	{  -29,  -41, 2898, 1441, -198,  25, },
	{  -26,  -60, 2862, 1500, -201,  21, },
	{  -24,  -78, 2823, 1558, -205,  22, },
	{  -21,  -94, 2784, 1617, -208,  18, },
	{  -19, -109, 2742, 1676, -210,  16, },
	{  -17, -123, 2699, 1734, -212,  15, },
	{  -14, -136, 2654, 1793, -214,  13, },
	{  -12, -148, 2608, 1852, -214,  10, },
	{  -10, -159, 2560, 1910, -215,  10, },
	{   -9, -168, 2512, 1968, -215,   8, },
	{   -7, -177, 2461, 2026, -214,   7, },
	{   -5, -185, 2410, 2083, -212,   5, },
	{   -3, -192, 2358, 2139, -209,   3, },
	{   -2, -197, 2304, 2195, -206,   2, },
	{    0, -202, 2250, 2250, -202,   0, },
	{    2, -206, 2195, 2304, -197,  -2, },
	{    3, -209, 2139, 2358, -192,  -3, },
	{    5, -212, 2083, 2410, -185,  -5, },
	{    6, -214, 2026, 2461, -177,  -6, },
	{    8, -215, 1968, 2512, -168,  -9, },
	{   10, -215, 1910, 2560, -159, -10, },
	{   11, -214, 1852, 2608, -148, -13, },
	{   13, -214, 1793, 2654, -136, -14, },
	{   15, -212, 1734, 2699, -123, -17, },
	{   17, -210, 1676, 2742, -109, -20, },
	{   18, -208, 1617, 2784,  -94, -21, },
	{   20, -205, 1558, 2823,  -78, -22, },
	{   22, -201, 1500, 2862,  -60, -27, },
	{   24, -198, 1441, 2898,  -41, -28, },
	{   26, -193, 1383, 2933,  -21, -32, },
	{   28, -189, 1326, 2965,    0, -34, },
	{   30, -184, 1269, 2996,   23, -38, },
	{   33, -180, 1212, 3025,   46, -40, },
	{   35, -174, 1157, 3052,   71, -45, },
	{   37, -169, 1101, 3077,   98, -48, },
	{   40, -164, 1047, 3100,  125, -52, },
	{   42, -158,  993, 3120,  154, -55, },
	{   44, -153,  941, 3139,  184, -59, },
	{   47, -147,  889, 3155,  216, -64, },
	{   50, -141,  838, 3169,  248, -68, },
	{   52, -136,  788, 3181,  282, -71, },
	{   55, -130,  739, 3191,  318, -77, },
	{   57, -124,  691, 3199,  354, -81, },
	{   60, -118,  645, 3205,  392, -88, },
	{   63, -113,  599, 3208,  431, -92, },
	{   65, -107,  555, 3209,  471, -97, }
};

/* ScalingRatio = 2.0 */
static const s16 XV_fixedcoeff_taps6_SR2[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_6] = {
	{   0, 970, 2235,  970,   0, -79, },
	{  -3, 943, 2233,  997,   3, -77, },
	{  -5, 915, 2231, 1025,   6, -76, },
	{  -8, 888, 2227, 1052,  10, -73, },
	{ -10, 861, 2223, 1079,  14, -71, },
	{ -12, 834, 2218, 1107,  18, -69, },
	{ -14, 808, 2213, 1134,  22, -67, },
	{ -15, 782, 2206, 1162,  27, -66, },
	{ -17, 756, 2199, 1189,  32, -63, },
	{ -18, 731, 2191, 1217,  37, -62, },
	{ -20, 706, 2182, 1245,  42, -59, },
	{ -21, 681, 2172, 1272,  48, -56, },
	{ -22, 657, 2162, 1300,  55, -56, },
	{ -22, 633, 2151, 1327,  61, -54, },
	{ -23, 609, 2139, 1355,  68, -52, },
	{ -24, 586, 2126, 1382,  76, -50, },
	{ -25, 564, 2113, 1410,  83, -49, },
	{ -25, 541, 2099, 1437,  91, -47, },
	{ -26, 520, 2084, 1464, 100, -46, },
	{ -26, 498, 2069, 1491, 109, -45, },
	{ -27, 477, 2053, 1517, 118, -42, },
	{ -27, 457, 2036, 1544, 128, -42, },
	{ -27, 437, 2019, 1570, 138, -41, },
	{ -28, 418, 2001, 1596, 148, -39, },
	{ -28, 399, 1983, 1622, 160, -40, },
	{ -29, 380, 1964, 1647, 171, -37, },
	{ -29, 362, 1944, 1672, 183, -36, },
	{ -29, 345, 1924, 1697, 195, -36, },
	{ -30, 328, 1903, 1722, 208, -35, },
	{ -30, 311, 1882, 1746, 221, -34, },
	{ -31, 295, 1860, 1770, 235, -33, },
	{ -31, 279, 1838, 1793, 249, -32, },
	{ -32, 264, 1816, 1816, 264, -32, },
	{ -32, 249, 1793, 1838, 279, -31, },
	{ -33, 235, 1770, 1860, 295, -31, },
	{ -34, 221, 1746, 1882, 311, -30, },
	{ -35, 208, 1722, 1903, 328, -30, },
	{ -35, 195, 1697, 1924, 345, -30, },
	{ -36, 183, 1672, 1944, 362, -29, },
	{ -37, 171, 1647, 1964, 380, -29, },
	{ -38, 160, 1622, 1983, 399, -30, },
	{ -39, 148, 1596, 2001, 418, -28, },
	{ -40, 138, 1570, 2019, 437, -28, },
	{ -42, 128, 1544, 2036, 457, -27, },
	{ -43, 118, 1517, 2053, 477, -26, },
	{ -44, 109, 1491, 2069, 498, -27, },
	{ -46, 100, 1464, 2084, 520, -26, },
	{ -47,  91, 1437, 2099, 541, -25, },
	{ -49,  83, 1410, 2113, 564, -25, },
	{ -50,  76, 1382, 2126, 586, -24, },
	{ -52,  68, 1355, 2139, 609, -23, },
	{ -54,  61, 1327, 2151, 633, -22, },
	{ -55,  55, 1300, 2162, 657, -23, },
	{ -57,  48, 1272, 2172, 681, -20, },
	{ -59,  42, 1245, 2182, 706, -20, },
	{ -61,  37, 1217, 2191, 731, -19, },
	{ -63,  32, 1189, 2199, 756, -17, },
	{ -65,  27, 1162, 2206, 782, -16, },
	{ -67,  22, 1134, 2213, 808, -14, },
	{ -69,  18, 1107, 2218, 834, -12, },
	{ -71,  14, 1079, 2223, 861, -10, },
	{ -73,  10, 1052, 2227, 888,  -8, },
	{ -75,   6, 1025, 2231, 915,  -6, },
	{ -77,   3,  997, 2233, 943,  -3, }
};

/* ScalingRatio = 3.0 */
static const s16 XV_fixedcoeff_taps6_SR3[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_6] = {
	{ 126, 1019, 1806, 1019,  126,   0, },
	{ 120, 1000, 1805, 1038,  132,   1, },
	{ 114,  980, 1804, 1057,  138,   3, },
	{ 108,  961, 1802, 1075,  145,   5, },
	{ 103,  942, 1800, 1094,  152,   5, },
	{  98,  922, 1797, 1113,  159,   7, },
	{  93,  903, 1794, 1131,  167,   8, },
	{  88,  884, 1790, 1150,  174,  10, },
	{  84,  865, 1786, 1168,  182,  11, },
	{  80,  846, 1782, 1187,  191,  10, },
	{  76,  827, 1777, 1205,  199,  12, },
	{  72,  809, 1771, 1223,  208,  13, },
	{  68,  790, 1766, 1241,  217,  14, },
	{  65,  772, 1759, 1259,  226,  15, },
	{  61,  753, 1753, 1277,  236,  16, },
	{  58,  735, 1746, 1295,  246,  16, },
	{  56,  717, 1738, 1313,  256,  16, },
	{  53,  699, 1730, 1330,  266,  18, },
	{  50,  682, 1722, 1347,  277,  18, },
	{  48,  664, 1713, 1364,  288,  19, },
	{  46,  647, 1704, 1381,  299,  19, },
	{  43,  630, 1694, 1398,  311,  20, },
	{  41,  613, 1684, 1414,  323,  21, },
	{  40,  596, 1674, 1430,  335,  21, },
	{  38,  580, 1663, 1446,  347,  22, },
	{  36,  563, 1652, 1462,  360,  23, },
	{  35,  547, 1641, 1478,  373,  22, },
	{  33,  531, 1629, 1493,  386,  24, },
	{  32,  516, 1617, 1508,  399,  24, },
	{  31,  500, 1604, 1523,  413,  25, },
	{  30,  485, 1592, 1537,  427,  25, },
	{  29,  470, 1578, 1551,  441,  27, },
	{  28,  455, 1565, 1565,  455,  28, },
	{  27,  441, 1551, 1578,  470,  29, },
	{  26,  427, 1537, 1592,  485,  29, },
	{  25,  413, 1523, 1604,  500,  31, },
	{  24,  399, 1508, 1617,  516,  32, },
	{  24,  386, 1493, 1629,  531,  33, },
	{  23,  373, 1478, 1641,  547,  34, },
	{  22,  360, 1462, 1652,  563,  37, },
	{  22,  347, 1446, 1663,  580,  38, },
	{  21,  335, 1430, 1674,  596,  40, },
	{  20,  323, 1414, 1684,  613,  42, },
	{  20,  311, 1398, 1694,  630,  43, },
	{  19,  299, 1381, 1704,  647,  46, },
	{  19,  288, 1364, 1713,  664,  48, },
	{  18,  277, 1347, 1722,  682,  50, },
	{  17,  266, 1330, 1730,  699,  54, },
	{  17,  256, 1313, 1738,  717,  55, },
	{  16,  246, 1295, 1746,  735,  58, },
	{  15,  236, 1277, 1753,  753,  62, },
	{  15,  226, 1259, 1759,  772,  65, },
	{  14,  217, 1241, 1766,  790,  68, },
	{  13,  208, 1223, 1771,  809,  72, },
	{  12,  199, 1205, 1777,  827,  76, },
	{  11,  191, 1187, 1782,  846,  79, },
	{  10,  182, 1168, 1786,  865,  85, },
	{   9,  174, 1150, 1790,  884,  89, },
	{   8,  167, 1131, 1794,  903,  93, },
	{   7,  159, 1113, 1797,  922,  98, },
	{   6,  152, 1094, 1800,  942, 102, },
	{   5,  145, 1075, 1802,  961, 108, },
	{   3,  138, 1057, 1804,  980, 114, },
	{   2,  132, 1038, 1805, 1000, 119, }
};

/* ScalingRatio = 4 */
static const s16 XV_fixedcoeff_taps6_SR4[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_6] = {
	{ 176, 1009, 1643, 1009, 176,  83, },
	{ 169,  993, 1644, 1026, 183,  81, },
	{ 162,  978, 1644, 1042, 190,  80, },
	{ 156,  962, 1643, 1058, 198,  79, },
	{ 150,  946, 1642, 1074, 205,  79, },
	{ 144,  930, 1641, 1091, 213,  77, },
	{ 138,  914, 1640, 1107, 222,  75, },
	{ 133,  898, 1638, 1123, 230,  74, },
	{ 128,  882, 1635, 1139, 239,  73, },
	{ 123,  866, 1633, 1154, 248,  72, },
	{ 118,  850, 1629, 1170, 257,  72, },
	{ 114,  834, 1626, 1186, 267,  69, },
	{ 109,  818, 1622, 1201, 276,  70, },
	{ 105,  802, 1618, 1217, 286,  68, },
	{ 101,  786, 1613, 1232, 297,  67, },
	{  97,  771, 1608, 1247, 307,  66, },
	{  94,  755, 1603, 1262, 318,  64, },
	{  91,  739, 1597, 1276, 328,  65, },
	{  87,  724, 1591, 1291, 339,  64, },
	{  85,  708, 1585, 1305, 351,  62, },
	{  82,  693, 1578, 1319, 362,  62, },
	{  79,  677, 1571, 1333, 374,  62, },
	{  77,  662, 1563, 1347, 386,  61, },
	{  75,  647, 1556, 1360, 398,  60, },
	{  73,  632, 1547, 1373, 410,  61, },
	{  71,  617, 1539, 1386, 423,  60, },
	{  69,  602, 1530, 1399, 436,  60, },
	{  68,  587, 1521, 1412, 449,  59, },
	{  66,  573, 1511, 1424, 462,  60, },
	{  65,  558, 1501, 1436, 475,  61, },
	{  64,  544, 1491, 1447, 488,  62, },
	{  63,  530, 1481, 1459, 502,  61, },
	{  62,  516, 1470, 1470, 516,  62, },
	{  62,  502, 1459, 1481, 530,  62, },
	{  61,  488, 1447, 1491, 544,  65, },
	{  61,  475, 1436, 1501, 558,  65, },
	{  60,  462, 1424, 1511, 573,  66, },
	{  60,  449, 1412, 1521, 587,  67, },
	{  60,  436, 1399, 1530, 602,  69, },
	{  60,  423, 1386, 1539, 617,  71, },
	{  61,  410, 1373, 1547, 632,  73, },
	{  61,  398, 1360, 1556, 647,  74, },
	{  61,  386, 1347, 1563, 662,  77, },
	{  62,  374, 1333, 1571, 677,  79, },
	{  62,  362, 1319, 1578, 693,  82, },
	{  63,  351, 1305, 1585, 708,  84, },
	{  64,  339, 1291, 1591, 724,  87, },
	{  64,  328, 1276, 1597, 739,  92, },
	{  65,  318, 1262, 1603, 755,  93, },
	{  66,  307, 1247, 1608, 771,  97, },
	{  67,  297, 1232, 1613, 786, 101, },
	{  68,  286, 1217, 1618, 802, 105, },
	{  69,  276, 1201, 1622, 818, 110, },
	{  70,  267, 1186, 1626, 834, 113, },
	{  71,  257, 1170, 1629, 850, 119, },
	{  72,  248, 1154, 1633, 866, 123, },
	{  73,  239, 1139, 1635, 882, 128, },
	{  75,  230, 1123, 1638, 898, 132, },
	{  76,  222, 1107, 1640, 914, 137, },
	{  77,  213, 1091, 1641, 930, 144, },
	{  78,  205, 1074, 1642, 946, 151, },
	{  79,  198, 1058, 1643, 962, 156, },
	{  80,  190, 1042, 1644, 978, 162, },
	{  82,  183, 1026, 1644, 993, 168, }
};

/* ScalingRatio = 2.0 */
static const s16 XV_fixedcoeff_taps8_SR2[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_8] = {
	{ -55,   0, 1078, 2049, 1078,    0, -55,   1, },
	{ -53,  -7, 1055, 2049, 1102,    7, -56,  -1, },
	{ -52, -13, 1032, 2048, 1126,   15, -58,  -2, },
	{ -50, -20, 1009, 2047, 1149,   22, -59,  -2, },
	{ -49, -26,  986, 2046, 1173,   31, -61,  -4, },
	{ -47, -31,  963, 2043, 1197,   39, -62,  -6, },
	{ -46, -37,  940, 2040, 1220,   48, -64,  -5, },
	{ -45, -42,  917, 2037, 1244,   57, -65,  -7, },
	{ -43, -47,  894, 2033, 1267,   66, -67,  -7, },
	{ -42, -51,  871, 2028, 1290,   76, -69,  -7, },
	{ -41, -55,  848, 2023, 1313,   86, -70,  -8, },
	{ -40, -59,  826, 2017, 1336,   97, -72,  -9, },
	{ -38, -63,  803, 2010, 1359,  108, -73, -10, },
	{ -37, -67,  781, 2003, 1382,  119, -75, -10, },
	{ -36, -70,  759, 1996, 1405,  130, -76, -12, },
	{ -35, -73,  737, 1987, 1427,  142, -78, -11, },
	{ -34, -76,  715, 1979, 1449,  154, -79, -12, },
	{ -33, -78,  693, 1969, 1471,  167, -81, -12, },
	{ -32, -81,  672, 1959, 1493,  180, -82, -13, },
	{ -31, -83,  650, 1949, 1514,  193, -83, -13, },
	{ -30, -85,  629, 1938, 1536,  207, -85, -14, },
	{ -29, -86,  609, 1926, 1557,  221, -86, -16, },
	{ -28, -88,  588, 1914, 1577,  235, -87, -15, },
	{ -28, -89,  568, 1902, 1598,  250, -88, -17, },
	{ -27, -90,  548, 1889, 1618,  265, -89, -18, },
	{ -26, -91,  528, 1875, 1638,  280, -90, -18, },
	{ -25, -92,  508, 1861, 1657,  296, -91, -18, },
	{ -24, -93,  489, 1846, 1676,  312, -92, -18, },
	{ -24, -93,  470, 1831, 1695,  328, -92, -19, },
	{ -23, -94,  451, 1816, 1714,  345, -93, -20, },
	{ -22, -94,  432, 1800, 1732,  361, -93, -20, },
	{ -22, -94,  414, 1783, 1749,  379, -94, -19, },
	{ -21, -94,  396, 1767, 1767,  396, -94, -21, },
	{ -21, -94,  379, 1749, 1783,  414, -94, -20, },
	{ -20, -93,  361, 1732, 1800,  432, -94, -22, },
	{ -19, -93,  345, 1714, 1816,  451, -94, -24, },
	{ -19, -92,  328, 1695, 1831,  470, -93, -24, },
	{ -18, -92,  312, 1676, 1846,  489, -93, -24, },
	{ -18, -91,  296, 1657, 1861,  508, -92, -25, },
	{ -17, -90,  280, 1638, 1875,  528, -91, -27, },
	{ -17, -89,  265, 1618, 1889,  548, -90, -28, },
	{ -16, -88,  250, 1598, 1902,  568, -89, -29, },
	{ -16, -87,  235, 1577, 1914,  588, -88, -27, },
	{ -15, -86,  221, 1557, 1926,  609, -86, -30, },
	{ -14, -85,  207, 1536, 1938,  629, -85, -30, },
	{ -14, -83,  193, 1514, 1949,  650, -83, -30, },
	{ -13, -82,  180, 1493, 1959,  672, -81, -32, },
	{ -13, -81,  167, 1471, 1969,  693, -78, -32, },
	{ -12, -79,  154, 1449, 1979,  715, -76, -34, },
	{ -12, -78,  142, 1427, 1987,  737, -73, -34, },
	{ -11, -76,  130, 1405, 1996,  759, -70, -37, },
	{ -10, -75,  119, 1382, 2003,  781, -67, -37, },
	{ -10, -73,  108, 1359, 2010,  803, -63, -38, },
	{  -9, -72,   97, 1336, 2017,  826, -59, -40, },
	{  -8, -70,   86, 1313, 2023,  848, -55, -41, },
	{  -8, -69,   76, 1290, 2028,  871, -51, -41, },
	{  -7, -67,   66, 1267, 2033,  894, -47, -43, },
	{  -6, -65,   57, 1244, 2037,  917, -42, -46, },
	{  -5, -64,   48, 1220, 2040,  940, -37, -46, },
	{  -5, -62,   39, 1197, 2043,  963, -31, -48, },
	{  -4, -61,   31, 1173, 2046,  986, -26, -49, },
	{  -3, -59,   22, 1149, 2047, 1009, -20, -49, },
	{  -2, -58,   15, 1126, 2048, 1032, -13, -52, },
	{  -1, -56,    7, 1102, 2049, 1055,  -7, -53, }
};

/* ScalingRatio = 3.0 */
static const s16 XV_fixedcoeff_taps8_SR3[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_8] = {
	{   0, 275, 1036, 1514, 1036,  275,   0, -40, },
	{  -1, 266, 1023, 1514, 1048,  283,   1, -38, },
	{  -2, 257, 1010, 1513, 1060,  292,   2, -36, },
	{  -3, 249,  997, 1512, 1073,  301,   3, -36, },
	{  -3, 241,  983, 1510, 1085,  310,   5, -35, },
	{  -4, 233,  970, 1509, 1097,  319,   6, -34, },
	{  -5, 225,  957, 1507, 1109,  329,   7, -33, },
	{  -6, 217,  944, 1505, 1121,  338,   9, -32, },
	{  -6, 210,  931, 1503, 1133,  348,  10, -33, },
	{  -7, 202,  917, 1500, 1144,  358,  12, -30, },
	{  -7, 195,  904, 1497, 1156,  368,  13, -30, },
	{  -8, 188,  891, 1494, 1167,  378,  15, -29, },
	{  -9, 181,  877, 1491, 1179,  388,  17, -28, },
	{  -9, 174,  864, 1487, 1190,  398,  19, -27, },
	{  -9, 168,  851, 1483, 1201,  409,  21, -28, },
	{ -10, 161,  837, 1479, 1212,  419,  23, -25, },
	{ -10, 155,  824, 1475, 1223,  430,  25, -26, },
	{ -11, 149,  811, 1470, 1233,  441,  27, -24, },
	{ -11, 142,  798, 1465, 1244,  452,  29, -23, },
	{ -12, 137,  784, 1460, 1254,  463,  32, -22, },
	{ -12, 131,  771, 1455, 1264,  474,  34, -21, },
	{ -12, 125,  758, 1449, 1275,  486,  37, -22, },
	{ -13, 120,  745, 1444, 1284,  497,  40, -21, },
	{ -13, 115,  732, 1438, 1294,  509,  42, -21, },
	{ -13, 109,  719, 1432, 1304,  520,  45, -20, },
	{ -14, 104,  706, 1425, 1313,  532,  48, -18, },
	{ -14, 100,  693, 1418, 1322,  544,  52, -19, },
	{ -14,  95,  680, 1412, 1332,  556,  55, -20, },
	{ -15,  90,  667, 1404, 1340,  568,  58, -16, },
	{ -15,  86,  655, 1397, 1349,  580,  62, -18, },
	{ -16,  82,  642, 1390, 1358,  592,  66, -18, },
	{ -16,  77,  630, 1382, 1366,  605,  69, -17, },
	{ -16,  73,  617, 1374, 1374,  617,  73, -16, },
	{ -17,  69,  605, 1366, 1382,  630,  77, -16, },
	{ -17,  66,  592, 1358, 1390,  642,  82, -17, },
	{ -18,  62,  580, 1349, 1397,  655,  86, -15, },
	{ -18,  58,  568, 1340, 1404,  667,  90, -13, },
	{ -18,  55,  556, 1332, 1412,  680,  95, -16, },
	{ -19,  52,  544, 1322, 1418,  693, 100, -14, },
	{ -19,  48,  532, 1313, 1425,  706, 104, -13, },
	{ -20,  45,  520, 1304, 1432,  719, 109, -13, },
	{ -20,  42,  509, 1294, 1438,  732, 115, -14, },
	{ -21,  40,  497, 1284, 1444,  745, 120, -13, },
	{ -22,  37,  486, 1275, 1449,  758, 125, -12, },
	{ -22,  34,  474, 1264, 1455,  771, 131, -11, },
	{ -23,  32,  463, 1254, 1460,  784, 137, -11, },
	{ -23,  29,  452, 1244, 1465,  798, 142, -11, },
	{ -24,  27,  441, 1233, 1470,  811, 149, -11, },
	{ -25,  25,  430, 1223, 1475,  824, 155, -11, },
	{ -26,  23,  419, 1212, 1479,  837, 161,  -9, },
	{ -26,  21,  409, 1201, 1483,  851, 168, -11, },
	{ -27,  19,  398, 1190, 1487,  864, 174,  -9, },
	{ -28,  17,  388, 1179, 1491,  877, 181,  -9, },
	{ -29,  15,  378, 1167, 1494,  891, 188,  -8, },
	{ -29,  13,  368, 1156, 1497,  904, 195,  -8, },
	{ -30,  12,  358, 1144, 1500,  917, 202,  -7, },
	{ -31,  10,  348, 1133, 1503,  931, 210,  -8, },
	{ -32,   9,  338, 1121, 1505,  944, 217,  -6, },
	{ -33,   7,  329, 1109, 1507,  957, 225,  -5, },
	{ -34,   6,  319, 1097, 1509,  970, 233,  -4, },
	{ -35,   5,  310, 1085, 1510,  983, 241,  -3, },
	{ -36,   3,  301, 1073, 1512,  997, 249,  -3, },
	{ -37,   2,  292, 1060, 1513, 1010, 257,  -1, },
	{ -38,   1,  283, 1048, 1514, 1023, 266,  -1, }
};

/* ScalingRatio = 4 */
static const s16 XV_fixedcoeff_taps8_SR4[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_8] = {
	{ 49, 366, 977, 1312,  977, 366,  49,  0, },
	{ 48, 357, 967, 1312,  986, 374,  51,  1, },
	{ 46, 349, 958, 1311,  995, 382,  54,  1, },
	{ 44, 342, 948, 1311, 1004, 390,  56,  1, },
	{ 42, 334, 939, 1310, 1013, 399,  58,  1, },
	{ 40, 326, 929, 1309, 1021, 407,  60,  4, },
	{ 39, 318, 919, 1308, 1030, 415,  63,  4, },
	{ 37, 311, 910, 1307, 1039, 424,  65,  3, },
	{ 36, 303, 900, 1305, 1047, 433,  68,  4, },
	{ 34, 296, 890, 1303, 1055, 442,  70,  6, },
	{ 33, 289, 880, 1301, 1064, 450,  73,  6, },
	{ 32, 282, 870, 1299, 1072, 459,  76,  6, },
	{ 31, 275, 861, 1297, 1080, 468,  79,  5, },
	{ 29, 268, 851, 1295, 1088, 477,  82,  6, },
	{ 28, 261, 841, 1292, 1096, 486,  85,  7, },
	{ 27, 254, 831, 1289, 1104, 496,  88,  7, },
	{ 26, 248, 821, 1287, 1112, 505,  91,  6, },
	{ 25, 241, 811, 1284, 1119, 514,  94,  8, },
	{ 24, 235, 800, 1280, 1127, 523,  98,  9, },
	{ 23, 228, 790, 1277, 1134, 533, 101, 10, },
	{ 22, 222, 780, 1273, 1141, 542, 105, 11, },
	{ 22, 216, 770, 1270, 1148, 552, 109,  9, },
	{ 21, 210, 760, 1266, 1155, 561, 112, 11, },
	{ 20, 204, 750, 1262, 1162, 571, 116, 11, },
	{ 19, 198, 740, 1257, 1169, 581, 120, 12, },
	{ 19, 193, 730, 1253, 1175, 590, 124, 12, },
	{ 18, 187, 720, 1248, 1182, 600, 129, 12, },
	{ 17, 182, 710, 1244, 1188, 610, 133, 12, },
	{ 17, 176, 700, 1239, 1194, 620, 137, 13, },
	{ 16, 171, 690, 1234, 1201, 630, 142, 12, },
	{ 16, 166, 680, 1229, 1206, 640, 146, 13, },
	{ 15, 161, 670, 1223, 1212, 650, 151, 14, },
	{ 15, 156, 660, 1218, 1218, 660, 156, 13, },
	{ 14, 151, 650, 1212, 1223, 670, 161, 15, },
	{ 14, 146, 640, 1206, 1229, 680, 166, 15, },
	{ 13, 142, 630, 1201, 1234, 690, 171, 15, },
	{ 13, 137, 620, 1194, 1239, 700, 176, 17, },
	{ 12, 133, 610, 1188, 1244, 710, 182, 17, },
	{ 12, 129, 600, 1182, 1248, 720, 187, 18, },
	{ 11, 124, 590, 1175, 1253, 730, 193, 20, },
	{ 11, 120, 581, 1169, 1257, 740, 198, 20, },
	{ 11, 116, 571, 1162, 1262, 750, 204, 20, },
	{ 10, 112, 561, 1155, 1266, 760, 210, 22, },
	{ 10, 109, 552, 1148, 1270, 770, 216, 21, },
	{ 10, 105, 542, 1141, 1273, 780, 222, 23, },
	{  9, 101, 533, 1134, 1277, 790, 228, 24, },
	{  9,  98, 523, 1127, 1280, 800, 235, 24, },
	{  8,  94, 514, 1119, 1284, 811, 241, 25, },
	{  8,  91, 505, 1112, 1287, 821, 248, 24, },
	{  8,  88, 496, 1104, 1289, 831, 254, 26, },
	{  7,  85, 486, 1096, 1292, 841, 261, 28, },
	{  7,  82, 477, 1088, 1295, 851, 268, 28, },
	{  6,  79, 468, 1080, 1297, 861, 275, 30, },
	{  6,  76, 459, 1072, 1299, 870, 282, 32, },
	{  5,  73, 450, 1064, 1301, 880, 289, 34, },
	{  5,  70, 442, 1055, 1303, 890, 296, 35, },
	{  4,  68, 433, 1047, 1305, 900, 303, 36, },
	{  4,  65, 424, 1039, 1307, 910, 311, 36, },
	{  3,  63, 415, 1030, 1308, 919, 318, 40, },
	{  3,  60, 407, 1021, 1309, 929, 326, 41, },
	{  2,  58, 399, 1013, 1310, 939, 334, 41, },
	{  2,  56, 390, 1004, 1311, 948, 342, 43, },
	{  1,  54, 382,  995, 1311, 958, 349, 46, },
	{  1,  51, 374,  986, 1312, 967, 357, 48, }
};

/* ScalingRatio = 3.0 */
static const s16 XV_fixedcoeff_taps10_SR3[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_10] = {
	{ -31,   0, 359, 1033, 1399, 1033,  359,   0, -31, -25, },
	{ -31,  -2, 350, 1022, 1398, 1043,  368,   3, -31, -24, },
	{ -30,  -4, 341, 1012, 1398, 1053,  378,   5, -32, -25, },
	{ -30,  -6, 333, 1002, 1398, 1062,  387,   8, -32, -26, },
	{ -30,  -8, 324,  992, 1397, 1072,  396,  10, -32, -25, },
	{ -30, -10, 315,  981, 1396, 1082,  406,  13, -33, -24, },
	{ -29, -12, 307,  971, 1395, 1091,  415,  16, -33, -25, },
	{ -29, -13, 298,  960, 1393, 1101,  425,  18, -33, -24, },
	{ -29, -15, 290,  949, 1392, 1110,  434,  21, -34, -22, },
	{ -28, -17, 282,  939, 1390, 1120,  444,  25, -34, -25, },
	{ -28, -18, 274,  928, 1388, 1129,  454,  28, -34, -25, },
	{ -28, -20, 266,  917, 1386, 1138,  464,  31, -34, -24, },
	{ -27, -21, 258,  906, 1384, 1147,  474,  34, -35, -24, },
	{ -27, -22, 250,  895, 1381, 1156,  484,  38, -35, -24, },
	{ -27, -23, 242,  885, 1379, 1164,  494,  41, -35, -24, },
	{ -27, -25, 235,  874, 1376, 1173,  504,  45, -35, -24, },
	{ -26, -26, 227,  863, 1373, 1181,  515,  49, -36, -24, },
	{ -26, -27, 220,  852, 1369, 1190,  525,  53, -36, -24, },
	{ -26, -28, 213,  841, 1366, 1198,  535,  56, -36, -23, },
	{ -26, -29, 206,  830, 1362, 1206,  546,  61, -36, -24, },
	{ -25, -29, 199,  819, 1358, 1214,  556,  65, -36, -25, },
	{ -25, -30, 192,  808, 1354, 1222,  567,  69, -36, -25, },
	{ -25, -31, 185,  797, 1350, 1229,  577,  73, -36, -23, },
	{ -25, -32, 178,  785, 1346, 1237,  588,  78, -36, -23, },
	{ -25, -32, 172,  774, 1341, 1244,  599,  83, -36, -24, },
	{ -25, -33, 165,  763, 1336, 1252,  610,  87, -36, -23, },
	{ -24, -33, 159,  752, 1331, 1259,  620,  92, -36, -24, },
	{ -24, -34, 153,  741, 1326, 1266,  631,  97, -36, -24, },
	{ -24, -34, 147,  730, 1321, 1272,  642, 102, -36, -24, },
	{ -24, -35, 141,  719, 1315, 1279,  653, 107, -36, -23, },
	{ -24, -35, 135,  708, 1310, 1285,  664, 113, -36, -24, },
	{ -24, -35, 129,  697, 1304, 1292,  675, 118, -36, -24, },
	{ -24, -36, 124,  686, 1298, 1298,  686, 124, -36, -24, },
	{ -24, -36, 118,  675, 1292, 1304,  697, 129, -35, -24, },
	{ -24, -36, 113,  664, 1285, 1310,  708, 135, -35, -24, },
	{ -24, -36, 107,  653, 1279, 1315,  719, 141, -35, -23, },
	{ -24, -36, 102,  642, 1272, 1321,  730, 147, -34, -24, },
	{ -23, -36,  97,  631, 1266, 1326,  741, 153, -34, -25, },
	{ -23, -36,  92,  620, 1259, 1331,  752, 159, -33, -25, },
	{ -23, -36,  87,  610, 1252, 1336,  763, 165, -33, -25, },
	{ -23, -36,  83,  599, 1244, 1341,  774, 172, -32, -26, },
	{ -23, -36,  78,  588, 1237, 1346,  785, 178, -32, -25, },
	{ -23, -36,  73,  577, 1229, 1350,  797, 185, -31, -25, },
	{ -23, -36,  69,  567, 1222, 1354,  808, 192, -30, -27, },
	{ -23, -36,  65,  556, 1214, 1358,  819, 199, -29, -27, },
	{ -24, -36,  61,  546, 1206, 1362,  830, 206, -29, -26, },
	{ -24, -36,  56,  535, 1198, 1366,  841, 213, -28, -25, },
	{ -24, -36,  53,  525, 1190, 1369,  852, 220, -27, -26, },
	{ -24, -36,  49,  515, 1181, 1373,  863, 227, -26, -26, },
	{ -24, -35,  45,  504, 1173, 1376,  874, 235, -25, -27, },
	{ -24, -35,  41,  494, 1164, 1379,  885, 242, -23, -27, },
	{ -24, -35,  38,  484, 1156, 1381,  895, 250, -22, -27, },
	{ -24, -35,  34,  474, 1147, 1384,  906, 258, -21, -27, },
	{ -24, -34,  31,  464, 1138, 1386,  917, 266, -20, -28, },
	{ -24, -34,  28,  454, 1129, 1388,  928, 274, -18, -29, },
	{ -24, -34,  25,  444, 1120, 1390,  939, 282, -17, -29, },
	{ -24, -34,  21,  434, 1110, 1392,  949, 290, -15, -27, },
	{ -24, -33,  18,  425, 1101, 1393,  960, 298, -13, -29, },
	{ -24, -33,  16,  415, 1091, 1395,  971, 307, -12, -30, },
	{ -25, -33,  13,  406, 1082, 1396,  981, 315, -10, -29, },
	{ -25, -32,  10,  396, 1072, 1397,  992, 324,  -8, -30, },
	{ -25, -32,   8,  387, 1062, 1398, 1002, 333,  -6, -31, },
	{ -25, -32,   5,  378, 1053, 1398, 1012, 341,  -4, -30, },
	{ -25, -31,   3,  368, 1043, 1398, 1022, 350,  -2, -30, }
};

/* ScalingRatio = 4 */
static const s16 XV_fixedcoeff_taps10_SR4[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_10] = {
	{   0, 107, 454, 924, 1150,  924, 454, 107,   0, -24, },
	{   0, 104, 446, 917, 1149,  930, 461, 110,   0, -21, },
	{  -1, 100, 439, 910, 1149,  936, 468, 114,   1, -20, },
	{  -1,  97, 432, 904, 1149,  942, 475, 117,   2, -21, },
	{  -2,  94, 425, 897, 1148,  948, 482, 121,   2, -19, },
	{  -2,  91, 418, 890, 1147,  954, 490, 125,   3, -20, },
	{  -3,  88, 411, 883, 1147,  960, 497, 128,   3, -18, },
	{  -3,  85, 404, 876, 1146,  966, 504, 132,   4, -18, },
	{  -3,  82, 397, 869, 1145,  972, 512, 136,   5, -19, },
	{  -4,  79, 390, 862, 1144,  978, 519, 140,   5, -17, },
	{  -4,  76, 384, 855, 1142,  983, 526, 144,   6, -16, },
	{  -4,  74, 377, 848, 1141,  989, 534, 148,   7, -18, },
	{  -5,  71, 370, 841, 1139,  995, 541, 152,   7, -15, },
	{  -5,  68, 364, 834, 1138, 1000, 549, 156,   8, -16, },
	{  -5,  66, 357, 827, 1136, 1005, 556, 160,   9, -15, },
	{  -6,  63, 350, 820, 1134, 1011, 564, 165,  10, -15, },
	{  -6,  61, 344, 812, 1132, 1016, 571, 169,  11, -14, },
	{  -6,  59, 338, 805, 1130, 1021, 579, 174,  12, -16, },
	{  -6,  56, 331, 798, 1128, 1026, 586, 178,  13, -14, },
	{  -7,  54, 325, 790, 1126, 1031, 594, 183,  14, -14, },
	{  -7,  52, 319, 783, 1124, 1036, 601, 187,  15, -14, },
	{  -7,  50, 312, 776, 1121, 1041, 609, 192,  16, -14, },
	{  -7,  48, 306, 768, 1119, 1045, 617, 197,  17, -14, },
	{  -8,  46, 300, 761, 1116, 1050, 624, 202,  18, -13, },
	{  -8,  44, 294, 753, 1113, 1054, 632, 207,  19, -12, },
	{  -8,  42, 288, 746, 1110, 1059, 639, 212,  20, -12, },
	{  -8,  40, 282, 738, 1107, 1063, 647, 217,  22, -12, },
	{  -9,  38, 277, 731, 1104, 1067, 655, 222,  23, -12, },
	{  -9,  36, 271, 723, 1101, 1071, 662, 227,  24, -10, },
	{  -9,  35, 265, 715, 1097, 1075, 670, 232,  26, -10, },
	{  -9,  33, 259, 708, 1094, 1079, 677, 238,  27, -10, },
	{ -10,  32, 254, 700, 1091, 1083, 685, 243,  28, -10, },
	{ -10,  30, 248, 693, 1087, 1087, 693, 248,  30, -10, },
	{ -10,  28, 243, 685, 1083, 1091, 700, 254,  32, -10, },
	{ -10,  27, 238, 677, 1079, 1094, 708, 259,  33,  -9, },
	{ -11,  26, 232, 670, 1075, 1097, 715, 265,  35,  -8, },
	{ -11,  24, 227, 662, 1071, 1101, 723, 271,  36,  -8, },
	{ -11,  23, 222, 655, 1067, 1104, 731, 277,  38, -10, },
	{ -12,  22, 217, 647, 1063, 1107, 738, 282,  40,  -8, },
	{ -12,  20, 212, 639, 1059, 1110, 746, 288,  42,  -8, },
	{ -12,  19, 207, 632, 1054, 1113, 753, 294,  44,  -8, },
	{ -12,  18, 202, 624, 1050, 1116, 761, 300,  46,  -9, },
	{ -13,  17, 197, 617, 1045, 1119, 768, 306,  48,  -8, },
	{ -13,  16, 192, 609, 1041, 1121, 776, 312,  50,  -8, },
	{ -13,  15, 187, 601, 1036, 1124, 783, 319,  52,  -8, },
	{ -14,  14, 183, 594, 1031, 1126, 790, 325,  54,  -7, },
	{ -14,  13, 178, 586, 1026, 1128, 798, 331,  56,  -6, },
	{ -14,  12, 174, 579, 1021, 1130, 805, 338,  59,  -8, },
	{ -15,  11, 169, 571, 1016, 1132, 812, 344,  61,  -5, },
	{ -15,  10, 165, 564, 1011, 1134, 820, 350,  63,  -6, },
	{ -16,   9, 160, 556, 1005, 1136, 827, 357,  66,  -4, },
	{ -16,   8, 156, 549, 1000, 1138, 834, 364,  68,  -5, },
	{ -16,   7, 152, 541,  995, 1139, 841, 370,  71,  -4, },
	{ -17,   7, 148, 534,  989, 1141, 848, 377,  74,  -5, },
	{ -17,   6, 144, 526,  983, 1142, 855, 384,  76,  -3, },
	{ -18,   5, 140, 519,  978, 1144, 862, 390,  79,  -3, },
	{ -18,   5, 136, 512,  972, 1145, 869, 397,  82,  -4, },
	{ -19,   4, 132, 504,  966, 1146, 876, 404,  85,  -2, },
	{ -19,   3, 128, 497,  960, 1147, 883, 411,  88,  -2, },
	{ -20,   3, 125, 490,  954, 1147, 890, 418,  91,  -2, },
	{ -20,   2, 121, 482,  948, 1148, 897, 425,  94,  -1, },
	{ -21,   2, 117, 475,  942, 1149, 904, 432,  97,  -1, },
	{ -21,   1, 114, 468,  936, 1149, 910, 439, 100,   0, },
	{ -22,   0, 110, 461,  930, 1149, 917, 446, 104,   1, }
};

/* ScalingRatio = 4 */
static const s16 XV_fixedcoeff_taps12_SR4[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_TAPS_12] = {
	{ -19,   0, 152, 498, 893, 1070,  893, 498, 152,   0, -19, -22, },
	{ -19,  -1, 147, 487, 879, 1059,  889, 499, 155,   1, -19,  19, },
	{ -19,  -2, 143, 480, 874, 1059,  894, 506, 159,   2, -19,  19, },
	{ -19,  -3, 139, 474, 869, 1059,  899, 512, 163,   3, -19,  19, },
	{ -19,  -4, 136, 468, 863, 1059,  904, 519, 167,   4, -19,  18, },
	{ -19,  -5, 132, 461, 858, 1058,  909, 525, 171,   5, -19,  20, },
	{ -19,  -5, 128, 455, 853, 1058,  913, 531, 175,   7, -19,  19, },
	{ -18,  -6, 125, 449, 847, 1057,  918, 538, 180,   8, -19,  17, },
	{ -18,  -7, 121, 443, 842, 1056,  923, 544, 184,   9, -19,  18, },
	{ -18,  -8, 118, 436, 836, 1056,  927, 551, 188,  10, -19,  19, },
	{ -18,  -8, 114, 430, 831, 1055,  932, 557, 193,  12, -19,  17, },
	{ -18,  -9, 111, 424, 825, 1054,  936, 564, 197,  13, -19,  18, },
	{ -18, -10, 107, 418, 819, 1053,  941, 570, 202,  14, -19,  19, },
	{ -18, -10, 104, 412, 814, 1052,  945, 577, 206,  16, -19,  17, },
	{ -18, -11, 101, 406, 808, 1050,  949, 583, 211,  17, -19,  19, },
	{ -18, -11,  98, 400, 802, 1049,  954, 590, 216,  19, -19,  16, },
	{ -18, -12,  95, 394, 796, 1048,  958, 596, 220,  20, -19,  18, },
	{ -18, -12,  92, 388, 791, 1046,  962, 603, 225,  22, -19,  16, },
	{ -18, -13,  89, 382, 785, 1045,  966, 609, 230,  24, -19,  16, },
	{ -18, -13,  86, 376, 779, 1043,  970, 616, 235,  25, -19,  16, },
	{ -18, -14,  83, 370, 773, 1041,  973, 622, 240,  27, -19,  18, },
	{ -18, -14,  80, 364, 767, 1039,  977, 629, 244,  29, -19,  18, },
	{ -18, -15,  77, 358, 761, 1037,  981, 635, 249,  31, -19,  19, },
	{ -18, -15,  74, 352, 755, 1035,  984, 642, 255,  33, -19,  18, },
	{ -18, -15,  71, 347, 749, 1033,  988, 648, 260,  35, -19,  17, },
	{ -18, -16,  69, 341, 743, 1031,  991, 654, 265,  36, -19,  19, },
	{ -18, -16,  66, 335, 736, 1029,  995, 661, 270,  38, -19,  19, },
	{ -18, -16,  64, 330, 730, 1026,  998, 667, 275,  41, -18,  17, },
	{ -18, -17,  61, 324, 724, 1024, 1001, 674, 280,  43, -18,  18, },
	{ -18, -17,  59, 318, 718, 1021, 1004, 680, 286,  45, -18,  18, },
	{ -18, -17,  56, 313, 712, 1019, 1007, 686, 291,  47, -18,  18, },
	{ -18, -17,  54, 307, 705, 1016, 1010, 693, 296,  49, -18,  19, },
	{ -18, -18,  51, 302, 699, 1013, 1013, 699, 302,  51, -18,  20, },
	{ -18, -18,  49, 296, 693, 1010, 1016, 705, 307,  54, -17,  19, },
	{ -18, -18,  47, 291, 686, 1007, 1019, 712, 313,  56, -17,  18, },
	{ -18, -18,  45, 286, 680, 1004, 1021, 718, 318,  59, -17,  18, },
	{ -18, -18,  43, 280, 674, 1001, 1024, 724, 324,  61, -17,  18, },
	{ -18, -18,  41, 275, 667,  998, 1026, 730, 330,  64, -16,  17, },
	{ -18, -19,  38, 270, 661,  995, 1029, 736, 335,  66, -16,  19, },
	{ -19, -19,  36, 265, 654,  991, 1031, 743, 341,  69, -16,  20, },
	{ -19, -19,  35, 260, 648,  988, 1033, 749, 347,  71, -15,  18, },
	{ -19, -19,  33, 255, 642,  984, 1035, 755, 352,  74, -15,  19, },
	{ -19, -19,  31, 249, 635,  981, 1037, 761, 358,  77, -15,  20, },
	{ -19, -19,  29, 244, 629,  977, 1039, 767, 364,  80, -14,  19, },
	{ -19, -19,  27, 240, 622,  973, 1041, 773, 370,  83, -14,  19, },
	{ -19, -19,  25, 235, 616,  970, 1043, 779, 376,  86, -13,  17, },
	{ -19, -19,  24, 230, 609,  966, 1045, 785, 382,  89, -13,  17, },
	{ -19, -19,  22, 225, 603,  962, 1046, 791, 388,  92, -12,  17, },
	{ -19, -19,  20, 220, 596,  958, 1048, 796, 394,  95, -12,  19, },
	{ -20, -19,  19, 216, 590,  954, 1049, 802, 400,  98, -11,  18, },
	{ -20, -19,  17, 211, 583,  949, 1050, 808, 406, 101, -11,  21, },
	{ -20, -19,  16, 206, 577,  945, 1052, 814, 412, 104, -10,  19, },
	{ -20, -19,  14, 202, 570,  941, 1053, 819, 418, 107, -10,  21, },
	{ -20, -19,  13, 197, 564,  936, 1054, 825, 424, 111,  -9,  20, },
	{ -20, -19,  12, 193, 557,  932, 1055, 831, 430, 114,  -8,  19, },
	{ -21, -19,  10, 188, 551,  927, 1056, 836, 436, 118,  -8,  22, },
	{ -21, -19,   9, 184, 544,  923, 1056, 842, 443, 121,  -7,  21, },
	{ -21, -19,   8, 180, 538,  918, 1057, 847, 449, 125,  -6,  20, },
	{ -21, -19,   7, 175, 531,  913, 1058, 853, 455, 128,  -5,  21, },
	{ -21, -19,   5, 171, 525,  909, 1058, 858, 461, 132,  -5,  22, },
	{ -21, -19,   4, 167, 519,  904, 1059, 863, 468, 136,  -4,  20, },
	{ -22, -19,   3, 163, 512,  899, 1059, 869, 474, 139,  -3,  22, },
	{ -22, -19,   2, 159, 506,  894, 1059, 874, 480, 143,  -2,  22, },
	{ -22, -19,   1, 155, 499,  889, 1059, 879, 487, 147,  -1,  22, }
};

#define XV_HSCALER_CTRL_WIDTH_HWREG_HFLTCOEFF			(16)
#define XV_HSCALER_CTRL_DEPTH_HWREG_HFLTCOEFF			(384)
#define XV_HSCALER_CTRL_ADDR_HWREG_PHASESH_V_BASE		(0x2000)
#define XV_HSCALER_CTRL_ADDR_HWREG_PHASESH_V_HIGH		(0x3fff)
#define XV_HSCALER_CTRL_WIDTH_HWREG_PHASESH_V			(18)
#define XV_HSCALER_CTRL_DEPTH_HWREG_PHASESH_V			(1920)
#define XV_HSCALER_CTRL_ADDR_HWREG_PHASEH_FIX			(0x4000)

/* H-scaler masks */
#define XV_HSCALER_PHASESH_V_OUTPUT_WR_EN			BIT(8)

/* V-scaler registers */
#define XV_VSCALER_CTRL_ADDR_AP_CTRL			(0x000)
#define XV_VSCALER_CTRL_ADDR_GIE			(0x004)
#define XV_VSCALER_CTRL_ADDR_IER			(0x008)
#define XV_VSCALER_CTRL_ADDR_ISR			(0x00c)
#define XV_VSCALER_CTRL_ADDR_HWREG_HEIGHTIN_DATA	(0x010)
#define XV_VSCALER_CTRL_ADDR_HWREG_WIDTH_DATA		(0x018)
#define XV_VSCALER_CTRL_ADDR_HWREG_HEIGHTOUT_DATA	(0x020)
#define XV_VSCALER_CTRL_ADDR_HWREG_LINERATE_DATA	(0x028)
#define XV_VSCALER_CTRL_ADDR_HWREG_COLORMODE_DATA	(0x030)
#define XV_VSCALER_CTRL_ADDR_HWREG_VFLTCOEFF_BASE	(0x800)
#define XV_VSCALER_CTRL_ADDR_HWREG_VFLTCOEFF_HIGH	(0xbff)

#define XV_VSCALER_CTRL_WIDTH_HWREG_VFLTCOEFF		(16)
#define XV_VSCALER_CTRL_DEPTH_HWREG_VFLTCOEFF		(384)

/* These bits are for xscaler feature flags */
#define XSCALER_CLK_PROP	BIT(0)
#define XSCALER_HPHASE_FIX	BIT(1)

/**
 * struct xscaler_feature - dt or IP property structure
 * @flags: Bitmask of properties enabled in IP or dt
 */
struct xscaler_feature {
	u32 flags;
};

/**
 * struct xscaler_device - Xilinx Scaler device structure
 * @xvip: Xilinx Video IP device
 * @pads: Scaler sub-device media pads
 * @formats: V4L2 media bus formats at the sink and source pads
 * @default_formats: default V4L2 media bus formats
 * @vip_formats: Xilinx Video IP format retrieved from the DT
 * @num_hori_taps: number of horizontal taps
 * @num_vert_taps: number of vertical taps
 * @max_num_phases: maximum number of phases
 * @pix_per_clk: Pixels per Clock cycle the IP operates upon
 * @max_pixels: The maximum number of pixels that the H-scaler examines
 * @max_lines: The maximum number of lines that the V-scaler examines
 * @H_phases: The phases needed to program the H-scaler for different taps
 * @H_phases_h: The phases needed to program the H-scaler for non step 4 use case
 * @hscaler_coeff: The complete array of H-scaler coefficients
 * @vscaler_coeff: The complete array of V-scaler coefficients
 * @is_polyphase: Track if scaling algorithm is polyphase or not
 * @rst_gpio: GPIO reset line to bring VPSS Scaler out of reset
 * @cfg: Pointer to scaler config structure
 * @aclk_axis: AXI4-Stream video interface clock
 * @aclk_ctrl: AXI4-Lite control interface clock
 */
struct xscaler_device {
	struct xvip_device xvip;

	struct media_pad pads[2];
	struct v4l2_mbus_framefmt formats[2];
	struct v4l2_mbus_framefmt default_formats[2];
	const struct xvip_video_format *vip_formats[2];

	u32 num_hori_taps;
	u32 num_vert_taps;
	u32 max_num_phases;
	u32 pix_per_clk;
	u32 max_pixels;
	u32 max_lines;
	u64 H_phases[XV_HSCALER_MAX_LINE_WIDTH];
	u64 H_phases_h[XV_HSCALER_MAX_LINE_WIDTH];
	short hscaler_coeff[XV_HSCALER_MAX_H_PHASES][XV_HSCALER_MAX_H_TAPS];
	short vscaler_coeff[XV_VSCALER_MAX_V_PHASES][XV_VSCALER_MAX_V_TAPS];
	bool is_polyphase;

	struct gpio_desc *rst_gpio;
	const struct xscaler_feature *cfg;
	struct clk *aclk_axis;
	struct clk *aclk_ctrl;
};

static const struct xscaler_feature xlnx_scaler_v2_2 = {
	.flags = XSCALER_CLK_PROP | XSCALER_HPHASE_FIX,
};

static const struct xscaler_feature xlnx_scaler_v1_0 = {
	.flags = XSCALER_CLK_PROP,
};

static const struct xscaler_feature xlnx_scaler = {
	.flags = 0,
};

static const struct of_device_id xscaler_of_id_table[] = {
	{ .compatible = "xlnx,v-vpss-scaler",
		.data = &xlnx_scaler},
	{ .compatible = "xlnx,v-vpss-scaler-1.0",
		.data = &xlnx_scaler_v1_0},
	{ .compatible = "xlnx,v-vpss-scaler-2.2",
		.data = &xlnx_scaler_v2_2},
	{ /* end of table */ }
};
MODULE_DEVICE_TABLE(of, xscaler_of_id_table);

static inline struct xscaler_device *to_scaler(struct v4l2_subdev *subdev)
{
	return container_of(subdev, struct xscaler_device, xvip.subdev);
}

static void
xv_hscaler_calculate_phases(struct xscaler_device *xscaler,
			    u32 width_in, u32 width_out, u32 pixel_rate)
{
	unsigned int loop_width;
	unsigned int x, s;
	int offset = 0;
	int xwrite_pos = 0;
	bool output_write_en;
	bool get_new_pix;
	u64 phaseH;
	u64 array_idx = 0;
	int nr_rds = 0;
	int nr_rds_clck;
	unsigned int nphases = xscaler->max_num_phases;
	unsigned int nppc = xscaler->pix_per_clk;
	unsigned int shift = XHSC_STEP_PRECISION_SHIFT - ilog2(nphases);

	memset(xscaler->H_phases, 0, sizeof(xscaler->H_phases));
	memset(xscaler->H_phases_h, 0, sizeof(xscaler->H_phases_h));

	loop_width = max_t(u32, width_in, width_out);
	loop_width = ALIGN(loop_width + nppc - 1, nppc);

	for (x = 0; x < loop_width; x++) {
		xscaler->H_phases[x] = 0;
		xscaler->H_phases_h[x] = 0;
		nr_rds_clck = 0;
		for (s = 0; s < nppc; s++) {
			phaseH = (offset >> shift) & (nphases - 1);
			get_new_pix = false;
			output_write_en = false;
			if ((offset >> XHSC_STEP_PRECISION_SHIFT) != 0) {
				/* read a new input sample */
				get_new_pix = true;
				offset -= (1 << XHSC_STEP_PRECISION_SHIFT);
				array_idx++;
			}

			if (((offset >> XHSC_STEP_PRECISION_SHIFT) == 0) &&
			    (xwrite_pos < width_out)) {
				/* produce a new output sample */
				offset += pixel_rate;
				output_write_en = true;
				xwrite_pos++;
			}

			if (nppc == XSCALER_PPC_8) {
				if (s < XHSC_HPHASE_STEP_4) {
					xscaler->H_phases[x] |=
						((u64)phaseH <<
						 (s * XHSC_HPHASE_MUL_8PPC));
					xscaler->H_phases[x] |=
						((u64)array_idx <<
						 (XHSC_HPHASE_SHIFT_BY_6 +
						  (s * XHSC_HPHASE_MUL_8PPC)));
					if (output_write_en)
						xscaler->H_phases[x] |=
							((u64)1 <<
							 (XHSC_HPHASE_MULTIPLIER_8PPC +
							  (s * XHSC_HPHASE_MUL_8PPC)));
				} else {
					xscaler->H_phases_h[x] |=
						((u64)phaseH <<
						 ((s - XHSC_HPHASE_STEP_4)
						  * XHSC_HPHASE_MUL_8PPC));
					xscaler->H_phases_h[x] |=
						((u64)array_idx <<
						 (XHSC_HPHASE_SHIFT_BY_6 +
						  ((s - XHSC_HPHASE_STEP_4)
						   * XHSC_HPHASE_MUL_8PPC)));
					if (output_write_en)
						xscaler->H_phases_h[x] |=
							((u64)1 <<
							 (XHSC_HPHASE_MULTIPLIER_8PPC +
							  ((s - XHSC_HPHASE_STEP_4)
							   * XHSC_HPHASE_MUL_8PPC)));
				}
			} else if (nppc == XSCALER_PPC_4) {
				xscaler->H_phases[x] |=
					((u64)phaseH <<
					 (s * XHSC_HPHASE_MUL_4PPC));
				xscaler->H_phases[x] |=
					((u64)array_idx <<
					 (XHSC_HPHASE_SHIFT_BY_6 +
					  (s * XHSC_HPHASE_MUL_4PPC)));
				if (output_write_en)
					xscaler->H_phases[x] |=
						((u64)1 <<
						 (XHSC_HPHASE_MULTIPLIER + s *
						  XHSC_HPHASE_MUL_4PPC));
			} else {
				xscaler->H_phases[x] |=
					(phaseH <<
					 (s * XHSC_HPHASE_MULTIPLIER));
				xscaler->H_phases[x] |=
					(array_idx <<
					 (XHSC_HPHASE_SHIFT_BY_6 +
					  (s * XHSC_HPHASE_MULTIPLIER)));

				if (output_write_en)
					xscaler->H_phases[x] |=
						(XV_HSCALER_PHASESH_V_OUTPUT_WR_EN <<
						 (s * XHSC_HPHASE_MULTIPLIER));
			}

			if (get_new_pix)
				nr_rds_clck++;
		}
		if (array_idx >= nppc)
			array_idx &= (nppc - 1);

		nr_rds += nr_rds_clck;
		if (nr_rds >= nppc)
			nr_rds -= nppc;
	}
}

static void
xv_hscaler_load_ext_coeff(struct xscaler_device *xscaler,
			  const short *coeff, u32 ntaps)
{
	unsigned int i, j, pad, offset;
	u32 nphases = xscaler->max_num_phases;

	/* Determine if coefficient needs padding (effective vs. max taps) */
	pad = XV_HSCALER_MAX_H_TAPS - ntaps;
	offset = pad >> 1;
	dev_dbg(xscaler->xvip.dev,
		"%s : Pad = %d Offset = %d Nphases = %d ntaps = %d",
			__func__, pad, offset, nphases, ntaps);

	/* Load coefficients into scaler coefficient table */
	for (i = 0; i < nphases; i++) {
		for (j = 0; j < ntaps; ++j)
			xscaler->hscaler_coeff[i][j + offset] =
						coeff[i * ntaps + j];
	}

	if (pad) { /* effective taps < max_taps */
		for (i = 0; i < nphases; i++) {
			/* pad left */
			for (j = 0; j < offset; j++)
				xscaler->hscaler_coeff[i][j] = 0;
			/* pad right */
			j = ntaps + offset;
			for (; j < XV_HSCALER_MAX_H_TAPS; j++)
				xscaler->hscaler_coeff[i][j] = 0;
		}
	}
}

static const short *xv_select_coeff(struct xscaler_device *xscaler,
				    u32 in, u32 out, u32 *ntaps)
{
	const short *coeff = NULL;

	/*
	 * Scale Down Mode will use dynamic filter selection logic
	 * Scale Up Mode (including 1:1) will always use 6 tap filter
	 */
	if (out < in) {
		u16 scale_ratio = (in * 10) / out;

		/* Since XV_HSCALER_TAPS_* is same as XV_VSCALER_TAPS_* */
		switch (*ntaps) {
		case XV_HSCALER_TAPS_6:
			*ntaps = XV_HSCALER_TAPS_6;
			if (scale_ratio > 35)
				coeff = &XV_fixedcoeff_taps6_SR4[0][0];
			else if (scale_ratio > 25)
				coeff = &XV_fixedcoeff_taps6_SR3[0][0];
			else if (scale_ratio > 15)
				coeff = &XV_fixedcoeff_taps6_SR2[0][0];
			else
				coeff = &XV_fixedcoeff_taps6_SR1p2[0][0];
			break;
		case XV_HSCALER_TAPS_8:
			if (scale_ratio > 35) {
				coeff = &XV_fixedcoeff_taps8_SR4[0][0];
				*ntaps = XV_HSCALER_TAPS_8;
			} else if (scale_ratio > 25) {
				coeff = &XV_fixedcoeff_taps8_SR3[0][0];
				*ntaps = XV_HSCALER_TAPS_8;
			} else if (scale_ratio > 15) {
				coeff = &XV_fixedcoeff_taps8_SR2[0][0];
				*ntaps = XV_HSCALER_TAPS_8;
			} else {
				coeff = &XV_fixedcoeff_taps6_SR1p2[0][0];
				*ntaps = XV_HSCALER_TAPS_6;
			}
			break;
		case XV_HSCALER_TAPS_10:
			if (scale_ratio > 35) {
				coeff = &XV_fixedcoeff_taps10_SR4[0][0];
				*ntaps = XV_HSCALER_TAPS_10;
			} else if (scale_ratio > 25) {
				coeff = &XV_fixedcoeff_taps10_SR3[0][0];
				*ntaps = XV_HSCALER_TAPS_10;
			} else if (scale_ratio > 15) {
				coeff = &XV_fixedcoeff_taps8_SR2[0][0];
				*ntaps = XV_HSCALER_TAPS_8;
			} else {
				coeff = &XV_fixedcoeff_taps6_SR1p2[0][0];
				*ntaps = XV_HSCALER_TAPS_6;
			}
			break;
		case XV_HSCALER_TAPS_12:
			if (scale_ratio > 35) {
				coeff = &XV_fixedcoeff_taps12_SR4[0][0];
				*ntaps = XV_HSCALER_TAPS_12;
			} else if (scale_ratio > 25) {
				coeff = &XV_fixedcoeff_taps10_SR3[0][0];
				*ntaps = XV_HSCALER_TAPS_10;
			} else if (scale_ratio > 15) {
				coeff = &XV_fixedcoeff_taps8_SR2[0][0];
				*ntaps = XV_HSCALER_TAPS_8;
			} else {
				coeff = &XV_fixedcoeff_taps6_SR1p2[0][0];
				*ntaps = XV_HSCALER_TAPS_6;
			}
			break;
		default:
			dev_err(xscaler->xvip.dev,
				"Unsupported number of taps = %d",
				*ntaps);
		}
	} else {
		dev_dbg(xscaler->xvip.dev, "scaler : scale up 6 tap");
		coeff = &XV_lanczos2_taps6[0][0];
		*ntaps = XV_HSCALER_TAPS_6;
	}

	return coeff;
}

/**
 * xv_hscaler_select_coeff - Selection of H-Scaler coefficients of operation
 * @xscaler: VPSS Scaler device information
 * @width_in: Width of input video
 * @width_out: Width of desired output video
 *
 * There are instances when a N-tap filter might operate in an M-tap
 * configuration where N > M.
 *
 * For example :
 * Depending on the ratio of scaling (while downscaling), a 12-tap
 * filter may operate with 10 tap coefficients and zero-pads the remaining
 * coefficients.
 *
 * While upscaling the driver will program 6-tap filter coefficients
 * in any N-tap configurations (for N >= 6).
 *
 * This selection is adopted by the as it gives optimal
 * video output determined by repeated testing of the IP
 *
 * Return: Will return 0 if successful. Returns -EINVAL on an unsupported
 * H-scaler number of taps.
 */
static int
xv_hscaler_select_coeff(struct xscaler_device *xscaler,
			u32 width_in, u32 width_out)
{
	const short *coeff;
	u32 ntaps = xscaler->num_hori_taps;

	coeff = xv_select_coeff(xscaler, width_in, width_out, &ntaps);
	if (!coeff)
		return -EINVAL;

	xv_hscaler_load_ext_coeff(xscaler, coeff, ntaps);
	return 0;
}

static void xv_hscaler_set_coeff(struct xscaler_device *xscaler)
{
	int val, i, j, offset, rd_indx;
	u32 ntaps = xscaler->num_hori_taps;
	u32 nphases = xscaler->max_num_phases;
	u32 base_addr;

	offset = (XV_HSCALER_MAX_H_TAPS - ntaps) / 2;
	base_addr = V_HSCALER_OFF + XV_HSCALER_CTRL_ADDR_HWREG_HFLTCOEFF_BASE;
	for (i = 0; i < nphases; i++) {
		for (j = 0; j < ntaps / 2; j++) {
			rd_indx = j * 2 + offset;
			val = (xscaler->hscaler_coeff[i][rd_indx + 1] <<
			       XSCALER_BITSHIFT_16) |
			       (xscaler->hscaler_coeff[i][rd_indx] &
			       XHSC_MASK_LOW_16BITS);
			 xvip_write(&xscaler->xvip, base_addr +
				    ((i * ntaps / 2 + j) * 4), val);
		}
	}
}

static void
xv_vscaler_load_ext_coeff(struct xscaler_device *xscaler,
			  const short *coeff, u32 ntaps)
{
	int i, j, pad, offset;
	u32 nphases = xscaler->max_num_phases;

	/* Determine if coefficient needs padding (effective vs. max taps) */
	pad = XV_VSCALER_MAX_V_TAPS - ntaps;
	offset = pad ? (pad >> 1) : 0;

	dev_dbg(xscaler->xvip.dev,
		"%s : Pad = %d Offset = %d Nphases = %d ntaps = %d",
			__func__, pad, offset, nphases, ntaps);

	/* Load User defined coefficients into scaler coefficient table */
	for (i = 0; i < nphases; i++) {
		for (j = 0; j < ntaps; ++j)
			xscaler->vscaler_coeff[i][j + offset] =
						coeff[i * ntaps + j];
	}

	if (pad) { /* effective taps < max_taps */
		for (i = 0; i < nphases; i++) {
			/* pad left */
			for (j = 0; j < offset; j++)
				xscaler->vscaler_coeff[i][j] = 0;
			/* pad right */
			j = ntaps + offset;
			for (; j < XV_VSCALER_MAX_V_TAPS; j++)
				xscaler->vscaler_coeff[i][j] = 0;
		}
	}
}

static void xv_vscaler_set_coeff(struct xscaler_device *xscaler)
{
	u32 nphases = xscaler->max_num_phases;
	u32 ntaps   = xscaler->num_vert_taps;
	int val, i, j, offset, rd_indx;
	u32 base_addr;

	offset = (XV_VSCALER_MAX_V_TAPS - ntaps) / 2;
	base_addr = V_VSCALER_OFF + XV_VSCALER_CTRL_ADDR_HWREG_VFLTCOEFF_BASE;

	for (i = 0; i < nphases; i++) {
		for (j = 0; j < ntaps / 2; j++) {
			rd_indx = j * 2 + offset;
			val = (xscaler->vscaler_coeff[i][rd_indx + 1] <<
			       XSCALER_BITSHIFT_16) |
			       (xscaler->vscaler_coeff[i][rd_indx] &
			       XVSC_MASK_LOW_16BITS);
			xvip_write(&xscaler->xvip,
				   base_addr + ((i * ntaps / 2 + j) * 4), val);
		}
	}
}

/**
 * xv_vscaler_select_coeff - Selection of V-Scaler coefficients of operation
 * @xscaler: VPSS Scaler device information
 * @height_in: Height of input video
 * @height_out: Height of desired output video
 *
 * There are instances when a N-tap filter might operate in an M-tap
 * configuration where N > M.
 *
 * For example :
 * Depending on the ratio of scaling (while downscaling), a 10-tap
 * filter may operate with 6 tap coefficients and zero-pads the remaining
 * coefficients.
 *
 * While upscaling the driver will program 6-tap filter coefficients
 * in any N-tap configurations (for N >= 6).
 *
 * This selection is adopted by the as it gives optimal
 * video output determined by repeated testing of the IP
 *
 * Return: Will return 0 if successful. Returns -EINVAL on an unsupported
 * V-scaler number of taps.
 */
static int
xv_vscaler_select_coeff(struct xscaler_device *xscaler,
			u32 height_in, u32 height_out)
{
	const short *coeff;
	u32 ntaps = xscaler->num_vert_taps;

	coeff = xv_select_coeff(xscaler, height_in, height_out, &ntaps);
	if (!coeff)
		return -EINVAL;

	xv_vscaler_load_ext_coeff(xscaler, coeff, ntaps);
	return 0;
}

/*
 * V4L2 Subdevice Video Operations
 */

static inline void
xv_procss_disable_block(struct xvip_device *xvip, u32 channel, u32 ip_block)
{
	xvip_clr(xvip, ((channel - 1) * XGPIO_CHAN_OFFSET) +
		 XGPIO_DATA_OFFSET + S_AXIS_RESET_OFF,
		 ip_block);
}

static inline void
xv_procss_enable_block(struct xvip_device *xvip, u32 channel, u32 ip_block)
{
	xvip_set(xvip, ((channel - 1) * XGPIO_CHAN_OFFSET) +
		 XGPIO_DATA_OFFSET + S_AXIS_RESET_OFF,
		 ip_block);
}

static void xscaler_reset(struct xscaler_device *xscaler)
{
	xv_procss_disable_block(&xscaler->xvip, XGPIO_CH_RESET_SEL,
				XGPIO_RESET_MASK_ALL_BLOCKS);
	xv_procss_enable_block(&xscaler->xvip, XGPIO_CH_RESET_SEL,
			       XGPIO_RESET_MASK_IP_AXIS);
}

static int
xv_vscaler_setup_video_fmt(struct xscaler_device *xscaler, u32 code_in)
{
	u32 video_in;

	switch (code_in) {
	case MEDIA_BUS_FMT_VYYUYY8_1X24:
	case MEDIA_BUS_FMT_VYYUYY10_4X20:
		dev_dbg(xscaler->xvip.dev,
			"Vscaler Input Media Format YUV 420");
		video_in = XVIDC_CSF_YCRCB_420;
		break;
	case MEDIA_BUS_FMT_UYVY8_1X16:
	case MEDIA_BUS_FMT_UYVY10_1X20:
		dev_dbg(xscaler->xvip.dev,
			"Vscaler Input Media Format YUV 422");
		video_in = XVIDC_CSF_YCRCB_422;
		break;
	case MEDIA_BUS_FMT_VUY8_1X24:
	case MEDIA_BUS_FMT_VUY10_1X30:
	case MEDIA_BUS_FMT_VUY12_1X36:
		dev_dbg(xscaler->xvip.dev,
			"Vscaler Input Media Format YUV 444");
		video_in = XVIDC_CSF_YCRCB_444;
		break;
	case MEDIA_BUS_FMT_RBG888_1X24:
	case MEDIA_BUS_FMT_RBG101010_1X30:
		dev_dbg(xscaler->xvip.dev,
			"Vscaler Input Media Format RGB");
		video_in = XVIDC_CSF_RGB;
		break;
	default:
		dev_err(xscaler->xvip.dev,
			"Vscaler Unsupported Input Media Format 0x%x",
			code_in);
		return -EINVAL;
	}
	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_HWREG_COLORMODE_DATA,
		   video_in);
	/*
	 * Vscaler will upscale to YUV 422 before
	 * Hscaler starts operation
	 */
	if (video_in == XVIDC_CSF_YCRCB_420)
		return XVIDC_CSF_YCRCB_422;
	return video_in;
}

static int xv_hscaler_setup_video_fmt(struct xscaler_device *xscaler,
				      u32 code_out, u32 vsc_out)
{
	u32 video_out;

	switch (vsc_out) {
	case XVIDC_CSF_YCRCB_422:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Input Media Format is YUV 422");
		break;
	case XVIDC_CSF_YCRCB_444:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Input Media Format is YUV 444");
		break;
	case XVIDC_CSF_RGB:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Input Media Format is RGB");
		break;
	default:
		dev_err(xscaler->xvip.dev,
			"Hscaler got unsupported format from Vscaler");
		return -EINVAL;
	}

	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		XV_HSCALER_CTRL_ADDR_HWREG_COLORMODE_DATA,
		vsc_out);

	switch (code_out) {
	case MEDIA_BUS_FMT_VYYUYY8_1X24:
	case MEDIA_BUS_FMT_VYYUYY10_4X20:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Output Media Format YUV 420\n");
		video_out = XVIDC_CSF_YCRCB_420;
		break;
	case MEDIA_BUS_FMT_UYVY8_1X16:
	case MEDIA_BUS_FMT_UYVY10_1X20:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Output Media Format YUV 422\n");
		video_out = XVIDC_CSF_YCRCB_422;
		break;
	case MEDIA_BUS_FMT_VUY8_1X24:
	case MEDIA_BUS_FMT_VUY10_1X30:
	case MEDIA_BUS_FMT_VUY12_1X36:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Output Media Format YUV 444\n");
		video_out = XVIDC_CSF_YCRCB_444;
		break;
	case MEDIA_BUS_FMT_RBG888_1X24:
	case MEDIA_BUS_FMT_RBG101010_1X30:
		dev_dbg(xscaler->xvip.dev,
			"Hscaler Output Media Format RGB\n");
		video_out = XVIDC_CSF_RGB;
		break;
	default:
		dev_err(xscaler->xvip.dev,
			"Hscaler Unsupported Output Media Format 0x%x",
			code_out);
		return -EINVAL;
	}
	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_HWREG_COLORMODEOUT_DATA,
		   video_out);
	return 0;
}

static void
xv_hscaler_set_phases(struct xscaler_device *xscaler)
{
	u32 loop_width;
	u32 index = 0, val;
	u32 offset = 0, i, j = 0, lsb, msb, msb2;
	u64 phasehdata, phasehdata_h;

	loop_width = xscaler->max_pixels / xscaler->pix_per_clk;

	if (xscaler->cfg->flags & XSCALER_HPHASE_FIX) {
		offset = V_HSCALER_OFF +
			XV_HSCALER_CTRL_ADDR_HWREG_PHASEH_FIX;
	} else {
		offset = V_HSCALER_OFF +
			XV_HSCALER_CTRL_ADDR_HWREG_PHASESH_V_BASE;
	}

	switch (xscaler->pix_per_clk) {
	case XSCALER_PPC_1:
		/*
		 * phaseH is 64 bits but only lower 16 bits of each entry
		 * is valid .Form a 32 bit word with 16bit LSB from 2
		 * consecutive entries. Need 1 32b write to get 2 entries
		 * into IP registers (i is array loc and index is
		 * address offset)
		 */
		for (i = 0; i < loop_width; i += 2) {
			lsb = lower_32_bits(xscaler->H_phases[i] &
					    XHSC_MASK_LOW_16BITS);
			msb = lower_32_bits(xscaler->H_phases[i + 1] &
					    XHSC_MASK_LOW_16BITS);
			val = (msb << 16 | lsb);
			xvip_write(&xscaler->xvip, offset + (index * 4), val);
			++index;
		}
		dev_dbg(xscaler->xvip.dev,
			"%s : Operating in 1 PPC design", __func__);
		return;
	case XSCALER_PPC_2:
		/*
		 * PhaseH is 64bits but only lower 32b of each entry is valid
		 * Need 1 32b write to get each entry into IP registers
		 */
		for (i = 0; i < loop_width; i++) {
			val = lower_32_bits(xscaler->H_phases[i] &
					    XHSC_MASK_LOW_32BITS);
			xvip_write(&xscaler->xvip, offset + (i * 4), val);
		}
		dev_dbg(xscaler->xvip.dev,
			"%s : Operating in 2 PPC design", __func__);
		return;
	case XSCALER_PPC_4:
		/*
		 * PhaseH is 64bits and each entry has valid 32b MSB & LSB
		 * Need 2 32b writes to get each entry into IP registers
		 * (index is array loc and offset is address offset)
		 */
		for (i = 0; i < loop_width; i++) {
			phasehdata = xscaler->H_phases[index++];
			lsb = (u32)(phasehdata & XHSC_MASK_LOW_32BITS);
			msb = (u32)((phasehdata >> 32) & XHSC_MASK_LOW_32BITS);
			xvip_write(&xscaler->xvip, offset + (j * 4), lsb);
			xvip_write(&xscaler->xvip, offset + ((j + 1) * 4), msb);
			j += 2;
		}
		dev_dbg(xscaler->xvip.dev,
			"%s : Operating in 4 PPC design", __func__);
		return;
	case XSCALER_PPC_8:
		/*
		 * phasehdata and phasehdata_h are 64bits and each entry has valid 44
		 * bits. phasehdata has lower 44 bits and phasehdata_h has higher 44 bits.
		 * Need to form 3 32b writes from the total 88 bits and
		 * Write each 32bits into IP registers.
		 * index is array loc and offset is address offset.
		 */
		for (i = 0; i < loop_width; i++) {
			phasehdata = xscaler->H_phases[index];
			phasehdata_h = xscaler->H_phases_h[index];

			lsb = lower_32_bits(phasehdata);
			msb = upper_32_bits(phasehdata);
			val = (u32)(FIELD_PREP(XHSC_MASK_LOW_20BITS, phasehdata_h));
			msb |= (val << XSCALER_PHDATA_SHIFT);
			msb2 = (lower_32_bits(phasehdata_h)) >> XSCALER_PHDATA_H_SHIFT;
			val = ((u32)(FIELD_PREP(XHSC_MASK_LOW_12BITS,
							upper_32_bits(phasehdata_h))));
			msb2 |= (val << XSCALER_PHDATA_SHIFT);

			xvip_write(&xscaler->xvip, offset + (j * 4), lsb);
			xvip_write(&xscaler->xvip, offset + ((j + 1) * 4), msb);
			xvip_write(&xscaler->xvip, offset + ((j + 2) * 4), msb2);
			/* offset + ((j + 3) * 4) register is reserved, so increment offset by 4 */
			j += 4;
			index++;
		}
		dev_dbg(xscaler->xvip.dev, "Operating in 8 ppc design");
		return;
	default:
		dev_warn(xscaler->xvip.dev, "%s : %d unsupported ppc design!!!\n",
			 __func__, xscaler->pix_per_clk);
	}
}

static int xscaler_s_stream(struct v4l2_subdev *subdev, int enable)
{
	struct xscaler_device *xscaler = to_scaler(subdev);
	u32 width_in, width_out;
	u32 height_in, height_out;
	u32 code_in, code_out;
	u32 pixel_rate;
	u32 line_rate;
	int ret;

	if (!enable) {
		dev_dbg(xscaler->xvip.dev, "%s: Stream Off", __func__);
		/* Reset the Global IP Reset through PS GPIO */
		gpiod_set_value_cansleep(xscaler->rst_gpio,
					 XSCALER_RESET_ASSERT);
		gpiod_set_value_cansleep(xscaler->rst_gpio,
					 XSCALER_RESET_DEASSERT);
		xscaler_reset(xscaler);
		memset(xscaler->H_phases, 0, sizeof(xscaler->H_phases));
		memset(xscaler->H_phases_h, 0, sizeof(xscaler->H_phases_h));
		return 0;
	}

	dev_dbg(xscaler->xvip.dev, "%s: Stream On", __func__);

	/* Extract Sink Pad Information */
	width_in = xscaler->formats[XVIP_PAD_SINK].width;
	height_in = xscaler->formats[XVIP_PAD_SINK].height;
	code_in = xscaler->formats[XVIP_PAD_SINK].code;

	/* Extract Source Pad Information */
	width_out = xscaler->formats[XVIP_PAD_SOURCE].width;
	height_out = xscaler->formats[XVIP_PAD_SOURCE].height;
	code_out = xscaler->formats[XVIP_PAD_SOURCE].code;

	/*
	 * V Scaler is before H Scaler
	 * V-Scaler_setup
	 */
	line_rate = (height_in * STEP_PRECISION) / height_out;

	if (xscaler->is_polyphase) {
		ret = xv_vscaler_select_coeff(xscaler, height_in, height_out);
		if (ret < 0)
			return ret;
		xv_vscaler_set_coeff(xscaler);
	}

	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_HWREG_HEIGHTIN_DATA, height_in);
	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_HWREG_WIDTH_DATA, width_in);
	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_HWREG_HEIGHTOUT_DATA, height_out);
	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_HWREG_LINERATE_DATA, line_rate);
	ret = xv_vscaler_setup_video_fmt(xscaler, code_in);
	if (ret < 0)
		return ret;

	/* H-Scaler_setup */
	pixel_rate = (width_in * STEP_PRECISION) / width_out;

	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_HWREG_HEIGHT_DATA, height_out);
	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_HWREG_WIDTHIN_DATA, width_in);
	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_HWREG_WIDTHOUT_DATA, width_out);
	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_HWREG_PIXELRATE_DATA, pixel_rate);
	ret = xv_hscaler_setup_video_fmt(xscaler, code_out, ret);
	if (ret < 0)
		return ret;

	if (xscaler->is_polyphase) {
		ret = xv_hscaler_select_coeff(xscaler, width_in, width_out);
		if (ret < 0)
			return ret;
		xv_hscaler_set_coeff(xscaler);
	}

	xv_hscaler_calculate_phases(xscaler, width_in, width_out, pixel_rate);
	xv_hscaler_set_phases(xscaler);

	/* Start Scaler sub-cores */
	xvip_write(&xscaler->xvip, V_HSCALER_OFF +
		   XV_HSCALER_CTRL_ADDR_AP_CTRL, XSCALER_STREAM_ON);
	xvip_write(&xscaler->xvip, V_VSCALER_OFF +
		   XV_VSCALER_CTRL_ADDR_AP_CTRL, XSCALER_STREAM_ON);
	xv_procss_enable_block(&xscaler->xvip, XGPIO_CH_RESET_SEL,
			       XGPIO_RESET_MASK_VIDEO_IN);
	return 0;
}

/*
 * V4L2 Subdevice Pad Operations
 */

static int xscaler_enum_frame_size(struct v4l2_subdev *subdev,
				   struct v4l2_subdev_state *sd_state,
				   struct v4l2_subdev_frame_size_enum *fse)
{
	struct v4l2_mbus_framefmt *format;
	struct xscaler_device *xscaler = to_scaler(subdev);

	format = v4l2_subdev_state_get_format(sd_state, fse->pad);
	if (fse->index || fse->code != format->code)
		return -EINVAL;

	fse->min_width = XSCALER_MIN_WIDTH;
	fse->max_width = xscaler->max_pixels;
	fse->min_height = XSCALER_MIN_HEIGHT;
	fse->max_height = xscaler->max_lines;

	return 0;
}

static struct v4l2_mbus_framefmt *
__xscaler_get_pad_format(struct xscaler_device *xscaler,
			 struct v4l2_subdev_state *sd_state,
			 unsigned int pad, u32 which)
{
	struct v4l2_mbus_framefmt *format;

	switch (which) {
	case V4L2_SUBDEV_FORMAT_TRY:
		format = v4l2_subdev_state_get_format(sd_state, pad);
		break;
	case V4L2_SUBDEV_FORMAT_ACTIVE:
		format = &xscaler->formats[pad];
		break;
	default:
		format = NULL;
		break;
	}

	return format;
}

static int xscaler_get_format(struct v4l2_subdev *subdev,
			      struct v4l2_subdev_state *sd_state,
			      struct v4l2_subdev_format *fmt)
{
	struct xscaler_device *xscaler = to_scaler(subdev);
	struct v4l2_mbus_framefmt *format;

	format = __xscaler_get_pad_format(xscaler, sd_state, fmt->pad,
					  fmt->which);
	if (!format)
		return -EINVAL;

	fmt->format = *format;

	return 0;
}

static int xscaler_set_format(struct v4l2_subdev *subdev,
			      struct v4l2_subdev_state *sd_state,
			      struct v4l2_subdev_format *fmt)
{
	struct xscaler_device *xscaler = to_scaler(subdev);
	struct v4l2_mbus_framefmt *format;

	format = __xscaler_get_pad_format(xscaler, sd_state, fmt->pad,
					  fmt->which);
	if (!format)
		return -EINVAL;

	*format = fmt->format;

	format->width = clamp_t(unsigned int, fmt->format.width,
				XSCALER_MIN_WIDTH, xscaler->max_pixels);
	format->height = clamp_t(unsigned int, fmt->format.height,
				 XSCALER_MIN_HEIGHT, xscaler->max_lines);
	format->code = fmt->format.code;
	fmt->format = *format;
	return 0;
}

/*
 * V4L2 Subdevice Operations
 */

static int
xscaler_open(struct v4l2_subdev *subdev, struct v4l2_subdev_fh *fh)
{
	struct xscaler_device *xscaler = to_scaler(subdev);
	struct v4l2_mbus_framefmt *format;

	/* Initialize with default formats */
	format = v4l2_subdev_state_get_format(fh->state, XVIP_PAD_SINK);
	*format = xscaler->default_formats[XVIP_PAD_SINK];

	format = v4l2_subdev_state_get_format(fh->state, XVIP_PAD_SOURCE);
	*format = xscaler->default_formats[XVIP_PAD_SOURCE];

	return 0;
}

static int
xscaler_close(struct v4l2_subdev *subdev, struct v4l2_subdev_fh *fh)
{
	return 0;
}

static struct v4l2_subdev_video_ops xscaler_video_ops = {
	.s_stream = xscaler_s_stream,
};

static struct v4l2_subdev_pad_ops xscaler_pad_ops = {
	.enum_mbus_code		= xvip_enum_mbus_code,
	.enum_frame_size	= xscaler_enum_frame_size,
	.get_fmt		= xscaler_get_format,
	.set_fmt		= xscaler_set_format,
};

static struct v4l2_subdev_ops xscaler_ops = {
	.video  = &xscaler_video_ops,
	.pad    = &xscaler_pad_ops,
};

static const struct v4l2_subdev_internal_ops xscaler_internal_ops = {
	.open	= xscaler_open,
	.close	= xscaler_close,
};

/*
 * Media Operations
 */

static const struct media_entity_operations xscaler_media_ops = {
	.link_validate = v4l2_subdev_link_validate,
};

/*
 * Platform Device Driver
 */

static int xscaler_parse_of(struct xscaler_device *xscaler)
{
	struct device *dev = xscaler->xvip.dev;
	struct device_node *node = xscaler->xvip.dev->of_node;
	const struct xvip_video_format *vip_format;
	struct device_node *ports;
	struct device_node *port;
	int ret;
	u32 port_id, dt_ppc = 0;

	if (xscaler->cfg->flags & XSCALER_CLK_PROP) {
		xscaler->aclk_axis = devm_clk_get(dev, "aclk_axis");
		if (IS_ERR(xscaler->aclk_axis)) {
			ret = PTR_ERR(xscaler->aclk_axis);
			dev_err(dev, "failed to get aclk_axis (%d)\n", ret);
			return ret;
		}
		xscaler->aclk_ctrl = devm_clk_get(dev, "aclk_ctrl");
		if (IS_ERR(xscaler->aclk_ctrl)) {
			ret = PTR_ERR(xscaler->aclk_ctrl);
			dev_err(dev, "failed to get aclk_ctrl (%d)\n", ret);
			return ret;
		}
	} else {
		dev_info(dev, "assuming all required clocks are enabled!\n");
	}

	ret = of_property_read_u32(node, "xlnx,max-height",
				   &xscaler->max_lines);
	if (ret < 0) {
		dev_err(dev, "xlnx,max-height is missing!");
		return -EINVAL;
	} else if (xscaler->max_lines > XSCALER_MAX_HEIGHT ||
		   xscaler->max_lines < XSCALER_MIN_HEIGHT) {
		dev_err(dev, "Invalid height in dt");
		return -EINVAL;
	}

	ret = of_property_read_u32(node, "xlnx,max-width",
				   &xscaler->max_pixels);
	if (ret < 0) {
		dev_err(dev, "xlnx,max-width is missing!");
		return -EINVAL;
	} else if (xscaler->max_pixels > XSCALER_MAX_WIDTH ||
		   xscaler->max_pixels < XSCALER_MIN_WIDTH) {
		dev_err(dev, "Invalid width in dt");
		return -EINVAL;
	}

	ports = of_get_child_by_name(node, "ports");
	if (!ports)
		ports = node;

	/* Get the format description for each pad */
	for_each_child_of_node(ports, port) {
		if (port->name && (of_node_cmp(port->name, "port") == 0)) {
			vip_format = xvip_of_get_format(port);
			if (IS_ERR(vip_format)) {
				dev_err(dev, "invalid format in DT");
				return PTR_ERR(vip_format);
			}

			ret = of_property_read_u32(port, "reg", &port_id);
			if (ret < 0) {
				dev_err(dev, "No reg in DT");
				return ret;
			}

			if (port_id != 0 && port_id != 1) {
				dev_err(dev, "Invalid reg in DT");
				return -EINVAL;
			}
			xscaler->vip_formats[port_id] = vip_format;

			/*
			 * TODO: Add check for xlnx,video-width property
			 * just like VPSS-CSC driver here
			 */
		}
	}

	ret = of_property_read_u32(node, "xlnx,num-hori-taps",
				   &xscaler->num_hori_taps);
	if (ret < 0)
		return ret;

	switch (xscaler->num_hori_taps) {
	case XV_HSCALER_TAPS_2:
	case XV_HSCALER_TAPS_4:
		xscaler->is_polyphase = false;
		break;
	case XV_HSCALER_TAPS_6:
	case XV_HSCALER_TAPS_8:
	case XV_HSCALER_TAPS_10:
	case XV_HSCALER_TAPS_12:
		xscaler->is_polyphase = true;
		break;
	default:
		dev_err(dev, "Unsupported num-hori-taps %d",
			xscaler->num_hori_taps);
		return -EINVAL;
	}

	ret = of_property_read_u32(node, "xlnx,num-vert-taps",
				   &xscaler->num_vert_taps);
	if (ret < 0)
		return ret;

	/*
	 * For Bilinear and Bicubic case
	 * number of vertical and horizontal taps must match
	 */
	switch (xscaler->num_vert_taps) {
	case XV_HSCALER_TAPS_2:
	case XV_VSCALER_TAPS_4:
		if (xscaler->num_vert_taps != xscaler->num_hori_taps) {
			dev_err(dev,
				"H-scaler taps %d mismatches V-scaler taps %d",
				 xscaler->num_hori_taps,
				 xscaler->num_vert_taps);
			return -EINVAL;
		}
		break;
	case XV_VSCALER_TAPS_6:
	case XV_VSCALER_TAPS_8:
	case XV_VSCALER_TAPS_10:
	case XV_VSCALER_TAPS_12:
		xscaler->is_polyphase = true;
		break;
	default:
		dev_err(dev, "Unsupported num-vert-taps %d",
			xscaler->num_vert_taps);
		return -EINVAL;
	}

	ret = of_property_read_u32(node, "xlnx,pix-per-clk", &dt_ppc);
	if (ret < 0)
		return ret;

	/* Driver supports 1, 2, 4 and 8 ppc */
	if (dt_ppc != XSCALER_PPC_1 && dt_ppc != XSCALER_PPC_2 &&
	    dt_ppc != XSCALER_PPC_4 && dt_ppc != XSCALER_PPC_8) {
		dev_err(xscaler->xvip.dev,
			"Unsupported xlnx,pix-per-clk(%d) value in DT", dt_ppc);
		return -EINVAL;
	}
	xscaler->pix_per_clk = dt_ppc;

	/* Reset GPIO */
	xscaler->rst_gpio = devm_gpiod_get(dev, "reset", GPIOD_OUT_HIGH);
	if (IS_ERR(xscaler->rst_gpio)) {
		if (PTR_ERR(xscaler->rst_gpio) != -EPROBE_DEFER)
			dev_err(dev, "Reset GPIO not setup in DT");
		return PTR_ERR(xscaler->rst_gpio);
	}

	return 0;
}

static int xscaler_probe(struct platform_device *pdev)
{
	struct xscaler_device *xscaler;
	struct v4l2_subdev *subdev;
	struct v4l2_mbus_framefmt *default_format;
	int ret;
	const struct of_device_id *match;
	struct device_node *node = pdev->dev.of_node;
	struct resource *res;

	xscaler = devm_kzalloc(&pdev->dev, sizeof(*xscaler), GFP_KERNEL);
	if (!xscaler)
		return -ENOMEM;

	xscaler->xvip.dev = &pdev->dev;

	match = of_match_node(xscaler_of_id_table, node);
	if (!match)
		return -ENODEV;

	if (!strcmp(xscaler_of_id_table[0].compatible, match->compatible)) {
		dev_warn(&pdev->dev,
			 "%s - compatible string is getting deprecated!\n",
			 match->compatible);
	}

	xscaler->cfg = match->data;

	ret = xscaler_parse_of(xscaler);
	if (ret < 0)
		return ret;

	/* Initialize coefficient parameters */
	xscaler->max_num_phases = XSCALER_MAX_PHASES;

	if (xscaler->cfg->flags & XSCALER_CLK_PROP) {
		res = platform_get_resource(pdev, IORESOURCE_MEM, 0);
		xscaler->xvip.iomem = devm_ioremap_resource(xscaler->xvip.dev,
							    res);
		if (IS_ERR(xscaler->xvip.iomem))
			return PTR_ERR(xscaler->xvip.iomem);

		ret = clk_prepare_enable(xscaler->aclk_axis);
		if (ret) {
			dev_err(&pdev->dev, "failed to enable aclk_axis (%d)\n",
				ret);
			goto res_cleanup;
		}

		ret = clk_prepare_enable(xscaler->aclk_ctrl);
		if (ret) {
			dev_err(&pdev->dev, "failed to enable aclk_ctrl (%d)\n",
				ret);
			goto axis_clk_cleanup;
		}
	} else {
		ret = xvip_init_resources(&xscaler->xvip);
		if (ret < 0)
			return ret;
	}

	/* Reset the Global IP Reset through a PS GPIO */
	gpiod_set_value_cansleep(xscaler->rst_gpio, XSCALER_RESET_DEASSERT);
	/* Reset internal GPIO within the IP */
	xscaler_reset(xscaler);

	/* Initialize V4L2 subdevice and media entity */
	subdev = &xscaler->xvip.subdev;
	v4l2_subdev_init(subdev, &xscaler_ops);
	subdev->dev = &pdev->dev;
	subdev->internal_ops = &xscaler_internal_ops;
	strscpy(subdev->name, dev_name(&pdev->dev), sizeof(subdev->name));
	v4l2_set_subdevdata(subdev, xscaler);
	subdev->flags |= V4L2_SUBDEV_FL_HAS_DEVNODE;

	/* Initialize default and active formats */
	default_format = &xscaler->default_formats[XVIP_PAD_SINK];
	default_format->code = xscaler->vip_formats[XVIP_PAD_SINK]->code;
	default_format->field = V4L2_FIELD_NONE;
	default_format->colorspace = V4L2_COLORSPACE_SRGB;
	default_format->width = XSCALER_DEF_IN_WIDTH;
	default_format->height = XSCALER_DEF_IN_HEIGHT;
	xscaler->formats[XVIP_PAD_SINK] = *default_format;

	default_format = &xscaler->default_formats[XVIP_PAD_SOURCE];
	*default_format = xscaler->default_formats[XVIP_PAD_SINK];
	default_format->code = xscaler->vip_formats[XVIP_PAD_SOURCE]->code;
	default_format->width = XSCALER_DEF_OUT_WIDTH;
	default_format->height = XSCALER_DEF_OUT_HEIGHT;
	xscaler->formats[XVIP_PAD_SOURCE] = *default_format;

	xscaler->pads[XVIP_PAD_SINK].flags = MEDIA_PAD_FL_SINK;
	xscaler->pads[XVIP_PAD_SOURCE].flags = MEDIA_PAD_FL_SOURCE;
	subdev->entity.ops = &xscaler_media_ops;

	ret = media_entity_pads_init(&subdev->entity, 2, xscaler->pads);
	if (ret < 0)
		goto error;

	platform_set_drvdata(pdev, xscaler);

	ret = v4l2_async_register_subdev(subdev);
	if (ret < 0) {
		dev_err(&pdev->dev, "failed to register subdev");
		goto error;
	}
	dev_info(xscaler->xvip.dev, "Num Hori Taps %d",
		 xscaler->num_hori_taps);
	dev_info(xscaler->xvip.dev, "Num Vert Taps %d",
		 xscaler->num_vert_taps);
	dev_info(&pdev->dev, "VPSS Scaler Probe Successful");
	return 0;

error:
	media_entity_cleanup(&subdev->entity);
	clk_disable_unprepare(xscaler->aclk_ctrl);
axis_clk_cleanup:
	clk_disable_unprepare(xscaler->aclk_axis);
res_cleanup:
	xvip_cleanup_resources(&xscaler->xvip);
	return ret;
}

static void xscaler_remove(struct platform_device *pdev)
{
	struct xscaler_device *xscaler = platform_get_drvdata(pdev);
	struct v4l2_subdev *subdev = &xscaler->xvip.subdev;

	v4l2_async_unregister_subdev(subdev);
	media_entity_cleanup(&subdev->entity);
	clk_disable_unprepare(xscaler->aclk_ctrl);
	clk_disable_unprepare(xscaler->aclk_axis);
	xvip_cleanup_resources(&xscaler->xvip);
}

static struct platform_driver xscaler_driver = {
	.driver			= {
		.name		= "xilinx-vpss-scaler",
		.of_match_table	= xscaler_of_id_table,
	},
	.probe			= xscaler_probe,
	.remove			= xscaler_remove,
};

module_platform_driver(xscaler_driver);
MODULE_DESCRIPTION("Xilinx Scaler VPSS Driver");
MODULE_LICENSE("GPL");
