import csv
import shutil

from tqdm import tqdm

import module.config.server as server

server.server = 'cn' # Don't need to edit, it's used to avoid error.

from module.base.decorator import cached_property
from module.base.utils import load_image
from module.logger import logger
from module.ocr.al_ocr import AlOcr
from module.ocr.ocr import Ocr
from module.statistics.assets import CAMPAIGN_BONUS, CAMPAIGN_BONUS_SINGLE
from module.statistics.battle_status import BattleStatusStatistics
from module.statistics.campaign_bonus import CampaignBonusStatistics
from module.statistics.get_items import GetItemsStatistics
from module.statistics.utils import *


class DropStatistics:
    DROP_FOLDER = './screenshots'
    TEMPLATE_FOLDER = 'item_templates'
    TEMPLATE_BASIC = './assets/stats_basic'
    SKIP_IMAGE_FOLDER = 'skip_images'
    IMAGE_EXTENSION = '.png'
    CNOCR_CONTEXT = 'cpu'
    CSV_FILE = 'drop_result.csv'
    CSV_OVERWRITE = True
    CSV_ENCODING = 'utf-8'

    def __init__(self):
        AlOcr.CNOCR_CONTEXT = DropStatistics.CNOCR_CONTEXT
        Ocr.SHOW_LOG = False
        if not os.path.exists(self.template_folder):
            shutil.copytree(DropStatistics.TEMPLATE_BASIC, self.template_folder)
        os.makedirs(self.skip_image_folder, exist_ok=True)

        self.battle_status = BattleStatusStatistics()
        self.get_items = GetItemsStatistics()
        self.campaign_bonus = CampaignBonusStatistics()
        self.get_items.load_template_folder(self.template_folder)

    @property
    def template_folder(self):
        return os.path.join(DropStatistics.DROP_FOLDER, DropStatistics.TEMPLATE_FOLDER)

    @property
    def skip_image_folder(self):
        return os.path.join(DropStatistics.DROP_FOLDER, DropStatistics.SKIP_IMAGE_FOLDER)

    @property
    def csv_file(self):
        return os.path.join(DropStatistics.DROP_FOLDER, DropStatistics.CSV_FILE)

    @staticmethod
    def drop_folder(campaign):
        return os.path.join(DropStatistics.DROP_FOLDER, campaign)

    @staticmethod
    def is_template_folder(folder):
        return 'template' in folder or 'skip' in folder

    def skip_file_folder(self, campaign):
        return os.path.join(self.skip_image_folder, campaign)

    @cached_property
    def csv_overwrite_check(self):
        """
        Remove existing csv file. This method only run once.
        """
        if DropStatistics.CSV_OVERWRITE:
            if os.path.exists(self.csv_file):
                logger.info(f'Remove existing csv file: {self.csv_file}')
                os.remove(self.csv_file)
        return True

    def check_server(self, campaign):
        """
        Call server.set_server() by folder name

        Args:
            campaign (str):
        """
        name = campaign.split()
        target_server = 'cn'
        if len(name) >= 3:
            target_server = name[-1]

        if server.server != target_server:
            server.set_server(target_server)

    def drop_image(self, file):
        """
        Move a image file to {SKIP_IMAGE_FOLDER}/{CAMPAIGN}.
        """
        if not self.SKIP_IMAGE_FOLDER:
            return False
        campaign = os.path.basename(os.path.abspath(os.path.join(file, '../')))
        folder = self.skip_file_folder(campaign)
        os.makedirs(folder, exist_ok=True)
        shutil.move(file, os.path.join(folder, os.path.basename(file)))

    def parse_template(self, file):
        """
        Extract template from a single file.
        New templates will be given an auto-increased ID.
        """
        images = unpack(load_image(file))[-1::]
        similarities = [get_similarity(image) for image in images]
        images = [resize_image(image) for image in images]
        for image, similarity in zip(images, similarities):
            # if self.get_items.appear_on(image):
            #     self.get_items.extract_template(image, folder=self.template_folder)
            if self.campaign_bonus.appear_on(image, similarity=similarity):
                for button in [CAMPAIGN_BONUS_SINGLE, CAMPAIGN_BONUS]:
                    self.campaign_bonus.bonus_button = button
                    self.campaign_bonus.extract_template(image, folder=self.template_folder)
            else:
                raise ImageError('No campaign bonus on image.')

    def parse_drop(self, file):
        """
        Parse a single file.

        Args:
            file (str):

        Yields:
            list: [timestamp, campaign, enemy_name, drop_type, item, amount]
        """
        ts = os.path.splitext(os.path.basename(file))[0]
        campaign = os.path.basename(os.path.abspath(os.path.join(file, '../')))
        images = unpack(load_image(file))[-1::]
        similarities = [get_similarity(image) for image in images]
        images = [resize_image(image) for image in images]
        enemy_name = 'unknown'
        for image, similarity in zip(images, similarities):
            # if self.battle_status.appear_on(image):
            #     enemy_name = self.battle_status.stats_battle_status(image)
            # if self.get_items.appear_on(image):
            #     for item in self.get_items.stats_get_items(image):
            #         yield [ts, campaign, enemy_name, 'GET_ITEMS', item.name, item.amount]
            if self.campaign_bonus.appear_on(image, similarity=similarity):
                for button in [CAMPAIGN_BONUS_SINGLE, CAMPAIGN_BONUS]:
                    self.campaign_bonus.bonus_button = button
                    for item in self.campaign_bonus.stats_get_items(image, mode='known'):
                        yield [ts, campaign, enemy_name, button.name, item.name, item.amount]
            else:
                raise ImageError('No campaign bonus on image.')

    def extract_template(self, campaign):
        """
        Extract images from a given folder.

        Args:
            campaign (str):
        """
        print('')
        logger.hr(f'Extract templates from {campaign}', level=1)
        self.check_server(campaign)
        drop_folder = load_folder(self.drop_folder(campaign), ext=DropStatistics.IMAGE_EXTENSION)
        for ts, file in tqdm(drop_folder.items()):
            try:
                self.parse_template(file)
            except ImageError as e:
                logger.warning(f'{e} image file: {ts}')
                self.drop_image(file)
                continue
            except Exception as e:
                logger.exception(e)
                logger.warning(f'Error on image {ts}')
                self.drop_image(file)
                continue

    def extract_drop(self, campaign):
        """
        Parse images from a given folder.

        Args:
            campaign (str):
        """
        print('')
        logger.hr(f'extract drops from {campaign}', level=1)
        self.check_server(campaign)
        _ = self.csv_overwrite_check

        with open(self.csv_file, 'a', newline='', encoding=DropStatistics.CSV_ENCODING) as csv_file:
            writer = csv.writer(csv_file)
            drop_folder = load_folder(self.drop_folder(campaign), ext=DropStatistics.IMAGE_EXTENSION)
            for ts, file in tqdm(drop_folder.items()):
                try:
                    rows = list(self.parse_drop(file))
                    writer.writerows(rows)
                except ImageError as e:
                    logger.warning(f'{e} image file: {ts}')
                    self.drop_image(file)
                    continue
                except Exception as e:
                    logger.exception(e)
                    logger.warning(f'Error on image {ts}')
                    self.drop_image(file)
                    continue


if __name__ == '__main__':
    # Drop screenshot folder. Default to './screenshots'
    DropStatistics.DROP_FOLDER = './screenshots'
    # Folder to load templates and save new templates.
    # This will load {DROP_FOLDER}/{TEMPLATE_FOLDER}.
    # If folder doesn't exist, auto copy from './assets/stats_basic'
    DropStatistics.TEMPLATE_FOLDER = 'template'
    # Folder to save dropped images.
    # This will save images {DROP_FOLDER}/{SKIP_IMAGE_FOLDER}/{CAMPAIGN}.
    # If folder doesn't exist, auto create
    DropStatistics.SKIP_IMAGE_FOLDER = 'skip_images'
    # image file extension suck as '.png', '.jpg'
    DropStatistics.IMAGE_EXTENSION = ['.png', '.jpg', '.PNG']
    # 'cpu' or 'gpu', default to 'cpu'.
    # Use 'gpu' for faster prediction, but you must have the gpu version of mxnet installed.
    DropStatistics.CNOCR_CONTEXT = 'cpu'
    # Name of the output csv file.
    # This will write to {DROP_FOLDER}/{CSV_FILE}.
    DropStatistics.CSV_FILE = 'drop_results.csv'
    # If True, remove existing file before extraction.
    DropStatistics.CSV_OVERWRITE = True
    # Usually to be 'utf-8'.
    # For better Chinese export to Excel, use 'gbk'.
    DropStatistics.CSV_ENCODING = 'utf-8-sig'
    # campaign names to export under DROP_FOLDER.
    # This will load {DROP_FOLDER}/{CAMPAIGN}.
    # Just a demonstration here, you should modify it to your own.
    CAMPAIGNS = ['campaign_13_1']

    stat = DropStatistics()

    """
    Step 1:
        Uncomment these code and run.
        After run, comment again.
    """
    # with os.scandir(stat.DROP_FOLDER) as entries:
    #     for i in entries:
    #         if i.is_dir() and not stat.is_template_folder(i.name):
    #             if not i.name.startswith('9-'):
    #                 continue
    #             stat.extract_template(i.name)

    """
    Step 2:
        Goto {DROP_FOLDER}/{TEMPLATE_FOLDER}.
        Manually rename the templates you interested in.
    """
    pass

    """
    Step 3:
        Uncomment these code and run.
        After run, comment again.
        Results are saved in {DROP_FOLDER}/{CSV_FILE}.
    """
    # with os.scandir(stat.DROP_FOLDER) as entries:
    #     for i in entries:
    #         if i.is_dir() and not stat.is_template_folder(i.name):
    #             if i.name not in CAMPAIGNS:
    #                 continue
    #             stat.extract_drop(i.name)
