//
//  CLSActivity.h
//  ClassKit
//
//  Copyright © 2018-2020 Apple Inc. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <ClassKit/CLSObject.h>
#import <ClassKit/CLSContext.h>
#import <ClassKit/CLSActivityItem.h>

@class CLSActivityItem;

NS_ASSUME_NONNULL_BEGIN

/*!
 @abstract      CLSActivity represents user generated activity data for a given context.
 @discussion    CLSActivity has three major components.

             (1) Progress, used to measure the amount of activity a user generates as a percentage.
             (2) Duration, used to measure the length of time a user spent on this activity. Use @c -start and @c -stop methods.
             (3) Activity Items, used to gather information about the activity generated by a user.
 */
API_AVAILABLE(ios(11.3), macos(11.0), macCatalyst(14.0)) API_UNAVAILABLE(watchos, tvos)
@interface CLSActivity : CLSObject

/*!
 @abstract      Current progress as a decimal representation of a percentage.
 @discussion    Should be [0.0, 1.0].
 */
@property (nonatomic, assign) double progress;

/*!
 @abstract      Returns the total time tracked in this activity (excluding any previous activities).
 @discussion    The time between calling @c -start and @c -stop.
 */
@property (nonatomic, assign, readonly) NSTimeInterval duration;

/*!
 @abstract      The primary activityItem to be reported on.
 @discussion    This can be nil indicating @c progress property is the primary data instead of any activityItems.
 */
@property (nullable, nonatomic, strong) __kindof CLSActivityItem *primaryActivityItem;

/*!
 @abstract      Adds progress to this activity.
 @discussion    The progress should be a decimal representation of the start and ending percentage [0.0, 1.0].
 @param         start      Starting percentage.
 @param         end        Ending percentage.
 */
- (void)addProgressRangeFromStart:(double)start
                            toEnd:(double)end;

/*!
 @abstract      Add an activity item to this CLSActivity.
 */
- (void)addAdditionalActivityItem:(CLSActivityItem *)activityItem;

/*!
 @abstract      Array of all additional activity items on this CLSActivity.
 */
@property (nonatomic, strong, readonly) NSArray<__kindof CLSActivityItem *> *additionalActivityItems;

@end

API_AVAILABLE(ios(11.3), macos(11.0), macCatalyst(14.0)) API_UNAVAILABLE(watchos, tvos)
@interface CLSActivity (Activation)

/*!
 @abstract      Returns whether this Activity has been started or not.
 */
@property (nonatomic, assign, readonly, getter=isStarted) BOOL started;

/*!
 @abstract      Start Activity.
 @discussion    Starts the activity (or resumes if previously stopped).
 */
- (void)start;

/*!
 @abstract      Stop Activity.
 @discussion    Stops or pauses the activity and ends the time being tracked on it.
 */
- (void)stop;

/*!
@abstract Deletes all activity items.
@discussion Convenience method to delete all activity items associated with the current activity.
*/
- (void)removeAllActivityItems API_AVAILABLE(ios(14.5), macos(11.3), macCatalyst(14.5)) API_UNAVAILABLE(watchos, tvos);

@end

API_AVAILABLE(ios(11.3), macos(11.0), macCatalyst(14.0)) API_UNAVAILABLE(watchos, tvos)
@interface CLSContext (Activity)

/*!
 @abstract      Returns the current activity.
 @discussion    Activity associated with a context.  If no activity was ever created this is nil. See: @c -[CLSContext createNewActivity]; for more details.
 */
@property (nullable, nonatomic, strong, readonly) CLSActivity *currentActivity;

/*!
 @abstract      Creates a new activity
 @discussion    Creates a new activity and sets it as the current activity.
 */
- (CLSActivity *)createNewActivity;

@end

NS_ASSUME_NONNULL_END
