//
//  HKLiveWorkoutBuilder.h
//  HealthKit
//
//  Copyright © 2018 Apple. All rights reserved.
//

#import <HealthKit/HealthKit.h>
#import <HealthKit/HKWorkoutBuilder.h>
#import <HealthKit/HKLiveWorkoutDataSource.h>

NS_ASSUME_NONNULL_BEGIN

@class HKLiveWorkoutBuilder;

HK_EXTERN API_AVAILABLE(watchos(5.0)) API_UNAVAILABLE(ios)
@protocol HKLiveWorkoutBuilderDelegate <NSObject>

/*!
 @method        workoutBuilder:didCollectDataOfTypes:
 @abstract      Called every time new samples are added to the workout builder.
 @discussion    With new samples added, statistics for the collectedTypes may have changed and should be read again
 
 @param         workoutBuilder    The workout builder to which samples were added.
 @param         collectedTypes    The sample types that were added.
 */
- (void)workoutBuilder:(HKLiveWorkoutBuilder *)workoutBuilder didCollectDataOfTypes:(NSSet<HKSampleType *> *)collectedTypes;

/*!
 @method        workoutBuilderDidCollectEvent:
 @abstract      Called every time a new event is added to the workout builder.
 
 @param         workoutBuilder    The workout builder to which an event was added.
 */
- (void)workoutBuilderDidCollectEvent:(HKLiveWorkoutBuilder *)workoutBuilder;

@end

HK_EXTERN API_AVAILABLE(watchos(5.0)) API_UNAVAILABLE(ios)
@interface HKLiveWorkoutBuilder : HKWorkoutBuilder

- (instancetype)initWithHealthStore:(HKHealthStore *)healthStore
                      configuration:(HKWorkoutConfiguration *)configuration
                             device:(nullable HKDevice *)device NS_UNAVAILABLE;

/*!
 @property      delegate
 @abstract      The delegate object which will be notified of changes to collected data and elapsed workout time.
 */
@property (weak) id<HKLiveWorkoutBuilderDelegate> delegate;

/*!
 @property      workoutSession
 @abstract      The workout session with which this builder is associated.
 */
@property (weak, readonly) HKWorkoutSession *workoutSession;

/*!
 @property      shouldCollectWorkoutEvents
 @abstract      Whether or not the builder should automatically incorporate events generated by the session.
 @discussion    YES if the builder should automatically incorporate events from the associated session. If NO, the client
                is expected to add relevant events directly to the builder and those generated by the session will be
                ignored. Defaults to YES. The result of changing this value after a call to
                beginCollectionWithStartDate:completion: is undefined.
 */
@property (assign) BOOL shouldCollectWorkoutEvents;

/*!
 @property      dataSource
 @abstract      The data source that will provide data to this workout builder.
 */
@property (strong, nullable) HKLiveWorkoutDataSource *dataSource;

/*!
 @property      elapsedTime
 @abstract      The elapsed duration of the workout as constructed thus far, considering paused time. When the builder
                is reading events from an active session, it is possible for this value to increase or decrease;
                when this occurs, the workoutBuilderDidCollectEvent: delegate method will be called.
 */
@property (assign, readonly) NSTimeInterval elapsedTime;

@end

NS_ASSUME_NONNULL_END
