/* DLib float setup header for math functionality
   Copyright 2015-2017 IAR Systems AB.
*/
#ifndef _DLIB_FLOAT_SETUP
#define _DLIB_FLOAT_SETUP

#ifndef _SYSTEM_BUILD
  #pragma system_include
#endif

/*
  Possible Overrides
  (All in a packet must be overrided or not.)
*/

/* Overrides on the trigonometrics (sin, cos, and tan) defaults */
/* Default */
   #define _DLIB_trigonometric_IS_DEFAULT
/* or accurate
   #define _DLIB_trigonometric_accurate_IS_DEFAULT
   or small
   #define _DLIB_trigonometric_small_IS_DEFAULT
*/

/* Overrides on the exp-pow defaults */
/* Default */
   #define _DLIB_exp_pow_IS_DEFAULT
/* or accurate
   #define _DLIB_exp_pow_accurate_IS_DEFAULT
   or small
   #define _DLIB_exp_pow_small_IS_DEFAULT
*/

/* Overrides on the logx defaults */
/* Default */
   #define _DLIB_log_IS_DEFAULT
/* or small
   #define _DLIB_log_small_IS_DEFAULT
*/

/* Low-level math types
   (Used to contain flag-bits or small numbers.)
*/
typedef __UINT32_T_TYPE__ __iar_FlagUType;
typedef __INT32_T_TYPE__  __iar_FlagSType;

typedef __INT32_T_TYPE__  __iar_ExpType;


/*
  The macro FP_FAST_FMA
  is optionally defined. If defined, it indicates that the fma function generally executes
  about as fast as, or faster than, a multiply and an add of double operands.230) The
  macros
  
  FP_FAST_FMAF
  FP_FAST_FMAL

  are, respectively, float and long double analogs of FP_FAST_FMA. If defined,
  these macros expand to the integer constant 1.
*/

#ifdef __ARM_FEATURE_FMA
  #if __ARM_FP & 0x4
    #define FP_FAST_FMAF  1
  #endif
  #if __ARM_FP & 0x8
    #define FP_FAST_FMA   1
    #define FP_FAST_FMAL  1
  #endif
#endif  

/* Low-level math functions
   (remove 64-bit versions if sizeof(float) == sizeof(double).)
*/

#define _FLT_MAX_EXP 0xFF
#define _FLT_BIAS    0x7E
#define _FLT_OFF     23

#define _DBL_MAX_EXP 0x7FF
#define _DBL_BIAS    0x3FE
#define _DBL_OFF     52

#pragma diag_suppress = Pe549 // variable used before set.

// Use _NOFORCEINLINE to get result from a compilation of only this header.
#ifdef _NOFORCEINLINE
  #define _FORCEINLINE
#else
  #define _FORCEINLINE _Pragma("inline=forced")
#endif

// Choose between the AARCH64 alternative implementations.
#ifndef _NOBETTER
#ifdef __ARM_FP
  #define _BETTER
#endif
#endif

#ifdef __cplusplus
extern "C" {
#endif // C++

/*
   Return 0 if x is INF or NaN, !0 if subnormal, normal, or 0.
   It may return any integer type.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isfinite32(float _X)
{
  #if defined(__aarch64__) && defined(_BETTER)
    unsigned long long _R;
    __asm("FMOV %w0, %s1             \n"
          "UBFX %w0, %w0, 23, 8      \n"
          "CMP  %w0, #0xFF           \n" /* INF */
          "CSET %x0, LO              \n" /* less than, not unordered */
          : "=r"(_R) : "w"(_X) : "cc");
    return (__UINT32_T_TYPE__)_R;
  #else
    __INT32_T_TYPE__ _Ix = __iar_fp2bits32(_X);
    return ((_Ix << 1) >> (_FLT_OFF + 1)) + 1;
  #endif
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isfinite64(double _X)
{
  #if defined(__aarch64__) && defined(_BETTER)
    unsigned long long _R;
    __asm("FMOV %x0, %d1             \n"
          "UBFX %x0, %x0, 52, 11     \n"
          "CMP  %x0, #0x7FF          \n" /* INF */
          "CSET %x0, LO              \n" /* less than, not unordered */
          : "=r"(_R) : "w"(_X) : "cc");
    return (__UINT32_T_TYPE__)_R;
  #else
    __INT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
    return ((_Ix << 1) >> (_DBL_OFF + 1)) + 1;
  #endif
}
/*
   Return !0 if NaN, else 0.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isnan32(float _X)
{
  #ifdef __aarch64__
    #ifndef _BETTER
      __INT32_T_TYPE__ _Ix = __iar_fp2bits32(_X) << 1;
      return (_Ix >> (_FLT_OFF + 1)) + 1 ? 0 : (_Ix << (31 - _FLT_OFF));
    #else
      unsigned long long _R;
      __asm("FCMP %s1, #0.0 \n"
            "CSET %x0, VS   \n" /* unordered */
            : "=r"(_R) : "w"(_X) : "cc");
      return (__UINT32_T_TYPE__)_R;
    #endif
  #else
    __INT32_T_TYPE__ _Ix = __iar_fp2bits32(_X) << 1;
    return (_Ix >> (_FLT_OFF + 1)) + 1 ? 0 : (_Ix << (31 - _FLT_OFF));
  #endif
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isnan64(double _X)
{
  #if defined(__aarch64__) && defined(_BETTER)
    unsigned long long _R;
    __asm("FCMP %d1, #0.0 \n"
          "CSET %x0, VS   \n" /* unordered */
          : "=r"(_R) : "w"(_X) : "cc");
    return (__UINT32_T_TYPE__)_R;
  #else
    __INT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
    return ((_Ix << 1) >> (_DBL_OFF + 1)) + 1
           ? 0
           : ((_Ix << (64 - _DBL_OFF)) != 0);
  #endif
}

/*
   Return !0 if +/-0, else 0. Do not need to check for NaN.
   It may return any integer type.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_iszero32(float _X)
{
  #ifdef __aarch64__
    #ifndef _BETTER
      __UINT32_T_TYPE__ _Ix = __iar_fp2bits32(_X);
      return (_Ix << 1) == 0;
    #else
      unsigned long long _R;
      __asm("FCMP %s1, #0.0 \n"
            "CSET %x0, EQ   \n" /* equal to */
            : "=r"(_R) : "w"(_X) : "cc");
      return (__UINT32_T_TYPE__)_R;
    #endif
  #else
    __UINT32_T_TYPE__ _Ix = __iar_fp2bits32(_X);
    return (_Ix << 1) == 0;
  #endif
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_iszero64(double _X)
{
  #ifdef __aarch64__
    #ifndef _BETTER
      __UINT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
      return (_Ix << 1) == 0;
    #else
      unsigned long long _R;
      __asm("FCMP %d1, #0.0 \n"
            "CSET %x0, EQ   \n" /* equal to */
            : "=r"(_R) : "w"(_X) : "cc");
      return (__UINT32_T_TYPE__)_R;
    #endif
  #else
    __UINT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
    return (_Ix & ~(1ULL << 63)) == 0;
  #endif
}


/*
   Return !0 if INF, else 0. Do not need to check for NaN.
   It may return any integer type.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isinf32(float _X)
{
  #if defined(__aarch64__) && defined(_BETTER)
    unsigned long long _R;
    float _F;
    __asm("FABS %s1, %s1             \n"
          "MOVZ %w0, #0x7F80, LSL 16 \n" /* INF */
          "FMOV %s2, %w0             \n"
          "FCMP %s1, %s2             \n"
          "CSET %x0, EQ              \n" /* equal to */
          : "=r"(_R),"+w"(_X), "+w"(_F) :: "cc");
    return (__UINT32_T_TYPE__)_R;
  #else
    __INT32_T_TYPE__ _Ix = __iar_fp2bits32(_X);
    return ((_Ix << 1) >> (_FLT_OFF + 1)) + 1
           ? 0
           : ((_Ix << (32 - _FLT_OFF)) == 0);
  #endif
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isinf64(double _X)
{
  #ifdef __aarch64__
    #ifndef _BETTER
      __INT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
      return ((_Ix << 1) >> (_DBL_OFF + 1)) + 1 
               ? 0 
               : ((_Ix << (64 - _DBL_OFF)) == 0);
    #else
      unsigned long long _R;
      double _D;
      __asm("FABS %d1, %d1             \n"
            "MOVZ %x0, #0x7FF0, LSL 48 \n" /* INF */
            "FMOV %d2, %x0             \n"
            "FCMP %d1, %d2             \n"
            "CSET %x0, EQ              \n" /* equal to */
            : "=r"(_R),  "+w"(_X), "+w"(_D) :: "cc");
      return (__UINT32_T_TYPE__)_R;
    #endif
  #else
    __INT32_T_TYPE__ _Ix = __iar_fpgethi64(_X);
    return ((_Ix << 1) >> (_DBL_OFF - 31)) + 1 
             ? 0 
             : ((_Ix << (64 - _DBL_OFF)) == 0);
  #endif
}


/*
  Return !0 if subnormal, else 0. Do not need to check for NaN.
   It may return any integer type.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_issubnormal32(float _X)
{
  #if __SUBNORMAL_FLOATING_POINTS__
    __UINT32_T_TYPE__ _Ix = __iar_fp2bits32(_X) & ~(1 << 31);
    return (_Ix != 0) && (_Ix < (1 << _FLT_OFF));
  #else
    return 0;
  #endif
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_issubnormal64(double _X)
{
  #if __SUBNORMAL_FLOATING_POINTS__
    __UINT64_T_TYPE__ _Ix = __iar_fp2bits64(_X) & ~(1ULL << 63);
    return (_Ix != 0) && (_Ix < (1ULL << _DBL_OFF));
  #else
    return 0;
  #endif
}

/*
   Return !0 if normal, else 0. Do not need to check for NaN.
   It may return any integer type.
*/
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isnormal32(float _X)
{
  __INT32_T_TYPE__ _Exp = ((__INT32_T_TYPE__)
                          (__iar_fp2bits32(_X) << 1) >> (_FLT_OFF + 1));
  return ((_Exp + 1) >> 1);
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_isnormal64(double _X)
{
  #ifdef __aarch64__
    __INT64_T_TYPE__ _Exp =
      ((__INT64_T_TYPE__)(__iar_fp2bits64(_X) << 1) >> (_DBL_OFF + 1));
    return ((_Exp + 1) >> 1);
  #else
    __INT32_T_TYPE__ _Exp = 
      ((__INT32_T_TYPE__)(__iar_fpgethi64(_X) << 1) >> (_DBL_OFF - 31));
    return ((_Exp + 1) >> 1);
  #endif
}


/*
   Return the signbit (without NaN-check).
 */
#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_signbit32(float _X)
{
  __UINT32_T_TYPE__ _Ix = __iar_fp2bits32(_X);
  return (_Ix >> 31);
}

#pragma no_arith_checks
_FORCEINLINE __intrinsic __UINT32_T_TYPE__ __iar_signbit64(double _X)
{
  __UINT64_T_TYPE__ _Ix = __iar_fp2bits64(_X);
  return (_Ix >> 63);
}

#ifdef __cplusplus
} // extern "C"
#endif // C++

#endif /* _DLIB_FLOAT_SETUP */
