/**************************************************
 *
 * This file declares the ARM intrinsic inline functions.
 *
 * Copyright 1999-2018 IAR Systems. All rights reserved.
 *
 * $Revision$
 *
 **************************************************/

#ifndef __INTRINSICS_INCLUDED
#define __INTRINSICS_INCLUDED

#ifndef _SYSTEM_BUILD
  #pragma system_include
#endif

/*
 * Check that the correct C compiler is used.
 */
#ifndef __ICCARM__ 
  #error "File intrinsics.h can only be used together with iccarm."
#endif

#ifndef __ICCARM_INTRINSICS_VERSION__
  #error "Unknown compiler intrinsics version"
#elif __ICCARM_INTRINSICS_VERSION__ != 2
  #error "Compiler intrinsics version does not match this file"
#endif

#include <iccarm_builtin.h>

#define __no_operation        __iar_builtin_no_operation
#define __disable_interrupt   __iar_builtin_disable_interrupt
#define __enable_interrupt    __iar_builtin_enable_interrupt
#define __get_interrupt_state __iar_builtin_get_interrupt_state
#define __set_interrupt_state __iar_builtin_set_interrupt_state

#pragma inline=forced
void __disable_fiq(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s | 0x40);
}
#pragma inline=forced
void __enable_fiq(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s & ~0x40);
}

#pragma inline=forced
void __disable_irq(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s | 0x80);
}
#pragma inline=forced
void __enable_irq(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s & ~0x80);
}

#pragma inline=forced
void __disable_SError(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s | 0x100);
}
#pragma inline=forced
void __enable_SError(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s & ~0x100);
}

#pragma inline=forced
void __disable_debug(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s | 0x200);
}
#pragma inline=forced
void __enable_debug(void)
{
  unsigned long long s = __get_interrupt_state();
  __set_interrupt_state(s & ~0x200);
}

#pragma inline=forced
void __DGH(void)
{
  __asm("HINT #6");
}

/* ACLE: Reads a system register */
#define __arm_rsr __iar_builtin_rsr
#define __arm_rsr64 __iar_builtin_rsr64
#define __arm_rsrp __iar_builtin_rsrp

/* ACLE: Writes a system register */
#define __arm_wsr __iar_builtin_wsr
#define __arm_wsr64 __iar_builtin_wsr64
#define __arm_wsrp __iar_builtin_wsrp

/* ACLE:  Floating-point data-processing intrinsics */
#ifdef __ARM_FP

  /* ACLE: square root */
  #if (__ARM_FP & 0x8)
    #define __sqrt      __iar_builtin_VSQRT_F64
  #endif
  #if (__ARM_FP & 0x4)
    #define  __sqrtf   __iar_builtin_VSQRT_F32
  #endif

  /* ACLE: fused mac */
  #ifdef __ARM_FEATURE_FMA
    #if (__ARM_FP & 0x8)
      #define __fma    __iar_builtin_VFMA_F64
    #endif
    #if (__ARM_FP & 0x4)
      #define __fmaf    __iar_builtin_VFMA_F32
    #endif
  #endif

  /* ACLE: directed rounding */
  #ifdef __ARM_FEATURE_DIRECTED_ROUNDING
    #define __rintn  __iar_builtin_VRINTN_F64
    #define __rintnf __iar_builtin_VRINTN_F32
  #endif

#endif


/* ACLE: CRC32 extension */
#ifdef __ARM_FEATURE_CRC32
  #define __crc32b(x,y)  __iar_builtin_CRC32B((x), (y))
  #define __crc32h(x,y)  __iar_builtin_CRC32H((x), (y))
  #define __crc32w(x,y)  __iar_builtin_CRC32W((x), (y))
  #define __crc32cb(x,y) __iar_builtin_CRC32CB((x), (y))
  #define __crc32ch(x,y) __iar_builtin_CRC32CH((x), (y))
  #define __crc32cw(x,y) __iar_builtin_CRC32CW((x), (y))

  #ifdef __aarch64__
    #define __crc32d(x,y)  __iar_builtin_CRC32X((x), (y))
    #define __crc32cd(x,y) __iar_builtin_CRC32CX((x), (y))
  #else
    #pragma inline=forced
    unsigned int __crc32d(unsigned int crc, unsigned long long data)
    {
      crc = __crc32w(crc, (unsigned int)(data));
      crc = __crc32w(crc, (unsigned int)(data >> 32));
      return crc;
    }
    #pragma inline=forced
    unsigned int __crc32cd(unsigned int crc, unsigned long long data)
    {
      crc = __crc32cw(crc, (unsigned int)(data));
      crc = __crc32cw(crc, (unsigned int)(data >> 32));
      return crc;
    }
  #endif
#endif

#endif  /* __INTRINSICS_INCLUDED */
