/*
 * Decompiled with CFR 0.152.
 */
package org.jumpmind.symmetric.io.stage;

import java.io.File;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.DirectoryStream;
import java.nio.file.FileSystems;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.util.Iterator;
import java.util.Map;
import java.util.Set;
import java.util.TreeSet;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.time.DurationFormatUtils;
import org.jumpmind.symmetric.io.stage.IStagedResource;
import org.jumpmind.symmetric.io.stage.IStagingManager;
import org.jumpmind.symmetric.io.stage.StagedResource;
import org.jumpmind.symmetric.io.stage.StagingFileLock;
import org.jumpmind.symmetric.io.stage.StagingLowFreeSpace;
import org.jumpmind.symmetric.io.stage.StagingPurgeContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class StagingManager
implements IStagingManager {
    protected static final String LOCK_EXTENSION = ".lock";
    private static final Logger log = LoggerFactory.getLogger(StagingManager.class);
    protected File directory;
    protected Set<String> resourcePathsCache;
    protected Map<String, IStagedResource> inUse = new ConcurrentHashMap<String, IStagedResource>();
    protected boolean clusterEnabled;
    protected long lowFreeSpaceThresholdMegabytes;
    protected static final DirectoryStream.Filter<Path> STAGING_FILE_FILTER = new DirectoryStream.Filter<Path>(){

        @Override
        public boolean accept(Path entry) {
            try {
                boolean accept = Files.isDirectory(entry, new LinkOption[0]) || entry.getFileName().toString().endsWith(".create") || entry.getFileName().toString().endsWith(".done");
                return accept;
            }
            catch (NullPointerException ex) {
                return false;
            }
        }
    };

    public StagingManager(String directory, boolean clusterEnabled, long lowFreeSpaceThresholdMegabytes) {
        log.info("The staging directory was initialized at the following location: " + directory);
        this.directory = new File(directory);
        this.directory.mkdirs();
        this.clusterEnabled = clusterEnabled;
        this.lowFreeSpaceThresholdMegabytes = lowFreeSpaceThresholdMegabytes;
        this.resourcePathsCache = ConcurrentHashMap.newKeySet();
    }

    public StagingManager(String directory, boolean clusterEnabled) {
        this(directory, clusterEnabled, 0L);
    }

    @Override
    public Set<String> getResourceReferences() {
        return new TreeSet<String>(this.resourcePathsCache);
    }

    @Override
    public long clean(long ttlInMs) {
        return this.clean(ttlInMs, null);
    }

    public synchronized long clean(long ttlInMs, StagingPurgeContext context) {
        try {
            log.info("Cleaning staging...");
            if (context == null) {
                context = new StagingPurgeContext();
            }
            long start = System.currentTimeMillis();
            context.setStartTime(start);
            this.resourcePathsCache.clear();
            this.clean(FileSystems.getDefault().getPath(this.directory.getAbsolutePath(), new String[0]), ttlInMs, context);
            this.logCleaningProgress(context);
            this.cleanInUseCache(ttlInMs, context);
            long end = System.currentTimeMillis();
            log.info("Finished cleaning staging in " + DurationFormatUtils.formatDurationWords((long)(end - start), (boolean)true, (boolean)true) + ".");
            return context.getPurgedFileSize() + context.getPurgedMemSize();
        }
        catch (Exception ex) {
            throw new RuntimeException("Failure while cleaning staging.", ex);
        }
    }

    protected void logCleaningProgress(StagingPurgeContext context) {
        if (context.getPurgedFileCount() > 0L) {
            log.info("Purged {} staging files, freed {} of disk space.", (Object)context.getPurgedFileCount(), (Object)FileUtils.byteCountToDisplaySize((long)context.getPurgedFileSize()));
        }
        if (context.getPurgedMemCount() > 0L) {
            log.info("Purged {} staging memory buffers, freed {}.", (Object)context.getPurgedMemCount(), (Object)FileUtils.byteCountToDisplaySize((long)context.getPurgedMemSize()));
        }
    }

    protected void clean(Path path, long ttlInMs, StagingPurgeContext context) throws IOException {
        DirectoryStream<Path> stream = Files.newDirectoryStream(path, STAGING_FILE_FILTER);
        if (context.shouldLogStatus()) {
            this.logCleaningProgress(context);
            context.setLastLogTime(System.currentTimeMillis());
        }
        for (Path entry : stream) {
            if (Files.isDirectory(entry, new LinkOption[0])) {
                this.clean(entry, ttlInMs, context);
                continue;
            }
            try {
                String parentDirectory = "";
                Path parentPath = entry.getParent();
                if (parentPath != null) {
                    parentDirectory = parentPath.toString();
                }
                String entryName = "";
                Path entryPath = entry.getFileName();
                if (entryPath != null) {
                    entryName = entryPath.toString();
                }
                String stagingPath = StagedResource.toPath(this.directory, new File(parentDirectory + "/" + entryName));
                IStagedResource resource = this.createStagedResource(stagingPath);
                if (stagingPath == null) continue;
                if (this.shouldCleanPath(resource, ttlInMs, context)) {
                    if (resource.isMemoryResource()) {
                        context.incrementPurgedMemoryCount();
                        context.addPurgedMemoryBytes(resource.getSize());
                    } else {
                        context.incrementPurgedFileCount();
                        context.addPurgedFileBytes(resource.getSize());
                    }
                    this.cleanPath(resource, ttlInMs, context);
                    continue;
                }
                this.resourcePathsCache.add(stagingPath);
            }
            catch (IllegalStateException ex) {
                log.warn("Failure during clean ", (Throwable)ex);
            }
        }
        stream.close();
    }

    protected void cleanInUseCache(long ttlInMs, StagingPurgeContext context) {
        long resourceCount = 0L;
        long memoryBytes = 0L;
        Iterator<Map.Entry<String, IStagedResource>> iter = this.inUse.entrySet().iterator();
        while (iter.hasNext()) {
            Map.Entry<String, IStagedResource> entry = iter.next();
            IStagedResource resource = entry.getValue();
            if (!this.shouldCleanInUseCache(resource, ttlInMs, context)) continue;
            ++resourceCount;
            if (resource.isMemoryResource()) {
                memoryBytes += resource.getSize();
                context.incrementPurgedMemoryCount();
                context.addPurgedMemoryBytes(resource.getSize());
            }
            iter.remove();
        }
        if (resourceCount > 0L) {
            log.info("Cleared {} cache objects, freed {}.", (Object)resourceCount, (Object)FileUtils.byteCountToDisplaySize((long)memoryBytes));
        }
    }

    protected boolean shouldCleanPath(IStagedResource resource, long ttlInMs, StagingPurgeContext context) {
        boolean resourceIsOld = System.currentTimeMillis() - resource.getLastUpdateTime() > ttlInMs;
        return resourceIsOld && resource.getState() == IStagedResource.State.DONE && !resource.isInUse();
    }

    protected boolean shouldCleanInUseCache(IStagedResource resource, long ttlInMs, StagingPurgeContext context) {
        boolean resourceIsOld = System.currentTimeMillis() - resource.getLastUpdateTime() > ttlInMs;
        return resourceIsOld && resource.getState() == IStagedResource.State.DONE && !resource.isInUse();
    }

    protected boolean cleanPath(IStagedResource resource, long ttlInMs, StagingPurgeContext context) {
        boolean success = resource.delete();
        if (!success) {
            log.warn("Failed to delete the '{}' staging resource", (Object)resource.getPath());
        }
        return success;
    }

    @Override
    public IStagedResource create(Object ... path) {
        String filePath = this.buildFilePath(path);
        IStagedResource resource = this.createStagedResource(filePath);
        if (resource.exists()) {
            resource.delete();
        } else {
            resource.getFile().getParentFile().mkdirs();
        }
        if (this.lowFreeSpaceThresholdMegabytes > 0L) {
            long freeSpace = 0L;
            freeSpace = path.length == 0 ? this.directory.getFreeSpace() / 1000000L : new File(this.directory, (String)path[0]).getFreeSpace() / 1000000L;
            if (freeSpace <= this.lowFreeSpaceThresholdMegabytes) {
                throw new StagingLowFreeSpace(String.format("Free disk space of %d MB is below threshold of %d MB", freeSpace, this.lowFreeSpaceThresholdMegabytes));
            }
        }
        this.inUse.put(filePath, resource);
        this.resourcePathsCache.add(filePath);
        return resource;
    }

    protected IStagedResource createStagedResource(String filePath) {
        return new StagedResource(this.directory, filePath, this);
    }

    protected String buildFilePath(Object ... path) {
        StringBuilder buffer = new StringBuilder();
        for (int i = 0; i < path.length; ++i) {
            Object part = path[i];
            if (part instanceof Number) {
                part = StringUtils.leftPad((String)part.toString(), (int)10, (String)"0");
            }
            buffer.append(part);
            if (i >= path.length - 1) continue;
            buffer.append("/");
        }
        return buffer.toString();
    }

    @Override
    public IStagedResource find(String path) {
        IStagedResource fileResource;
        IStagedResource resource = this.inUse.get(path);
        if (resource == null && (fileResource = this.createStagedResource(path)).exists()) {
            resource = fileResource;
            this.inUse.put(path, resource);
            this.resourcePathsCache.add(path);
        }
        return resource;
    }

    @Override
    public IStagedResource find(Object ... path) {
        return this.find(this.buildFilePath(path));
    }

    public void removeResourcePath(String path) {
        this.resourcePathsCache.remove(path);
        this.inUse.remove(path);
    }

    @Override
    public StagingFileLock acquireFileLock(String serverInfo, Object ... path) {
        String lockFilePath = String.format("%s/%s%s", this.directory, this.buildFilePath(path), LOCK_EXTENSION);
        log.debug("About to acquire lock at {}", (Object)lockFilePath);
        StagingFileLock stagingFileLock = new StagingFileLock();
        File lockFile = new File(lockFilePath);
        File containingDirectory = lockFile.getParentFile();
        if (containingDirectory != null) {
            containingDirectory.mkdirs();
        }
        boolean acquired = false;
        try {
            acquired = lockFile.createNewFile();
            if (acquired) {
                FileUtils.write((File)lockFile, (CharSequence)serverInfo, (Charset)Charset.defaultCharset(), (boolean)false);
            }
        }
        catch (IOException ex) {
            log.debug("Failed to create lock file  (" + lockFilePath + ")", (Throwable)ex);
        }
        stagingFileLock.setAcquired(acquired);
        stagingFileLock.setLockFile(lockFile);
        if (!acquired) {
            if (lockFile.exists()) {
                try {
                    String lockFileContents = FileUtils.readFileToString((File)lockFile, (String)"UTF8");
                    stagingFileLock.setLockFailureMessage("Lock file exists: " + lockFileContents);
                }
                catch (Exception ex) {
                    stagingFileLock.setLockFailureMessage("Lock file exists but could not read contents: " + ex.getMessage());
                    if (log.isDebugEnabled()) {
                        log.debug("Failed to read lock file contents (" + lockFilePath + ")", (Throwable)ex);
                    }
                }
            } else {
                stagingFileLock.setLockFailureMessage("Lock file does not exist, but could not be created. Check directory permissions.");
            }
        }
        return stagingFileLock;
    }

    @Override
    public File getStagingDirectory() {
        return this.directory;
    }
}

