/*-
 * Copyright (C) 2008 Erik Larsson
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

package org.catacombae.hfsexplorer.types.hfs;

import java.io.PrintStream;
import org.catacombae.csjc.structelements.ASCIIStringField;
import org.catacombae.csjc.structelements.Dictionary;
import org.catacombae.hfsexplorer.Util;

/**
 * A directory thread record in the HFS file system model.
 * 
 * This class was generated by CStructToJavaClass.
 */
public class CdrThdRec extends CatDataRec {
    /*
     * struct CdrThdRec
     * size: 46 bytes
     * description: 
     * 
     * BP  Size  Type       Identifier  Description                           
     * -----------------------------------------------------------------------
     * 0   1     SInt8      cdrType     record type (SignedByte)              
     * 1   1     SInt8      cdrResrv2   reserved (SignedByte)                 
     * 2   4*2   SInt32[2]  thdResrv    reserved (ARRAY[1..2] OF LongInt)     
     * 10  4     SInt32     thdParID    parent ID for this directory (LongInt)
     * 14  1     UInt8      thdCNameLen length of thdCName (part of Str31)
     * 15  1*31  Char[31]   thdCName    name of this directory (part of Str31)
     */
    
    public static final int STRUCTSIZE = 46;
    
    private final byte[] thdResrv = new byte[4*2];
    private final byte[] thdParID = new byte[4];
    private final byte[] thdCNameLen = new byte[1];
    private final byte[] thdCName = new byte[1*31];
    
    /**
     * Creates a new HFS folder thread record by reading from the given data at
     * the specified offset.
     * 
     * @param data
     * @param offset
     */
    public CdrThdRec(byte[] data, int offset) {
        super(data, offset);
        System.arraycopy(data, offset + 2, thdResrv, 0, 4 * 2);
        System.arraycopy(data, offset + 10, thdParID, 0, 4);
        System.arraycopy(data, offset + 14, thdCNameLen, 0, 1);
        System.arraycopy(data, offset + 15, thdCName, 0, 1 * 31);
    }

    public static int length() { return STRUCTSIZE; }
    
    /** reserved (ARRAY[1..2] OF LongInt) */
    public int[] getThdResrv() { return Util.readIntArrayBE(thdResrv); }
    /** parent ID for this directory (LongInt) */
    public int getThdParID() { return Util.readIntBE(thdParID); }
    /** length of thdCName (part of Str31) */
    public byte getThdCNameLen() { return Util.readByteBE(thdCNameLen); }
    /** name of this directory (part of Str31) */
    public byte[] getThdCName() { return Util.readByteArrayBE(thdCName); }
    
    @Override
    public void printFields(PrintStream ps, String prefix) {
        super.printFields(ps, prefix);
        ps.println(prefix + " thdResrv: " + getThdResrv());
        ps.println(prefix + " thdParID: " + Util.unsign(getThdParID()));
        ps.println(prefix + " thdCNameLen: " + Util.unsign(getThdCNameLen()));
        ps.println(prefix + " thdCName: \"" + Util.toASCIIString(getThdCName()) + "\"");
    }

    @Override
    public void print(PrintStream ps, String prefix) {
        ps.println(prefix + "CdrThdRec:");
        printFields(ps, prefix);
    }
    
    @Override
    public byte[] getBytes() {
        byte[] result = new byte[STRUCTSIZE];
        int offset = 0;
        
        byte[] superData = super.getBytes();
        System.arraycopy(superData, 0, result, offset, superData.length); offset += superData.length;
        System.arraycopy(thdResrv, 0, result, offset, thdResrv.length); offset += thdResrv.length;
        System.arraycopy(thdParID, 0, result, offset, thdParID.length); offset += thdParID.length;
        System.arraycopy(thdCNameLen, 0, result, offset, thdCNameLen.length); offset += thdCNameLen.length;
        System.arraycopy(thdCName, 0, result, offset, thdCName.length); offset += thdCName.length;
        return result;
    }
    
    @Override
    public int size() {
        return length();
    }
    
    @Override
    public Dictionary getStructElements() {
        DictionaryBuilder db = new DictionaryBuilder(CdrThdRec.class.getSimpleName());
        
        super.addSuperStructElements(db);
        db.addIntArray("thdResrv", thdResrv, BITS_32, UNSIGNED, BIG_ENDIAN, "Reserved", HEXADECIMAL);
        db.addUIntBE("thdParID", thdParID, "Parent ID");
        db.addUIntBE("thdCNameLen", thdCNameLen, "Length of record name");
        db.add("thdCName", new ASCIIStringField(thdCName), "Record name");
        
        return db.getResult();
    }
}
